/**
 * 2007, Digitalis Informatica. All rights reserved. Distribuicao e Gestao de Informatica, Lda. Estrada de Paco de Arcos
 * num.9 - Piso -1 2780-666 Paco de Arcos Telefone: (351) 21 4408990 Fax: (351) 21 4408999 http://www.digitalis.pt
 */

package pt.digitalis.dif.model.authorization;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.hibernate.Session;
import org.hibernate.Transaction;

import com.google.inject.Inject;

import pt.digitalis.dif.controller.security.managers.IIdentityManager;
import pt.digitalis.dif.controller.security.managers.impl.AuthorizationManagerStaticImpl;
import pt.digitalis.dif.controller.security.objects.ACLEntry;
import pt.digitalis.dif.dem.Entity;
import pt.digitalis.dif.dem.managers.IDEMManager;
import pt.digitalis.dif.dem.managers.impl.model.DIFRepositoryFactory;
import pt.digitalis.dif.dem.managers.impl.model.dao.IAclDAO;
import pt.digitalis.dif.dem.managers.impl.model.data.Acl;
import pt.digitalis.dif.dem.managers.impl.model.impl.AuthorizationServiceImpl;
import pt.digitalis.dif.exception.security.AuthorizationManagerException;
import pt.digitalis.dif.exception.security.IdentityManagerException;
import pt.digitalis.dif.model.dataset.DataSetException;
import pt.digitalis.dif.utils.logging.DIFLogger;

/**
 * Authorization Manager implementation with data base persistence.
 * 
 * @author Rodrigo Gonalves <a href="mailto:rgoncalves@digitalis.pt">rgoncalves@digitalis.pt</a><br/>
 * @author Joo Galaio <a href="mailto:jgalaio@digitalis.pt">jgalaio@digitalis.pt</a><br/>
 * @created 2008/03/19
 */
public class AuthorizationManagerDataBaseImpl extends AuthorizationManagerStaticImpl {

    /**
     * Used to control initialization
     */
    private static boolean ENTRIES_INITIALIZED = false;

    /**
     * Converts an ACLEntry object into an Acl.
     * 
     * @param entry
     *            the entry to convert
     * @param id
     *            the entry's id
     * @return the acl object
     */
    static private Acl convertACLEntryToAcl(ACLEntry entry, Long id)
    {
        Acl acl = new Acl();

        if (entry.getUserID() != null)
            acl.setUserId(entry.getUserID());

        if (entry.getGroupID() != null)
            acl.setGroupId(entry.getGroupID());

        if (entry.getEntityType() != null)
            acl.setEntityType(entry.getEntityType().toString());

        if (entry.getEntityID() != null)
            acl.setEntityId(entry.getEntityID());

        acl.setPublicAccess(entry.isPublicAccess() ? "S" : "N");
        acl.setIsEnabled(entry.isEnabled() ? "S" : "N");
        acl.setIsDefault(entry.isDefault() ? "S" : "N");

        if (id != null)
            acl.setId(id);

        return acl;
    }

    /**
     * Converts a list of Acl objects to a list of ACLEntry objects.
     * 
     * @param acl
     *            the original list
     * @return the converted list
     */
    static private List<ACLEntry> convertAclListToACLEntryList(List<Acl> acl)
    {
        List<ACLEntry> result = new ArrayList<ACLEntry>(acl.size());

        for (Acl instance: acl)
            result.add(convertAclToACLEntry(instance));

        return result;
    }

    /**
     * Converts an Acl object into an ACLEntry.
     * 
     * @param acl
     *            the instance to convert
     * @return the entry object
     */
    static private ACLEntry convertAclToACLEntry(Acl acl)
    {
        ACLEntry entry = new ACLEntry();

        // String userId, String groupId, String entityType, String entityId, Boolean publicAccess, Boolean isEnabled,
        // Boolean isDefault
        if (acl.getUserId() != null)
            entry.setUserID(acl.getUserId());

        if (acl.getGroupId() != null)
            entry.setGroupID(acl.getGroupId());

        if (acl.getEntityType() != null)
            entry.setEntityType(Entity.valueOf(acl.getEntityType()));

        if (acl.getEntityId() != null)
            entry.setEntityID(acl.getEntityId());

        if ("S".equals(acl.getPublicAccess()))
            entry.setPublicAccess();

        entry.setEnabled("S".equals(acl.getIsEnabled()));
        entry.setDefault("S".equals(acl.getIsDefault()));

        return entry;
    }

    /** The ACL DAO. */
    IAclDAO aclDao = new AuthorizationServiceImpl().getAclDAO();

    /**
     * Ctor.
     * 
     * @param identityManager
     *            the identity manager
     * @param demManager
     *            the DEM manager
     */
    @Inject
    public AuthorizationManagerDataBaseImpl(IIdentityManager identityManager, IDEMManager demManager)
    {
        super(identityManager, demManager);
    }

    /**
     * @see pt.digitalis.dif.controller.security.managers.impl.AbstractAuthorizationManagerImpl#addACLEntryToGroup(pt.digitalis.dif.controller.security.objects.ACLEntry,
     *      boolean)
     */
    @Override
    protected boolean addACLEntryToGroup(ACLEntry entry, boolean overrideIfExistant)
            throws AuthorizationManagerException
    {
        Session session = DIFRepositoryFactory.getSession();
        session.beginTransaction();

        try
        {
            Acl acl = aclDao.getAclByGroup(entry.getGroupID(),
                    entry.getEntityType() != null ? entry.getEntityType().toString() : null, entry.getEntityID());

            if (acl == null)
            {
                entry.setEnabled(true);
                persistACLEntry(entry, null);
            }
            else
            {
                // Override => Turn enabled ON in the database so that this access is active from now on
                if (overrideIfExistant)
                {
                    acl.setIsEnabled("S");
                    aclDao.merge(acl);
                }
                else
                {
                    // If the ACL was not enabled, since no override was asked we must not add this entry. Previous
                    // DISABLED ACL represent no ACL in the Manager cache
                    if (!"S".equals(acl.getIsEnabled()))
                    {
                        entry.setEnabled(false);
                    }
                }
            }

            session.getTransaction().commit();
        }
        catch (DataSetException e)
        {
            session.getTransaction().rollback();
            throw new AuthorizationManagerException(e);
        }

        return super.addACLEntryToGroup(entry, true);
    }

    /**
     * @see pt.digitalis.dif.controller.security.managers.impl.AbstractAuthorizationManagerImpl#addACLEntryToPublic(pt.digitalis.dif.controller.security.objects.ACLEntry)
     */
    @Override
    protected boolean addACLEntryToPublic(ACLEntry entry) throws AuthorizationManagerException
    {
        Session session = DIFRepositoryFactory.getSession();
        session.beginTransaction();

        try
        {
            Acl acl = aclDao.getAclByPublic(entry.getEntityType() != null ? entry.getEntityType().toString() : null,
                    entry.getEntityID());

            entry = this.processAcl(entry, acl);

            session.getTransaction().commit();
        }
        catch (DataSetException e)
        {
            session.getTransaction().rollback();
            throw new AuthorizationManagerException(e);
        }

        return super.addACLEntryToPublic(entry);
    }

    /**
     * @see pt.digitalis.dif.controller.security.managers.impl.AbstractAuthorizationManagerImpl#addACLEntryToUser(pt.digitalis.dif.controller.security.objects.ACLEntry)
     */
    @Override
    protected boolean addACLEntryToUser(ACLEntry entry) throws AuthorizationManagerException
    {
        boolean result = false;
        try
        {
            if (identityManager.userExists(entry.getUserID()))
            {
                Session session = DIFRepositoryFactory.getSession();
                session.beginTransaction();

                try
                {
                    Acl acl = aclDao.getAclByUser(entry.getUserID(),
                            entry.getEntityType() != null ? entry.getEntityType().toString() : null,
                            entry.getEntityID());

                    entry = this.processAcl(entry, acl);

                    session.getTransaction().commit();
                    result = true;

                }
                catch (DataSetException e)
                {
                    session.getTransaction().rollback();
                    throw new AuthorizationManagerException(e);
                }
            }
            else
                result = false;

        }
        catch (IdentityManagerException identityManagerException)
        {
            throw new AuthorizationManagerException("Could not access the identity manager to verify user existance!",
                    identityManagerException);
        }

        return result && super.addACLEntryToUser(entry);
    }

    /**
     * @see pt.digitalis.dif.controller.security.managers.impl.AbstractAuthorizationManagerImpl#deleteGroup(java.lang.String,
     *      pt.digitalis.dif.controller.security.objects.ACLEntry)
     */
    @Override
    protected boolean deleteGroup(String groupID, ACLEntry entry)
    {
        Transaction transaction = DIFRepositoryFactory.getSession().beginTransaction();

        try
        {
            aclDao.deleteGroupAccess(groupID, entry.getEntityType().toString(), entry.getEntityID());
        }
        catch (DataSetException e)
        {
            e.printStackTrace();
        }

        transaction.commit();
        return super.deleteGroup(groupID, entry);
    }

    /**
     * @see pt.digitalis.dif.controller.security.managers.impl.AbstractAuthorizationManagerImpl#disableGroup(java.lang.String,
     *      pt.digitalis.dif.controller.security.objects.ACLEntry,
     *      pt.digitalis.dif.controller.security.objects.ACLEntry)
     */
    @Override
    protected boolean disableGroup(String groupID, ACLEntry existentEntry, ACLEntry entryToDisable)
    {
        Session session = DIFRepositoryFactory.getSession();
        session.beginTransaction();
        try
        {
            Acl acl = aclDao.getAclByGroup(entryToDisable.getGroupID(),
                    entryToDisable.getEntityType() != null ? entryToDisable.getEntityType().toString() : null,
                    entryToDisable.getEntityID());
            acl.setIsEnabled("N");
            aclDao.merge(acl);

            session.getTransaction().commit();
        }
        catch (DataSetException e)
        {
            session.getTransaction().rollback();
        }

        return super.disableGroup(groupID, existentEntry, entryToDisable);
    }

    /**
     * Inspector for the 'groupAccessControlList' attribute.
     * 
     * @return the groupAccessControlList value
     */
    @Override
    public Map<String, Set<ACLEntry>> getGroupAccessControlList()
    {
        this.loadPersistedEntries();
        return super.getGroupAccessControlList();
    }

    /**
     * Inspector for the 'publicAccessControlList' attribute.
     * 
     * @return the publicAccessControlList value
     */
    @Override
    public Map<String, ACLEntry> getPublicAccessControlList()
    {
        this.loadPersistedEntries();
        return super.getPublicAccessControlList();
    }

    /**
     * Inspector for the 'userAccessControlList' attribute.
     * 
     * @return the userAccessControlList value
     */
    @Override
    public Map<String, Set<ACLEntry>> getUserAccessControlList()
    {
        this.loadPersistedEntries();
        return super.getUserAccessControlList();
    }

    /**
     * Load persisted entries from Database
     */
    synchronized private void loadPersistedEntries()
    {
        if (!ENTRIES_INITIALIZED)
        {
            DIFLogger.getLogger().debug("[AuthorizationManager] Start loading persistent Entries from Database...");
            ENTRIES_INITIALIZED = true;
            Session session = DIFRepositoryFactory.getSession();
            session.beginTransaction();

            List<ACLEntry> aclEntryList = convertAclListToACLEntryList(this.aclDao.findAll());

            try
            {
                for (ACLEntry entry: aclEntryList)
                {
                    if (entry.isPublicAccess())
                    {
                        super.addACLEntryToPublic(entry);
                    }
                    else if (entry.getUserID() != null)
                    {
                        super.addACLEntryToUser(entry);
                    }
                    else if (entry.getGroupID() != null)
                    {
                        super.addACLEntryToGroup(entry, true);
                    }
                }
                session.getTransaction().commit();

            }
            catch (Exception e)
            {
                ENTRIES_INITIALIZED = false;
                session.getTransaction().rollback();
                DIFLogger.getLogger()
                        .info("AuthorizationManager - Failed loading persistent Entries from Database... Reason: "
                                + e.getMessage());
            }
            DIFLogger.getLogger().info("[AuthorizationManager] Finished loading persistent Entries from Database...");
        }
    }

    /**
     * Persists an entry on the persistence layer.
     * 
     * @param entry
     *            the entry to persist
     * @param id
     *            the id
     */
    private void persistACLEntry(ACLEntry entry, Long id)
    {
        Acl acl = convertACLEntryToAcl(entry, id);
        aclDao.persist(acl);
    }

    /**
     * Persist the object if not persisted. Otherwise the entry already persisted, so must be updated. <br />
     * Note: ACLEntry's cannot be updated
     * 
     * @param entry
     *            the {@link ACLEntry}
     * @param acl
     *            the {@link Acl}
     * @return {@link ACLEntry}
     */
    private ACLEntry processAcl(ACLEntry entry, Acl acl)
    {
        if (acl == null)
            persistACLEntry(entry, null);
        else if (!entry.isDefault())
        {
            updateACLEntry(entry, acl.getId());
        }
        else
            entry = convertAclToACLEntry(acl);
        return entry;
    }

    /**
     * @see pt.digitalis.dif.controller.security.managers.IAuthorizationManager#revokeAccessFromPublic(pt.digitalis.dif.dem.Entity,
     *      java.lang.String)
     */
    @Override
    public boolean revokeAccessFromPublic(Entity entityType, String entityID)
    {

        Transaction transaction = DIFRepositoryFactory.getSession().beginTransaction();

        boolean status = aclDao.deletePublicAccess(entityType.toString(), entityID);

        transaction.commit();

        return status && super.revokeAccessFromPublic(entityType, entityID);
    }

    /**
     * @see pt.digitalis.dif.controller.security.managers.IAuthorizationManager#revokeAccessFromUser(java.lang.String,
     *      pt.digitalis.dif.dem.Entity, java.lang.String)
     */
    @Override
    public boolean revokeAccessFromUser(String userID, Entity entityType, String entityID)
    {
        Transaction transaction = DIFRepositoryFactory.getSession().beginTransaction();

        boolean status = false;
        try
        {
            status = aclDao.deleteUserAccess(userID, entityType.toString(), entityID);
        }
        catch (DataSetException e)
        {
            e.printStackTrace();
        }

        transaction.commit();

        return status && super.revokeAccessFromUser(userID, entityType, entityID);
    }

    /**
     * @see pt.digitalis.dif.controller.security.managers.IAuthorizationManager#revokeAllAccessFromGroup(java.lang.String)
     */
    @Override
    public boolean revokeAllAccessFromGroup(String groupID)
    {

        Transaction transaction = DIFRepositoryFactory.getSession().beginTransaction();

        boolean status = aclDao.deleteAllGroupAccess(groupID);

        transaction.commit();

        return status && super.revokeAllAccessFromGroup(groupID);
    }

    /**
     * @see pt.digitalis.dif.controller.security.managers.IAuthorizationManager#revokeAllAccessFromUser(java.lang.String)
     */
    @Override
    public boolean revokeAllAccessFromUser(String userID)
    {
        Transaction transaction = DIFRepositoryFactory.getSession().beginTransaction();

        boolean status = aclDao.deleteAllUserAccess(userID);

        transaction.commit();

        return status && super.revokeAllAccessFromUser(userID);
    }

    /**
     * Update an existing entry on the persistence layer.
     * 
     * @param entry
     *            the entry to update
     * @param id
     *            the entry's id
     */
    private void updateACLEntry(ACLEntry entry, Long id)
    {
        Acl acl = convertACLEntryToAcl(entry, id);
        aclDao.merge(acl);
    }

}
