/**
 * 2010, Digitalis Informatica. All rights reserved. Distribuicao e Gestao de Informatica, Lda. Estrada de Paco de Arcos
 * num.9 - Piso -1 2780-666 Paco de Arcos Telefone: (351) 21 4408990 Fax: (351) 21 4408999 http://www.digitalis.pt
 */

package pt.digitalis.dif.content;

import java.io.IOException;
import java.io.InputStream;
import java.math.BigDecimal;
import java.sql.Blob;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import org.apache.commons.io.IOUtils;
import org.hibernate.Hibernate;
import org.hibernate.Query;
import org.hibernate.Session;
import org.hibernate.exception.ConstraintViolationException;

import pt.digitalis.dif.content.model.IDocumentsService;
import pt.digitalis.dif.content.model.data.ContentItemAcl;
import pt.digitalis.dif.content.model.impl.DocumentsServiceImpl;
import pt.digitalis.dif.controller.security.objects.IDIFUser;
import pt.digitalis.dif.model.dataset.DataSetException;
import pt.digitalis.dif.model.dataset.Filter;
import pt.digitalis.dif.model.dataset.FilterType;
import pt.digitalis.dif.utils.extensions.cms.ACLEntry;
import pt.digitalis.dif.utils.extensions.cms.AbstractContentManager;
import pt.digitalis.dif.utils.extensions.cms.ContentItem;
import pt.digitalis.dif.utils.extensions.cms.Node;
import pt.digitalis.dif.utils.extensions.cms.exception.ContentItemNotFoundException;
import pt.digitalis.dif.utils.extensions.cms.exception.ContentItemWithDuplicateNameAndParentNodeException;
import pt.digitalis.dif.utils.extensions.cms.exception.ContentManagerException;
import pt.digitalis.dif.utils.extensions.cms.exception.InvalidParentNodeException;
import pt.digitalis.dif.utils.extensions.cms.exception.NodeNotFoundException;
import pt.digitalis.dif.utils.extensions.cms.exception.NodeWithDuplicatePathException;
import pt.digitalis.dif.utils.extensions.cms.exception.NodeWithNodesException;
import pt.digitalis.utils.common.CollectionUtils;
import pt.digitalis.utils.common.StringUtils;

/**
 * @author Pedro Viegas <a href="mailto:pviegas@digitalis.pt">pviegas@digitalis.pt</a><br/>
 * @created 2010/10/06
 */
public class ContentManagerDBImpl extends AbstractContentManager {

    /** */
    private IDocumentsService documentsService = null;

    /**
     * @see pt.digitalis.dif.utils.extensions.cms.IContentManager#beginTransaction()
     */
    public void beginTransaction()
    {
        getSession().beginTransaction();
    }

    /**
     * @see pt.digitalis.dif.utils.extensions.cms.IContentManager#commitTransaction()
     */
    public void commitTransaction()
    {
        getSession().getTransaction().commit();
    }

    /**
     * Converts a content item list from DB data object to DIF form
     * 
     * @param contentItemList
     *            content item list to convert
     * @param user
     *            current user
     * @return the content item list converted
     */
    private List<ContentItem> convertContentItemListToDIFObject(
            List<pt.digitalis.dif.content.model.data.ContentItem> contentItemList, IDIFUser user)
    {
        List<ContentItem> contentItemConvertedList = new ArrayList<ContentItem>();

        if (contentItemList == null)
            return null;

        for (pt.digitalis.dif.content.model.data.ContentItem contentItem: contentItemList)
            contentItemConvertedList.add(convertContentItemToDIFObject(contentItem, user));

        return contentItemConvertedList;
    }

    /**
     * Converts a content item from DIF form to DB data object
     * 
     * @param contentItem
     *            content item to convert
     * @return the content item converted
     * @throws ContentManagerException
     */
    private pt.digitalis.dif.content.model.data.ContentItem convertContentItemToDBObject(ContentItem contentItem)
            throws ContentManagerException
    {
        pt.digitalis.dif.content.model.data.Node node;
        try
        {
            node = getDocumentsService().getNodeDataSet().get(contentItem.getParentNodeId().toString());
        }
        catch (DataSetException e)
        {
            throw new ContentManagerException(e);
        }

        pt.digitalis.dif.content.model.data.ContentItem convertedContentItem = new pt.digitalis.dif.content.model.data.ContentItem(
                node, contentItem.getName(), contentItem.getCreationDate(), contentItem.getCreatorID());

        Blob blob;
        String contentToCreate = contentItem.getContent();
        if (contentToCreate == null)
        {
            contentToCreate = "";
        }
        blob = Hibernate.createBlob(contentToCreate.getBytes());
        convertedContentItem.setContent(blob);

        convertedContentItem.setDescription(contentItem.getDescription());
        convertedContentItem.setId(contentItem.getId());

        return convertedContentItem;
    }

    /**
     * Converts a content item from DB data object to DIF form
     * 
     * @param contentItem
     *            content item to convert
     * @param user
     *            currentUser
     * @return the content item converted
     */
    private ContentItem convertContentItemToDIFObject(pt.digitalis.dif.content.model.data.ContentItem contentItem,
            IDIFUser user)
    {
        ContentItem convertedContentItem = new ContentItem(contentItem.getNode().getId(), contentItem.getName(),
                contentItem.getCreatorId(), user);

        convertedContentItem.setId(contentItem.getId());
        if (contentItem.getContent() != null)
        {
            try
            {
                InputStream inputStream = contentItem.getContent().getBinaryStream();
                convertedContentItem.setContent(IOUtils.toString(inputStream));
            }
            catch (SQLException e)
            {
                convertedContentItem.setContent("");
                e.printStackTrace();
            }
            catch (IOException e)
            {
                convertedContentItem.setContent("");
                e.printStackTrace();
            }
        }
        else
        {
            convertedContentItem.setContent("");
        }
        convertedContentItem.setCreationDate(contentItem.getCreationDate());
        convertedContentItem.setDescription(contentItem.getDescription());

        return convertedContentItem;
    }

    /**
     * Converts a List<Long> object to a string separated by commas
     * 
     * @param contentList
     *            a ContentItem List to convert
     * @return the list returned (null if List is null or empty
     */
    private String convertContentListToString(List<pt.digitalis.dif.content.model.data.ContentItem> contentList)
    {

        if (contentList == null || contentList.size() == 0)
            return null;

        StringBuffer toRetrun = new StringBuffer();

        for (pt.digitalis.dif.content.model.data.ContentItem item: contentList)
        {
            toRetrun.append(",'" + item.getId() + "'");
        }

        return toRetrun.substring(1);
    }

    /**
     * ContentItem ACL list from DB to DIF format
     * 
     * @param implList
     *            ContentItem ACL list to convert
     * @return the ACL list converted
     */
    private List<ACLEntry> convertFromContentItemAclToDIFAcl(
            List<pt.digitalis.dif.content.model.data.ContentItemAcl> implList)
    {
        List<ACLEntry> convertedList = new ArrayList<ACLEntry>();

        for (pt.digitalis.dif.content.model.data.ContentItemAcl contentItemAcl: implList)
        {
            ACLEntry entry = new ACLEntry();
            entry.setGroupID(contentItemAcl.getGroupId());
            entry.setUserID(contentItemAcl.getUserId());
            convertedList.add(entry);
        }

        return convertedList;
    }

    /**
     * Converts a List<Lonverts Converts node ACL list from DB to DIF format
     * 
     * @param implList
     *            node ACL list to convert
     * @return the ACL list converted
     */
    private List<ACLEntry> convertFromNodeAclToDIFAcl(List<pt.digitalis.dif.content.model.data.NodeAcl> implList)
    {
        List<ACLEntry> convertedList = new ArrayList<ACLEntry>();

        for (pt.digitalis.dif.content.model.data.NodeAcl nodeAcl: implList)
        {
            ACLEntry entry = new ACLEntry();
            entry.setGroupID(nodeAcl.getGroupId());
            entry.setUserID(nodeAcl.getUserId());
            convertedList.add(entry);
        }

        return convertedList;
    }

    /**
     * Converts a List<Long> object to a string separated by commas
     * 
     * @param decimalList
     *            a Long List to convert
     * @return the list returned (null if List is null or empty
     */
    private String convertIdListToString(List<BigDecimal> decimalList)
    {

        if (decimalList == null || decimalList.size() == 0)
            return null;

        StringBuffer buffer = new StringBuffer();

        for (BigDecimal decimal: decimalList)
        {
            buffer.append("," + decimal);
        }

        return buffer.substring(1);
    }

    /**
     * Converts a node list from DB to DIF
     * 
     * @param nodeList
     *            node list to convert
     * @param user
     *            current user
     * @return the node list converted
     */
    private List<Node> convertNodeListToDIFObjects(List<pt.digitalis.dif.content.model.data.Node> nodeList,
            IDIFUser user)
    {
        List<Node> nodeConvertedList = new ArrayList<Node>();

        if (nodeList != null)
        {
            for (pt.digitalis.dif.content.model.data.Node node: nodeList)
                nodeConvertedList.add(convertNodeToDIFObject(node, user));
        }

        return nodeConvertedList;
    }

    /**
     * Converts a node from DIF form to DB data object
     * 
     * @param node
     *            node to convert
     * @return the node converted
     */
    private pt.digitalis.dif.content.model.data.Node convertNodeToDBObject(Node node)
    {
        pt.digitalis.dif.content.model.data.Node convertedNode = new pt.digitalis.dif.content.model.data.Node(
                node.getName(), node.getCreationDate(), node.getCreatorID(), node.isPublic() ? 'S' : 'N');

        convertedNode.setId(node.getId());
        convertedNode.setDescription(node.getDescription());
        convertedNode.setFullPath(node.getFullPathName());
        convertedNode.setParentNodeId(node.getParentNodeId());

        return convertedNode;
    }

    /**
     * Converts a node from DB to DIF
     * 
     * @param node
     *            node to convert
     * @param user
     *            user instantiating the Node
     * @return the node converted
     */
    private Node convertNodeToDIFObject(pt.digitalis.dif.content.model.data.Node node, IDIFUser user)
    {

        Node convertedNode = new Node(node.getParentNodeId(), node.getName(), node.getCreatorId(), user);

        convertedNode.setCreationDate(node.getCreationDate());
        convertedNode.setDescription(node.getDescription());
        convertedNode.setFullPathName(node.getFullPath());
        convertedNode.setId(node.getId());
        convertedNode.setPublic(node.getIsPublic() == 'S' ? true : false);

        return convertedNode;
    }

    /**
     * @see pt.digitalis.dif.utils.extensions.cms.AbstractContentManager#deleteContentInRepository(java.lang.String,
     *      pt.digitalis.dif.controller.security.objects.IDIFUser)
     */
    @Override
    protected boolean deleteContentInRepository(String contentId, IDIFUser user) throws ContentManagerException
    {
        // delete node
        try
        {
            getDocumentsService().getContentItemDataSet().delete(contentId.toString());

            return true;
        }
        catch (DataSetException e)
        {
            throw new ContentManagerException(e);
        }

    }

    /**
     * @see pt.digitalis.dif.utils.extensions.cms.AbstractContentManager#deleteNodeInRepository(java.lang.Long,
     *      pt.digitalis.dif.controller.security.objects.IDIFUser, boolean)
     */
    @Override
    @SuppressWarnings("unchecked")
    protected boolean deleteNodeInRepository(Long nodeId, IDIFUser user, boolean cascadeDelete)
            throws ContentManagerException
    {
        boolean result = false;
        /* tem de ter controlo transacional devido a eliminao hierarquica de nodes */
        Session session = getDocumentsService().getNodeDAO().getSession();
        boolean doCommit = false;
        boolean wasActive = session.getTransaction().isActive();
        if (!wasActive)
        {
            session.beginTransaction();
        }

        try
        {

            pt.digitalis.dif.content.model.data.Node node = getDocumentsService().getNodeDataSet().get(
                    nodeId.toString());

            if (!cascadeDelete)
            {
                List<Node> childNodes = getNodesByParentNodeNoPermissions(node.getId(), user);

                if (childNodes.size() > 0)
                    throw new NodeWithNodesException(node.getId());

                result = deleteNodeInRepository(node);
            }
            else
            {
                // get all node IDs to delete
                Query query = getSession().createSQLQuery(
                        " SELECT id " + " FROM cms.node " + " START WITH id = ? "
                                + " CONNECT BY PRIOR id = parent_node_id");

                query.setLong(0, node.getId());
                List<BigDecimal> nodeIdList = query.list();

                // nothing to delete...
                if (nodeIdList.size() == 0)
                    return true;

                String nodeListStr = convertIdListToString(nodeIdList);

                // get all content ids to delete
                pt.digitalis.dif.model.dataset.Query<pt.digitalis.dif.content.model.data.ContentItem> queryContents = getDocumentsService()
                        .getContentItemDataSet().query();
                queryContents.addFilter(new Filter(StringUtils
                        .toLowerFirstChar(pt.digitalis.dif.content.model.data.Node.class.getSimpleName() + ".id"),
                        FilterType.IN, CollectionUtils.listToCommaSeparatedString(nodeIdList)));

                String contentListStr = convertContentListToString(queryContents.asList());

                if (contentListStr != null)
                {
                    // delete all content acls
                    getSession().createQuery(
                            "DELETE FROM "
                                    + pt.digitalis.dif.content.model.data.ContentItemAcl.class.getSimpleName()
                                    + " WHERE "
                                    + StringUtils
                                            .toLowerFirstChar(pt.digitalis.dif.content.model.data.ContentItem.class
                                                    .getSimpleName()) + " in (" + contentListStr + ")").executeUpdate();

                    // delete all contents
                    getSession().createQuery(
                            "DELETE FROM " + pt.digitalis.dif.content.model.data.ContentItem.class.getSimpleName()
                                    + " WHERE " + pt.digitalis.dif.content.model.data.ContentItem.Fields.ID + " in ("
                                    + contentListStr + ")").executeUpdate();
                }

                // delete all node acls
                getSession().createQuery(
                        "DELETE FROM "
                                + pt.digitalis.dif.content.model.data.NodeAcl.class.getSimpleName()
                                + " WHERE "
                                + StringUtils.toLowerFirstChar(pt.digitalis.dif.content.model.data.Node.class
                                        .getSimpleName()) + " in (" + nodeListStr + ")").executeUpdate();

                // delete all nodes
                getSession().createQuery(
                        "DELETE FROM " + pt.digitalis.dif.content.model.data.Node.class.getSimpleName() + " WHERE "
                                + pt.digitalis.dif.content.model.data.Node.Fields.ID + " in (" + nodeListStr + ")")
                        .executeUpdate();

            }
            doCommit = true;
            result = true;
        }

        catch (DataSetException e)
        {
            throw new ContentManagerException(e);
        }
        finally
        {
            if (!wasActive)
            {
                if (doCommit)
                {
                    session.getTransaction().commit();
                }
                else
                {
                    session.getTransaction().rollback();
                }

            }
        }
        return result;

    }

    /**
     * Deletes an existing node from the repository
     * 
     * @param node
     *            the node to delete
     * @return T if all went well
     * @throws ContentManagerException
     */
    private boolean deleteNodeInRepository(pt.digitalis.dif.content.model.data.Node node)
            throws ContentManagerException
    {
        try
        {
            // delete node
            getDocumentsService().getNodeDataSet().delete(node.getId().toString());

            return true;
        }
        catch (DataSetException e)
        {
            throw new ContentManagerException(e);
        }
    }

    /**
     * @see pt.digitalis.dif.utils.extensions.cms.AbstractContentManager#getContentByDescriptionInRepository(java.lang.String,
     *      pt.digitalis.dif.controller.security.objects.IDIFUser)
     */
    @Override
    protected List<ContentItem> getContentByDescriptionInRepository(String description, IDIFUser user)
            throws ContentManagerException
    {
        try
        {
            pt.digitalis.dif.model.dataset.Query<pt.digitalis.dif.content.model.data.ContentItem> query = getDocumentsService()
                    .getContentItemDataSet().query();
            query.addFilter(new Filter(pt.digitalis.dif.content.model.data.ContentItem.Fields.DESCRIPTION.toString(),
                    FilterType.EQUALS, description));

            return convertContentItemListToDIFObject(query.asList(), user);
        }
        catch (DataSetException e)
        {
            throw new ContentManagerException(e);
        }
    }

    /**
     * @see pt.digitalis.dif.utils.extensions.cms.AbstractContentManager#getContentByIdInRepository(java.lang.String,
     *      pt.digitalis.dif.controller.security.objects.IDIFUser)
     */
    @Override
    protected ContentItem getContentByIdInRepository(String id, IDIFUser user) throws ContentManagerException
    {
        try
        {
            pt.digitalis.dif.content.model.data.ContentItem contentItemDB = getDocumentsService()
                    .getContentItemDataSet().get(id.toString());

            if (contentItemDB == null)
                throw new ContentItemNotFoundException(id);

            return convertContentItemToDIFObject(contentItemDB, user);
        }
        catch (DataSetException e)
        {
            throw new ContentManagerException(e);
        }
    }

    /**
     * @see pt.digitalis.dif.utils.extensions.cms.AbstractContentManager#getContentByIdNoPermissionsInRepository(java.lang.String,
     *      pt.digitalis.dif.controller.security.objects.IDIFUser)
     */
    @Override
    protected ContentItem getContentByIdNoPermissionsInRepository(String id, IDIFUser user)
            throws ContentManagerException
    {
        try
        {
            pt.digitalis.dif.content.model.data.ContentItem contentItem = getDocumentsService().getContentItemDataSet()
                    .get(id.toString());

            if (contentItem == null)
                throw new ContentItemNotFoundException(id);

            return convertContentItemToDIFObject(contentItem, user);
        }
        catch (DataSetException e)
        {
            throw new ContentManagerException(e);
        }
    }

    /**
     * @see pt.digitalis.dif.utils.extensions.cms.AbstractContentManager#getContentByNameInRepository(java.lang.String,
     *      pt.digitalis.dif.controller.security.objects.IDIFUser)
     */
    @Override
    protected List<ContentItem> getContentByNameInRepository(String name, IDIFUser user) throws ContentManagerException
    {
        try
        {
            pt.digitalis.dif.model.dataset.Query<pt.digitalis.dif.content.model.data.ContentItem> query = getDocumentsService()
                    .getContentItemDataSet().query();
            query.addFilter(new Filter(pt.digitalis.dif.content.model.data.ContentItem.Fields.NAME.toString(),
                    FilterType.EQUALS, name));

            return convertContentItemListToDIFObject(query.asList(), user);
        }
        catch (DataSetException e)
        {
            throw new ContentManagerException(e);
        }
    }

    /**
     * @see pt.digitalis.dif.utils.extensions.cms.AbstractContentManager#getContentByParentNodeInRepository(java.lang.Long,
     *      pt.digitalis.dif.controller.security.objects.IDIFUser)
     */
    @Override
    protected List<ContentItem> getContentByParentNodeInRepository(Long nodeId, IDIFUser user)
            throws ContentManagerException
    {
        try
        {
            pt.digitalis.dif.model.dataset.Query<pt.digitalis.dif.content.model.data.ContentItem> query = getDocumentsService()
                    .getContentItemDataSet().query();
            query.addFilter(new Filter(StringUtils.toLowerFirstChar(pt.digitalis.dif.content.model.data.Node.class
                    .getSimpleName() + ".id"), FilterType.EQUALS, nodeId.toString()));

            return convertContentItemListToDIFObject(query.asList(), user);
        }
        catch (DataSetException e)
        {
            throw new ContentManagerException(e);
        }
    }

    /**
     * Searches content items by parent node. Permission checks are ignored.
     * 
     * @param nodeId
     *            the parent id
     * @param user
     *            current user
     * @return the content list
     * @throws ContentManagerException
     */
    protected List<ContentItem> getContentByParentNodeNoPermissions(Long nodeId, IDIFUser user)
            throws ContentManagerException
    {
        getNodeByIdNoPermissions(nodeId, user);

        pt.digitalis.dif.model.dataset.Query<pt.digitalis.dif.content.model.data.ContentItem> query = getDocumentsService()
                .getContentItemDataSet().query();
        try
        {
            query.addFilter(new Filter(StringUtils.toLowerFirstChar(pt.digitalis.dif.content.model.data.Node.class

            .getSimpleName() + ".id"), FilterType.EQUALS, nodeId.toString()));
            return convertContentItemListToDIFObject(query.asList(), user);
        }
        catch (DataSetException e)
        {
            throw new ContentManagerException(e);
        }

    }

    /**
     * @see pt.digitalis.dif.utils.extensions.cms.AbstractContentManager#getContentFromNodePathByNameRepository(java.lang.String,
     *      java.lang.String, pt.digitalis.dif.controller.security.objects.IDIFUser)
     */
    @Override
    protected ContentItem getContentFromNodePathByNameRepository(String nodeFullPath, String name, IDIFUser user)
            throws ContentManagerException
    {
        try
        {
            pt.digitalis.dif.model.dataset.Query<pt.digitalis.dif.content.model.data.ContentItem> query = getDocumentsService()
                    .getContentItemDataSet().query();
            query.addFilter(new Filter(pt.digitalis.dif.content.model.data.ContentItem.Fields.NAME.toString(),
                    FilterType.EQUALS, name));
            query.addFilter(new Filter(pt.digitalis.dif.content.model.data.Node.Fields.FULLPATH.toString(),
                    FilterType.EQUALS, nodeFullPath));

            return convertContentItemToDIFObject(query.singleValue(), user);
        }
        catch (DataSetException e)
        {
            throw new ContentManagerException(e);
        }

    }

    /**
     * @see pt.digitalis.dif.utils.extensions.cms.AbstractContentManager#getContentItemACLInRepository(java.lang.String)
     */
    @Override
    protected List<ACLEntry> getContentItemACLInRepository(String contentId) throws ContentManagerException
    {
        try
        {
            pt.digitalis.dif.model.dataset.Query<pt.digitalis.dif.content.model.data.ContentItemAcl> query = getDocumentsService()
                    .getContentItemAclDataSet().query();
            query.addFilter(new Filter(StringUtils
                    .toLowerFirstChar(pt.digitalis.dif.content.model.data.ContentItem.class.getSimpleName() + ".id"),
                    FilterType.EQUALS, contentId.toString()));

            return convertFromContentItemAclToDIFAcl(query.asList());
        }
        catch (DataSetException e)
        {
            throw new ContentManagerException(e);
        }
    }

    /**
     * Get documentsService
     * 
     * @return service
     */
    private IDocumentsService getDocumentsService()
    {
        if (documentsService == null)
            documentsService = new DocumentsServiceImpl();

        return documentsService;
    }

    /**
     * @see pt.digitalis.dif.utils.extensions.cms.AbstractContentManager#getNodeACLInRepository(java.lang.Long)
     */
    @Override
    protected List<ACLEntry> getNodeACLInRepository(Long nodeId) throws ContentManagerException
    {
        try
        {
            pt.digitalis.dif.model.dataset.Query<pt.digitalis.dif.content.model.data.NodeAcl> query = getDocumentsService()
                    .getNodeAclDataSet().query();
            query.addFilter(new Filter(StringUtils.toLowerFirstChar(pt.digitalis.dif.content.model.data.Node.class
                    .getSimpleName() + ".id"), FilterType.EQUALS, nodeId.toString()));

            return convertFromNodeAclToDIFAcl(query.asList());
        }
        catch (DataSetException e)
        {
            throw new ContentManagerException(e);
        }
    }

    /**
     * @see pt.digitalis.dif.utils.extensions.cms.AbstractContentManager#getNodeByIdInRepository(java.lang.Long,
     *      pt.digitalis.dif.controller.security.objects.IDIFUser)
     */
    @Override
    protected Node getNodeByIdInRepository(Long id, IDIFUser user) throws ContentManagerException
    {
        try
        {
            pt.digitalis.dif.content.model.data.Node node = getDocumentsService().getNodeDataSet().get(id.toString());

            if (node == null)
                throw new NodeNotFoundException(id);

            return convertNodeToDIFObject(node, user);
        }
        catch (DataSetException e)
        {
            throw new ContentManagerException(e);
        }
    }

    /**
     * @see pt.digitalis.dif.utils.extensions.cms.AbstractContentManager#getNodeByIdNoPermissionsInRepository(java.lang.Long,
     *      pt.digitalis.dif.controller.security.objects.IDIFUser)
     */
    @Override
    protected Node getNodeByIdNoPermissionsInRepository(Long id, IDIFUser user) throws ContentManagerException
    {
        try
        {
            pt.digitalis.dif.content.model.data.Node node = getDocumentsService().getNodeDataSet().get(id.toString());

            if (node == null)
                throw new NodeNotFoundException(id);

            return convertNodeToDIFObject(node, user);
        }
        catch (DataSetException e)
        {
            throw new ContentManagerException(e);
        }
    }

    /**
     * @see pt.digitalis.dif.utils.extensions.cms.AbstractContentManager#getNodeByPathInRespository(java.lang.String,
     *      pt.digitalis.dif.controller.security.objects.IDIFUser)
     */
    @Override
    protected Node getNodeByPathInRespository(String fullPath, IDIFUser user) throws ContentManagerException
    {
        try
        {
            pt.digitalis.dif.model.dataset.Query<pt.digitalis.dif.content.model.data.Node> query = getDocumentsService()
                    .getNodeDataSet().query();
            query.addFilter(new Filter(pt.digitalis.dif.content.model.data.Node.Fields.FULLPATH.toString(),
                    FilterType.EQUALS, fullPath));

            pt.digitalis.dif.content.model.data.Node node = query.singleValue();

            if (node == null)
                throw new NodeNotFoundException("Node not found with path: " + fullPath);

            return convertNodeToDIFObject(node, user);
        }
        catch (DataSetException e)
        {
            throw new ContentManagerException(e);
        }
    }

    /**
     * @see pt.digitalis.dif.utils.extensions.cms.AbstractContentManager#getNodesByDescriptionInRepository(java.lang.String,
     *      pt.digitalis.dif.controller.security.objects.IDIFUser)
     */
    @Override
    protected List<Node> getNodesByDescriptionInRepository(String description, IDIFUser user)
            throws ContentManagerException
    {
        try
        {
            pt.digitalis.dif.model.dataset.Query<pt.digitalis.dif.content.model.data.Node> query = getDocumentsService()
                    .getNodeDataSet().query();
            query.addFilter(new Filter(pt.digitalis.dif.content.model.data.Node.Fields.DESCRIPTION.toString(),
                    FilterType.EQUALS, description));

            return convertNodeListToDIFObjects(query.asList(), user);
        }
        catch (DataSetException e)
        {
            throw new ContentManagerException(e);
        }
    }

    /**
     * @see pt.digitalis.dif.utils.extensions.cms.AbstractContentManager#getNodesByDescriptionInRepository(java.lang.String,
     *      java.lang.String, pt.digitalis.dif.controller.security.objects.IDIFUser)
     */
    @Override
    protected List<Node> getNodesByDescriptionInRepository(String basePathToSearch, String description, IDIFUser user)
            throws ContentManagerException
    {
        try
        {
            // for performance reasons; if searches in root, uses the base method
            if (basePathToSearch == null || basePathToSearch.length() == 0 || basePathToSearch.equals(SEPARATOR))
                return getNodesByDescriptionInRepository(description, user);

            pt.digitalis.dif.model.dataset.Query<pt.digitalis.dif.content.model.data.Node> query = getDocumentsService()
                    .getNodeDataSet().query();
            query.addFilter(new Filter(pt.digitalis.dif.content.model.data.Node.Fields.DESCRIPTION.toString(),
                    FilterType.EQUALS, description));
            query.addFilter(new Filter(pt.digitalis.dif.content.model.data.Node.Fields.FULLPATH.toString(),
                    FilterType.LESSER_OR_EQUALS_THAN, basePathToSearch));

            return convertNodeListToDIFObjects(query.asList(), user);
        }
        catch (DataSetException e)
        {
            throw new ContentManagerException(e);
        }
    }

    /**
     * @see pt.digitalis.dif.utils.extensions.cms.AbstractContentManager#getNodesByNameInRepository(java.lang.String,
     *      pt.digitalis.dif.controller.security.objects.IDIFUser)
     */
    @Override
    protected List<Node> getNodesByNameInRepository(String name, IDIFUser user) throws ContentManagerException
    {

        pt.digitalis.dif.model.dataset.Query<pt.digitalis.dif.content.model.data.Node> query = getDocumentsService()
                .getNodeDataSet().query();
        try
        {
            query.addFilter(new Filter(pt.digitalis.dif.content.model.data.Node.Fields.NAME.toString(),
                    FilterType.EQUALS, name));
            return convertNodeListToDIFObjects(query.asList(), user);
        }
        catch (DataSetException e)
        {
            throw new ContentManagerException(e);
        }
    }

    /**
     * @see pt.digitalis.dif.utils.extensions.cms.AbstractContentManager#getNodesByNameInRepository(java.lang.String,
     *      java.lang.String, pt.digitalis.dif.controller.security.objects.IDIFUser)
     */
    @Override
    protected List<Node> getNodesByNameInRepository(String basePathToSearch, String name, IDIFUser user)
            throws ContentManagerException
    {
        // for performance reasons, if search base is root, uses the base method
        if (basePathToSearch == null || basePathToSearch.length() == 0 || basePathToSearch.equals(SEPARATOR))
            return getNodesByNameInRepository(name, user);

        pt.digitalis.dif.model.dataset.Query<pt.digitalis.dif.content.model.data.Node> query = getDocumentsService()
                .getNodeDataSet().query();
        try
        {
            query.addFilter(new Filter(pt.digitalis.dif.content.model.data.Node.Fields.NAME.toString(),
                    FilterType.EQUALS, name));
            query.addFilter(new Filter(pt.digitalis.dif.content.model.data.Node.Fields.FULLPATH.toString(),
                    FilterType.LESSER_OR_EQUALS_THAN, basePathToSearch));

            return convertNodeListToDIFObjects(query.asList(), user);
        }
        catch (DataSetException e)
        {
            throw new ContentManagerException(e);
        }
    }

    /**
     * @see pt.digitalis.dif.utils.extensions.cms.AbstractContentManager#getNodesByParentNodeInRepository(java.lang.Long,
     *      pt.digitalis.dif.controller.security.objects.IDIFUser)
     */
    @Override
    protected List<Node> getNodesByParentNodeInRepository(Long nodeId, IDIFUser user) throws ContentManagerException
    {
        try
        {
            pt.digitalis.dif.model.dataset.Query<pt.digitalis.dif.content.model.data.Node> query = getDocumentsService()
                    .getNodeDataSet().query();
            query.addFilter(new Filter(pt.digitalis.dif.content.model.data.Node.Fields.PARENTNODEID.toString(),
                    FilterType.EQUALS, nodeId.toString()));

            return convertNodeListToDIFObjects(query.asList(), user);
        }
        catch (DataSetException e)
        {
            throw new ContentManagerException(e);
        }
    }

    /**
     * @see pt.digitalis.dif.utils.extensions.cms.AbstractContentManager#getNodesByParentNodeNoPermissionsInRepository(java.lang.Long,
     *      pt.digitalis.dif.controller.security.objects.IDIFUser)
     */
    @Override
    protected List<Node> getNodesByParentNodeNoPermissionsInRepository(Long nodeId, IDIFUser user)
            throws ContentManagerException
    {
        try
        {
            pt.digitalis.dif.model.dataset.Query<pt.digitalis.dif.content.model.data.Node> query = getDocumentsService()
                    .getNodeDataSet().query();
            query.addFilter(new Filter(pt.digitalis.dif.content.model.data.Node.Fields.PARENTNODEID.toString(),
                    FilterType.EQUALS, nodeId.toString()));

            return convertNodeListToDIFObjects(query.asList(), user);
        }
        catch (DataSetException e)
        {
            throw new ContentManagerException(e);
        }
    }

    /**
     * @see pt.digitalis.dif.utils.extensions.cms.AbstractContentManager#getParentFullPathInRepository(pt.digitalis.dif.utils.extensions.cms.Node)
     */
    @Override
    protected String getParentFullPathInRepository(Node node) throws ContentManagerException
    {
        try
        {
            pt.digitalis.dif.content.model.data.Node parentNodeDb = getDocumentsService().getNodeDataSet().get(
                    node.getParentNodeId().toString());

            if (parentNodeDb == null)
                throw new InvalidParentNodeException(node.getId());

            return parentNodeDb.getFullPath();
        }
        catch (DataSetException e)
        {
            throw new ContentManagerException(e);
        }
    }

    /**
     * @see pt.digitalis.dif.utils.extensions.cms.AbstractContentManager#getRootNodesInRepository(pt.digitalis.dif.controller.security.objects.IDIFUser)
     */
    @Override
    protected List<Node> getRootNodesInRepository(IDIFUser user) throws ContentManagerException
    {
        try
        {
            pt.digitalis.dif.model.dataset.Query<pt.digitalis.dif.content.model.data.Node> query = getDocumentsService()
                    .getNodeDataSet().query();
            query.addFilter(new Filter(pt.digitalis.dif.content.model.data.Node.Fields.PARENTNODEID.toString(),
                    FilterType.IS_NULL));

            return convertNodeListToDIFObjects(query.asList(), user);
        }
        catch (DataSetException e)
        {
            throw new ContentManagerException(e);
        }
    }

    /**
     * get Session
     * 
     * @return Session
     */
    private Session getSession()
    {
        return getDocumentsService().getContentItemAclDAO().getSession();
    }

    /**
     * @see pt.digitalis.dif.utils.extensions.cms.AbstractContentManager#grantContentAccessToGroupInRepository(pt.digitalis.dif.utils.extensions.cms.ContentItem,
     *      java.lang.String)
     */
    @Override
    protected boolean grantContentAccessToGroupInRepository(ContentItem contentItemParam, String groupId)
            throws ContentManagerException
    {
        Set<String> groups = new HashSet<String>();
        groups.add(groupId);
        if (!hasContentAccessGroupsInRepository(contentItemParam.getId(), groups))
        {
            pt.digitalis.dif.content.model.data.ContentItem contentItem = this
                    .convertContentItemToDBObject(contentItemParam);

            pt.digitalis.dif.content.model.data.ContentItemAcl contentItemAcl = new pt.digitalis.dif.content.model.data.ContentItemAcl();
            contentItemAcl.setContentItem(contentItem);
            contentItemAcl.setGroupId(groupId);

            try
            {
                getDocumentsService().getContentItemAclDataSet().insert(contentItemAcl);
            }
            catch (DataSetException e)
            {
                throw new ContentManagerException(e);
            }

        }
        return true;
    }

    /**
     * @see pt.digitalis.dif.utils.extensions.cms.AbstractContentManager#grantContentAccessToUserInRepository(pt.digitalis.dif.utils.extensions.cms.ContentItem,
     *      pt.digitalis.dif.controller.security.objects.IDIFUser)
     */
    @Override
    protected boolean grantContentAccessToUserInRepository(ContentItem contentItemParam, IDIFUser user)
            throws ContentManagerException
    {
        if (!hasContentAccessUserInRepository(contentItemParam.getId(), user))
        {
            pt.digitalis.dif.content.model.data.ContentItem contentItem = convertContentItemToDBObject(contentItemParam);

            pt.digitalis.dif.content.model.data.ContentItemAcl contentItemAcl = new pt.digitalis.dif.content.model.data.ContentItemAcl();

            contentItemAcl.setContentItem(contentItem);
            contentItemAcl.setUserId(user.getID());

            try
            {
                getDocumentsService().getContentItemAclDataSet().insert(contentItemAcl);
            }
            catch (DataSetException e)
            {
                throw new ContentManagerException(e);
            }
        }
        return true;
    }

    /**
     * @see pt.digitalis.dif.utils.extensions.cms.AbstractContentManager#grantNodeAccessToGroupInRepository(pt.digitalis.dif.utils.extensions.cms.Node,
     *      java.lang.String)
     */
    @Override
    protected boolean grantNodeAccessToGroupInRepository(Node nodeParam, String groupId) throws ContentManagerException
    {
        pt.digitalis.dif.content.model.data.Node node = convertNodeToDBObject(nodeParam);

        pt.digitalis.dif.content.model.data.NodeAcl nodeAcl = new pt.digitalis.dif.content.model.data.NodeAcl();

        nodeAcl.setNode(node);
        nodeAcl.setGroupId(groupId);
        nodeAcl.setUserId(null);
        try
        {
            getDocumentsService().getNodeAclDataSet().insert(nodeAcl);
        }
        catch (DataSetException e)
        {
            throw new ContentManagerException(e);
        }

        return true;
    }

    /**
     * @see pt.digitalis.dif.utils.extensions.cms.AbstractContentManager#grantNodeAccessToUserInRepository(pt.digitalis.dif.utils.extensions.cms.Node,
     *      pt.digitalis.dif.controller.security.objects.IDIFUser)
     */
    @Override
    protected boolean grantNodeAccessToUserInRepository(Node nodeParam, IDIFUser user) throws ContentManagerException
    {
        pt.digitalis.dif.content.model.data.Node node = convertNodeToDBObject(nodeParam);

        pt.digitalis.dif.content.model.data.NodeAcl nodeAcl = new pt.digitalis.dif.content.model.data.NodeAcl();

        nodeAcl.setNode(node);
        nodeAcl.setUserId(user.getID());
        try
        {
            getDocumentsService().getNodeAclDataSet().insert(nodeAcl);
        }
        catch (DataSetException e)
        {
            throw new ContentManagerException(e);
        }
        return true;
    }

    /**
     * @see pt.digitalis.dif.utils.extensions.cms.AbstractContentManager#hasContentAccessGroupsInRepository(java.lang.String,
     *      java.util.Set)
     */
    @Override
    protected boolean hasContentAccessGroupsInRepository(String contentId, Set<String> groups)
            throws ContentManagerException
    {
        try
        {
            pt.digitalis.dif.model.dataset.Query<ContentItemAcl> query = getDocumentsService()
                    .getContentItemAclDataSet().query();
            query.addFilter(new Filter(StringUtils.toLowerFirstChar(ContentItem.class.getSimpleName() + ".id"),
                    FilterType.EQUALS, contentId.toString()));
            query.addFilter(new Filter(ContentItemAcl.Fields.GROUPID.toString(), FilterType.IN, CollectionUtils
                    .setToCommaSeparatedString(groups)));

            long result = query.count();
            return result > 0;

        }
        catch (DataSetException e)
        {
            throw new ContentManagerException(e);
        }
    }

    /**
     * @see pt.digitalis.dif.utils.extensions.cms.AbstractContentManager#hasContentAccessUserInRepository(java.lang.String,
     *      pt.digitalis.dif.controller.security.objects.IDIFUser)
     */
    @Override
    protected boolean hasContentAccessUserInRepository(String contentId, IDIFUser user) throws ContentManagerException
    {
        try
        {
            pt.digitalis.dif.model.dataset.Query<pt.digitalis.dif.content.model.data.ContentItemAcl> query = getDocumentsService()
                    .getContentItemAclDataSet().query();
            query.addFilter(new Filter(StringUtils.toLowerFirstChar(ContentItem.class.getSimpleName() + ".id"),
                    FilterType.EQUALS, contentId.toString()));
            query.addFilter(new Filter(pt.digitalis.dif.content.model.data.ContentItemAcl.Fields.USERID.toString(),
                    FilterType.EQUALS, user.getID()));

            return query.count() > 0;
        }
        catch (DataSetException e)
        {
            throw new ContentManagerException(e);
        }
    }

    /**
     * @see pt.digitalis.dif.utils.extensions.cms.AbstractContentManager#hasNodeAccessGroupsInRepository(java.lang.Long,
     *      java.util.Set)
     */
    @Override
    protected boolean hasNodeAccessGroupsInRepository(Long nodeId, Set<String> groups) throws ContentManagerException
    {
        try
        {
            pt.digitalis.dif.model.dataset.Query<pt.digitalis.dif.content.model.data.NodeAcl> query = getDocumentsService()
                    .getNodeAclDataSet().query();
            query.addFilter(new Filter(StringUtils.toLowerFirstChar(pt.digitalis.dif.content.model.data.Node.class
                    .getSimpleName() + ".id"), FilterType.EQUALS, nodeId.toString()));
            query.addFilter(new Filter(pt.digitalis.dif.content.model.data.NodeAcl.Fields.GROUPID.toString(),
                    FilterType.IN, CollectionUtils.setToCommaSeparatedString(groups)));

            return query.count() > 0;
        }
        catch (DataSetException e)
        {
            throw new ContentManagerException(e);
        }
    }

    /**
     * @see pt.digitalis.dif.utils.extensions.cms.AbstractContentManager#hasNodeAccessUserInRespository(java.lang.Long,
     *      pt.digitalis.dif.controller.security.objects.IDIFUser)
     */
    @Override
    protected boolean hasNodeAccessUserInRespository(Long nodeId, IDIFUser user) throws ContentManagerException
    {
        try
        {
            pt.digitalis.dif.model.dataset.Query<pt.digitalis.dif.content.model.data.NodeAcl> query = getDocumentsService()
                    .getNodeAclDataSet().query();
            query.addFilter(new Filter(StringUtils.toLowerFirstChar(pt.digitalis.dif.content.model.data.Node.class
                    .getSimpleName() + ".id"), FilterType.EQUALS, nodeId.toString()));
            query.addFilter(new Filter(pt.digitalis.dif.content.model.data.NodeAcl.Fields.USERID.toString(),
                    FilterType.EQUALS, user.getID()));

            return query.count() > 0;
        }
        catch (DataSetException e)
        {
            throw new ContentManagerException(e);
        }
    }

    /**
     * @see pt.digitalis.dif.utils.extensions.cms.AbstractContentManager#mergeContentInRepository(pt.digitalis.dif.utils.extensions.cms.ContentItem,
     *      pt.digitalis.dif.controller.security.objects.IDIFUser)
     */
    @Override
    protected ContentItem mergeContentInRepository(ContentItem contentParam, IDIFUser user)
            throws ContentManagerException
    {
        pt.digitalis.dif.content.model.data.ContentItem content = convertContentItemToDBObject(contentParam);
        ContentItem result = null;
        try
        {

            Session session = getDocumentsService().getNodeDAO().getSession();
            boolean wasActive = session.getTransaction().isActive();
            if (!wasActive)
            {
                session.beginTransaction();
            }
            /*
             * Couldn't use dataset.update because it uses the hibernate attachDirty and this method doesn't work well
             * with the convert objects strategy used on this implementation The merge can't be used because it's made
             * for ignore BLOB fields.
             */
            session.saveOrUpdate(content);
            content = getDocumentsService().getContentItemDAO().findById(content.getId());
            result = convertContentItemToDIFObject(content, user);

            if (!wasActive)
            {
                session.getTransaction().commit();
            }
            return result;
        }
        catch (ConstraintViolationException e)
        {
            throw new ContentItemWithDuplicateNameAndParentNodeException(e);
        }

    }

    /**
     * @see pt.digitalis.dif.utils.extensions.cms.AbstractContentManager#mergeNodeInRepository(pt.digitalis.dif.utils.extensions.cms.Node,
     *      pt.digitalis.dif.controller.security.objects.IDIFUser)
     */
    @Override
    protected Node mergeNodeInRepository(Node nodeParam, IDIFUser user)
    {
        pt.digitalis.dif.content.model.data.Node node = convertNodeToDBObject(nodeParam);

        Session session = getDocumentsService().getNodeDAO().getSession();
        boolean wasActive = session.getTransaction().isActive();
        if (!wasActive)
        {
            session.beginTransaction();
        }
        /*
         * Couldn't use dataset.update because it uses the hibernate attachDirty and this method doesn't work well with
         * the convert objects strategy used on this implementation
         */
        node = getDocumentsService().getNodeDAO().merge(node);

        if (!wasActive)
        {
            session.getTransaction().commit();
        }
        return convertNodeToDIFObject(node, user);
    }

    /**
     * @see pt.digitalis.dif.utils.extensions.cms.AbstractContentManager#moveContentInRepository(pt.digitalis.dif.utils.extensions.cms.ContentItem,
     *      pt.digitalis.dif.utils.extensions.cms.Node, pt.digitalis.dif.controller.security.objects.IDIFUser)
     */
    @Override
    protected ContentItem moveContentInRepository(ContentItem contentParam, Node nodeParam, IDIFUser user)
            throws ContentManagerException
    {
        try
        {
            pt.digitalis.dif.content.model.data.ContentItem currentContent = getDocumentsService()
                    .getContentItemDataSet().get(contentParam.getId().toString());

            if (currentContent == null)
                throw new ContentItemNotFoundException();

            currentContent.setNode(convertNodeToDBObject(nodeParam));

            getDocumentsService().getContentItemDataSet().update(currentContent);

            return convertContentItemToDIFObject(currentContent, user);
        }
        catch (DataSetException e)
        {
            throw new ContentManagerException(e);
        }
    }

    /**
     * @see pt.digitalis.dif.utils.extensions.cms.AbstractContentManager#moveNodeInRepository(pt.digitalis.dif.utils.extensions.cms.Node,
     *      pt.digitalis.dif.controller.security.objects.IDIFUser)
     */
    @Override
    protected Node moveNodeInRepository(Node nodeParam, IDIFUser user)
    {
        pt.digitalis.dif.content.model.data.Node node = convertNodeToDBObject(nodeParam);
        Session session = getDocumentsService().getNodeDAO().getSession();
        boolean wasActive = session.getTransaction().isActive();
        if (!wasActive)
        {
            session.beginTransaction();
        }
        /*
         * Couldn't use dataset.update because it uses the hibernate attachDirty and this method doesn't work well with
         * the convert objects strategy used on this implementation
         */
        getDocumentsService().getNodeDAO().merge(node);

        if (!wasActive)
        {
            session.getTransaction().commit();
        }
        return convertNodeToDIFObject(node, user);
    }

    /**
     * @see pt.digitalis.dif.utils.extensions.cms.AbstractContentManager#nodeExistsInRepository(java.lang.Long,
     *      pt.digitalis.dif.controller.security.objects.IDIFUser)
     */
    @Override
    protected boolean nodeExistsInRepository(Long id, IDIFUser user) throws ContentManagerException
    {
        try
        {
            pt.digitalis.dif.content.model.data.Node node = getDocumentsService().getNodeDataSet().get(id.toString());

            return node != null;
        }
        catch (DataSetException e)
        {
            throw new ContentManagerException(e);
        }
    }

    /**
     * @see pt.digitalis.dif.utils.extensions.cms.AbstractContentManager#persistContentInRepository(pt.digitalis.dif.utils.extensions.cms.ContentItem,
     *      pt.digitalis.dif.controller.security.objects.IDIFUser)
     */
    @Override
    protected ContentItem persistContentInRepository(ContentItem content, IDIFUser user) throws ContentManagerException
    {
        ContentItem result = null;
        boolean isActive = this.getDocumentsService().getContentItemDAO().getSession().getTransaction().isActive();

        if (!isActive)
        {
            this.getDocumentsService().getContentItemDAO().getSession().beginTransaction();
        }

        try
        {
            pt.digitalis.dif.content.model.data.ContentItem contentItemDb = this.convertContentItemToDBObject(content);
            contentItemDb.setCreationDate(new java.util.Date());
            contentItemDb.setCreatorId(user.getID());

            this.getDocumentsService().getContentItemDAO().persist(contentItemDb);
            contentItemDb = this.getDocumentsService().getContentItemDAO().findById(contentItemDb.getId());
            result = this.convertContentItemToDIFObject(contentItemDb, user);

            if (!isActive)
            {
                this.getDocumentsService().getContentItemDAO().getSession().getTransaction().commit();
            }

            return result;
        }
        catch (ConstraintViolationException e)
        {
            if (!isActive)
            {
                this.getDocumentsService().getContentItemDAO().getSession().getTransaction().rollback();
            }

            throw new ContentItemWithDuplicateNameAndParentNodeException(e);
        }

    }

    /**
     * @see pt.digitalis.dif.utils.extensions.cms.AbstractContentManager#persistNodeInRepository(pt.digitalis.dif.utils.extensions.cms.Node,
     *      pt.digitalis.dif.controller.security.objects.IDIFUser)
     */
    @Override
    protected Node persistNodeInRepository(Node node, IDIFUser user) throws ContentManagerException
    {
        try
        {
            pt.digitalis.dif.content.model.data.Node nodeDb = this.convertNodeToDBObject(node);

            nodeDb.setCreationDate(new java.util.Date());
            nodeDb.setCreatorId(user.getID());

            nodeDb = this.getDocumentsService().getNodeDataSet().insert(nodeDb);

            // now inserts a default ACL
            // the alternative is to assume that the creator always have access

            pt.digitalis.dif.content.model.data.NodeAcl nodeAcl = new pt.digitalis.dif.content.model.data.NodeAcl();
            nodeAcl.setUserId(node.getCreatorID());
            nodeAcl.setNode(nodeDb);

            this.getDocumentsService().getNodeAclDataSet().insert(nodeAcl);

            return this.convertNodeToDIFObject(nodeDb, user);
        }
        catch (ConstraintViolationException e)
        {
            throw new NodeWithDuplicatePathException(e);
        }

        catch (DataSetException e)
        {
            throw new ContentManagerException(e);
        }
    }

    /**
     * Revokes group access to content item
     * 
     * @param contentId
     *            the id of the content to revoke access
     * @param groupId
     *            the group to revoke access
     * @return T if access was revoked, F otherwise.
     */
    @Override
    protected boolean revokeContentAccessToGroupInRepository(String contentId, String groupId)
    {
        boolean wasActive = getSession().getTransaction().isActive();
        if (!wasActive)
        {
            getSession().beginTransaction();
        }

        try
        {
            pt.digitalis.dif.model.dataset.Query<pt.digitalis.dif.content.model.data.ContentItemAcl> query = getDocumentsService()
                    .getContentItemAclDataSet().query();
            query.addFilter(new Filter(StringUtils
                    .toLowerFirstChar(pt.digitalis.dif.content.model.data.ContentItem.class.getSimpleName() + ".id"),
                    FilterType.EQUALS, contentId.toString()));
            query.addFilter(new Filter(pt.digitalis.dif.content.model.data.ContentItemAcl.Fields.GROUPID.toString(),
                    FilterType.EQUALS, groupId));

            for (pt.digitalis.dif.content.model.data.ContentItemAcl toDelete: query.asList())
            {
                getDocumentsService().getContentItemAclDataSet().delete(toDelete.getId().toString());
            }

            if (!wasActive)
            {
                getSession().getTransaction().commit();
            }
        }
        catch (Exception e)
        {
            e.printStackTrace();
            if (!wasActive)
            {
                getSession().getTransaction().rollback();
            }
            return false;
        }

        return true;
    }

    /**
     * @see pt.digitalis.dif.utils.extensions.cms.AbstractContentManager#revokeContentAccessToUserInRepository(java.lang.String,
     *      pt.digitalis.dif.controller.security.objects.IDIFUser)
     */
    @Override
    protected boolean revokeContentAccessToUserInRepository(String contentId, IDIFUser user)
    {
        boolean wasActive = getSession().getTransaction().isActive();
        if (!wasActive)
        {
            getSession().beginTransaction();
        }

        try
        {
            pt.digitalis.dif.model.dataset.Query<pt.digitalis.dif.content.model.data.ContentItemAcl> query = getDocumentsService()
                    .getContentItemAclDataSet().query();
            query.addFilter(new Filter(StringUtils
                    .toLowerFirstChar(pt.digitalis.dif.content.model.data.ContentItem.class.getSimpleName() + ".id"),
                    FilterType.EQUALS, contentId.toString()));
            query.addFilter(new Filter(pt.digitalis.dif.content.model.data.ContentItemAcl.Fields.USERID.toString(),
                    FilterType.EQUALS, user.getID()));

            for (pt.digitalis.dif.content.model.data.ContentItemAcl toDelete: query.asList())
            {
                getDocumentsService().getContentItemAclDataSet().delete(toDelete.getId().toString());
            }

            if (!wasActive)
            {
                getSession().getTransaction().commit();
            }
        }
        catch (Exception e)
        {
            e.printStackTrace();
            if (!wasActive)
            {
                getSession().getTransaction().rollback();
            }
            return false;
        }

        return true;
    }

    /**
     * @see pt.digitalis.dif.utils.extensions.cms.AbstractContentManager#revokeNodeAccessToGroupInRepository(java.lang.Long,
     *      java.lang.String)
     */
    @Override
    protected boolean revokeNodeAccessToGroupInRepository(Long nodeId, String groupId) throws ContentManagerException
    {
        try
        {
            pt.digitalis.dif.model.dataset.Query<pt.digitalis.dif.content.model.data.NodeAcl> query = getDocumentsService()
                    .getNodeAclDataSet().query();
            query.addFilter(new Filter(StringUtils.toLowerFirstChar(pt.digitalis.dif.content.model.data.Node.class
                    .getSimpleName() + ".id"), FilterType.EQUALS, nodeId.toString()));
            query.addFilter(new Filter(pt.digitalis.dif.content.model.data.NodeAcl.Fields.GROUPID.toString(),
                    FilterType.EQUALS, groupId.toString()));

            for (pt.digitalis.dif.content.model.data.NodeAcl toDelete: query.asList())
                getDocumentsService().getNodeAclDataSet().delete(toDelete.getId().toString());

            return true;
        }
        catch (DataSetException e)
        {
            throw new ContentManagerException(e);
        }
    }

    /**
     * @see pt.digitalis.dif.utils.extensions.cms.AbstractContentManager#revokeNodeAccessToUserInRepository(java.lang.Long,
     *      pt.digitalis.dif.controller.security.objects.IDIFUser)
     */
    @Override
    protected boolean revokeNodeAccessToUserInRepository(Long nodeId, IDIFUser user) throws ContentManagerException
    {
        try
        {
            pt.digitalis.dif.model.dataset.Query<pt.digitalis.dif.content.model.data.NodeAcl> query = getDocumentsService()
                    .getNodeAclDataSet().query();
            query.addFilter(new Filter(StringUtils.toLowerFirstChar(pt.digitalis.dif.content.model.data.Node.class
                    .getSimpleName() + ".id"), FilterType.EQUALS, nodeId.toString()));
            query.addFilter(new Filter(pt.digitalis.dif.content.model.data.NodeAcl.Fields.USERID.toString(),
                    FilterType.EQUALS, user.getID()));

            for (pt.digitalis.dif.content.model.data.NodeAcl toDelete: query.asList())
                getDocumentsService().getNodeAclDataSet().delete(toDelete.getId().toString());

            return true;
        }
        catch (DataSetException e)
        {
            throw new ContentManagerException(e);
        }
    }

    /**
     * @see pt.digitalis.dif.utils.extensions.cms.IContentManager#rollbackTransaction()
     */
    public void rollbackTransaction()
    {
        getSession().getTransaction().commit();
    }
}
