/**
 * - Digitalis Internal Framework v2.0 - (C) 2007, Digitalis Informatica. Distribuicao e Gestao de Informatica, Lda.
 * Estrada de Paco de Arcos num.9 - Piso -1 2780-666 Paco de Arcos Telefone: (351) 21 4408990 Fax: (351) 21 4408999
 * http://www.digitalis.pt
 */
package pt.digitalis.dif.codegen;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;

import pt.digitalis.dif.codegen.util.DEMLoaderEntityRegistry;
import pt.digitalis.dif.codegen.util.DEMLoaderHelper;
import pt.digitalis.dif.codegen.util.IClassEnhancer;
import pt.digitalis.dif.dem.DEMRegistryImpl;
import pt.digitalis.dif.dem.Entity;
import pt.digitalis.dif.dem.annotations.AnnotationMemberTags;
import pt.digitalis.dif.dem.annotations.entities.ApplicationDefinition;
import pt.digitalis.dif.dem.annotations.entities.ServiceDefinition;
import pt.digitalis.dif.dem.annotations.entities.StageDefinition;
import pt.digitalis.dif.dem.config.IDEMRegistrator;
import pt.digitalis.dif.dem.config.IEntityRegistration;
import pt.digitalis.dif.dem.managers.impl.UsageIssuesManagerImpl;
import pt.digitalis.dif.dem.objects.issues.IssueScope;
import pt.digitalis.dif.dem.objects.issues.IssueType;
import pt.digitalis.dif.exception.codegen.AnnotationMisuseException;
import pt.digitalis.dif.exception.codegen.DIFCodeGenerationException;
import pt.digitalis.dif.ioc.DIFIoCRegistry;
import pt.digitalis.dif.utils.logging.DIFLogger;
import pt.digitalis.utils.bytecode.exceptions.CodeGenerationException;
import pt.digitalis.utils.bytecode.holders.AnnotationHolder;
import pt.digitalis.utils.bytecode.holders.ClassHolder;
import pt.digitalis.utils.bytecode.holders.HolderRepository;
import pt.digitalis.utils.inspection.exception.ResourceNotFoundException;

import com.google.inject.Inject;

/**
 * The framework's main code generation facility. It is an utility class that is called by the frameworks initialization
 * procedure
 * 
 * @author Pedro Viegas <a href="mailto:pviegas@digitalis.pt">pviegas@digitalis.pt</a>
 * @author Rodrigo Gonalves <a href="mailto:rgoncalves@digitalis.pt">rgoncalves@digitalis.pt</a>
 * @created Jul 23, 2007
 */
public class DIFCodeGenerator {

    /** The class enhancer */
    @Inject
    private IClassEnhancer classEnhancer;

    /** The package registrator */
    @Inject
    private IDEMRegistrator demRegistrator;

    /**
     * Performs cleanup operations.
     */
    public void cleanUp()
    {
        // Perform memory cleanup
        DEMLoaderEntityRegistry.cleanUp();
        DEMLoaderHelper.cleanUp();
        HolderRepository.cleanUp();

        /*
         * TODO: After these cleanups, some containers are set to null and not reinitialized again. This might originate
         * NullPointerExceptions. The unit tests might not run properly. If such a situation arises it will be
         * appropriated to move these clean-up operations to a separate method that will be called explicitly by the
         * developer.
         */
    }

    /**
     * Collects all contributed package registrations in the registry
     * 
     * @throws ResourceNotFoundException
     *             if it can't get no classes to enhancement
     * @throws CodeGenerationException
     *             if a class can't be enhanced
     */
    public void collectRegisteredPackages() throws ResourceNotFoundException, CodeGenerationException
    {

        // Get all contributed implementation of the EntityRegistration class
        List<IEntityRegistration> entityRegistrators = DIFIoCRegistry.getRegistry().getImplementations(
                IEntityRegistration.class);

        // ...and run them
        for (IEntityRegistration entityRegistration: entityRegistrators)
            entityRegistration.registerEntitys(demRegistrator);
    }

    /**
     * Searches all registered packages for DEM entities and performs the Class enhance operations.
     * 
     * @throws CodeGenerationException
     *             if a class can't be enhanced
     * @throws ResourceNotFoundException
     *             if it can't get no classes to enhancement
     * @throws DIFCodeGenerationException
     */
    public void enhanceDEMClasses() throws CodeGenerationException, ResourceNotFoundException,
            DIFCodeGenerationException
    {

        List<String> stagesToRemove = new ArrayList<String>();
        List<String> servicesToRemove = new ArrayList<String>();
        List<String> applicationsToRemove = new ArrayList<String>();

        /* Remove from Registry the incoherent entities */
        for (Entry<String, ClassHolder> entry: DEMLoaderEntityRegistry.getStages().entrySet())
        {
            String stageID = entry.getKey();
            ClassHolder clazz = entry.getValue();

            AnnotationHolder annotationHolder = clazz.getAnnotations().get(StageDefinition.class.getCanonicalName());
            String serviceID = annotationHolder.getMembers().get(AnnotationMemberTags.STAGE_DEFINITION_SERVICE)
                    .toString();

            if (DEMLoaderEntityRegistry.getServices().containsKey(serviceID))
            {
                clazz = DEMLoaderEntityRegistry.getServices().get(serviceID);

                annotationHolder = clazz.getAnnotations().get(ServiceDefinition.class.getCanonicalName());
                String applicationID = annotationHolder.getMembers()
                        .get(AnnotationMemberTags.SERVICE_DEFINITION_APPLICATION).toString();

                if (DEMLoaderEntityRegistry.getApplications().containsKey(applicationID))
                {
                    clazz = DEMLoaderEntityRegistry.getApplications().get(applicationID);

                    annotationHolder = clazz.getAnnotations().get(ApplicationDefinition.class.getCanonicalName());
                    String providerID = annotationHolder.getMembers()
                            .get(AnnotationMemberTags.APPLICATION_DEFINITION_PROVIDER).toString();

                    if (!DEMLoaderEntityRegistry.getProviders().containsKey(providerID))
                    {
                        applicationsToRemove.add(applicationID);
                        servicesToRemove.add(serviceID);
                        stagesToRemove.add(stageID);
                    }
                }
                else
                {
                    servicesToRemove.add(serviceID);
                    stagesToRemove.add(stageID);
                }
            }
            else
            {
                stagesToRemove.add(stageID);
            }
        }

        for (String applicationId: applicationsToRemove)
        {
            DEMLoaderEntityRegistry.getServices().remove(applicationId);
            DIFLogger.getLogger().info(
                    "The application '" + applicationId + "' has been excluded due to the lack of suitable provider");
        }

        for (String serviceId: servicesToRemove)
        {
            DEMLoaderEntityRegistry.getServices().remove(serviceId);
            DIFLogger.getLogger().info(
                    "The service '" + serviceId + "' has been excluded due to the lack of suitable application");
        }

        for (String stageId: stagesToRemove)
        {
            DEMLoaderEntityRegistry.getStages().remove(stageId);
            DIFLogger.getLogger().info(
                    "The stage '" + stageId + "' has been excluded due to the lack of suitable service");
        }

        /*
         * Enhance each entity class. Note that the enhancement order is relevant and respects the DEM hierarchy. Each
         * class is also added to the respective slot in the DEMRegistry
         */

        enhanceEntities(Entity.VALIDATOR, DEMLoaderEntityRegistry.getValidators());
        enhanceEntities(Entity.PROVIDER, DEMLoaderEntityRegistry.getProviders());
        enhanceEntities(Entity.APPLICATION, DEMLoaderEntityRegistry.getApplications());
        enhanceEntities(Entity.SERVICE, DEMLoaderEntityRegistry.getServices());
        enhanceEntities(Entity.STAGE, DEMLoaderEntityRegistry.getStages());

    }

    /**
     * Performs the bytecode enhancement to the given entity classes
     * 
     * @param entityType
     *            the entity type
     * @param entityMap
     *            the map of entities to add
     * @throws CodeGenerationException
     *             if a class can't be enhanced
     * @throws ResourceNotFoundException
     *             if it can't get no classes to enhancement
     * @throws DIFCodeGenerationException
     */
    private void enhanceEntities(Entity entityType, Map<String, ClassHolder> entityMap) throws CodeGenerationException,
            ResourceNotFoundException, DIFCodeGenerationException
    {

        ClassHolder entityClazz;

        if (entityMap != null)
            for (ClassHolder clazz: entityMap.values())
            {
                try
                {

                    entityClazz = classEnhancer.enhance(clazz);

                    if (!entityType.equals(Entity.VALIDATOR))
                    {
                        DEMRegistryImpl.addEntity(entityType, entityClazz);
                    }

                }
                catch (AnnotationMisuseException annotationMisuseException)
                {
                    UsageIssuesManagerImpl.getInstance().addIssue(
                            IssueType.ERROR,
                            IssueScope.LOADTIME,
                            annotationMisuseException.getExceptionContext()
                                    .get(AnnotationMisuseException.ContextKeys.CLASS).toString(),
                            annotationMisuseException.getMessage(), annotationMisuseException);
                }
            }
    }

    /**
     * Searches all registered packages for DEM entities and other DIF elements.
     * 
     * @throws ResourceNotFoundException
     *             if it can't get no classes to enhancement
     * @throws CodeGenerationException
     *             if a class can't be enhanced
     */
    public void searchRegisteredPackages() throws ResourceNotFoundException, CodeGenerationException
    {

        /*
         * Iterate through the DEM-registered folders for annotated classes and separate them by their corresponding
         * entity types in the DEM load time registry
         */
        for (String packageName: DEMLoaderHelper.getPackageList())
        {

            // Get all classes in this folder
            List<ClassHolder> classes = new ArrayList<ClassHolder>();

            try
            {
                classes = DEMLoaderHelper.getDEMEntityClassesInPackage(packageName);

            }
            catch (ResourceNotFoundException resourceNotFoundException)
            {
                throw new ResourceNotFoundException("Could not find any classes for enhancement in package "
                        + packageName, resourceNotFoundException);
            }

            // Load the classes which are entities
            try
            {
                DEMLoaderEntityRegistry.loadEntityClasses(classes, demRegistrator);
            }
            catch (ClassNotFoundException e)
            {
                throw new CodeGenerationException(e);
            }
            catch (InstantiationException e)
            {
                throw new CodeGenerationException(e);
            }
            catch (IllegalAccessException e)
            {
                throw new CodeGenerationException(e);
            }
        }
    }
}
