/**
 * - Digitalis Internal Framework v2.0 - (C) 2007, Digitalis Informatica. Distribuicao e Gestao de Informatica, Lda.
 * Estrada de Paco de Arcos num.9 - Piso -1 2780-666 Paco de Arcos Telefone: (351) 21 4408990 Fax: (351) 21 4408999
 * http://www.digitalis.pt
 */
package pt.digitalis.dif.controller;

import pt.digitalis.dif.controller.interfaces.IChAL;
import pt.digitalis.dif.controller.interfaces.IDIFDispatcher;
import pt.digitalis.dif.controller.interfaces.IDIFRequest;
import pt.digitalis.dif.controller.interfaces.IDIFResponse;
import pt.digitalis.dif.controller.interfaces.IDispatcherErrorHandler;
import pt.digitalis.dif.controller.objects.ClientDescriptor;
import pt.digitalis.dif.controller.objects.RESTAction;
import pt.digitalis.dif.dem.annotations.controller.Channel;
import pt.digitalis.dif.dem.managers.IMessageManager;
import pt.digitalis.dif.exception.controller.BusinessFlowException;
import pt.digitalis.dif.exception.controller.ControllerException;
import pt.digitalis.dif.ioc.DIFIoCRegistry;

import com.newrelic.api.agent.Trace;

/**
 * Base implementation for all ChALs The API exposed to the Listener entities is the <code>serve()</code> method. All
 * the communication between AbstractChAL and Listener is done through this API.
 * 
 * @param <RequestType>
 *            type original request type
 * @param <ResponseType>
 *            the original response type
 * @author Rodrigo Gonalves <a href="mailto:rgoncalves@digitalis.pt">rgoncalves@digitalis.pt</a>
 * @author Pedro Viegas <a href="mailto:pviegas@digitalis.pt">pviegas@digitalis.pt</a>
 * @created 2007/03/27
 */
abstract public class AbstractChAL<RequestType, ResponseType> implements IChAL<RequestType, ResponseType> {

    /** The channel ID */
    private String channelID = null;

    /** The message manager. */
    protected IMessageManager messageManager = DIFIoCRegistry.getRegistry().getImplementation(IMessageManager.class);

    /**
     * Get the id of the channel associated to the ChAL. If the id is unset the method reads the channel id from the
     * annotation.
     * 
     * @return the channel ID read from the {@link Channel} annotation
     */
    public String getChannelID()
    {
        if (channelID == null)
            channelID = this.getClass().getAnnotation(Channel.class).value();

        return channelID;
    }

    /**
     * Creates a client descriptor object identifying the client agent who issued the request
     * 
     * @param originalRequest
     *            the original request object
     * @return the client descriptor for the current request
     * @throws ControllerException
     *             when any runtime exception is thrown
     */
    abstract protected ClientDescriptor getClientDescriptor(RequestType originalRequest) throws ControllerException;

    /**
     * Inspector of the dispatcher assigned to this channel.
     * 
     * @return theDispatcher attribute
     */
    protected IDIFDispatcher getDispatcher()
    {
        return DIFIoCRegistry.getRegistry().getImplementation(IDIFDispatcher.class, getChannelID());
    }

    /**
     * Inspector of the dispatcher assigned to this channel.
     * 
     * @return theDispatcher attribute
     */
    protected IDispatcherErrorHandler getErrorHandler()
    {
        return DIFIoCRegistry.getRegistry().getImplementation(IDispatcherErrorHandler.class, getChannelID());
    }

    /**
     * @see pt.digitalis.dif.controller.interfaces.IChAL#serve(java.lang.Object, java.lang.Object,
     *      pt.digitalis.dif.controller.objects.RESTAction)
     * @param originalRequest
     *            the original request object
     * @param finalResponse
     *            the final response object
     * @param restAction
     *            the REST action, if specified
     * @return the processed DIFResponse
     */
    @Trace(metricName = "DIF:ChAL:Serve", dispatcher = true)
    public IDIFResponse serve(RequestType originalRequest, ResponseType finalResponse, RESTAction restAction)
    {

        IDIFResponse difResponse = null;
        IDIFRequest difRequest = null;

        try
        {
            difRequest = translateRequest(originalRequest);
            difRequest.setRestAction(restAction);

            // Initializes the client descriptor if ins't filled. The translateRequest is responsible for filling it.
            if (difRequest.getClient() == null)
                difRequest.setClient(getClientDescriptor(originalRequest));

            if (validateRequest(originalRequest))
            {

                difResponse = getDispatcher().dispatch(difRequest);

                // Fail-over. Should never return null.
                if (difResponse == null)
                    difResponse = getErrorHandler().getDefaultErrorResponse(difRequest, null);

            }
            else
                difResponse = getErrorHandler().getDefaultErrorResponse(difRequest, null);

        }
        catch (BusinessFlowException exception)
        {
            difResponse = getErrorHandler().processException(difRequest, exception);
        }
        catch (RuntimeException exception)
        {
            difResponse = getErrorHandler().processException(difRequest, exception);
        }
        catch (ControllerException exception)
        {
            difResponse = getErrorHandler().processException(difRequest, exception);
        }

        this.publish(difResponse, originalRequest, finalResponse);

        return difResponse;
    }

    /**
     * Validates the original request. It can be used to check if a particular request is valid for the channel. It may
     * also perform other validations, as the request parameters integrity.
     * 
     * @param originalRequest
     *            The original request received from the client.
     * @return T case the request is well formed, F otherwise
     * @throws ControllerException
     *             when any runtime exception is thrown
     */
    abstract protected boolean validateRequest(RequestType originalRequest) throws ControllerException;
}
