/**
 * 2009, Digitalis Informatica. All rights reserved. Distribuicao e Gestao de Informatica, Lda. Estrada de Paco de Arcos
 * num.9 - Piso -1 2780-666 Paco de Arcos Telefone: (351) 21 4408990 Fax: (351) 21 4408999 http://www.digitalis.pt
 */

package pt.digitalis.dif.utils.extensions.cms;

import java.util.ArrayList;
import java.util.Date;
import java.util.List;

import pt.digitalis.dif.controller.security.objects.IDIFUser;
import pt.digitalis.dif.exception.security.IdentityManagerException;
import pt.digitalis.dif.ioc.DIFIoCRegistry;
import pt.digitalis.dif.utils.extensions.cms.exception.AlreadyDeletedException;
import pt.digitalis.dif.utils.extensions.cms.exception.ContentItemNotFoundException;
import pt.digitalis.dif.utils.extensions.cms.exception.ContentManagerException;

/**
 * Content Item definition class
 * 
 * @author Pedro Viegas <a href="mailto:pviegas@digitalis.pt">pviegas@digitalis.pt</a><br/>
 * @created 2010/10/06
 */
public class ContentItem {

    /** the node ACL */
    private List<ACLEntry> accessList;

    /** The content */
    private String content;

    /** The contentManager */
    private IContentManager contentManager;

    /** The date when the document was created */
    private Date creationDate;

    /** the creator user ID */
    private String creatorID;

    /** The document description */
    private String description;

    /** the document unique identifier in the document repository */
    private String id;

    /** True if the content item object has already been deleted in the repository */
    private boolean isDeleted = false;

    /** The document name */
    private String name;

    /** The parent node ID */
    private Long parentNodeId;

    /** The current user */
    private IDIFUser user;

    /**
     * Default constructor for <code>DocumentRepositoryEntry</code>.
     * 
     * @param user
     *            The current user
     */
    public ContentItem(IDIFUser user)
    {
        super();

        this.creationDate = new Date();
        this.user = user;
    }

    /**
     * @param parentNodeId
     *            the parent node ID
     * @param name
     *            the content name
     * @param creatorID
     *            the creator ID
     * @param user
     *            The current user
     */
    public ContentItem(Long parentNodeId, String name, String creatorID, IDIFUser user)
    {
        this.creatorID = creatorID;
        this.name = name;
        this.parentNodeId = parentNodeId;
        this.creationDate = new Date();
        this.user = user;
    }

    /**
     * ContentItem is deleted in DB. Object ID property is set to null
     * 
     * @return T if update was succeeded
     * @throws IdentityManagerException
     * @throws ContentManagerException
     */
    public boolean delete() throws IdentityManagerException, ContentManagerException
    {
        if (isDeleted)
            throw new AlreadyDeletedException();

        boolean ret = getContentManager().deleteContent(id, user);

        // object ID is set to null
        if (ret)
        {
            isDeleted = true;
            id = null;
        }

        return ret;
    }

    /**
     * Get ACL.
     * 
     * @return the accessList value
     * @throws ContentManagerException
     */
    public List<ACLEntry> getACL() throws ContentManagerException
    {
        if (accessList == null)
        {

            accessList = getContentManager().getContentItemACL(id);
        }

        return accessList;
    }

    /**
     * Inspector for the 'content' attribute.
     * 
     * @return the content value
     */
    public String getContent()
    {
        return content;
    }

    /**
     * Lazy load for content manager
     * 
     * @return the content manager
     */
    private IContentManager getContentManager()
    {
        if (contentManager == null)
            contentManager = DIFIoCRegistry.getRegistry().getImplementation(IContentManager.class);

        return contentManager;
    }

    /**
     * Inspector for the 'creationDate' attribute.
     * 
     * @return the creationDate value
     */
    public Date getCreationDate()
    {
        return creationDate;
    }

    /**
     * Inspector for the 'creatorID' attribute.
     * 
     * @return the creatorID value
     */
    public String getCreatorID()
    {
        return creatorID;
    }

    /**
     * Inspector for the 'description' attribute.
     * 
     * @return the description value
     */
    public String getDescription()
    {
        return description;
    }

    /**
     * Inspector for the 'id' attribute.
     * 
     * @return the id value
     */
    public String getId()
    {
        return id;
    }

    /**
     * Inspector for the 'name' attribute.
     * 
     * @return the name value
     */
    public String getName()
    {
        return name;
    }

    /**
     * @return the parent Node
     * @throws IdentityManagerException
     * @throws ContentManagerException
     */
    public Node getParentNode() throws IdentityManagerException, ContentManagerException
    {
        if (isDeleted)
            throw new AlreadyDeletedException();

        return getContentManager().getNodeById(parentNodeId, user);
    }

    /**
     * Inspector for the 'parentNodeId' attribute.
     * 
     * @return the parentNodeId value
     */
    public Long getParentNodeId()
    {
        return parentNodeId;
    }

    /**
     * Inspector for the 'userId' attribute.
     * 
     * @return the userId value
     */
    public IDIFUser getUser()
    {
        return user;
    }

    /**
     * Grants ContentItem access to group. Grant access is done immediately
     * 
     * @param groupId
     *            the group to grant access
     * @return T if grant was succeeded
     * @throws ContentManagerException
     */
    public boolean grantAccessGroup(String groupId) throws ContentManagerException
    {
        ACLEntry entry = null;

        if (isDeleted)
            throw new AlreadyDeletedException();

        boolean ret = getContentManager().grantContentAccessToGroup(id, groupId);

        if (ret && accessList != null)
        {
            entry = new ACLEntry();
            entry.setGroupID(groupId);
            accessList.add(entry);
        }

        return ret;
    }

    /**
     * Grants ContentItem access to user. Grant access is done immediately
     * 
     * @param user
     *            the user to grant access
     * @return T if grant was succeeded
     * @throws ContentManagerException
     */
    public boolean grantAccessUser(IDIFUser user) throws ContentManagerException
    {
        ACLEntry entry = null;

        if (isDeleted)
            throw new AlreadyDeletedException();

        boolean ret = getContentManager().grantContentAccessToUser(id, user);

        if (ret && accessList != null)
        {
            entry = new ACLEntry();
            entry.setUserID(user.getID());
            accessList.add(entry);
        }

        return ret;
    }

    /**
     * Inspector for the 'isDeleted' attribute.
     * 
     * @return the isDeleted value
     */
    public boolean isDeleted()
    {
        return isDeleted;
    }

    /**
     * Revokes ContentItem access to group. revoke access is done immediately
     * 
     * @param groupId
     *            the group to revoke access
     * @return T if revoke was succeeded
     * @throws ContentManagerException
     */
    public boolean revokeAccessGroup(String groupId) throws ContentManagerException
    {
        if (isDeleted)
            throw new AlreadyDeletedException();

        boolean ret = getContentManager().revokeContentAccessToGroup(id, groupId);

        if (ret && accessList != null)
        {
            List<ACLEntry> newList = new ArrayList<ACLEntry>();
            for (ACLEntry entry: accessList)
            {
                if (entry.isUserEntry() || !entry.getGroupID().equals(groupId))
                    newList.add(entry);
            }
            accessList = newList;
        }

        return ret;
    }

    /**
     * Revokes ContentItem access to group. revoke access is done immediately
     * 
     * @param user
     *            the user to revoke access
     * @return T if revoke was succeeded
     * @throws ContentManagerException
     */
    public boolean revokeAccessUser(IDIFUser user) throws ContentManagerException
    {
        if (isDeleted)
            throw new AlreadyDeletedException();

        boolean ret = getContentManager().revokeContentAccessToUser(id, user);

        if (ret && accessList != null)
        {
            List<ACLEntry> newList = new ArrayList<ACLEntry>();
            for (ACLEntry entry: accessList)
            {
                if (entry.isGroupEntry() || !entry.getUserID().equals(user.getID()))
                    newList.add(entry);
            }
            accessList = newList;
        }

        return ret;
    }

    /**
     * Modifier for the 'content' attribute.
     * 
     * @param content
     *            the new content value to set
     */
    public void setContent(String content)
    {
        this.content = content;
    }

    /**
     * Modifier for the 'creationDate' attribute.
     * 
     * @param creationDate
     *            the new creationDate value to set
     */
    public void setCreationDate(Date creationDate)
    {
        this.creationDate = creationDate;
    }

    /**
     * Modifier for the 'creatorID' attribute.
     * 
     * @param creatorID
     *            the new creatorID value to set
     */
    public void setCreatorID(String creatorID)
    {
        this.creatorID = creatorID;
    }

    /**
     * Modifier for the 'description' attribute.
     * 
     * @param description
     *            the new description value to set
     */
    public void setDescription(String description)
    {
        this.description = description;
    }

    /**
     * Modifier for the 'id' attribute.
     * 
     * @param id
     *            the new id value to set
     */
    public void setId(String id)
    {
        this.id = id;
    }

    /**
     * Modifier for the 'name' attribute.
     * 
     * @param name
     *            the new name value to set
     */
    public void setName(String name)
    {
        this.name = name;
    }

    /**
     * Sets the 'userId' attribute.
     * 
     * @param user
     *            the user currently using the object
     */
    public void setUserId(IDIFUser user)
    {
        this.user = user;
    }

    /**
     * Updates the ContentItem with current object values. If the ContentItem doesn't exist than it is inserted
     * 
     * @return T if update was succeeded
     * @throws IdentityManagerException
     * @throws ContentManagerException
     */
    public boolean update() throws IdentityManagerException, ContentManagerException
    {
        ContentItem contentItem = null;
        if (isDeleted || id == null)
        {
            contentItem = getContentManager().addContent(this);
            id = contentItem.getId();
        }
        else
        {
            try
            {
                contentItem = getContentManager().updateContent(this);
            }
            catch (ContentItemNotFoundException e)
            {
                contentItem = getContentManager().addContent(this);
                id = contentItem.getId();
            }
        }
        return true;
    }

}
