/**
 * 2007, Digitalis Informatica. All rights reserved. Distribuicao e Gestao de Informatica, Lda. Estrada de Paco de Arcos
 * num.9 - Piso -1 2780-666 Paco de Arcos Telefone: (351) 21 4408990 Fax: (351) 21 4408999 http://www.digitalis.pt
 */

package pt.digitalis.dif.utils.identity.tests;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import junit.framework.TestCase;
import pt.digitalis.dif.controller.security.managers.IIdentityManager;
import pt.digitalis.dif.controller.security.managers.impl.IdentityManagerStaticImpl;
import pt.digitalis.dif.controller.security.objects.DIFGroupImpl;
import pt.digitalis.dif.controller.security.objects.DIFUserImpl;
import pt.digitalis.dif.controller.security.objects.IDIFGroup;
import pt.digitalis.dif.controller.security.objects.IDIFUser;
import pt.digitalis.dif.exception.InternalFrameworkException;
import pt.digitalis.dif.exception.security.IdentityManagerException;
import pt.digitalis.dif.ioc.DIFIoCRegistry;
import pt.digitalis.dif.utils.Pagination;

/**
 * Defines unit tests for the identity manager implementation.
 * 
 * @author Luis Pinto <a href="mailto:lpinto@digitalis.pt">lpinto@digitalis.pt</a><br/>
 * @created 2012/10/30
 */
public abstract class AbstractIdentityManagerImplTest extends TestCase {

    /** A bulk parameter name. */
    final static protected String A_BULK_ATTRIBUTE_NAME = "aBulkAttributeName";

    /** A bulk parameter value. */
    final static protected String A_BULK_ATTRIBUTE_VALUE = "aBulkAttributeValue";

    /** A changed attribute value. */
    final static protected String A_CHANGED_ATTRIBUTE_VALUE = "aChangedAttributeValue";

    /** A changed bulk parameter value. */
    final static protected String A_CHANGED_BULK_ATTRIBUTE_VALUE = "aChangedBulkAttributeValue";

    /** An attribute name. */
    final static protected String AN_ATTRIBUTE_NAME = "anAttributeName";

    /** An attribute value. */
    final static protected String AN_ATTRIBUTE_VALUE = "anAttributeValue";

    /** Another attribute name. */
    final static protected String ANOTHER_ATTRIBUTE_NAME = "anotherAttributeName";

    /** Another parameter value. */
    final static protected String ANOTHER_ATTRIBUTE_VALUE = "anotherAttributeValue";

    /** Another bulk parameter name. */
    final static protected String ANOTHER_BULK_ATTRIBUTE_NAME = "anotherBulkAttributeName";

    /** Another bulk parameter value. */
    final static protected String ANOTHER_BULK_ATTRIBUTE_VALUE = "anotherBulkAttributeValue";

    /** Another changed parameter value. */
    final static protected String ANOTHER_CHANGED_ATTRIBUTE_VALUE = "anotherChangedAttributeValue";

    /** Another changed bulk parameter value. */
    final static protected String ANOTHER_CHANGED_BULK_PARAMETER_VALUE = "anotherChangedBulkAttributeValue";

    /** The new user ID. */
    final static private String ATTRIBUTE_KEY = "key";

    /**  */
    final static private String ATTRIBUTE_VALUE = "some attribute value";

    /** A bad group ID. */
    final static private String BAD_GROUP = "badGroup";

    /** A bad password. */
    final static private String BAD_PASS = "badPass";

    /** A bad username. */
    final static private String BAD_USER = "badUser";

    /** Changed group description. */
    final static private String CHANGED_DESC = "changedGroupDesc";

    /** Changed group name. */
    final static private String CHANGED_GROUP_NAME = "newGroup";

    /** The changed mail. */
    final static private String CHANGED_MAIL = "changed@mail.pt";

    /** Changed group name. */
    final static private String CHANGED_PARENT_GROUP_NAME = "childGroup";

    /** A changed password. */
    final static private String CHANGED_PASS = "changedPass";

    /** Changed attributes value. */
    final static private String CHANGED_USER = "changedUser";

    /** A child group ID. */
    final static private String CHILD_GROUP_1_ID = "childGroup1ID";

    /** Another child group ID. */
    final static private String CHILD_GROUP_2_ID = "childGroup2ID";

    /** Group description. */
    final static private String DESC = "aGroupDesc";

    /** A parent group ID. */
    final static private String GRAND_PARENT_GROUP_ID = "grandParentGroupID";

    /** The 'N/A' String literal. */
    final static public String N_A = "N/A";

    /** New group name. */
    final static private String NEW_GROUP_NAME = "newGroup";

    /** The new user ID. */
    final static private String NEW_USER = "newUser";

    /** A parent group ID. */
    final static private String PARENT_GROUP_ID = "parentGroupID";

    /** The user mail. */
    final static private String USER_MAIL = "user@mail.pt";

    /** The user password. */
    final static private String USER_PASS = "passwd";

    /** The test object. */
    protected IIdentityManager identityManagerImpl = null;

    /** Main group name. */
    protected String MAIN_GROUP_NAME = "mainGroup";

    /** Test group 1. */
    private String TEST_GRP_1_ID = "testGroup1";

    /** Test group 2. */
    private String TEST_GRP_2_ID = "testGroup2";

    /** Test group 3. */
    private String TEST_GRP_3_ID = "testGroup3";

    /** Test user 1. */
    private String TEST_USER_1_ID = "testUser1";

    /** Test user 2. */
    private String TEST_USER_2_ID = "testUser2";

    /** Test user 3. */
    private String TEST_USER_3_ID = "testUser3";

    /**
     * Creates a group and adds it to the identity manager.
     * 
     * @throws IdentityManagerException
     *             if the user can't be created
     */
    protected void createGroup() throws IdentityManagerException
    {
        createGroup(null);
    }

    /**
     * Creates a group and adds it to the identity manager.
     * 
     * @param groupName
     * @throws IdentityManagerException
     *             if the user can't be created
     */
    protected void createGroup(String groupName) throws IdentityManagerException
    {
        createGroup(groupName, MAIN_GROUP_NAME);
    }

    /**
     * Creates a group and adds it to the identity manager.
     * 
     * @param groupName
     *            the group name
     * @param parentGroup
     *            the group parent group
     * @throws IdentityManagerException
     *             if the user can't be created
     */
    protected void createGroup(String groupName, String parentGroup) throws IdentityManagerException
    {
        String finalGroupName = groupName;
        if (finalGroupName == null || "".equals(finalGroupName))
        {
            finalGroupName = NEW_GROUP_NAME;
        }

        // Create group for addition
        IDIFGroup transferGroup = new DIFGroupImpl();
        transferGroup.setID(finalGroupName);
        transferGroup.setName(finalGroupName);
        transferGroup.setDescription(DESC);
        if (!finalGroupName.equals(parentGroup))
        {
            transferGroup.setParentGroupID(parentGroup);

            if (!identityManagerImpl.groupExists(parentGroup))
            {
                createGroup(parentGroup);
            }
        }

        // Add group if it doesn't exist and verify it's existence
        if (identityManagerImpl.groupExists(finalGroupName))
        {
            removeGroup(finalGroupName);
            assertFalse(identityManagerImpl.groupExists(finalGroupName));
        }
        identityManagerImpl.addGroup(transferGroup);
        assertTrue(identityManagerImpl.groupExists(finalGroupName));
    }

    /**
     * Creates a user and adds it to the identity manager.
     * 
     * @throws IdentityManagerException
     *             if the user can't be created
     * @throws InternalFrameworkException
     */
    protected void createUser() throws IdentityManagerException, InternalFrameworkException
    {
        this.createUser(NEW_USER);
    }

    /**
     * Creates a user and adds it to the identity manager.
     * 
     * @param userID
     *            the user ID
     * @throws IdentityManagerException
     *             if the user can't be created
     * @throws InternalFrameworkException
     */
    protected void createUser(String userID) throws IdentityManagerException, InternalFrameworkException
    {
        // Create user for addition
        IDIFUser newUser = new DIFUserImpl();
        newUser.setID(userID);
        newUser.setName(userID);
        newUser.setNick(userID);
        newUser.setEmail(USER_MAIL);
        newUser.setPassword(USER_PASS);
        newUser.setProfileID(MAIN_GROUP_NAME);

        if (!identityManagerImpl.groupExists(MAIN_GROUP_NAME))
        {
            createGroup(MAIN_GROUP_NAME);
        }

        // Add user if it doesn't exist and verify it's existence
        if (identityManagerImpl.userExists(userID))
        {
            identityManagerImpl.removeUser(userID);
        }

        assertFalse(identityManagerImpl.userExists(userID));

        identityManagerImpl.addUser(newUser);
        newUser.setAttribute(ATTRIBUTE_KEY, ATTRIBUTE_VALUE);

        assertTrue(identityManagerImpl.userExists(userID));
    }

    /**
     * Removes the group NEW_GROUP_NAME from the identity manager.
     * 
     * @throws IdentityManagerException
     *             if the user can't be created
     */
    protected void removeGroup() throws IdentityManagerException
    {

        removeGroup(NEW_GROUP_NAME);
    }

    /**
     * Removes a group from the identity manager.
     * 
     * @param groupId
     *            the group ID to remove
     * @throws IdentityManagerException
     */
    protected void removeGroup(String groupId) throws IdentityManagerException
    {
        for (String groupID: identityManagerImpl.getGroupGroups(groupId).keySet())
        {
            removeGroup(groupID);
        }

        // Remove group and assure it's been removed
        if (identityManagerImpl.groupExists(groupId))
        {
            // First remove any pending test users
            for (String userID: identityManagerImpl.getGroup(groupId).getUserIDs())
                identityManagerImpl.removeUser(userID);

            identityManagerImpl.removeGroup(groupId);
        }
        assertFalse(identityManagerImpl.groupExists(groupId));
    }

    /**
     * Removes a user from the identity manager.
     * 
     * @param userId
     *            the id of the user to remove
     * @throws IdentityManagerException
     *             if the user can't be created
     */
    protected void removeUser(String userId) throws IdentityManagerException
    {
        // Remove user and assure it's been removed
        identityManagerImpl.removeUser(userId);
        assertFalse(identityManagerImpl.userExists(userId));
    }

    /**
     * @see junit.framework.TestCase#setUp()
     */
    @Override
    protected void setUp() throws Exception
    {
        System.out.print(".");
        super.setUp();
    }

    /**
     * @see junit.framework.TestCase#tearDown()
     */
    @Override
    public void tearDown()
    {
        System.out.println(".");
        try
        {
            if (identityManagerImpl.userExists(NEW_USER))
                removeUser(NEW_USER);
            if (identityManagerImpl.userExists(CHANGED_USER))
                removeUser(CHANGED_USER);
            if (identityManagerImpl.userExists(TEST_USER_1_ID))
                removeUser(TEST_USER_1_ID);
            if (identityManagerImpl.userExists(TEST_USER_2_ID))
                removeUser(TEST_USER_2_ID);
            if (identityManagerImpl.userExists(TEST_USER_3_ID))
                removeUser(TEST_USER_3_ID);
            /*
             * if (identityManagerImpl.groupExists(NEW_GROUP_NAME)) identityManagerImpl.removeGroup(NEW_GROUP_NAME); if
             * (identityManagerImpl.groupExists(TEST_GRP_1_ID)) identityManagerImpl.removeGroup(TEST_GRP_1_ID); if
             * (identityManagerImpl.groupExists(TEST_GRP_2_ID)) identityManagerImpl.removeGroup(TEST_GRP_2_ID); if
             * (identityManagerImpl.groupExists(TEST_GRP_3_ID)) identityManagerImpl.removeGroup(TEST_GRP_3_ID); if
             * (identityManagerImpl.groupExists(PARENT_GROUP_ID)) identityManagerImpl.removeGroup(PARENT_GROUP_ID); if
             * (identityManagerImpl.groupExists(CHILD_GROUP_1_ID)) identityManagerImpl.removeGroup(CHILD_GROUP_1_ID); if
             * (identityManagerImpl.groupExists(CHILD_GROUP_2_ID)) identityManagerImpl.removeGroup(CHILD_GROUP_2_ID); if
             * (identityManagerImpl.groupExists(CHANGED_PARENT_GROUP_NAME))
             * identityManagerImpl.removeGroup(CHANGED_PARENT_GROUP_NAME);
             */

            if (identityManagerImpl.groupExists(MAIN_GROUP_NAME))
                removeGroup(MAIN_GROUP_NAME);

        }
        catch (IdentityManagerException identityManagerException)
        {
            identityManagerException.printStackTrace();
        }

        identityManagerImpl.resetIdentityManager();
        identityManagerImpl = null;
    }

    /**
     * Tests if the obtainance of all groups in the system work correctly
     * 
     * @throws IdentityManagerException
     */
    public void testFindAllGroups() throws IdentityManagerException
    {
        IDIFGroup group1 = DIFIoCRegistry.getRegistry().getImplementation(IDIFGroup.class);
        IDIFGroup group2 = DIFIoCRegistry.getRegistry().getImplementation(IDIFGroup.class);
        IDIFGroup group3 = DIFIoCRegistry.getRegistry().getImplementation(IDIFGroup.class);

        int numElements = identityManagerImpl.getAllGroups().size();

        assertEquals(numElements, identityManagerImpl.countAllGroups());

        group1.setID(TEST_GRP_1_ID);
        if (identityManagerImpl.getGroup(TEST_GRP_1_ID) == null)
        {
            identityManagerImpl.addGroup(group1);
            numElements++;
        }

        group2.setID(TEST_GRP_2_ID);
        if (identityManagerImpl.getGroup(TEST_GRP_2_ID) == null)
        {
            identityManagerImpl.addGroup(group2);
            numElements++;
        }

        group3.setID(TEST_GRP_3_ID);
        if (identityManagerImpl.getGroup(TEST_GRP_3_ID) == null)
        {
            identityManagerImpl.addGroup(group3);
            numElements++;
        }

        Set<IDIFGroup> allGroups = identityManagerImpl.getAllGroups();

        assertTrue(allGroups.contains(group1));
        assertTrue(allGroups.contains(group2));
        assertTrue(allGroups.contains(group3));
        assertEquals(allGroups.size(), numElements);

        // TODO: Waiting for Pagination bug resolution
        // Pagination pagination = new Pagination(1, numElements);
        // assertEquals(identityManagerImpl.getGroups(pagination).size(), numElements);

        // TODO: Waiting for Pagination bug resolution
        // pagination = new Pagination(1, 2);
        // assertEquals(identityManagerImpl.getGroups(pagination).size(), 2);

        // TODO: Waiting for Pagination bug resolution
        // for (int i = 1; i <= numElements; i++)
        // {
        // pagination = new Pagination(i, 1);
        // assertEquals(identityManagerImpl.getGroups(pagination).size(), 1);
        // }

        identityManagerImpl.removeGroup(TEST_GRP_1_ID);
        numElements--;
        identityManagerImpl.removeGroup(TEST_GRP_2_ID);
        numElements--;
        identityManagerImpl.removeGroup(TEST_GRP_3_ID);
        numElements--;

        assertFalse(identityManagerImpl.getAllGroups().contains(group1));
        assertFalse(identityManagerImpl.getAllGroups().contains(group2));
        assertFalse(identityManagerImpl.getAllGroups().contains(group3));
        assertEquals(identityManagerImpl.getAllGroups().size(), numElements);
    }

    /**
     * Tests if the obtainance of all users in the system work correctly
     * 
     * @throws IdentityManagerException
     */
    public void testFindllUsers() throws IdentityManagerException
    {
        createGroup(MAIN_GROUP_NAME);

        IDIFUser testUser1 = new DIFUserImpl();
        testUser1.setID(TEST_USER_1_ID);
        testUser1.setName(TEST_USER_1_ID);
        testUser1.setNick(TEST_USER_1_ID);
        testUser1.setEmail(USER_MAIL);
        testUser1.setPassword(USER_PASS);
        testUser1.setProfileID(MAIN_GROUP_NAME);

        IDIFUser testUser2 = new DIFUserImpl();
        testUser2.setID(TEST_USER_2_ID);
        testUser2.setName(TEST_USER_2_ID);
        testUser2.setNick(TEST_USER_2_ID);
        testUser2.setEmail(USER_MAIL);
        testUser2.setPassword(USER_PASS);
        testUser2.setProfileID(MAIN_GROUP_NAME);

        IDIFUser testUser3 = new DIFUserImpl();
        testUser3.setID(TEST_USER_3_ID);
        testUser3.setName(TEST_USER_3_ID);
        testUser3.setNick(TEST_USER_3_ID);
        testUser3.setEmail(USER_MAIL);
        testUser3.setPassword(USER_PASS);
        testUser3.setProfileID(MAIN_GROUP_NAME);

        int numElements = identityManagerImpl.getAllUsers().size();

        assertEquals(numElements, identityManagerImpl.countAllUsers());

        identityManagerImpl.addUser(testUser1);
        identityManagerImpl.addUser(testUser2);
        identityManagerImpl.addUser(testUser3);

        numElements += 3;

        assertTrue(identityManagerImpl.getAllUsers().contains(testUser1));
        assertTrue(identityManagerImpl.getAllUsers().contains(testUser2));
        assertTrue(identityManagerImpl.getAllUsers().contains(testUser3));
        assertEquals(identityManagerImpl.getAllUsers().size(), numElements);

        // TODO: Waiting for Pagination bug resolution
        // Pagination pagination = new Pagination(1, numElements);
        // assertEquals(identityManagerImpl.getUsers(pagination).size(), numElements);

        // TODO: Waiting for Pagination bug resolution
        // for (int i = 1; i <= numElements; i++)
        // {
        // pagination = new Pagination(i, 1);
        // assertEquals(identityManagerImpl.getUsers(pagination).size(), 1);
        // }

        identityManagerImpl.removeUser(TEST_USER_1_ID);
        identityManagerImpl.removeUser(TEST_USER_2_ID);
        identityManagerImpl.removeUser(TEST_USER_3_ID);

        numElements -= 3;

        assertFalse(identityManagerImpl.getAllUsers().contains(testUser1));
        assertFalse(identityManagerImpl.getAllUsers().contains(testUser2));
        assertFalse(identityManagerImpl.getAllUsers().contains(testUser3));
        assertEquals(identityManagerImpl.getAllUsers().size(), numElements);

    }

    /**
     * Test the find user groups methods
     */
    public void testFindUserGroups()
    {
        final String GROUP1 = "testFindUserGroupsGroup1";
        final String GROUP2 = "testFindUserGroupsGroup2";
        final String GROUP3 = "testFindUserGroupsGroup3";
        try
        {
            createUser();

            createGroup(GRAND_PARENT_GROUP_ID);
            createGroup(PARENT_GROUP_ID, GRAND_PARENT_GROUP_ID);

            createGroup(GROUP1);
            createGroup(GROUP2);
            createGroup(GROUP3, PARENT_GROUP_ID);

            identityManagerImpl.addUserToGroup(NEW_USER, GROUP1);
            identityManagerImpl.addUserToGroup(NEW_USER, GROUP2);
            identityManagerImpl.addUserToGroup(NEW_USER, GROUP3);

            int totalGroups = identityManagerImpl.countAllGroupsOfUser(NEW_USER);
            assertEquals(totalGroups, 4);

            totalGroups = identityManagerImpl.countAllGroupsOfUser(NEW_USER, true);
            assertEquals(totalGroups, 6);

            totalGroups = identityManagerImpl.countAllGroupsOfUser(NEW_USER, false);
            assertEquals(totalGroups, 4);

            Set<String> groupIDs = identityManagerImpl.getUserGroupsIDs(NEW_USER);
            assertEquals(groupIDs.size(), 4);
            assertTrue(groupIDs.contains(GROUP1));
            assertTrue(groupIDs.contains(GROUP2));
            assertTrue(groupIDs.contains(GROUP3));
            assertFalse(groupIDs.contains(PARENT_GROUP_ID));
            assertFalse(groupIDs.contains(GRAND_PARENT_GROUP_ID));
            assertTrue(groupIDs.contains(MAIN_GROUP_NAME));

            groupIDs = identityManagerImpl.getUserGroupsIDs(NEW_USER, true);
            assertEquals(groupIDs.size(), 6);
            assertTrue(groupIDs.contains(GROUP1));
            assertTrue(groupIDs.contains(GROUP2));
            assertTrue(groupIDs.contains(GROUP3));
            assertTrue(groupIDs.contains(PARENT_GROUP_ID));
            assertTrue(groupIDs.contains(GRAND_PARENT_GROUP_ID));
            assertTrue(groupIDs.contains(MAIN_GROUP_NAME));

            groupIDs = identityManagerImpl.getUserGroupsIDs(NEW_USER, false);
            assertEquals(groupIDs.size(), 4);
            assertTrue(groupIDs.contains(GROUP1));
            assertTrue(groupIDs.contains(GROUP2));
            assertTrue(groupIDs.contains(GROUP3));
            assertFalse(groupIDs.contains(PARENT_GROUP_ID));
            assertFalse(groupIDs.contains(GRAND_PARENT_GROUP_ID));
            assertTrue(groupIDs.contains(MAIN_GROUP_NAME));

            Map<String, IDIFGroup> groups = identityManagerImpl.getUserGroups(NEW_USER);
            assertEquals(groups.size(), 4);
            assertTrue(groups.containsKey(GROUP1));
            assertTrue(groups.containsKey(GROUP2));
            assertTrue(groups.containsKey(GROUP3));
            assertFalse(groups.containsKey(PARENT_GROUP_ID));
            assertFalse(groups.containsKey(GRAND_PARENT_GROUP_ID));
            assertTrue(groups.containsKey(MAIN_GROUP_NAME));

            groups = identityManagerImpl.getUserGroups(NEW_USER, true);
            assertEquals(groups.size(), 6);
            assertTrue(groups.containsKey(GROUP1));
            assertTrue(groups.containsKey(GROUP2));
            assertTrue(groups.containsKey(GROUP3));
            assertTrue(groups.containsKey(PARENT_GROUP_ID));
            assertTrue(groups.containsKey(GRAND_PARENT_GROUP_ID));
            assertTrue(groups.containsKey(MAIN_GROUP_NAME));

            groups = identityManagerImpl.getUserGroups(NEW_USER, false);
            assertEquals(groups.size(), 4);
            assertTrue(groups.containsKey(GROUP1));
            assertTrue(groups.containsKey(GROUP2));
            assertTrue(groups.containsKey(GROUP3));
            assertFalse(groups.containsKey(PARENT_GROUP_ID));
            assertFalse(groups.containsKey(GRAND_PARENT_GROUP_ID));
            assertTrue(groups.containsKey(MAIN_GROUP_NAME));

            List<String> groupList = new ArrayList<String>();

            Pagination page = new Pagination(1, 2);
            groups = identityManagerImpl.getUserGroupsPagination(NEW_USER, page);
            assertEquals(groups.size(), 2);
            groupList.addAll(groups.keySet());

            page = new Pagination(2, 2);
            assertEquals(page.getEndRow(), 4);
            assertEquals(page.getPage(), 2);

            groups = identityManagerImpl.getUserGroupsPagination(NEW_USER, page);

            // Only one group exists because the getUserGroupsPagination doesn't consider the profile as a group
            assertEquals(groups.size(), 1);
            groupList.addAll(groups.keySet());

            assertEquals(groupList.size(), 3);
            assertTrue(groupList.contains(GROUP1));
            assertTrue(groupList.contains(GROUP2));
            assertTrue(groupList.contains(GROUP3));
            assertFalse(groupList.contains(MAIN_GROUP_NAME));
            assertFalse(groupList.contains(PARENT_GROUP_ID));

            assertEquals(identityManagerImpl.countAllUsers(GROUP1), 1);
            assertEquals(identityManagerImpl.countAllUsers(GROUP2), 1);
            assertEquals(identityManagerImpl.countAllUsers(GROUP3), 1);
            assertEquals(identityManagerImpl.countAllUsers(MAIN_GROUP_NAME), 0);
            assertEquals(identityManagerImpl.countAllUsers(PARENT_GROUP_ID), 0);
        }
        catch (IdentityManagerException e)
        {
            e.printStackTrace();
            assertFalse(true);
        }
        catch (InternalFrameworkException e)
        {
            e.printStackTrace();
            assertFalse(true);
        }
    }

    /**
     * Tests searching for groups with a given parent.
     * 
     * @throws IdentityManagerException
     *             if the identity manager can't be accessed
     */
    public void testGroupGroups() throws IdentityManagerException
    {
        createGroup(GRAND_PARENT_GROUP_ID);
        createGroup(PARENT_GROUP_ID, GRAND_PARENT_GROUP_ID);
        createGroup(CHILD_GROUP_1_ID, PARENT_GROUP_ID);
        createGroup(CHILD_GROUP_2_ID, PARENT_GROUP_ID);

        assertTrue(identityManagerImpl.groupExists(PARENT_GROUP_ID));
        assertTrue(identityManagerImpl.groupExists(CHILD_GROUP_1_ID));
        assertTrue(identityManagerImpl.groupExists(CHILD_GROUP_2_ID));

        Map<String, IDIFGroup> groupGroups = identityManagerImpl.getGroupGroups(GRAND_PARENT_GROUP_ID);
        assertNotNull(groupGroups);
        assertEquals(groupGroups.size(), 3);
        assertFalse(groupGroups.containsKey(GRAND_PARENT_GROUP_ID));
        assertTrue(groupGroups.containsKey(PARENT_GROUP_ID));
        assertTrue(groupGroups.containsKey(CHILD_GROUP_1_ID));
        assertTrue(groupGroups.containsKey(CHILD_GROUP_2_ID));
        assertEquals(groupGroups.get(CHILD_GROUP_1_ID).getParentGroupID(), PARENT_GROUP_ID);
        assertEquals(groupGroups.get(CHILD_GROUP_2_ID).getParentGroupID(), PARENT_GROUP_ID);

        identityManagerImpl.removeGroup(CHILD_GROUP_1_ID);
        identityManagerImpl.removeGroup(CHILD_GROUP_2_ID);
        identityManagerImpl.removeGroup(PARENT_GROUP_ID);
        identityManagerImpl.removeGroup(GRAND_PARENT_GROUP_ID);

        assertFalse(identityManagerImpl.groupExists(CHILD_GROUP_1_ID));
        assertFalse(identityManagerImpl.groupExists(CHILD_GROUP_2_ID));
        assertFalse(identityManagerImpl.groupExists(PARENT_GROUP_ID));
        assertFalse(identityManagerImpl.groupExists(GRAND_PARENT_GROUP_ID));
    }

    /**
     * Tests the group operations.
     */
    public void testGroupOperations()
    {
        try
        {
            createGroup();

            // Test group integrity
            IDIFGroup retrievedGroup = identityManagerImpl.getGroup(NEW_GROUP_NAME);
            assertEquals(retrievedGroup.getID(), NEW_GROUP_NAME);
            assertEquals(retrievedGroup.getName(), NEW_GROUP_NAME);
            assertEquals(retrievedGroup.getDescription(), DESC);
            assertTrue(retrievedGroup.getParentGroupID().contains(MAIN_GROUP_NAME));

            createGroup(CHANGED_PARENT_GROUP_NAME);

            // Set group data for modification
            IDIFGroup group = new DIFGroupImpl();
            group.setID(CHANGED_GROUP_NAME);
            group.setName(CHANGED_GROUP_NAME);
            group.setDescription(CHANGED_DESC);
            group.setParentGroupID(CHANGED_PARENT_GROUP_NAME);

            // Modify group data
            identityManagerImpl.updateGroup(group);
            assertTrue(identityManagerImpl.groupExists(NEW_GROUP_NAME));

            // Test group integrity
            retrievedGroup = identityManagerImpl.getGroup(CHANGED_GROUP_NAME);
            assertEquals(retrievedGroup.getID(), CHANGED_GROUP_NAME);
            assertEquals(retrievedGroup.getName(), CHANGED_GROUP_NAME);
            assertEquals(retrievedGroup.getDescription(), CHANGED_DESC);
            assertTrue(retrievedGroup.getParentGroupID().contains(CHANGED_PARENT_GROUP_NAME));

            removeGroup();

        }
        catch (IdentityManagerException e)
        {
            e.printStackTrace();
        }
    }

    /**
     * Tests the group operations forcing failure behavior.
     * 
     * @throws IdentityManagerException
     *             if the identity manager can't be accessed
     */
    @SuppressWarnings("unused")
    public void testGroupOperationsInFailureBehaviour() throws IdentityManagerException
    {
        // Create group for addition
        IDIFGroup transferGroup = new DIFGroupImpl();

        // Tries to group an user without id...must fail!!
        try
        {
            identityManagerImpl.addGroup(transferGroup);
        }
        catch (IdentityManagerException e)
        {
            // User must not have been added...
            assertFalse(identityManagerImpl.groupExists(""));
        }

        // Try to fetch data for an inexistent group
        try
        {
            IDIFGroup retrievedGroup = identityManagerImpl.getGroup(BAD_GROUP);
        }
        catch (IdentityManagerException identityManagerException)
        {
        }

        // Tries to modify a bad group...must fail!!
        try
        {
            identityManagerImpl.updateGroup(transferGroup);
        }
        catch (IdentityManagerException e)
        {
            // User must not have been added...
            assertFalse(identityManagerImpl.groupExists(BAD_GROUP));
        }

        // Tries to remove a bad user...must fail!!
        try
        {
            identityManagerImpl.removeGroup(BAD_GROUP);
        }
        catch (IdentityManagerException e)
        {
            // User must not exist
            assertFalse(identityManagerImpl.groupExists(BAD_GROUP));
        }
    }

    /**
     * Tests the illegal operations.
     * 
     * @throws InternalFrameworkException
     */
    public void testIllegalOperations() throws InternalFrameworkException
    {

        IDIFGroup badGroup = null;
        IDIFUser badUser = null;

        // Force exception throwing on addGroup
        try
        {
            createGroup();
            badGroup = new DIFGroupImpl();
            badGroup.setID(NEW_GROUP_NAME);
            identityManagerImpl.addGroup(badGroup);
        }
        catch (IdentityManagerException e)
        {
        }
        finally
        {
            try
            {
                removeGroup();
            }
            catch (IdentityManagerException e1)
            {
                e1.printStackTrace();
            }
        }

        // Force exception throwing on addUser
        try
        {
            createUser();
            badUser = new DIFUserImpl();
            badUser.setID(NEW_USER);
            identityManagerImpl.addUser(badUser);
        }
        catch (IdentityManagerException e)
        {
        }
        finally
        {
            try
            {
                removeUser(NEW_USER);
            }
            catch (IdentityManagerException e1)
            {
                e1.printStackTrace();
            }
        }

        // Receive null group
        try
        {
            identityManagerImpl.getGroup(null);
        }
        catch (IdentityManagerException e)
        {
        }

        // Force exception throwing on convertFromXXXUserToDIFUser
        try
        {
            final String BAD_USER_ID = "BAD_USER";
            identityManagerImpl.updateUserAttribute(BAD_USER_ID, AN_ATTRIBUTE_NAME, new Object());
            assertTrue(false);
        }
        catch (IdentityManagerException e)
        {
            assertTrue(true);
        }

    }

    /**
     * Tests the static Identity Manager Initialization
     */
    public void testInitialization()
    {
        // executions for increasing code coverage purpose
        identityManagerImpl.getExclusionCharaters();
        identityManagerImpl.getGroupAttributeName();
        identityManagerImpl.gatherManagedAttributes();
        assertFalse(identityManagerImpl.isReadOnly());

        int managedAttributes = identityManagerImpl.getManagedAttributes().size();
        identityManagerImpl.addManagedAttributes("dummy1,dummy2");
        identityManagerImpl.addManagedAttribute("dummy3");
        assertEquals(identityManagerImpl.getManagedAttributes().size(), managedAttributes + 3);

        // Only the static implementation have a initialization
        if (identityManagerImpl instanceof IdentityManagerStaticImpl)
        {
            ((IdentityManagerStaticImpl) identityManagerImpl).initializeStaticCustomUsers();

            try
            {
                IDIFUser user = identityManagerImpl.getUser("pedro");
                Map<String, IDIFGroup> groups = user.getGroups();
                assertTrue(groups.containsKey("alunos"));
                assertTrue(groups.containsKey("docentes"));
                assertEquals(user.getAttribute("cd_aluno"), "1");
                assertEquals(user.getAttribute("cd_curso"), "2");
                assertEquals(user.getAttribute("cd_docente"), "3");

                user = identityManagerImpl.getUser("viegas");
                groups = user.getGroups();
                assertTrue(groups.containsKey("alunos"));
                assertTrue(groups.containsKey("docentes"));
                assertEquals(user.getAttribute("cd_aluno"), "11");
                assertEquals(user.getAttribute("cd_curso"), "12");
                assertEquals(user.getAttribute("cd_docente"), "13");

                assertNotNull(identityManagerImpl.getGroup("alunos"));
                assertNotNull(identityManagerImpl.getGroup("candidatos"));
            }
            catch (IdentityManagerException e)
            {
                e.printStackTrace();
                assertFalse(true);
            }
        }

    }

    /**
     * Not available test condition
     * 
     * @throws IdentityManagerException
     */
    public void testNotAvailable() throws IdentityManagerException
    {
        assertNull(identityManagerImpl.getGroup(N_A));
    }

    /**
     * Tests the identity manager's relation operations.
     * 
     * @throws IdentityManagerException
     *             if the operation can't be executed
     * @throws InternalFrameworkException
     */
    public void testRelationOperations() throws IdentityManagerException, InternalFrameworkException
    {

        // Test settings
        createUser();
        createGroup(PARENT_GROUP_ID, GRAND_PARENT_GROUP_ID);
        createGroup(NEW_GROUP_NAME, PARENT_GROUP_ID);

        // Add user to group
        identityManagerImpl.addUserToGroup(NEW_USER, NEW_GROUP_NAME);
        assertTrue(identityManagerImpl.isUserInGroup(NEW_USER, NEW_GROUP_NAME));
        assertFalse(identityManagerImpl.isUserInGroup(NEW_USER, PARENT_GROUP_ID));
        assertFalse(identityManagerImpl.isUserInGroup(NEW_USER, GRAND_PARENT_GROUP_ID));
        assertFalse(identityManagerImpl.isUserInGroup(NEW_USER, "InexistentGroupMustReturnFalse"));

        assertTrue(identityManagerImpl.isUserInGroup(NEW_USER, NEW_GROUP_NAME, true));
        assertTrue(identityManagerImpl.isUserInGroup(NEW_USER, PARENT_GROUP_ID, true));
        assertTrue(identityManagerImpl.isUserInGroup(NEW_USER, GRAND_PARENT_GROUP_ID, true));
        assertFalse(identityManagerImpl.isUserInGroup(NEW_USER, "InexistentGroupMustReturnFalse", true));

        assertTrue(identityManagerImpl.isUserInGroup(NEW_USER, NEW_GROUP_NAME, false));
        assertFalse(identityManagerImpl.isUserInGroup(NEW_USER, PARENT_GROUP_ID, false));
        assertFalse(identityManagerImpl.isUserInGroup(NEW_USER, GRAND_PARENT_GROUP_ID, false));
        assertFalse(identityManagerImpl.isUserInGroup(NEW_USER, "InexistentGroupMustReturnFalse", false));

        // Verify user is in group and group contains user
        assertTrue(identityManagerImpl.getGroupUsers(NEW_GROUP_NAME).keySet().contains(NEW_USER));
        assertTrue(identityManagerImpl.getUserIDsInGroup(NEW_GROUP_NAME).contains(NEW_USER));
        assertTrue(identityManagerImpl.getUserGroups(NEW_USER).keySet().contains(NEW_GROUP_NAME));
        assertTrue(identityManagerImpl.getUserGroupsIDs(NEW_USER).contains(NEW_GROUP_NAME));
        assertFalse(identityManagerImpl.getUserGroups(NEW_USER).keySet().contains(PARENT_GROUP_ID));
        assertFalse(identityManagerImpl.getUserGroupsIDs(NEW_USER).contains(PARENT_GROUP_ID));
        assertFalse(identityManagerImpl.getUserGroups(NEW_USER).keySet().contains(GRAND_PARENT_GROUP_ID));
        assertFalse(identityManagerImpl.getUserGroupsIDs(NEW_USER).contains(GRAND_PARENT_GROUP_ID));

        assertTrue(identityManagerImpl.getUserGroups(NEW_USER, true).keySet().contains(PARENT_GROUP_ID));
        assertTrue(identityManagerImpl.getUserGroupsIDs(NEW_USER, true).contains(PARENT_GROUP_ID));
        assertTrue(identityManagerImpl.getUserGroups(NEW_USER, true).keySet().contains(GRAND_PARENT_GROUP_ID));
        assertTrue(identityManagerImpl.getUserGroupsIDs(NEW_USER, true).contains(GRAND_PARENT_GROUP_ID));

        // Remove user from group
        identityManagerImpl.removeUserFromGroup(NEW_USER, NEW_GROUP_NAME);
        assertFalse(identityManagerImpl.isUserInGroup(NEW_USER, NEW_GROUP_NAME));

        // Clean test settings
        removeUser(NEW_USER);
        removeGroup();
    }

    /**
     * Tests the finders by attributes
     * 
     * @throws InternalFrameworkException
     * @throws IdentityManagerException
     */
    public void testUserAttributes() throws IdentityManagerException, InternalFrameworkException
    {
        Map<String, String> attribToSearch = new HashMap<String, String>();

        createUser();

        attribToSearch.put(identityManagerImpl.getMailAttributeName(), USER_MAIL);
        attribToSearch.put(identityManagerImpl.getUserLoginAttributeName(), NEW_USER);
        attribToSearch.put(identityManagerImpl.getNameAttributeName(), NEW_USER);
        attribToSearch.put(identityManagerImpl.getUserParentGroupAttributeName(), MAIN_GROUP_NAME);
        attribToSearch.put(ATTRIBUTE_KEY, ATTRIBUTE_VALUE);
        assertEquals(identityManagerImpl.getUsersByAttributes(attribToSearch).size(), 1);
        attribToSearch.clear();

        attribToSearch.put(ATTRIBUTE_KEY, ATTRIBUTE_VALUE);
        int usersWithAttrib = identityManagerImpl.countUsers(attribToSearch);

        createUser(TEST_USER_1_ID);
        createUser(TEST_USER_2_ID);
        createUser(TEST_USER_3_ID);

        usersWithAttrib += 3;

        assertEquals(identityManagerImpl.countUsers(attribToSearch), usersWithAttrib);
        assertEquals(identityManagerImpl.getUsersByAttribute(ATTRIBUTE_KEY, ATTRIBUTE_VALUE).size(), usersWithAttrib);
        assertEquals(identityManagerImpl.getUsersByAttributes(attribToSearch).size(), usersWithAttrib);

        Pagination page = new Pagination(1, 4);
        // TODO: Waiting for Pagination bug resolution
        // assertEquals(identityManagerImpl.getUsersByAttributes(attribToSearch, page).size(), 4);

        page = new Pagination(1, 2);
        assertEquals(identityManagerImpl.getUsersByAttributes(attribToSearch, page).size(), 2);

        page = new Pagination(2, 2);
        // TODO: Waiting for Pagination bug resolution
        // assertEquals(identityManagerImpl.getUsersByAttributes(attribToSearch, page).size(), 2);

        // Test find user by email
        boolean userFound = false;

        for (IDIFUser difUser: identityManagerImpl.getUsersByEmail(USER_MAIL))
            if (difUser.getID().equals(NEW_USER) && difUser.getEmail().equals(USER_MAIL))
            {
                userFound = true;
                break;
            }

        assertTrue(userFound);

        // Test find user by attribute (key)
        userFound = false;

        for (IDIFUser difUser: identityManagerImpl.getUsersByAttribute(ATTRIBUTE_KEY, ATTRIBUTE_VALUE))
            if (difUser.getID().equals(NEW_USER) && difUser.getAttribute(ATTRIBUTE_KEY).equals(ATTRIBUTE_VALUE))
            {
                userFound = true;
                break;
            }
        assertTrue(userFound);

        // Test find user by several attributes

        removeUser(NEW_USER);
    }

    /**
     * Tests the user operations.
     * 
     * @throws IdentityManagerException
     *             if the attribute values can't be changed
     * @throws InternalFrameworkException
     */
    @SuppressWarnings("serial")
    public void testUserOperations() throws IdentityManagerException, InternalFrameworkException
    {
        String USER_DEFAULT_PASSWORD = "userDefaultPassword";

        createUser(USER_DEFAULT_PASSWORD);
        assertFalse(identityManagerImpl.isIdentityValid(USER_DEFAULT_PASSWORD, "password"));
        removeUser(USER_DEFAULT_PASSWORD);

        createUser();

        // User identity MUST be valid! (This is the correct way to validate the password).
        assertTrue(identityManagerImpl.isIdentityValid(NEW_USER, USER_PASS));
        // Test user integrity
        IDIFUser retrievedUser = identityManagerImpl.getUser(NEW_USER);

        /* Test user data integrity */
        assertEquals(retrievedUser.getID(), NEW_USER);
        assertEquals(retrievedUser.getName(), NEW_USER);
        assertEquals(retrievedUser.getNick(), NEW_USER);
        assertEquals(retrievedUser.getEmail(), USER_MAIL);
        assertEquals(retrievedUser.getProfileID(), MAIN_GROUP_NAME);

        try
        {
            retrievedUser.setAttribute(AN_ATTRIBUTE_NAME, AN_ATTRIBUTE_VALUE);
        }
        catch (InternalFrameworkException internalFrameworkException)
        {
            internalFrameworkException.printStackTrace();
        }

        retrievedUser = identityManagerImpl.getUser(retrievedUser.getID());
        assertEquals(retrievedUser.getAttribute(AN_ATTRIBUTE_NAME), AN_ATTRIBUTE_VALUE);
        assertEquals(retrievedUser.getAttributes().get(AN_ATTRIBUTE_NAME), AN_ATTRIBUTE_VALUE);
        assertEquals(retrievedUser.getAttribute(AN_ATTRIBUTE_NAME.toUpperCase()), AN_ATTRIBUTE_VALUE);
        assertEquals(retrievedUser.getAttributes().get(AN_ATTRIBUTE_NAME.toUpperCase()), AN_ATTRIBUTE_VALUE);
        assertEquals(retrievedUser.getAttribute(AN_ATTRIBUTE_NAME.toLowerCase()), AN_ATTRIBUTE_VALUE);
        assertEquals(retrievedUser.getAttributes().get(AN_ATTRIBUTE_NAME.toLowerCase()), AN_ATTRIBUTE_VALUE);

        try
        {
            retrievedUser.setAttribute(ANOTHER_ATTRIBUTE_NAME, ANOTHER_ATTRIBUTE_VALUE);
        }
        catch (InternalFrameworkException internalFrameworkException)
        {
            internalFrameworkException.printStackTrace();
        }

        retrievedUser = identityManagerImpl.getUser(retrievedUser.getID());
        assertEquals(retrievedUser.getAttribute(ANOTHER_ATTRIBUTE_NAME), ANOTHER_ATTRIBUTE_VALUE);
        assertEquals(retrievedUser.getAttributes().get(ANOTHER_ATTRIBUTE_NAME), ANOTHER_ATTRIBUTE_VALUE);
        assertEquals(retrievedUser.getAttribute(ANOTHER_ATTRIBUTE_NAME.toUpperCase()), ANOTHER_ATTRIBUTE_VALUE);
        assertEquals(retrievedUser.getAttributes().get(ANOTHER_ATTRIBUTE_NAME.toUpperCase()), ANOTHER_ATTRIBUTE_VALUE);
        assertEquals(retrievedUser.getAttribute(ANOTHER_ATTRIBUTE_NAME.toLowerCase()), ANOTHER_ATTRIBUTE_VALUE);
        assertEquals(retrievedUser.getAttributes().get(ANOTHER_ATTRIBUTE_NAME.toLowerCase()), ANOTHER_ATTRIBUTE_VALUE);

        try
        {
            retrievedUser.setAttribute(A_BULK_ATTRIBUTE_NAME, A_BULK_ATTRIBUTE_VALUE);
        }
        catch (InternalFrameworkException internalFrameworkException)
        {
            internalFrameworkException.printStackTrace();
        }

        retrievedUser = identityManagerImpl.getUser(retrievedUser.getID());
        assertEquals(retrievedUser.getAttribute(A_BULK_ATTRIBUTE_NAME), A_BULK_ATTRIBUTE_VALUE);
        assertEquals(retrievedUser.getAttributes().get(A_BULK_ATTRIBUTE_NAME), A_BULK_ATTRIBUTE_VALUE);
        assertEquals(retrievedUser.getAttribute(A_BULK_ATTRIBUTE_NAME.toUpperCase()), A_BULK_ATTRIBUTE_VALUE);
        assertEquals(retrievedUser.getAttributes().get(A_BULK_ATTRIBUTE_NAME.toUpperCase()), A_BULK_ATTRIBUTE_VALUE);
        assertEquals(retrievedUser.getAttribute(A_BULK_ATTRIBUTE_NAME.toLowerCase()), A_BULK_ATTRIBUTE_VALUE);
        assertEquals(retrievedUser.getAttributes().get(A_BULK_ATTRIBUTE_NAME.toLowerCase()), A_BULK_ATTRIBUTE_VALUE);

        try
        {
            retrievedUser.setAttribute(ANOTHER_BULK_ATTRIBUTE_NAME, ANOTHER_BULK_ATTRIBUTE_VALUE);
        }
        catch (InternalFrameworkException internalFrameworkException)
        {
            internalFrameworkException.printStackTrace();
        }

        retrievedUser = identityManagerImpl.getUser(retrievedUser.getID());
        assertEquals(retrievedUser.getAttribute(ANOTHER_BULK_ATTRIBUTE_NAME), ANOTHER_BULK_ATTRIBUTE_VALUE);
        assertEquals(retrievedUser.getAttributes().get(ANOTHER_BULK_ATTRIBUTE_NAME), ANOTHER_BULK_ATTRIBUTE_VALUE);

        /* Removes the attributes one by one */
        retrievedUser.removeAttribute(AN_ATTRIBUTE_NAME);
        identityManagerImpl.updateUser(retrievedUser, retrievedUser.getID());
        retrievedUser = identityManagerImpl.getUser(retrievedUser.getID());
        assertNull(retrievedUser.getAttribute(AN_ATTRIBUTE_NAME));
        assertEquals(retrievedUser.getAttribute(ANOTHER_ATTRIBUTE_NAME), ANOTHER_ATTRIBUTE_VALUE);
        assertEquals(retrievedUser.getAttribute(A_BULK_ATTRIBUTE_NAME), A_BULK_ATTRIBUTE_VALUE);
        assertEquals(retrievedUser.getAttribute(ANOTHER_BULK_ATTRIBUTE_NAME), ANOTHER_BULK_ATTRIBUTE_VALUE);

        retrievedUser.removeAttribute(ANOTHER_ATTRIBUTE_NAME);
        identityManagerImpl.updateUser(retrievedUser, retrievedUser.getID());
        retrievedUser = identityManagerImpl.getUser(retrievedUser.getID());
        assertNull(retrievedUser.getAttribute(AN_ATTRIBUTE_NAME));
        assertNull(retrievedUser.getAttribute(ANOTHER_ATTRIBUTE_NAME));
        assertEquals(retrievedUser.getAttribute(A_BULK_ATTRIBUTE_NAME), A_BULK_ATTRIBUTE_VALUE);
        assertEquals(retrievedUser.getAttribute(ANOTHER_BULK_ATTRIBUTE_NAME), ANOTHER_BULK_ATTRIBUTE_VALUE);

        retrievedUser.removeAttribute(A_BULK_ATTRIBUTE_NAME);
        identityManagerImpl.updateUser(retrievedUser, retrievedUser.getID());
        retrievedUser = identityManagerImpl.getUser(retrievedUser.getID());
        assertNull(retrievedUser.getAttribute(AN_ATTRIBUTE_NAME));
        assertNull(retrievedUser.getAttribute(ANOTHER_ATTRIBUTE_NAME));
        assertNull(retrievedUser.getAttribute(A_BULK_ATTRIBUTE_NAME));
        assertEquals(retrievedUser.getAttribute(ANOTHER_BULK_ATTRIBUTE_NAME), ANOTHER_BULK_ATTRIBUTE_VALUE);

        retrievedUser.removeAttribute(ANOTHER_BULK_ATTRIBUTE_NAME);
        identityManagerImpl.updateUser(retrievedUser, retrievedUser.getID());
        retrievedUser = identityManagerImpl.getUser(retrievedUser.getID());
        assertNull(retrievedUser.getAttribute(AN_ATTRIBUTE_NAME));
        assertNull(retrievedUser.getAttribute(ANOTHER_ATTRIBUTE_NAME));
        assertNull(retrievedUser.getAttribute(A_BULK_ATTRIBUTE_NAME));
        assertNull(retrievedUser.getAttribute(ANOTHER_BULK_ATTRIBUTE_NAME));

        /*
         * Method 'getPassword()' is undefined for type IDIFUser. Commented for reference.
         * assertEquals(user.getPassword(),"passwd");
         */

        if (identityManagerImpl.userExists(CHANGED_USER))
        {
            identityManagerImpl.removeUser(CHANGED_USER);
        }
        assertFalse(identityManagerImpl.userExists(CHANGED_USER));

        createGroup(CHANGED_PARENT_GROUP_NAME);

        // Set user data for modification- change the user id
        IDIFUser user = new DIFUserImpl();
        user.setID(CHANGED_USER);
        user.setName(CHANGED_USER);
        user.setNick(CHANGED_USER);
        user.setEmail(CHANGED_MAIL);
        user.setPassword(CHANGED_PASS);
        user.setProfileID(CHANGED_PARENT_GROUP_NAME);

        // Modify user data
        identityManagerImpl.updateUser(user, NEW_USER);
        assertTrue(identityManagerImpl.userExists(CHANGED_USER));

        // Update attributes AFTER changing user data (setting attributes requires that the user exists on the
        // server).
        try
        {
            user.setAttribute(ATTRIBUTE_KEY, ATTRIBUTE_VALUE);
            user.setAttribute(A_BULK_ATTRIBUTE_NAME, A_CHANGED_BULK_ATTRIBUTE_VALUE);
            user.setAttribute(ANOTHER_BULK_ATTRIBUTE_NAME, ANOTHER_CHANGED_BULK_PARAMETER_VALUE);
            user.setAttribute(AN_ATTRIBUTE_NAME, A_CHANGED_ATTRIBUTE_VALUE);
            user.setAttribute(ANOTHER_ATTRIBUTE_NAME, ANOTHER_CHANGED_ATTRIBUTE_VALUE);
        }
        catch (InternalFrameworkException internalFrameworkException)
        {
            throw new IdentityManagerException("Could not set attribute!!", internalFrameworkException);
        }

        // Force attribute if branch on convertFromDIFUserToXUser
        identityManagerImpl.updateUser(user, CHANGED_USER);
        // Test user integrity
        assertTrue(identityManagerImpl.isIdentityValid(CHANGED_USER, CHANGED_PASS));

        retrievedUser = identityManagerImpl.getUser(CHANGED_USER);
        assertEquals(retrievedUser.getID(), CHANGED_USER);
        assertEquals(retrievedUser.getName(), CHANGED_USER);
        assertEquals(retrievedUser.getNick(), CHANGED_USER);
        assertEquals(retrievedUser.getEmail(), CHANGED_MAIL);
        assertEquals(retrievedUser.getProfile().getID(), CHANGED_PARENT_GROUP_NAME);

        /* Attributes */
        assertEquals(retrievedUser.getAttributes().get(A_BULK_ATTRIBUTE_NAME), A_CHANGED_BULK_ATTRIBUTE_VALUE);
        assertEquals(retrievedUser.getAttributes().get(ANOTHER_BULK_ATTRIBUTE_NAME),
                ANOTHER_CHANGED_BULK_PARAMETER_VALUE);
        assertEquals(retrievedUser.getAttributes().get(AN_ATTRIBUTE_NAME), A_CHANGED_ATTRIBUTE_VALUE);
        assertEquals(retrievedUser.getAttributes().get(ANOTHER_ATTRIBUTE_NAME), ANOTHER_CHANGED_ATTRIBUTE_VALUE);

        // Change attributes (restore old values)
        identityManagerImpl.updateUserAttribute(CHANGED_USER, AN_ATTRIBUTE_NAME, AN_ATTRIBUTE_VALUE);
        identityManagerImpl.updateUserAttribute(CHANGED_USER, ANOTHER_ATTRIBUTE_NAME, ANOTHER_ATTRIBUTE_VALUE);
        identityManagerImpl.updateUserAttribute(CHANGED_USER, A_BULK_ATTRIBUTE_NAME, A_BULK_ATTRIBUTE_VALUE);
        identityManagerImpl
                .updateUserAttribute(CHANGED_USER, ANOTHER_BULK_ATTRIBUTE_NAME, ANOTHER_BULK_ATTRIBUTE_VALUE);

        retrievedUser = identityManagerImpl.getUser(CHANGED_USER);
        assertEquals(retrievedUser.getAttributes().get(A_BULK_ATTRIBUTE_NAME), A_BULK_ATTRIBUTE_VALUE);
        assertEquals(retrievedUser.getAttributes().get(ANOTHER_BULK_ATTRIBUTE_NAME), ANOTHER_BULK_ATTRIBUTE_VALUE);
        assertEquals(retrievedUser.getAttributes().get(AN_ATTRIBUTE_NAME), AN_ATTRIBUTE_VALUE);
        assertEquals(retrievedUser.getAttributes().get(ANOTHER_ATTRIBUTE_NAME), ANOTHER_ATTRIBUTE_VALUE);

        /* Removes the attributes one by one */
        retrievedUser.removeAttribute(AN_ATTRIBUTE_NAME);
        identityManagerImpl.updateUser(retrievedUser, retrievedUser.getID());
        retrievedUser = identityManagerImpl.getUser(retrievedUser.getID());
        assertNull(retrievedUser.getAttribute(AN_ATTRIBUTE_NAME));
        assertEquals(retrievedUser.getAttribute(ANOTHER_ATTRIBUTE_NAME), ANOTHER_ATTRIBUTE_VALUE);
        assertEquals(retrievedUser.getAttribute(A_BULK_ATTRIBUTE_NAME), A_BULK_ATTRIBUTE_VALUE);
        assertEquals(retrievedUser.getAttribute(ANOTHER_BULK_ATTRIBUTE_NAME), ANOTHER_BULK_ATTRIBUTE_VALUE);

        retrievedUser.removeAttribute(ANOTHER_ATTRIBUTE_NAME);
        identityManagerImpl.updateUser(retrievedUser, retrievedUser.getID());
        retrievedUser = identityManagerImpl.getUser(retrievedUser.getID());
        assertNull(retrievedUser.getAttribute(AN_ATTRIBUTE_NAME));
        assertNull(retrievedUser.getAttribute(ANOTHER_ATTRIBUTE_NAME));
        assertEquals(retrievedUser.getAttribute(A_BULK_ATTRIBUTE_NAME), A_BULK_ATTRIBUTE_VALUE);
        assertEquals(retrievedUser.getAttribute(ANOTHER_BULK_ATTRIBUTE_NAME), ANOTHER_BULK_ATTRIBUTE_VALUE);

        retrievedUser.removeAttribute(A_BULK_ATTRIBUTE_NAME);
        identityManagerImpl.updateUser(retrievedUser, retrievedUser.getID());
        retrievedUser = identityManagerImpl.getUser(retrievedUser.getID());
        assertNull(retrievedUser.getAttribute(AN_ATTRIBUTE_NAME));
        assertNull(retrievedUser.getAttribute(ANOTHER_ATTRIBUTE_NAME));
        assertNull(retrievedUser.getAttribute(A_BULK_ATTRIBUTE_NAME));
        assertEquals(retrievedUser.getAttribute(ANOTHER_BULK_ATTRIBUTE_NAME), ANOTHER_BULK_ATTRIBUTE_VALUE);

        retrievedUser.removeAttribute(ANOTHER_BULK_ATTRIBUTE_NAME);
        identityManagerImpl.updateUser(retrievedUser, retrievedUser.getID());
        retrievedUser = identityManagerImpl.getUser(retrievedUser.getID());
        assertNull(retrievedUser.getAttribute(AN_ATTRIBUTE_NAME));
        assertNull(retrievedUser.getAttribute(ANOTHER_ATTRIBUTE_NAME));
        assertNull(retrievedUser.getAttribute(A_BULK_ATTRIBUTE_NAME));
        assertNull(retrievedUser.getAttribute(ANOTHER_BULK_ATTRIBUTE_NAME));

        // Mass attribute change
        Map<String, Object> massAttributes = new HashMap<String, Object>() {

            {
                put(A_BULK_ATTRIBUTE_NAME, A_CHANGED_BULK_ATTRIBUTE_VALUE);
                put(ANOTHER_BULK_ATTRIBUTE_NAME, ANOTHER_CHANGED_BULK_PARAMETER_VALUE);
                put(AN_ATTRIBUTE_NAME, A_CHANGED_ATTRIBUTE_VALUE);
                put(ANOTHER_ATTRIBUTE_NAME, ANOTHER_CHANGED_ATTRIBUTE_VALUE);
            }
        };

        try
        {
            identityManagerImpl.updateUserAttributes("InexistentUserMustReturnExpcetion", massAttributes);
            assertFalse(true);
        }
        catch (IdentityManagerException e)
        {
            assertTrue(true);
        }
        identityManagerImpl.updateUserAttributes(CHANGED_USER, massAttributes);

        retrievedUser = identityManagerImpl.getUser(CHANGED_USER);
        assertEquals(retrievedUser.getAttributes().get(A_BULK_ATTRIBUTE_NAME), A_CHANGED_BULK_ATTRIBUTE_VALUE);
        assertEquals(retrievedUser.getAttributes().get(ANOTHER_BULK_ATTRIBUTE_NAME),
                ANOTHER_CHANGED_BULK_PARAMETER_VALUE);
        assertEquals(retrievedUser.getAttributes().get(AN_ATTRIBUTE_NAME), A_CHANGED_ATTRIBUTE_VALUE);
        assertEquals(retrievedUser.getAttributes().get(ANOTHER_ATTRIBUTE_NAME), ANOTHER_CHANGED_ATTRIBUTE_VALUE);

        // Test password changing
        assertFalse(identityManagerImpl.isIdentityValid(CHANGED_USER, "WrongPassword"));
        identityManagerImpl.changePassword(CHANGED_USER, USER_PASS);
        assertTrue(identityManagerImpl.isIdentityValid(CHANGED_USER, USER_PASS));

        removeUser(CHANGED_USER);
    }

    /**
     * Tests the user operations forcing failure behavior.
     * 
     * @throws IdentityManagerException
     *             if the identity manager can't be accessed
     */
    @SuppressWarnings("unused")
    public void testUserOperationsInFailureBehaviour() throws IdentityManagerException
    {
        // Create user for addition
        IDIFUser transferUser = new DIFUserImpl();

        // Tries to add an user without id...must fail!!
        try
        {
            identityManagerImpl.addUser(transferUser);
        }
        catch (IdentityManagerException e)
        {
            // User must not have been added...
            assertFalse(identityManagerImpl.userExists(""));
        }

        try
        {
            assertFalse(identityManagerImpl.isIdentityValid(BAD_USER, BAD_PASS));
        }
        catch (IdentityManagerException e)
        {

        }

        // Try to fetch data for an inexistent user
        try
        {
            IDIFUser retrievedUser = identityManagerImpl.getUser(BAD_USER);
        }
        catch (IdentityManagerException identityManagerException)
        {
        }

        // Tries to modify a bad user...must fail!!
        try
        {
            identityManagerImpl.updateUser(transferUser, BAD_USER);
        }
        catch (IdentityManagerException e)
        {
            // User must not have been added...
            assertFalse(identityManagerImpl.userExists(BAD_USER));
        }

        // Tries to remove a bad user...must fail!!
        try
        {
            identityManagerImpl.removeUser(CHANGED_USER);
        }
        catch (IdentityManagerException e)
        {
            // User must not exist
            assertFalse(identityManagerImpl.userExists(NEW_USER));
        }
    }
}
