/**
 * - Digitalis Internal Framework v2.0 - (C) 2007, Digitalis Informatica. Distribuicao e Gestao de Informatica, Lda.
 * Estrada de Paco de Arcos num.9 - Piso -1 2780-666 Paco de Arcos Telefone: (351) 21 4408990 Fax: (351) 21 4408999
 * http://www.digitalis.pt
 */
package pt.digitalis.dif.utils.logging;

import org.aspectj.lang.Signature;

import pt.digitalis.dif.controller.AbstractDIFDispatcher;
import pt.digitalis.dif.controller.objects.DIFContext;
import pt.digitalis.dif.controller.objects.DIFRequest;
import pt.digitalis.dif.controller.objects.DIFResponse;
import pt.digitalis.log.LogLevel;

/**
 * This aspect performs logging duties on the framework's core associated with the so-called "Debug" logging level. The
 * aspect is declared privileged so it can access the Dispatcher's difContext (which is declared with protected scope).
 *
 * @author Rodrigo Gonalves <a href="mailto:rgoncalves@digitalis.pt">rgoncalves@digitalis.pt</a>
 * @created 2007/05/22
 */

privileged public aspect DIFCoreDebugAspect extends DIFLogAspect {

    /* --- Pointcuts --- */

    /**
     * Captures the execution of validate, authenticate and authorize methods in any Dispatcher.
     */
    private pointcut dispatcherMethodsWithReturn() : execution(* pt.digitalis.dif.controller.AbstractDIFDispatcher+.validate(..)) ||
    execution(* pt.digitalis.dif.controller.AbstractDIFDispatcher+.authenticate(..)) ||
    execution(* pt.digitalis.dif.controller.AbstractDIFDispatcher+.authorize(..));

    /**
     * This pointcut captures the execution of the createContext() method. It also exposes the method's argument.
     *
     * @param difRequest
     *            the methods argument
     */
    private pointcut contextCreation(DIFRequest difRequest) : execution (* pt.digitalis.dif.controller.AbstractDIFDispatcher+.createContext(..)) && args (difRequest);

    /**
     * Captures the execution of the execute() method of any Dispatcher. Exposes the Dispatcher so the context can be
     * printed.
     *
     * @param dispatcher
     *            the dispatcher
     */
    private pointcut stageExecution(AbstractDIFDispatcher dispatcher) : execution (* pt.digitalis.dif.controller.AbstractDIFDispatcher+.execute(..)) && this(dispatcher);

    /** Captures the execution of the DIFRequest construction method. */
    private pointcut difResponseCreation() : call (* pt.digitalis.dif.controller.AbstractDIFDispatcher+.createResponse(..));

    /**
     * Captures the execution of the AbstractChAL method <code>serve()</code>.
     *
     * @param request
     *            the original request
     * @param response
     *            the response
     * @param restAction
     *            the REST action
     */
    private pointcut serve(Object request, Object response, Object restAction) : execution (* pt.digitalis.dif.controller.AbstractChAL+.serve(..)) && args (request, response, restAction);

    /* --- Advice --- */

    /**
     * This advice logs the return of the methods captured by the pointcut
     *
     * @param returnValue
     *            the method's return value
     */
    after() returning(Object returnValue): dispatcherMethodsWithReturn()
    {
        super.getLogger().increaseIndentation();

        if (super.getLogger().isDebugEnabled())
        {
            Signature signature = thisJoinPointStaticPart.getSignature();
            super.getLogger().log(
                    LogLevel.DEBUG,
                    super.getLogger().getIndentationString() + "Method " + signature.getDeclaringType().getName() + "."
                            + signature.getName());
        }

        super.getLogger().decreaseIndentation();
    }

    /**
     * Logs debugging information about the received request and the created context.
     *
     * @param difRequest
     *            the request received
     * @param context
     *            the created context
     */
    after(DIFRequest difRequest) returning (DIFContext context) : contextCreation(difRequest)
    {
        super.getLogger().increaseIndentation();

        if (super.getLogger().isDebugEnabled())
        {

            Signature signature = thisJoinPointStaticPart.getSignature();

            // Logging the request
            super.getLogger().log(
                    LogLevel.DEBUG,
                    super.getLogger().getIndentationString() + "\n------------------------" + difRequest.toString()
                            + "\nat: " + signature.getDeclaringType().getName() + "." + signature.getName()
                            + "\n------------------------");
            // Logging the context object
            super.getLogger().log(
                    LogLevel.DEBUG,
                    super.getLogger().getIndentationString() + "\n------------------------" + context.toString()
                            + "\nat: " + signature.getDeclaringType().getName() + "." + signature.getName()
                            + "\n------------------------");
        }

        super.getLogger().decreaseIndentation();
    }

    /**
     * Logs the execution context before the stage execution. This logging effort reflects the changes the context
     * suffered since it's creation through the <code>validate()</code>,<code>authenticate()</code>,<code>authorize()</code>
     *
     * @param dispatcher
     *            the dispatcher
     */
    before(AbstractDIFDispatcher dispatcher) : stageExecution(dispatcher)
    {
        super.getLogger().increaseIndentation();

        if (super.getLogger().isDebugEnabled())
        {

            Signature signature = thisJoinPointStaticPart.getSignature();

            super.getLogger().log(
                    LogLevel.DEBUG,
                    super.getLogger().getIndentationString() + "\n------------------------"
                            + dispatcher.difContext.toString() + "\nbefore entering: "
                            + signature.getDeclaringType().getName() + "." + signature.getName()
                            + "\n------------------------");
        }
    }

    /**
     * Logs the execution context after the stage execution. This logging effort reflects the changes the context
     * suffered during the stage execution (redirections, etc...)
     *
     * @param dispatcher
     *            the dispatcher
     */
    after(AbstractDIFDispatcher dispatcher) : stageExecution(dispatcher)
    {
        if (super.getLogger().isDebugEnabled())
        {

            Signature signature = thisJoinPointStaticPart.getSignature();

            super.getLogger().log(
                    LogLevel.DEBUG,
                    super.getLogger().getIndentationString() + "\n------------------------"
                            + dispatcher.difContext.toString() + "\nafter exiting: "
                            + signature.getDeclaringType().getName() + "." + signature.getName()
                            + "\n------------------------");
        }

        super.getLogger().decreaseIndentation();
    }

    /**
     * @param response
     *            the returned response.
     */
    after() returning (DIFResponse response) : difResponseCreation()
    {
        super.getLogger().increaseIndentation();

        if (super.getLogger().isDebugEnabled())
        {
            Signature signature = thisJoinPointStaticPart.getSignature();

            super.getLogger().log(
                    LogLevel.DEBUG,
                    super.getLogger().getIndentationString() + "\n------------------------" + response.toString()
                            + "\nat: " + signature.getDeclaringType().getName() + "." + signature.getName()
                            + "\n------------------------");
        }

        super.getLogger().decreaseIndentation();
    }

    /**
     * Logs original request from the Listener received by the AbstractChAL.
     *
     * @param request
     *            the original request
     * @param response
     *            the response
     * @param restAction
     *            the REST action
     */
    before(Object request, Object response, Object restAction) : serve(request, response, restAction)
    {
        super.getLogger().increaseIndentation();

        if (super.getLogger().isDebugEnabled())
        {
            Signature signature = thisJoinPointStaticPart.getSignature();
            super.getLogger().log(
                    LogLevel.DEBUG,
                    super.getLogger().getIndentationString() + "\n------------------------\n" + request.toString()
                            + "\nat: " + signature.getDeclaringType().getName() + "." + signature.getName()
                            + "\n------------------------");
        }
    }

    /**
     * Logs response that will be sent to the Listener.
     *
     * @param request
     *            the request
     * @param response
     *            the response to send to the Listener
     * @param restAction
     *            the REST action
     */
    after(Object request, Object response, Object restAction) : serve(request, response, restAction)
    {
        if (super.getLogger().isDebugEnabled())
        {
            Signature signature = thisJoinPointStaticPart.getSignature();

            // Log request
            super.getLogger().log(
                    LogLevel.DEBUG,
                    super.getLogger().getIndentationString() + "\n------------------------\n" + request.toString()
                            + "\nat: " + signature.getDeclaringType().getName() + "." + signature.getName()
                            + "\n------------------------");
            // Log response
            super.getLogger().log(
                    LogLevel.DEBUG,
                    super.getLogger().getIndentationString() + "\n------------------------\n" + response.toString()
                            + "\nat: " + signature.getDeclaringType().getName() + "." + signature.getName()
                            + "\n------------------------");
        }

        super.getLogger().decreaseIndentation();
    }

}
