/**
 * - Digitalis Internal Framework v2.0 - (C) 2007, Digitalis Informatica. Distribuicao e Gestao de Informatica, Lda.
 * Estrada de Paco de Arcos num.9 - Piso -1 2780-666 Paco de Arcos Telefone: (351) 21 4408990 Fax: (351) 21 4408999
 * http://www.digitalis.pt
 */
package pt.digitalis.dif.controller.objects;

import java.util.HashMap;
import java.util.Map;

import pt.digitalis.dif.controller.http.HTTPConstants;
import pt.digitalis.dif.controller.interfaces.IDIFContext;
import pt.digitalis.dif.controller.interfaces.IDIFRequest;
import pt.digitalis.dif.controller.interfaces.IDIFSession;
import pt.digitalis.dif.dem.interfaces.IStage;
import pt.digitalis.dif.dem.managers.IDEMManager;
import pt.digitalis.dif.dem.managers.impl.UsageIssuesManagerImpl;
import pt.digitalis.dif.dem.objects.EventType;
import pt.digitalis.dif.dem.objects.ViewObject;
import pt.digitalis.dif.dem.objects.issues.IssueScope;
import pt.digitalis.dif.dem.objects.issues.IssueType;
import pt.digitalis.dif.dem.objects.issues.UsageIssue;
import pt.digitalis.dif.ioc.DIFIoCRegistry;
import pt.digitalis.dif.startup.DIFGeneralConfigurationParameters;
import pt.digitalis.dif.utils.ObjectFormatter;
import pt.digitalis.dif.utils.logging.DIFLogger;
import pt.digitalis.utils.common.CollectionUtils;
import pt.digitalis.utils.common.StringUtils;
import pt.digitalis.utils.common.SystemUtils;

/**
 * This class is the base class for the DIF execution context.
 * 
 * @author Rodrigo Gonalves <a href="mailto:rgoncalves@digitalis.pt">rgoncalves@digitalis.pt</a>
 * @created 2007/03/16
 */
public class DIFContext implements IDIFContext {

    /** The DIFRequest. */
    private IDIFRequest difRequest = null;

    /**
     * Marks the defined stage for redirection processing by the Dispatcher
     */
    private boolean hasRedirection = false;

    /** Aditional headers we can put on response. This headers will prevail over the headers already added in framework */
    private Map<String, String> httpHeaders = new HashMap<String, String>();

    /** the result message */
    private ResultMessage resultMessage = null;

    /**
     * Identifier of the Stage that should be executed.
     */
    private String stage = new String();

    /** The processing result set. */
    private Map<String, Object> stageResults = new HashMap<String, Object>();

    /**
     * Identifier of the Stage to redirect after execution of the current Stage.
     */
    private String stageToRedirect = new String();

    /** A map of custom attributes for stage usage. These will have the same life span of this DIFContext instance */
    private Map<String, Object> temporaryAttributes = new HashMap<String, Object>();

    /**
     * The ViewObject.
     */
    protected ViewObject theView = new ViewObject();

    /**
     * @see pt.digitalis.dif.controller.interfaces.IDIFContext#addHTTPHeader(java.lang.String, java.lang.String)
     */
    public void addHTTPHeader(String name, String value)
    {
        this.httpHeaders.put(name, value);
    }

    /**
     * @see pt.digitalis.dif.controller.interfaces.IDIFContext#addResultMessage(java.lang.String, java.lang.String,
     *      java.lang.String)
     */
    public void addResultMessage(String type, String title, String description)
    {
        addResultMessage(type, title, description, null);
    }

    /**
     * @see pt.digitalis.dif.controller.interfaces.IDIFContext#addResultMessage(java.lang.String, java.lang.String,
     *      java.lang.String, boolean)
     */
    public void addResultMessage(String type, String title, String description, boolean popupMode)
    {
        resultMessage = new ResultMessage(type, title, description, popupMode, null);
    }

    /**
     * @see pt.digitalis.dif.controller.interfaces.IDIFContext#addResultMessage(java.lang.String, java.lang.String,
     *      java.lang.String, java.lang.String)
     */
    public void addResultMessage(String type, String title, String description, String mode)
    {
        resultMessage = new ResultMessage(type, title, description, false, mode);
    }

    /**
     * @see pt.digitalis.dif.controller.interfaces.IDIFContext#addStageResult(String, Object)
     */
    public void addStageResult(String resultName, Object resultValue)
    {
        this.stageResults.put(resultName, resultValue);
    }

    /**
     * @see pt.digitalis.dif.controller.interfaces.IDIFContext#getHTTPHeaders()
     */
    public Map<String, String> getHTTPHeaders()
    {
        return this.httpHeaders;
    }

    /**
     * @see pt.digitalis.dif.controller.interfaces.IDIFContext#getLanguage()
     */
    public String getLanguage()
    {
        IDIFSession session = getSession();

        if (session == null)
            return DIFGeneralConfigurationParameters.getInstance().getDefaultLanguage();
        else
            return session.getLanguage();
    }

    /**
     * @see pt.digitalis.dif.controller.interfaces.IDIFContext#getRequest()
     */
    public IDIFRequest getRequest()
    {
        return difRequest;
    }

    /**
     * @see pt.digitalis.dif.controller.interfaces.IDIFContext#getResultMessage()
     */
    public ResultMessage getResultMessage()
    {
        return resultMessage;
    }

    /**
     * @see pt.digitalis.dif.controller.interfaces.IDIFContext#getSession()
     */
    public IDIFSession getSession()
    {
        if (this.difRequest != null)
        {
            return this.difRequest.getSession();
        }
        else
        {
            return null;
        }
    }

    /**
     * @see pt.digitalis.dif.controller.interfaces.IDIFContext#getStage()
     */
    public String getStage()
    {
        return this.stage;
    }

    /**
     * @see pt.digitalis.dif.controller.interfaces.IDIFContext#getStageResults()
     */
    public final Map<String, Object> getStageResults()
    {
        return stageResults;
    }

    /**
     * @see pt.digitalis.dif.controller.interfaces.IDIFContext#getTemporaryAttributes()
     */
    public Map<String, Object> getTemporaryAttributes()
    {
        return temporaryAttributes;
    }

    /**
     * @see pt.digitalis.dif.controller.interfaces.IDIFContext#getView()
     */
    public ViewObject getView()
    {
        return this.theView;
    }

    /**
     * This method sets the current stage equal to the redirection stage. The redirection stage is set to null and a the
     * redirection flag is set to T. The stage results are reset.
     * 
     * @see pt.digitalis.dif.controller.interfaces.IDIFContext#handleRedirection()
     */
    public void handleRedirection()
    {
        this.stage = this.stageToRedirect;
        this.stageToRedirect = null;
        this.hasRedirection = false;
    }

    /**
     * @see pt.digitalis.dif.controller.interfaces.IDIFContext#hasRedirection()
     */
    public boolean hasRedirection()
    {
        return this.hasRedirection;
    }

    /**
     * @see pt.digitalis.dif.controller.interfaces.IDIFContext#redirectTo(java.lang.String)
     */
    public void redirectTo(String newStage)
    {
        this.redirectTo(newStage, "");
    }

    /**
     * @see pt.digitalis.dif.controller.interfaces.IDIFContext#redirectTo(java.lang.String, boolean)
     */
    public void redirectTo(String newStage, boolean allowRedirectToSameStage)
    {
        this.redirectTo(newStage, "", allowRedirectToSameStage);
    }

    /**
     * @see pt.digitalis.dif.controller.interfaces.IDIFContext#redirectTo(java.lang.String, java.util.Map)
     */
    public void redirectTo(String newStage, Map<String, Object> redirectionParameters)
    {
        this.redirectTo(newStage, redirectionParameters, false);
    }

    /**
     * @see pt.digitalis.dif.controller.interfaces.IDIFContext#redirectTo(java.lang.String, java.util.Map, boolean)
     */
    public void redirectTo(String newStage, Map<String, Object> redirectionParameters, boolean allowRedirectToSameStage)
    {

        // Prevent redirect to the current stage and thus might allow indefinite cyclic redirects
        if (allowRedirectToSameStage || !this.getStage().equals(newStage))
        {
            this.stageToRedirect = newStage;
            this.hasRedirection = true;

            if (redirectionParameters != null && !redirectionParameters.isEmpty())
            {
                this.difRequest.setParameters(redirectionParameters);
            }
        }
        else
            DIFLogger.getLogger().warn(
                    "Prevented a redirection to the same stage: \"" + stageToRedirect
                            + "\" (cyclic redirection danger!).");
    }

    /**
     * @see pt.digitalis.dif.controller.interfaces.IDIFContext#redirectTo(java.lang.String, java.lang.String)
     */
    public void redirectTo(String newStage, String redirectionParameters)
    {
        redirectTo(newStage, redirectionParameters, false);
    }

    /**
     * @see pt.digitalis.dif.controller.interfaces.IDIFContext#redirectTo(java.lang.String, java.lang.String, boolean)
     */
    public void redirectTo(String newStage, String redirectionParameters, boolean allowRedirectToSameStage)
    {
        Map<String, Object> parameterMap = new HashMap<String, Object>();

        if (redirectionParameters != null && !"".equals(redirectionParameters))
            parameterMap = CollectionUtils.keyValueStringToMapObject(redirectionParameters);

        redirectTo(newStage, parameterMap, allowRedirectToSameStage);
    }

    /**
     * @see pt.digitalis.dif.controller.interfaces.IDIFContext#reportIssue(java.lang.String,
     *      pt.digitalis.dif.dem.objects.issues.IssueType, java.lang.String)
     */
    public UsageIssue reportIssue(String uID, IssueType type, String issueDescription)
    {
        return this.reportIssue(uID, type, issueDescription, null);
    }

    /**
     * @see pt.digitalis.dif.controller.interfaces.IDIFContext#reportIssue(java.lang.String,
     *      pt.digitalis.dif.dem.objects.issues.IssueType, java.lang.String, java.lang.Exception)
     */
    public UsageIssue reportIssue(String uID, IssueType type, String issueDescription, Exception exception)
    {
        return this.reportIssue(uID, type, issueDescription, exception, true);
    }

    /**
     * @see pt.digitalis.dif.controller.interfaces.IDIFContext#reportIssue(java.lang.String,
     *      pt.digitalis.dif.dem.objects.issues.IssueType, java.lang.String, java.lang.Exception, boolean)
     */
    public UsageIssue reportIssue(String uID, IssueType type, String issueDescription, Exception exception,
            boolean showStackTrace)
    {
        IStage stage = DIFIoCRegistry.getRegistry().getImplementation(IDEMManager.class).getStage(this.getStage());

        Object submitForm = this.getRequest().getParameter(HTTPConstants.FORM_SUBMIT_NAME);
        Object eventID = this.getRequest().getParameter(HTTPConstants.EVENT_ID);
        EventType eventType = null;
        String originalDescription = issueDescription;
        String requestSum = "";
        String traceContent = null;

        // If form submitted check if an event handler for it exists
        if (stage != null)
        {
            if (stage.getEventHandlers().get(EventType.FORM_SUBMIT).contains(submitForm))
                eventType = EventType.FORM_SUBMIT;
            else if (stage.getEventHandlers().get(EventType.FORM_SUBMIT_SAVE_ACTION).contains(submitForm))
                eventType = EventType.FORM_SUBMIT_SAVE_ACTION;
            else if (stage.getEventHandlers().get(EventType.FORM_SUBMIT_AJAX_REQUEST).contains(submitForm))
                eventType = EventType.FORM_SUBMIT_AJAX_REQUEST;
            else if (stage.getEventHandlers().get(EventType.AJAX_REQUEST).contains(eventID))
                eventType = EventType.AJAX_REQUEST;
            else if (stage.getEventHandlers().get(EventType.DOCUMENT_TYPE).contains(eventID))
                eventType = EventType.DOCUMENT_TYPE;
        }
        if (eventID != null)
            issueDescription += "<b>Event:</b> " + eventID + (eventType != null ? " [" + eventType + "]" : "");
        if (submitForm != null)
            issueDescription += "<b>Form:</b> " + submitForm;

        UsageIssue issue = new UsageIssue();
        issue.setUID(uID == null ? UsageIssuesManagerImpl.getInstance().generateUID() : this.getStage() + uID);

        if (showStackTrace)
        {
            Map<String, String> issueDetails = new HashMap<String, String>();

            if (exception == null)
            {
                StackTraceElement[] traceList = Thread.currentThread().getStackTrace();
                StackTraceElement[] parsedTraceList = new StackTraceElement[traceList.length - 1];
                int totalToRemove = 1;

                // remove the current call from the original stack
                while (traceList[totalToRemove].getMethodName().contains("reportIssue"))
                    totalToRemove++;
                System.arraycopy(traceList, totalToRemove, parsedTraceList, 0, parsedTraceList.length - totalToRemove);

                traceContent = SystemUtils.getStackTraceListHTML(parsedTraceList).toString();
            }
            else
                traceContent = SystemUtils.getStackTraceHTML(exception);

            if (stage != null)
                requestSum += "<b>Stage:</b> " + stage.getOriginalClassName() + "<br/>";
            if (eventID != null)
                requestSum += "<b>Event:</b> " + eventID + (eventType != null ? " [" + eventType + "]" : "") + "<br/>";
            if (submitForm != null)
                requestSum += "<b>Form:</b> " + submitForm + "<br/>";

            issueDetails.put("issueDescription", requestSum + "<br/>" + issueDescription);
            issueDetails.put("traceContent", traceContent == null ? "No stack trace available!" : traceContent);
            issueDetails.put("content", "<code>" + this.toString().replaceAll("\\n", "<br />") + "</code>");
            issue.setShowDetailsContent(issueDetails);
        }

        issue.setIssueScope(IssueScope.RUNTIME);
        issue.setIssueType(type);
        issue.setIssueDescription(issueDescription);
        issue.setIssueSmallDescription(originalDescription + "\n" + requestSum);
        issue.setIssuePlainDescription(originalDescription + "\n" + requestSum + "\n"
                + (StringUtils.isNotBlank(traceContent) ? traceContent : ""));
        if (stage != null)
        {
            issue.setLocation(stage.getOriginalClassName());
        }
        issue.setException(exception);
        issue.setContext(this);

        this.reportIssue(issue);

        return issue;
    }

    /**
     * @see pt.digitalis.dif.controller.interfaces.IDIFContext#reportIssue(pt.digitalis.dif.dem.objects.issues.UsageIssue)
     */
    public UsageIssue reportIssue(UsageIssue issue)
    {
        UsageIssuesManagerImpl.getInstance().addIssue(issue);

        return issue;
    }

    /**
     * @see pt.digitalis.dif.controller.interfaces.IDIFContext#setRequest(pt.digitalis.dif.controller.interfaces.IDIFRequest)
     */
    public void setRequest(IDIFRequest difRequest)
    {
        this.difRequest = difRequest;
    }

    /**
     * @see pt.digitalis.dif.controller.interfaces.IDIFContext#setSession(IDIFSession)
     */
    public void setSession(IDIFSession session)
    {
        this.difRequest.setSession(session);
    }

    /**
     * @see pt.digitalis.dif.controller.interfaces.IDIFContext#setStage(String)
     */
    public void setStage(String stage)
    {
        this.stage = stage;
    }

    /**
     * @see pt.digitalis.dif.controller.interfaces.IDIFContext#setStageResults(Map)
     */
    public final void setStageResults(Map<String, Object> results)
    {
        this.stageResults = results;
    }

    /**
     * @see pt.digitalis.dif.controller.interfaces.IDIFContext#setTemporaryAttributes(java.util.Map)
     */
    public void setTemporaryAttributes(Map<String, Object> temporaryAttributes)
    {
        this.temporaryAttributes = temporaryAttributes;
    }

    /**
     * @see pt.digitalis.dif.controller.interfaces.IDIFContext#setView(ViewObject)
     */
    public void setView(ViewObject newView)
    {
        this.theView = newView;
    }

    /**
     * Prints the context information on an human-readable form. Overrides java.lang.Object#toString().
     * 
     * @see java.lang.Object#toString()
     */
    @Override
    public String toString()
    {
        ObjectFormatter formatter = new ObjectFormatter();
        formatter.addItem("Request", difRequest);
        formatter.addItem("Has Redirection", hasRedirection);
        formatter.addItem("View", theView);
        formatter.addItem("Stage", stage);
        formatter.addItem("Stage Results", stageResults);
        formatter.addItemIfNotNull("Stage to redirect", stageToRedirect);

        return formatter.getFormatedObject();
    }
}