/** 2007, Digitalis Informatica. All rights reserved.
 *
 * Distribuicao e Gestao de Informatica, Lda.
 * Estrada de Paco de Arcos num.9 - Piso -1
 * 2780-666 Paco de Arcos
 * Telefone: (351) 21 4408990
 * Fax: (351) 21 4408999
 * http://www.digitalis.pt
 */

package pt.digitalis.dif.controller.security.managers;

import pt.digitalis.dif.controller.security.objects.IDIFUser;
import pt.digitalis.dif.exception.security.AuthenticationManagerException;

/**
 * Defines the behavior for an authentication manager.
 *
 * @author Pedro Viegas <a href="mailto:pviegas@digitalis.pt">pviegas@digitalis.pt</a><br/>
 * @author Rodrigo Gonalves <a href="mailto:rgoncalves@digitalis.pt">rgoncalves@digitalis.pt</a><br/>
 * @created 2007/12/03
 */
public interface IAuthenticationManager {

    /**
     * Checks if a given user is already authenticated on the Authentication Module.
     *
     * TODO: When integrating External Authentication/Authorization/Identity servers like LDAP, Kerberos and others this
     * may change a bit. See these APIs for good candidates for implementing these needs:
     *
     * http://www.ja-sig.org/products/cas/index.html
     *
     * http://www.acegisecurity.org/
     *
     * @param clientIdentifier
     *            the Id that identifies the specific client. Depends on the Channel used to communicate
     * @return T if a session with this ID is present and active.
     */
    public boolean isClientLogged(String clientIdentifier);

    /**
     * Performs the log in of a user with a given id on the framework.
     *
     * @param clientIdentifier
     *            the Id that identifies the specific client. Depends on the Channel used to communicate
     * @param userID
     *            the id of the user to log in
     * @param password
     *            the password for the user
     * @return the updated session object
     * @throws AuthenticationManagerException
     *             if the resources needed for authentication can't be accessed
     */
    public IDIFUser logIn(String clientIdentifier, String userID, String password)
            throws AuthenticationManagerException;

    /**
     * Performs the log out of a user with a given id on the framework.
     *
     * @param clientIdentifier
     *            the Id that identifies the specific client. Depends on the Channel used to communicate
     */
    public void logOut(String clientIdentifier);

    /**
     * Called to notify the authentication system that the client has disconnected. Each implementation will decide what
     * to do in this case. Either do nothing and keep the user connected (if the authentication repository is shared by
     * external systems), of logout the user from the authentication system (if it is a DIF specific authentication
     * system).
     *
     * @param clientIdentifier
     *            the Id that identifies the specific client. Depends on the Channel used to communicate
     */
    public void disconnectClient(String clientIdentifier);

    /**
     * Searches for the user authenticated for this client in the authentication system and returns it if present.
     *
     * @param clientIdentifier
     *            the Id that identifies the specific client. Depends on the Channel used to communicate
     * @return the authenticated user record
     * @throws AuthenticationManagerException
     *             if the resources needed for authentication can't be accessed
     */
    public IDIFUser getLoggedUser(String clientIdentifier) throws AuthenticationManagerException;
}
