/**
 * 2007, Digitalis Informatica. All rights reserved. Distribuicao e Gestao de Informatica, Lda. Estrada de Paco de Arcos
 * num.9 - Piso -1 2780-666 Paco de Arcos Telefone: (351) 21 4408990 Fax: (351) 21 4408999 http://www.digitalis.pt
 */

package pt.digitalis.dif.controller.security.managers.impl;

import java.util.HashMap;
import java.util.Map;

import pt.digitalis.dif.controller.security.managers.IAuthenticationManager;
import pt.digitalis.dif.controller.security.managers.IIdentityManager;
import pt.digitalis.dif.controller.security.objects.IDIFUser;
import pt.digitalis.dif.exception.security.AuthenticationManagerException;
import pt.digitalis.dif.exception.security.IdentityManagerException;
import pt.digitalis.dif.utils.ObjectFormatter;

import com.google.inject.Inject;

/**
 * DIF's default implementation of an authentication manager. This implementation will manage authentication records for
 * the DIF localy. No central authentication process is supported.
 * 
 * @author Pedro Viegas <a href="mailto:pviegas@digitalis.pt">pviegas@digitalis.pt</a>
 * @author Rodrigo Gonalves <a href="mailto:rgoncalves@digitalis.pt">rgoncalves@digitalis.pt</a><br/>
 * @created 2007/12/03
 */
public class AuthenticationManagerStaticImpl implements IAuthenticationManager {

    /** List of logged clients and their users Map<K=clientIdentifier, V=userID>. */
    private Map<String, String> loggedClients = new HashMap<String, String>();

    /** The identity manager. */
    @Inject
    private IIdentityManager theIDManager;

    /**
     * @see pt.digitalis.dif.controller.security.managers.IAuthenticationManager#disconnectClient(java.lang.String)
     */
    public void disconnectClient(String clientIdentifier)
    {
        // Our implementation is for DIF usage only. So when a DIF client disconnects the authentication is terminated
        logOut(clientIdentifier);
    }

    /**
     * @see pt.digitalis.dif.controller.security.managers.IAuthenticationManager#getLoggedUser(java.lang.String)
     */
    public IDIFUser getLoggedUser(String clientIdentifier) throws AuthenticationManagerException
    {
        try
        {
            String userID = loggedClients.get(clientIdentifier);

            if (userID == null)
                return null;
            else
                return theIDManager.getUser(userID);
        }
        catch (IdentityManagerException identityManagerException)
        {
            throw new AuthenticationManagerException("Could not access the identity manager for user authentication!",
                    identityManagerException);
        }
    }

    /**
     * @see pt.digitalis.dif.controller.security.managers.IAuthenticationManager#isClientLogged(java.lang.String)
     */
    public boolean isClientLogged(String clientIdentifier)
    {
        return loggedClients.containsKey(clientIdentifier);
    }

    /**
     * @see pt.digitalis.dif.controller.security.managers.IAuthenticationManager#logIn(java.lang.String,
     *      java.lang.String, java.lang.String)
     */
    public IDIFUser logIn(String clientSpecificID, String userID, String password)
            throws AuthenticationManagerException
    {
        try
        {
            if (loggedClients.containsKey(clientSpecificID))
            {

                // Already authenticated. No validation required.
                String presentUserID = loggedClients.get(clientSpecificID);

                if (presentUserID.equals(userID))
                    return theIDManager.getUser(userID);
            }

            // No authentication. Validate user and authenticate.
            if (theIDManager.isIdentityValid(userID, password))
            {

                // Valid user/pass. Register login and return the user. User is now authenticated.
                loggedClients.put(clientSpecificID, userID);

                return theIDManager.getUser(userID);
            }
            else
                // Invalid user/password
                return null;
        }
        catch (IdentityManagerException identityManagerException)
        {
            throw new AuthenticationManagerException("Could not access the identity manager for user authentication!",
                    identityManagerException);
        }
    }

    /**
     * @see pt.digitalis.dif.controller.security.managers.IAuthenticationManager#logOut(java.lang.String)
     */
    public void logOut(String clientIdentifier)
    {
        loggedClients.remove(clientIdentifier);
    }

    /**
     * @see java.lang.Object#toString()
     */
    @Override
    public String toString()
    {
        ObjectFormatter formatter = new ObjectFormatter();
        formatter.addItem("Logged Clients", loggedClients);

        return formatter.getFormatedObject();
    }
}
