/**
 * - Digitalis Internal Framework v2.0 - (C) 2007, Digitalis Informatica. Distribuicao e Gestao de Informatica, Lda.
 * Estrada de Paco de Arcos num.9 - Piso -1 2780-666 Paco de Arcos Telefone: (351) 21 4408990 Fax: (351) 21 4408999
 * http://www.digitalis.pt
 */
package pt.digitalis.dif.dem.managers.impl;

import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;

import pt.digitalis.dif.dem.Entity;
import pt.digitalis.dif.dem.interfaces.IApplication;
import pt.digitalis.dif.dem.interfaces.IEntity;
import pt.digitalis.dif.dem.interfaces.IProvider;
import pt.digitalis.dif.dem.interfaces.IService;
import pt.digitalis.dif.dem.interfaces.IStage;
import pt.digitalis.dif.dem.managers.IMessageManager;
import pt.digitalis.dif.dem.objects.messages.MessageList;
import pt.digitalis.dif.dem.objects.messages.MessageTranslations;
import pt.digitalis.dif.startup.DIFStartupConfiguration;

/**
 * Manages the DEM messages, providing operations for access, pooling and persistence.
 * 
 * @author Pedro Viegas <a href="mailto:pviegas@digitalis.pt">pviegas@digitalis.pt</a>
 * @author Rodrigo Gonalves <a href="mailto:rgoncalves@digitalis.pt">rgoncalves@digitalis.pt</a>
 * @created 2007/06/01
 */
abstract public class AbstractMessageManager implements IMessageManager {

    /** The default message file extension. */
    final static protected String DEFAULT_MESSAGE_FILE_EXTENSION = ".messages";

    /** The message repository structure: EntityID, Message list */
    static private Map<String, MessageList> messageRepository = new HashMap<String, MessageList>();

    /** The messages folder. */
    final static protected String MESSAGES_FOLDER = "messages/";

    /** The number sign character as a string. */
    final static protected String NUMBER_SIGN = "#";

    /** The list of supported languages. */
    static private Set<String> supportedLanguages = new HashSet<String>();

    /**
     * @see pt.digitalis.dif.dem.managers.IMessageManager#addSupportedLanguage(java.lang.String)
     */
    public void addSupportedLanguage(String language)
    {
        supportedLanguages.add(language);
    }

    /**
     * Generates the ID of a given entity.
     * 
     * @param entity
     *            the entity to generate the ID
     * @return the entity ID
     */
    protected String getEntityID(IEntity entity)
    {
        String entityID = null;

        Class<?>[] interfaces = entity.getClass().getInterfaces();

        if (interfaces.length > 0)
        {

            for (Class<?> entityInterface: interfaces)
            {
                if (entityInterface.getCanonicalName().equals(IProvider.class.getCanonicalName()))
                    entityID = Entity.getID(Entity.PROVIDER, entity.getID());
                else if (entityInterface.getCanonicalName().equals(IApplication.class.getCanonicalName()))
                    entityID = Entity.getID(Entity.APPLICATION, entity.getID());
                else if (entityInterface.getCanonicalName().equals(IService.class.getCanonicalName()))
                    entityID = Entity.getID(Entity.SERVICE, entity.getID());
                else if (entityInterface.getCanonicalName().equals(IStage.class.getCanonicalName()))
                    entityID = Entity.getID(Entity.STAGE, entity.getID());
            }
        }

        return entityID;
    }

    /**
     * Gets the messages from the repository and collects them on first use
     * 
     * @param type
     *            the entity type
     * @param entity
     *            the entity object
     * @return the messages collected
     */
    public MessageList getMessageList(Entity type, IEntity entity)
    {

        String entityID = Entity.getID(type, entity.getID());
        MessageList messages = null;

        // Only reads from the cache if not in developer mode (developer mode refreshes from repository every time)
        if (!DIFStartupConfiguration.getDeveloperMode())
            messages = messageRepository.get(entityID);

        if (messages == null)
        {
            // First time access to this entity. Collect messages from the repository
            messages = collectEntityMessagesFromRepository(type, entity);
            messageRepository.put(entityID, messages);
        }

        return messages;
    }

    /**
     * @see pt.digitalis.dif.dem.managers.IMessageManager#getMessageList(pt.digitalis.dif.dem.interfaces.IApplication)
     */
    public MessageList getMessageList(IApplication application)
    {

        MessageList messages = new MessageList();

        // Get the application's provider messages
        messages.addMessageList(getMessageList(application.getProvider()));

        // Add/override the application ones
        messages.addMessageList(getMessageList(Entity.APPLICATION, application));

        return messages;
    }

    /**
     * @see pt.digitalis.dif.dem.managers.IMessageManager#getMessageList(pt.digitalis.dif.dem.interfaces.IProvider)
     */
    public MessageList getMessageList(IProvider provider)
    {

        return getMessageList(Entity.PROVIDER, provider);
    }

    /**
     * @see pt.digitalis.dif.dem.managers.IMessageManager#getMessageList(pt.digitalis.dif.dem.interfaces.IService)
     */
    public MessageList getMessageList(IService service)
    {

        MessageList messages = new MessageList();

        // Get the service's application messages
        messages.addMessageList(getMessageList(service.getApplication()));

        // Add/override the service ones
        messages.addMessageList(getMessageList(Entity.SERVICE, service));

        return messages;
    }

    /**
     * @see pt.digitalis.dif.dem.managers.IMessageManager#getMessageList(pt.digitalis.dif.dem.interfaces.IStage)
     */
    public MessageList getMessageList(IStage stage)
    {

        MessageList messages = new MessageList();

        // Get the stage's service messages
        messages.addMessageList(getMessageList(stage.getService()));

        // Add/override the stage ones
        messages.addMessageList(getMessageList(Entity.STAGE, stage));

        return messages;
    }

    /**
     * @see pt.digitalis.dif.dem.managers.IMessageManager#getMessages(pt.digitalis.dif.dem.interfaces.IApplication,
     *      String)
     */
    public Map<String, String> getMessages(IApplication application, String language)
    {

        return getMessageList(application).getMessages(language);
    }

    /**
     * @see pt.digitalis.dif.dem.managers.IMessageManager#getMessages(pt.digitalis.dif.dem.interfaces.IProvider, String)
     */
    public Map<String, String> getMessages(IProvider provider, String language)
    {

        return getMessageList(provider).getMessages(language);
    }

    /**
     * @see pt.digitalis.dif.dem.managers.IMessageManager#getMessages(pt.digitalis.dif.dem.interfaces.IService, String)
     */
    public Map<String, String> getMessages(IService service, String language)
    {

        return getMessageList(service).getMessages(language);
    }

    /**
     * @see pt.digitalis.dif.dem.managers.IMessageManager#getMessages(pt.digitalis.dif.dem.interfaces.IStage, String)
     */
    public Map<String, String> getMessages(IStage stage, String language)
    {

        return getMessageList(stage).getMessages(language);
    }

    /**
     * @see pt.digitalis.dif.dem.managers.IMessageManager#getSupportedLanguages()
     */
    public Set<String> getSupportedLanguages()
    {
        return supportedLanguages;
    }

    /**
     * @see pt.digitalis.dif.dem.managers.IMessageManager#invalidateMessages()
     */
    public void invalidateMessages()
    {
        messageRepository = new HashMap<String, MessageList>();
    }

    /**
     * @see pt.digitalis.dif.dem.managers.IMessageManager#isLanguageSupported(java.lang.String)
     */
    public boolean isLanguageSupported(String language)
    {
        // FIXME: Might be useful to implement a system to check that "en-gb" and "en" are the same language
        return supportedLanguages.contains(language);
    }

    /**
     * @see pt.digitalis.dif.dem.managers.IMessageManager#updateMessage(java.lang.Class, java.lang.String,
     *      pt.digitalis.dif.dem.objects.messages.MessageTranslations)
     */
    public void updateMessage(Class<?> clazz, String messageID, MessageTranslations message)
    {

        if (messageRepository.containsKey(clazz.getSimpleName())
                && messageRepository.get(clazz.getSimpleName()).containsMessageWithID(messageID))
            messageRepository.get(clazz.getSimpleName()).addMessage(messageID, message);
    }

    /**
     * @see pt.digitalis.dif.dem.managers.IMessageManager#updateMessage(pt.digitalis.dif.dem.interfaces.IEntity,
     *      java.lang.String, pt.digitalis.dif.dem.objects.messages.MessageTranslations)
     */
    public void updateMessage(IEntity entity, String messageID, MessageTranslations message)
    {
        String entityID = getEntityID(entity);

        if (messageRepository.containsKey(entityID) && messageRepository.get(entityID).containsMessageWithID(messageID))
            messageRepository.get(entityID).addMessage(messageID, message);
    }
}
