/**
 * 2007, Digitalis Informatica. All rights reserved. Distribuicao e Gestao de Informatica, Lda. Estrada de Paco de Arcos
 * num.9 - Piso -1 2780-666 Paco de Arcos Telefone: (351) 21 4408990 Fax: (351) 21 4408999 http://www.digitalis.pt
 */
package pt.digitalis.dif.dem.objects.parameters.constraints;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import pt.digitalis.dif.dem.interfaces.IStageInstance;
import pt.digitalis.dif.dem.managers.IMessageManager;
import pt.digitalis.dif.dem.objects.messages.MessageList;
import pt.digitalis.dif.dem.objects.parameters.IParameter;
import pt.digitalis.dif.ioc.DIFIoCRegistry;
import pt.digitalis.dif.utils.annotations.JSONIgnore;
import pt.digitalis.utils.common.StringUtils;

/**
 * Base implementation for all parameter constraints. Will provide common features and resources.
 * 
 * @author Pedro Viegas <a href="mailto:pviegas@digitalis.pt">pviegas@digitalis.pt</a>
 * @created Nov 16, 2007
 */
public abstract class AbstractParameterConstraint implements IParameterConstraint {

    /** The message manager to get class messages */
    static IMessageManager messageManager;

    /** Messages cache for all constraints */
    static private MessageList messages;

    /**
     * Get's the Message Manager from the IoC
     * 
     * @return the message manager instance
     */
    static private IMessageManager getMessageManager()
    {
        if (messageManager == null)
            messageManager = DIFIoCRegistry.getRegistry().getImplementation(IMessageManager.class);

        return messageManager;
    }

    /**
     * Lazy loading getter of messages
     * 
     * @return the messages
     */
    protected static MessageList getMessages()
    {
        // If the messages have not yet been loaded do it now
        if (messages == null)
        {
            messages = getMessageManager().collectEntityMessagesFromRepository(AbstractParameterConstraint.class);
        }

        return messages;
    }

    /** The error ids. */
    private List<String> errorIds = new ArrayList<String>();

    /** The parameter that called the constraint. */
    private IParameter<?> param;

    /**
     * Get the error IDs
     * 
     * @return List of error IDs
     */
    protected List<String> getErrorIDs()
    {
        return errorIds;
    }

    /**
     * Compiles all values to be parsed in the error message to be generated on request
     * 
     * @param parseValues
     *            if T will replace the current parameter value in the message. Otherwise return the untouched message
     *            for custom parsing
     * @return the list of key/values to replace in the error message
     */
    abstract protected Map<String, String> getErrorMessageValues(boolean parseValues);

    /**
     * Inspector for the 'param' attribute.
     *
     * @return the param value
     */
    @JSONIgnore
    public IParameter<?> getParameter()
    {
        return param;
    }

    /**
     * @see pt.digitalis.dif.dem.objects.parameters.constraints.IParameterConstraint#getValidationResult(java.lang.Object,
     *      pt.digitalis.dif.dem.interfaces.IStageInstance)
     */
    public ParameterConstraintResult getValidationResult(Object value, IStageInstance stageInstance)
    {
        Boolean constraintResult = validateConstraint(value, stageInstance);
        String language = null;
        if (stageInstance != null && stageInstance.getContext() != null)
        {
            language = stageInstance.getContext().getLanguage();
        }

        return new ParameterConstraintResult(this, constraintResult, getErrorIDs(), language);
    }

    /**
     * @see pt.digitalis.dif.dem.objects.parameters.constraints.IParameterConstraint#setParameter(pt.digitalis.dif.dem.objects.parameters.IParameter)
     */
    public void setParameter(IParameter<?> param)
    {
        this.param = param;

    }

    /**
     * @see pt.digitalis.dif.dem.objects.parameters.constraints.IParameterConstraint#validateConstraint(java.lang.Object,
     *      pt.digitalis.dif.dem.interfaces.IStageInstance)
     */
    public boolean validateConstraint(Object value, IStageInstance stageInstance)
    {
        if (value == null)
            return validateConstraint((String) null, stageInstance);
        else
            return validateConstraint(value.toString(), stageInstance);
    }

    /**
     * @see pt.digitalis.dif.dem.objects.parameters.constraints.IParameterConstraint#validationErrorMessage()
     */
    public String validationErrorMessage()
    {
        return this.validationErrorMessage(true);
    }

    /**
     * @see pt.digitalis.dif.dem.objects.parameters.constraints.IParameterConstraint#validationErrorMessage(boolean)
     */
    public String validationErrorMessage(boolean parseValues)
    {
        String message = getMessages().getMessages().get(this.getClass().getSimpleName() + ".message");
        Map<String, String> substitutions = new HashMap<String, String>();
        substitutions.putAll(getErrorMessageValues(parseValues));

        if (!parseValues)
            substitutions.remove("value");

        message = StringUtils.parseMessage(message, substitutions);

        return message;
    }

    /**
     * @see pt.digitalis.dif.dem.objects.parameters.constraints.IParameterConstraint#validationErrorMessage(boolean,
     *      java.lang.String)
     */
    public String validationErrorMessage(boolean parseValues, String language)
    {
        String message = getMessages().getMessages(language).get(this.getClass().getSimpleName() + ".message");
        Map<String, String> substitutions = new HashMap<String, String>();
        substitutions.putAll(getErrorMessageValues(parseValues));

        if (!parseValues)
            substitutions.remove("value");

        message = StringUtils.parseMessage(message, substitutions);

        return message;
    }

    /**
     * @see pt.digitalis.dif.dem.objects.parameters.constraints.IParameterConstraint#validationErrorMessage(java.lang.String)
     */
    public String validationErrorMessage(String language)
    {
        return this.validationErrorMessage(true, language);
    }

}
