/**
 * 2007, Digitalis Informatica. All rights reserved. Distribuicao e Gestao de Informatica, Lda. Estrada de Paco de Arcos
 * num.9 - Piso -1 2780-666 Paco de Arcos Telefone: (351) 21 4408990 Fax: (351) 21 4408999 http://www.digitalis.pt
 */
package pt.digitalis.dif.dem.objects.parameters.types;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Date;
import java.util.List;

import pt.digitalis.dif.controller.interfaces.IDIFContext;
import pt.digitalis.dif.dem.interfaces.IStageInstance;
import pt.digitalis.dif.dem.objects.parameters.constraints.IParameterConstraint;
import pt.digitalis.dif.dem.objects.parameters.errors.ParameterError;
import pt.digitalis.dif.dem.objects.parameters.errors.ParameterErrorList;
import pt.digitalis.dif.dem.objects.parameters.errors.ParameterErrorType;
import pt.digitalis.dif.exception.objects.ParameterException;
import pt.digitalis.dif.startup.DIFGeneralConfigurationParameters;
import pt.digitalis.utils.common.DateUtils;
import pt.digitalis.utils.common.StringUtils;

/**
 * /** This class will define a numeric Parameter.
 * <p>
 * It will hold information relative to the Parameter value, ID key and validation constraints.
 * 
 * @author Pedro Viegas <a href="mailto:pviegas@digitalis.pt">pviegas@digitalis.pt</a>
 * @author Rodrigo Gon�alves <a href="mailto:rgoncalves@digitalis.pt">rgoncalves@digitalis.pt</a>
 * @created Nov 23, 2007
 */
public class DateParameter extends AbstractParameter<Date> {

    /** The list of supported classes to define in the concrete implementations */
    @SuppressWarnings("serial")
    final static private List<String> supportedClasses = new ArrayList<String>() {

        {
            add(Date.class.getCanonicalName());
        }
    };

    /**
     * @see pt.digitalis.dif.dem.objects.parameters.types.AbstractParameter#automaticConstraints()
     */
    @Override
    protected String automaticConstraints()
    {
        return "date";
    }

    /**
     * @see pt.digitalis.dif.dem.objects.parameters.types.AbstractParameter#convertObjectToString(java.lang.Object)
     */
    @Override
    protected String convertObjectToString(Object obj)
    {
        if (obj instanceof Date)
            return DateUtils.simpleDateToString((Date) obj);
        else
            return super.convertObjectToString(obj);
    }

    /**
     * @see pt.digitalis.dif.dem.objects.parameters.IParameter#getSupportedClasses()
     */
    public List<String> getSupportedClasses()
    {
        return supportedClasses;
    }

    /**
     * @see pt.digitalis.dif.dem.objects.parameters.IParameter#getValueAsBigDecimal(pt.digitalis.dif.controller.interfaces.IDIFContext)
     */
    public BigDecimal getValueAsBigDecimal(IDIFContext context) throws ParameterException
    {
        return BigDecimal.valueOf(getValue(context).getTime());
    }

    /**
     * @see pt.digitalis.dif.dem.objects.parameters.IParameter#getValueAsBoolean(IDIFContext)
     */
    public boolean getValueAsBoolean(IDIFContext context)
    {
        return false;
    }

    /**
     * @see pt.digitalis.dif.dem.objects.parameters.IParameter#getValueAsDate(IDIFContext)
     */
    public Date getValueAsDate(IDIFContext context) throws ParameterException
    {
        return getValue(context);
    }

    /**
     * @see pt.digitalis.dif.dem.objects.parameters.IParameter#getValueAsDouble(IDIFContext)
     */
    public Double getValueAsDouble(IDIFContext context) throws ParameterException
    {
        return Double.valueOf(getValue(context).getTime());
    }

    /**
     * @see pt.digitalis.dif.dem.objects.parameters.IParameter#getValueAsLong(IDIFContext)
     */
    public Long getValueAsLong(IDIFContext context) throws ParameterException
    {
        return getValue(context).getTime();
    }

    /**
     * @see pt.digitalis.dif.dem.objects.parameters.types.AbstractParameter#getValueAsString(IDIFContext)
     */
    @Override
    public String getValueAsString(IDIFContext context) throws ParameterException
    {

        Date currentDate = getValue(context);

        if (currentDate == null)
            return null;
        else
            return DateUtils.simpleDateToString(currentDate);
    }

    /**
     * @see pt.digitalis.dif.dem.objects.parameters.IParameter#isNumeric()
     */
    public boolean isNumeric()
    {
        return false;
    }

    /**
     * @see pt.digitalis.dif.dem.objects.parameters.IParameter#setValueFromString(java.lang.String,
     *      pt.digitalis.dif.dem.interfaces.IStageInstance, boolean)
     */
    public ParameterErrorList setValueFromString(String value, IStageInstance stageInstance,
            boolean initializationInProgress)
    {

        ParameterErrorList list = new ParameterErrorList(this, value);
        ParameterError error;
        IParameterConstraint constraint = getConstraints().get("date");
        IDIFContext context;

        if (stageInstance == null)
            context = null;
        else
            context = stageInstance.getContext();

        // If the date constraint does not validate...
        if (!constraint.validateConstraint(value, stageInstance))
        {
            String language;

            if (context == null)
                language = DIFGeneralConfigurationParameters.getInstance().getDefaultLanguage();
            else
                language = context.getLanguage();

            error = new ParameterError(constraint.validationErrorMessage(language), ParameterErrorType.CONSTRAINT);
            error.setConstraint("date", constraint);

            list.addError(error);
        }
        else
        {
            // Valid date format, see if date conversion has a format constraint to use
            if (getConstraints().containsKey("futuredate") || getConstraints().containsKey("pastdate")
                    || getConstraints().containsKey("absolutedate") || getConstraints().containsKey("relativedate"))
            {
                if (value != null)
                {
                    String[] parts = value.split("/");

                    if (parts[2].length() < 3)
                    {
                        // If single digit year fill with left zero
                        parts[2] = StringUtils.fillStringLeft(parts[2], 2, "0");
                        Calendar currentDate = Calendar.getInstance();
                        Integer currentCenturyPrefix = currentDate.get(Calendar.YEAR) / 100;
                        Integer valueDateYear2Digits = Integer.parseInt(parts[2]);
                        Integer currentDateYear2Digits = currentDate.get(Calendar.YEAR) % 100;
                        Calendar currentCenturyTempDate = Calendar.getInstance();
                        currentCenturyTempDate.set(currentCenturyPrefix * 100 + valueDateYear2Digits,
                                Integer.parseInt(parts[1]), Integer.parseInt(parts[0]));

                        if (getConstraints().containsKey("pastdate"))
                        {
                            if (currentCenturyTempDate.before(currentDate))
                                parts[2] = currentCenturyPrefix + parts[2];
                            else
                                parts[2] = (currentCenturyPrefix - 1) + parts[2];
                        }
                        else if (getConstraints().containsKey("futuredate"))
                        {
                            if (currentCenturyTempDate.after(currentDate))
                                parts[2] = currentCenturyPrefix + parts[2];
                            else
                                parts[2] = (currentCenturyPrefix + 1) + parts[2];
                        }
                        else if (getConstraints().containsKey("relativedate"))
                        {
                            // Relative date => compare to current year
                            if (Math.abs(valueDateYear2Digits - currentDateYear2Digits) <= Math
                                    .abs(currentDateYear2Digits - valueDateYear2Digits + 100))
                                // Closer to current century => add current century
                                parts[2] = currentCenturyPrefix + parts[2];
                            else
                                // Closer to previous century => add previous century
                                parts[2] = (currentCenturyPrefix - 1) + parts[2];
                        }
                        else if (getConstraints().containsKey("absolutedate"))
                            // Absolute Date => add current century
                            parts[2] = currentCenturyPrefix + parts[2];
                    }

                    // Reassemble the parsed date
                    value = parts[0] + "/" + parts[1] + "/" + parts[2];
                }
            }

            try
            {
                if (value == null || "".equals(value))
                    return setValue(null, stageInstance, initializationInProgress);
                else
                    return setValue(DateUtils.stringToSimpleDate(value), stageInstance, initializationInProgress);

            }
            catch (Exception e)
            {
                /*
                 * Unreachable exception: to get here DateUtils#stringToDate() would have to throw a ParseException.
                 * This exception is raised when the beginning of the passed string can't be parsed. This would happen
                 * if a bad date value was passed. This situation is avoided by the constraint validation mechanism. So
                 * if a bad value is passed the code never enters the 'else' branch and as such is never possible to get
                 * here.
                 */
                error = new ParameterError("Error parsing value", ParameterErrorType.OTHER);
                error.setException(e);
                list.addError(error);
            }
        }

        return list;
    }
}
