/**
 * 2010, Digitalis Informatica. All rights reserved. Distribuicao e Gestao de Informatica, Lda. Estrada de Paco de Arcos
 * num.9 - Piso -1 2780-666 Paco de Arcos Telefone: (351) 21 4408990 Fax: (351) 21 4408999 http://www.digitalis.pt
 */

package pt.digitalis.dif.utils.extensions.cms;

import java.util.ArrayList;
import java.util.Date;
import java.util.List;

import pt.digitalis.dif.controller.security.objects.IDIFUser;
import pt.digitalis.dif.exception.security.IdentityManagerException;
import pt.digitalis.dif.ioc.DIFIoCRegistry;
import pt.digitalis.dif.utils.extensions.cms.exception.AlreadyDeletedException;
import pt.digitalis.dif.utils.extensions.cms.exception.ContentManagerException;
import pt.digitalis.dif.utils.extensions.cms.exception.NodeNotFoundException;
import pt.digitalis.dif.utils.extensions.cms.exception.NodeWithNodesException;

/**
 * Content node definition class
 * 
 * @author Pedro Viegas <a href="mailto:pviegas@digitalis.pt">pviegas@digitalis.pt</a><br/>
 * @created 2010/10/06
 */
public class Node {

    /** the node ACL */
    private List<ACLEntry> accessList;

    /** The contentManager */
    private IContentManager contentManager;

    /** The date when the document was created */
    private Date creationDate;

    /** the creator user ID */
    private String creatorID;

    /** The node description */
    private String description;

    /** the full path name with all parent nodes */
    private String fullPathName;

    /** the node unique identifier */
    private Long id;

    /** True if the node object has already been deleted in the repository */
    private boolean isDeleted = false;

    /**  */
    private boolean isPublic;

    /** the node name. Unique within the node */
    private String name;

    /** The parent node ID, if it is not a root node */
    private Long parentNodeId;

    /** The current user */
    private IDIFUser user;

    /**
     * Constructor
     * 
     * @param parentNodeId
     *            the parent node ID
     * @param name
     *            the node Name
     * @param creatorID
     *            the creator ID
     * @param user
     *            the user instantiating the Node
     */
    public Node(Long parentNodeId, String name, String creatorID, IDIFUser user)
    {
        this.creatorID = creatorID;
        this.name = name;
        this.parentNodeId = parentNodeId;
        this.user = user;
        this.creationDate = new Date();
    }

    /**
     * Node is deleted in DB. Object ID property is set to null
     * 
     * @return T if update was succeeded
     * @throws IdentityManagerException
     * @throws ContentManagerException
     */
    public boolean delete() throws IdentityManagerException, ContentManagerException
    {
        if (isDeleted)
            throw new AlreadyDeletedException();

        boolean ret = getContentManager().deleteNode(id, user);

        // object ID is set to null
        if (ret)
        {
            isDeleted = true;
            id = null;
        }

        return ret;
    }

    /**
     * Node is deleted in DB, as well has its child nodes. Object ID property is set to null
     * 
     * @return T if update was succeeded
     * @throws IdentityManagerException
     * @throws ContentManagerException
     */
    public boolean deleteCascade() throws IdentityManagerException, ContentManagerException
    {
        if (isDeleted)
            throw new AlreadyDeletedException();

        boolean ret = false;

        try
        {
            ret = getContentManager().deleteNode(id, user, true);
        }
        catch (NodeWithNodesException e)
        {
            // never should happen, since it is a cascade delete...
        }

        // object ID is set to null
        id = null;

        return ret;
    }

    /**
     * Get ACL.
     * 
     * @return the accessList value
     * @throws ContentManagerException
     */
    public List<ACLEntry> getACL() throws ContentManagerException
    {
        if (accessList == null)
        {
            accessList = getContentManager().getNodeACL(id);
        }

        return accessList;
    }

    /**
     * Lazy load for content manager
     * 
     * @return the content manager
     */
    private IContentManager getContentManager()
    {
        if (contentManager == null)
            contentManager = DIFIoCRegistry.getRegistry().getImplementation(IContentManager.class);

        return contentManager;
    }

    /**
     * Inspector for the 'creationDate' attribute.
     * 
     * @return the creationDate value
     */
    public Date getCreationDate()
    {
        return creationDate;
    }

    /**
     * Inspector for the 'creatorID' attribute.
     * 
     * @return the creatorID value
     */
    public String getCreatorID()
    {
        return creatorID;
    }

    /**
     * Inspector for the 'description' attribute.
     * 
     * @return the description value
     */
    public String getDescription()
    {
        return description;
    }

    /**
     * Inspector for the 'fullPathName' attribute.
     * 
     * @return the fullPathName value
     */
    public String getFullPathName()
    {
        return fullPathName;
    }

    /**
     * Inspector for the 'id' attribute.
     * 
     * @return the id value
     */
    public Long getId()
    {
        return id;
    }

    /**
     * @return the node inner nodes
     * @throws IdentityManagerException
     * @throws ContentManagerException
     */
    public List<Node> getInnerNodes() throws IdentityManagerException, ContentManagerException
    {
        if (isDeleted)
            throw new AlreadyDeletedException();

        return getContentManager().getNodesByParentNode(id, user);
    }

    /**
     * @return the node content elements
     * @throws IdentityManagerException
     * @throws ContentManagerException
     */
    public List<ContentItem> getItems() throws IdentityManagerException, ContentManagerException
    {
        if (isDeleted)
            throw new AlreadyDeletedException();

        return getContentManager().getContentByParentNode(id, user);
    }

    /**
     * Inspector for the 'name' attribute.
     * 
     * @return the name value
     */
    public String getName()
    {
        return name;
    }

    /**
     * @return the parent Node
     * @throws IdentityManagerException
     * @throws ContentManagerException
     */
    public Node getParentNode() throws IdentityManagerException, ContentManagerException
    {
        if (isDeleted)
            throw new AlreadyDeletedException();

        return getContentManager().getNodeById(parentNodeId, user);
    }

    /**
     * Inspector for the 'parentNodeId' attribute.
     * 
     * @return the parentNodeId value
     */
    public Long getParentNodeId()
    {
        return parentNodeId;
    }

    /**
     * Inspector for the 'user' attribute.
     * 
     * @return the user value
     */
    public IDIFUser getUser()
    {
        return user;
    }

    /**
     * Grants Node access to group. Grant access is done immediately
     * 
     * @param groupId
     *            the group to grant access
     * @return T if grant was succeeded
     * @throws ContentManagerException
     */
    public boolean grantAccessGroup(String groupId) throws ContentManagerException
    {
        ACLEntry entry = null;

        if (isDeleted)
            throw new AlreadyDeletedException();

        boolean ret = getContentManager().grantNodeAccessToGroup(id, groupId);

        if (ret && accessList != null)
        {
            entry = new ACLEntry();
            entry.setGroupID(groupId);
            accessList.add(entry);
        }

        return ret;
    }

    /**
     * Grants Node access to user. Grant access is done immediately
     * 
     * @param user
     *            the user to grant access
     * @return T if grant was succeeded
     * @throws ContentManagerException
     */
    public boolean grantAccessUser(IDIFUser user) throws ContentManagerException
    {
        ACLEntry entry = null;

        if (isDeleted)
            throw new AlreadyDeletedException();

        boolean ret = getContentManager().grantNodeAccessToUser(id, user);

        if (ret && accessList != null)
        {
            entry = new ACLEntry();
            entry.setUserID(user.getID());
            accessList.add(entry);
        }

        return ret;
    }

    /**
     * Inspector for the 'isDeleted' attribute.
     * 
     * @return the isDeleted value
     */
    public boolean isDeleted()
    {
        return isDeleted;
    }

    /**
     * Inspector for the 'isPublic' attribute.
     * 
     * @return the isPublic value
     */
    public boolean isPublic()
    {
        return isPublic;
    }

    /**
     * Revokes Node access to group. revoke access is done immediately
     * 
     * @param groupId
     *            the group to revoke access
     * @return T if revoke was succeeded
     * @throws ContentManagerException
     */
    public boolean revokeAccessGroup(String groupId) throws ContentManagerException
    {
        if (isDeleted)
            throw new AlreadyDeletedException();

        boolean ret = getContentManager().revokeNodeAccessToGroup(id, groupId);

        if (ret && accessList != null)
        {
            List<ACLEntry> newList = new ArrayList<ACLEntry>();
            for (ACLEntry entry: accessList)
            {
                if (entry.isUserEntry() || !entry.getGroupID().equals(groupId))
                    newList.add(entry);
            }
            accessList = newList;
        }

        return ret;
    }

    /**
     * Revokes Node access to group. revoke access is done immediately
     * 
     * @param user
     *            the user to revoke access
     * @return T if revoke was succeeded
     * @throws ContentManagerException
     */
    public boolean revokeAccessUser(IDIFUser user) throws ContentManagerException
    {
        if (isDeleted)
            throw new AlreadyDeletedException();

        boolean ret = getContentManager().revokeNodeAccessToUser(id, user);

        if (ret && accessList != null)
        {
            List<ACLEntry> newList = new ArrayList<ACLEntry>();
            for (ACLEntry entry: accessList)
            {
                if (entry.isGroupEntry() || !entry.getUserID().equals(user.getID()))
                    newList.add(entry);
            }
            accessList = newList;
        }

        return ret;
    }

    /**
     * Modifier for the 'creationDate' attribute.
     * 
     * @param creationDate
     *            the new creationDate value to set
     */
    public void setCreationDate(Date creationDate)
    {
        this.creationDate = creationDate;
    }

    /**
     * Modifier for the 'creatorID' attribute.
     * 
     * @param creatorID
     *            the new creatorID value to set
     */
    public void setCreatorID(String creatorID)
    {
        this.creatorID = creatorID;
    }

    /**
     * Modifier for the 'description' attribute.
     * 
     * @param description
     *            the new description value to set
     */
    public void setDescription(String description)
    {
        this.description = description;
    }

    /**
     * Modifier for the 'fullPathName' attribute.
     * 
     * @param fullPathName
     *            the new fullPathName value to set
     */
    public void setFullPathName(String fullPathName)
    {
        this.fullPathName = fullPathName;
    }

    /**
     * Modifier for the 'id' attribute.
     * 
     * @param id
     *            the new id value to set
     */
    public void setId(Long id)
    {
        this.id = id;
    }

    /**
     * Modifier for the 'name' attribute.
     * 
     * @param name
     *            the new name value to set
     */
    public void setName(String name)
    {
        this.name = name;
    }

    /**
     * Modifier for the 'parentNodeId' attribute.
     * 
     * @param parentNodeId
     *            the new parentNodeId value to set
     */
    public void setParentNodeId(long parentNodeId)
    {
        this.parentNodeId = parentNodeId;
    }

    /**
     * Modifier for the 'isPublic' attribute.
     * 
     * @param isPublic
     *            the new isPublic value to set
     */
    public void setPublic(boolean isPublic)
    {
        this.isPublic = isPublic;
    }

    /**
     * Sets the 'user' attribute.
     * 
     * @param user
     *            the user currently using the object
     */
    public void setUser(IDIFUser user)
    {
        this.user = user;
    }

    /**
     * Updates the Node with current object values. If the Node doesn't exist than it is inserted
     * 
     * @return T if update was succeeded
     * @throws IdentityManagerException
     * @throws ContentManagerException
     */
    public boolean update() throws IdentityManagerException, ContentManagerException
    {
        Node node = null;
        if (isDeleted || id == null)
        {
            node = getContentManager().addNode(this);
            id = node.getId();
        }
        else
        {
            try
            {
                node = getContentManager().updateNode(this);
            }
            catch (NodeNotFoundException e)
            {
                node = getContentManager().addNode(this);
                id = node.getId();
            }
        }
        return true;
    }

}
