/**
 * 2007, Digitalis Informatica. All rights reserved. Distribuicao e Gestao de Informatica, Lda. Estrada de Paco de Arcos
 * num.9 - Piso -1 2780-666 Paco de Arcos Telefone: (351) 21 4408990 Fax: (351) 21 4408999 http://www.digitalis.pt
 */

package pt.digitalis.dif.codegen.templates;

import pt.digitalis.dif.codegen.CGAncillaries;
import pt.digitalis.dif.controller.ExceptionHandlers;
import pt.digitalis.dif.controller.http.HTTPConstants;
import pt.digitalis.dif.controller.interfaces.IDIFContext;
import pt.digitalis.dif.controller.interfaces.IDispatcherErrorHandler;
import pt.digitalis.dif.dem.CallbackType;
import pt.digitalis.dif.dem.interfaces.IApplication;
import pt.digitalis.dif.dem.interfaces.IProvider;
import pt.digitalis.dif.dem.interfaces.IService;
import pt.digitalis.dif.dem.interfaces.IStage;
import pt.digitalis.dif.dem.interfaces.IStageInstance;
import pt.digitalis.dif.dem.managers.IDEMManager;
import pt.digitalis.dif.dem.managers.IMessageManager;
import pt.digitalis.dif.dem.managers.IParameterManager;
import pt.digitalis.dif.dem.managers.IRegistrationManager;
import pt.digitalis.dif.dem.objects.EventType;
import pt.digitalis.dif.dem.objects.ViewObject;
import pt.digitalis.dif.dem.objects.parameters.IParameters;
import pt.digitalis.dif.dem.objects.parameters.ParameterScope;
import pt.digitalis.dif.exception.InternalFrameworkException;
import pt.digitalis.dif.ioc.DIFIoCRegistry;
import pt.digitalis.dif.startup.DIFStartupConfiguration;
import pt.digitalis.dif.utils.extensions.document.IDocumentRepositoryManager;
import pt.digitalis.dif.utils.logging.DIFLogger;
import pt.digitalis.dif.utils.templates.TemplateUtils;
import pt.digitalis.log.LogLevel;
import pt.digitalis.utils.common.Chronometer;
import pt.digitalis.utils.common.StringUtils;
import pt.digitalis.utils.config.ConfigurationException;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * Resources used by the DEM Entity templates on the code generation.
 *
 * @author Pedro Viegas <a href="mailto:pviegas@digitalis.pt">pviegas@digitalis.pt</a>
 * @author Rodrigo Gonalves <a href="mailto:rgoncalves@digitalis.pt">rgoncalves@digitalis.pt</a>
 * @created Nov 6, 2007
 */
public final class TemplateResources
{

    /*
     * Implementation Note: This class holds resources used by the DEM entity templates. Since copying from template
     * doesn't initialize properly the private attributes, the resources used by the enhanced entities were placed on
     * this class thus enabling the entities to access them as needed.
     */

    /**
     * The Java script templates for each stage/view.
     */
    static public Map<String, Map<String, Map<String, String>>> javaScriptTemplatesByView = null;

    /** The DEM manager. */
    static private IDEMManager demManager = DIFIoCRegistry.getRegistry().getImplementation(IDEMManager.class);

    /** The document repository manager. */
    static private IDocumentRepositoryManager documentRepositoryManager = null;

    /** Singleton instance */
    static private TemplateResources instance = null;

    /** The message manager. */
    static private IMessageManager messageManager =
            DIFIoCRegistry.getRegistry().getImplementation(IMessageManager.class);

    /** The parameter manager */
    static private IParameterManager parameterManager =
            DIFIoCRegistry.getRegistry().getImplementation(IParameterManager.class);

    /** The registration manager. */
    static private IRegistrationManager registrationManager =
            DIFIoCRegistry.getRegistry().getImplementation(IRegistrationManager.class);

    /** Private constructor. Singleton implementation */
    private TemplateResources()
    {
    }

    /**
     * Returns a callback type
     *
     * @param id the id of the call back to return
     *
     * @return the registrationManager
     */
    static public CallbackType getCallBack(String id)
    {
        return CallbackType.valueOf(id);
    }

    /**
     * Returns the DEM manager.
     *
     * @return the demManager
     */
    static public IDEMManager getDEMManager()
    {
        return TemplateResources.demManager;
    }

    /**
     * Inspector for the 'documentRepositoryManager' attribute.
     *
     * @return the documentRepositoryManager value
     */
    public static IDocumentRepositoryManager getDocumentRepositoryManager()
    {
        if (documentRepositoryManager == null)
            documentRepositoryManager =
                    DIFIoCRegistry.getRegistry().getImplementation(IDocumentRepositoryManager.class);

        return documentRepositoryManager;
    }

    /**
     * Returns an event type
     *
     * @param id the id of the event to return
     *
     * @return the registrationManager
     */
    static public EventType getEventType(String id)
    {
        return EventType.valueOf(id);
    }

    /**
     * Instanciates a new User object of the given userType
     *
     * @param userType the user type to create
     * @param context  the current stage context
     *
     * @return the instanciated user object
     *
     * @exception ConfigurationException the configuration exception
     */
    static public Object getInjectedUser(String userType, IDIFContext context) throws ConfigurationException
    {
        return DIFIoCRegistry.getRegistry().getImplementation(IInjectUserCreator.class, userType).newUser(context);
    }

    /**
     * Gets instance.
     *
     * @return the singleton instance
     */
    static public TemplateResources getInstance()
    {
        if (instance == null)
            instance = new TemplateResources();

        return instance;
    }

    /**
     * Returns an IoC implementation
     *
     * @param clazz the class that defines the implementation interface
     *
     * @return the implementation
     */
    static public Object getIoCImplementation(String clazz)
    {
        try
        {
            return DIFIoCRegistry.getRegistry().getImplementation(Class.forName(clazz));
        }
        catch (ClassNotFoundException e)
        {
            return null;
        }
    }

    /**
     * Returns the message manager.
     *
     * @return the messageManager
     */
    static public IMessageManager getMessageManager()
    {
        return TemplateResources.messageManager;
    }

    /**
     * Returns the parameter manager.
     *
     * @return the parameterManager
     */
    static public IParameterManager getParameterManager()
    {
        return TemplateResources.parameterManager;
    }

    /**
     * Returns the parameter scope
     *
     * @param scope the scope as a param
     *
     * @return the parameter scope
     */
    static public ParameterScope getParameterScope(String scope)
    {
        return ParameterScope.valueOf(scope);
    }

    /**
     * Returns a IParameters instance Development note: This is a workaround for a Javassist bug. Could not have this
     * line of the generated code.
     *
     * @param stage the stage to get the parameters for
     *
     * @return the registrationManager
     */
    static public IParameters getParametersInstance(IStageInstance stage)
    {
        IParameters parameters = DIFIoCRegistry.getRegistry().getImplementation(IParameters.class);
        parameters.initialize(stage);

        return parameters;
    }

    /**
     * Returns the providers as a List of {@link IProvider}
     *
     * @return the registrationManager
     */
    static public List<IProvider> getProvidersAsList()
    {
        return java.util.Arrays.asList(getDEMManager().getProviders().values().toArray(new IProvider[] {null}));
    }

    /**
     * Returns the registration manager.
     *
     * @return the registrationManager
     */
    static public IRegistrationManager getRegistrationManager()
    {
        return TemplateResources.registrationManager;
    }

    /**
     * returns a given parameter form the request
     *
     * @param context the current context
     * @param id      the Id of the desired parameter
     *
     * @return the parameter value as string
     */
    static public String getRequestParameter(IDIFContext context, String id)
    {
        if (context != null && context.getRequest() != null)
        {
            Object result = context.getRequest().getParameter(id.toLowerCase());

            if (result == null)
                return null;
            else
                return result.toString();
        }
        else
            return null;
    }

    /**
     * Gets java script templates by view for stage.
     *
     * @param stageCGTemplate the stage cg template
     *
     * @return the java script templates by view for stage
     */
    public static Map<String, Map<String, String>> getJavaScriptTemplatesByViewForStage(IStage stageCGTemplate)
    {
        initializeJavaScriptTemplatesForViews();

        return javaScriptTemplatesByView.get(stageCGTemplate.getID());
    }

    /**
     * Initialize java script templates for views.
     */
    public synchronized static void initializeJavaScriptTemplatesForViews()
    {
        if (javaScriptTemplatesByView == null)
        {
            Chronometer crono = new Chronometer();
            crono.start();
            DIFLogger.getLogger().info("Parsing DEM to cache JavaScript templates...");

            HashMap<String, Map<String, Map<String, String>>> tempJavaScriptTemplatesByView =
                    new HashMap<String, Map<String, Map<String, String>>>();

            for (IProvider provider : getDEMManager().getProviders().values())
            {
                for (IApplication application : provider.getApplications().values())
                {
                    for (IService service : application.getServices().values())
                    {
                        for (IStage stage : service.getStages().values())
                        {
                            List<String> viewFiles = new ArrayList<String>();
                            Map<String, Map<String, String>> stageJavaScriptTemplatesByView =
                                    new HashMap<String, Map<String, String>>();

                            if (stage.getDefaultView() != null)
                                viewFiles.add(stage.getDefaultView().getTarget());

                            if (stage.getInjectedViews() != null)
                                for (ViewObject view : stage.getInjectedViews())
                                {
                                    if (!viewFiles.contains(view.getTarget()))
                                        viewFiles.add(view.getTarget());
                                }

                            if (stage.getInjectedErrorViews() != null)
                                for (ViewObject view : stage.getInjectedErrorViews().values())
                                {
                                    if (!viewFiles.contains(view.getTarget()))
                                        viewFiles.add(view.getTarget());
                                }

                            for (String view : viewFiles)
                            {
                                String headJSTemplateName = view.replace(".jsp", ".js");
                                String headFTLJSTemplateName = view.replace(".jsp", ".ftljs");
                                String onLoadJSTemplateName = view.replace(".jsp", ".onload.js");
                                String onLoadFTLJSTemplateName = view.replace(".jsp", ".onload.ftljs");

                                String headJSTemplateContent =
                                        TemplateUtils.getTemplateStreamContent(headJSTemplateName);
                                String headFTLJSTemplateContent =
                                        TemplateUtils.getTemplateStreamContent(headFTLJSTemplateName);
                                String onLoadJSTemplateContent =
                                        TemplateUtils.getTemplateStreamContent(onLoadJSTemplateName);
                                String onLoadFTLJSTemplateContent =
                                        TemplateUtils.getTemplateStreamContent(onLoadFTLJSTemplateName);

                                if (StringUtils.isNotBlank(headJSTemplateContent) ||
                                    StringUtils.isNotBlank(onLoadJSTemplateContent) ||
                                    StringUtils.isNotBlank(headFTLJSTemplateContent) ||
                                    StringUtils.isNotBlank(onLoadFTLJSTemplateContent))
                                {
                                    HashMap<String, String> viewTemplates = new HashMap<String, String>();

                                    if (StringUtils.isNotBlank(headJSTemplateContent))
                                        viewTemplates
                                                .put(CGAncillaries.JAVASCRIPT_TEMPLATE_HEAD_ID, headJSTemplateName);
                                    if (StringUtils.isNotBlank(headFTLJSTemplateContent))
                                        viewTemplates.put(CGAncillaries.JAVASCRIPT_FREE_MARKER_TEMPLATE_HEAD_ID,
                                                headFTLJSTemplateName);
                                    if (StringUtils.isNotBlank(onLoadJSTemplateContent))
                                        viewTemplates
                                                .put(CGAncillaries.JAVASCRIPT_TEMPLATE_ONLOAD_ID, onLoadJSTemplateName);
                                    if (StringUtils.isNotBlank(onLoadFTLJSTemplateContent))
                                        viewTemplates.put(CGAncillaries.JAVASCRIPT_FREE_MARKER_TEMPLATE_ONLOAD_ID,
                                                onLoadFTLJSTemplateName);

                                    for (Map.Entry<String, String> viewTemplate : viewTemplates.entrySet())
                                    {
                                        DIFLogger.getLogger()
                                                .log(DIFStartupConfiguration.getDeveloperMode() ? LogLevel.INFO
                                                                                                : LogLevel.DEBUG,
                                                        "    => " + application.getID() + "  " + service.getID() +
                                                        "  " + stage.getOriginalClassName().substring(
                                                                stage.getOriginalClassName().lastIndexOf(".") + 1) +
                                                        " (" + view + ") = " + viewTemplate.getValue());
                                    }

                                    stageJavaScriptTemplatesByView.put(view, viewTemplates);
                                }
                            }
                            tempJavaScriptTemplatesByView.put(stage.getID(), stageJavaScriptTemplatesByView);
                        }
                    }
                }
            }

            javaScriptTemplatesByView = tempJavaScriptTemplatesByView;
            DIFLogger.getLogger()
                    .info("JavaScript templates cached in " + crono.getTimePassedAsFormattedString() + ".");
        }
    }

    /**
     * Gets java script head template.
     *
     * @param stage the stage
     * @param view  the view
     *
     * @return the java script head template
     */
    public static String getJavaScriptHeadTemplate(IStage stage, ViewObject view)
    {
        Map<String, Map<String, String>> templatesForStage = getJavaScriptTemplatesByViewForStage(stage);
        if (templatesForStage != null)
        {
            Map<String, String> templatesForView = templatesForStage.get(view.getTarget());

            if (templatesForView != null)
                return templatesForView.get(CGAncillaries.JAVASCRIPT_TEMPLATE_HEAD_ID);
        }

        return null;
    }

    /**
     * Gets java script head template.
     *
     * @param stage the stage
     * @param view  the view
     *
     * @return the java script head template
     */
    public static String getJavaScriptHeadFreeMarkerTemplate(IStage stage, ViewObject view)
    {
        Map<String, Map<String, String>> templatesForStage = getJavaScriptTemplatesByViewForStage(stage);
        if (templatesForStage != null)
        {
            Map<String, String> templatesForView = templatesForStage.get(view.getTarget());

            if (templatesForView != null)
                return templatesForView.get(CGAncillaries.JAVASCRIPT_FREE_MARKER_TEMPLATE_HEAD_ID);
        }

        return null;
    }

    /**
     * Gets java script on load template.
     *
     * @param stage the stage
     * @param view  the view
     *
     * @return the java script on load template
     */
    public static String getJavaScriptOnLoadTemplate(IStage stage, ViewObject view)
    {
        Map<String, Map<String, String>> templatesForStage = getJavaScriptTemplatesByViewForStage(stage);
        if (templatesForStage != null)
        {
            Map<String, String> templatesForView = templatesForStage.get(view.getTarget());

            if (templatesForView != null)
                return templatesForView.get(CGAncillaries.JAVASCRIPT_TEMPLATE_ONLOAD_ID);
        }

        return null;
    }

    /**
     * Gets java script on load template.
     *
     * @param stage the stage
     * @param view  the view
     *
     * @return the java script on load template
     */
    public static String getJavaScriptOnLoadFreeMarkerTemplate(IStage stage, ViewObject view)
    {
        Map<String, Map<String, String>> templatesForStage = getJavaScriptTemplatesByViewForStage(stage);
        if (templatesForStage != null)
        {
            Map<String, String> templatesForView = templatesForStage.get(view.getTarget());

            if (templatesForView != null)
                return templatesForView.get(CGAncillaries.JAVASCRIPT_FREE_MARKER_TEMPLATE_ONLOAD_ID);
        }

        return null;
    }

    /**
     * The stage execute logic
     *
     * @param stageInstance the stage instance
     * @param context       the context
     *
     * @return the view object to render the stage output in
     */
    public ViewObject stageExecute(IStageInstance stageInstance, IDIFContext context)
    {
        // Declare result variable
        ViewObject result = null;

        // Will determine if we should return the default view. According to the event of default execution handler.
        boolean returnDefaultView = true;

        try
        {
            // FIXME: Refactor so that the AbstractDIFDispatcher remains abstracted from the HTTP
            // implementation. Use generic EventHandlers, instead of formSubmit
            Object submitStage = context.getRequest().getParameter(HTTPConstants.FORM_SUBMIT_STAGE);
            Object submitForm = context.getRequest().getParameter(HTTPConstants.FORM_SUBMIT_NAME);
            Object submitFormValidationRequest = context.getRequest().getParameter(HTTPConstants.FORM_VALIDATION);
            boolean isFormValidationRequest = Boolean.parseBoolean(
                    StringUtils.nvl(StringUtils.toStringOrNull(submitFormValidationRequest), "false"));

            Object eventID = context.getRequest().getParameter(HTTPConstants.EVENT_ID);

            // If form submitted check if an event handler of validation type for it exists (must call it prior to the
            // submit itself)
            if (stageInstance.getID().equalsIgnoreCase((String) submitStage) &&
                stageInstance.hasValidationLogicForForm(StringUtils.toStringOrNull(submitForm)))
                stageInstance.callExecuteOnEventMethod(context, EventType.FORM_VALIDATION, submitForm.toString());

            // If form submitted check if an event handler for it exists
            if (isFormValidationRequest && stageInstance.getID().equalsIgnoreCase((String) submitStage) &&
                stageInstance.hasValidationLogicForForm(StringUtils.toStringOrNull(submitForm)))
            {
                // The validation function has already been called. No other action is needed
                // Result can only be null here so no action is required
            }
            // If form submitted check if an event handler for it exists
            else if (stageInstance.getID().equalsIgnoreCase((String) submitStage) &&
                     (stageInstance.getEventHandlers().get(EventType.FORM_SUBMIT).contains(submitForm) ||
                      stageInstance.getEventHandlers().get(EventType.FORM_SUBMIT_SAVE_ACTION).contains(submitForm) ||
                      stageInstance.getEventHandlers().get(EventType.FORM_SUBMIT_AJAX_REQUEST).contains(submitForm)))
            {
                // Execute specific on submit event handler
                if (context.getRequest().isAjaxMode())
                {
                    result = stageInstance.callExecuteOnEventMethod(context, EventType.FORM_SUBMIT_AJAX_REQUEST,
                            submitForm.toString());
                    returnDefaultView = true;
                }
                else
                {
                    result = stageInstance
                            .callExecuteOnEventMethod(context, EventType.FORM_SUBMIT, submitForm.toString());
                    if (context.hasRedirection())
                    {
                        context.getSession()
                                .addAttribute(HTTPConstants.FORM_SUBMIT_AND_REDIRECTION, Boolean.TRUE.toString());
                    }
                }
            }
            else
            {
                boolean resultProcessed = false;

                if (eventID != null)
                {
                    for (EventType type : stageInstance.getEventHandlers().keySet())
                    {
                        for (String currentEventID : stageInstance.getEventHandlers().get(type))
                        {
                            if (currentEventID.equalsIgnoreCase(eventID.toString()))
                            {
                                // Execute specific on submit event handler
                                DIFLogger.getLogger().debug("_CG_execute: " + stageInstance.getID());
                                DIFLogger.getLogger().debug("Event: " + eventID + "(" + type + ")");

                                try
                                {
                                    result = stageInstance.callExecuteOnEventMethod(context, type, currentEventID);
                                    resultProcessed = true;
                                    returnDefaultView = returnDefaultView && (type != EventType.AJAX_REQUEST) &&
                                                        (type != EventType.FORM_SUBMIT_AJAX_REQUEST) &&
                                                        (type != EventType.DOCUMENT_TYPE);

                                    break;
                                }
                                catch (Exception exception)
                                {
                                    if (type == EventType.AJAX_REQUEST || type == EventType.FORM_SUBMIT_AJAX_REQUEST)
                                    {
                                        context.addStageResult(IDispatcherErrorHandler.EXCEPTION, exception);
                                        resultProcessed = true;

                                        break;
                                    }
                                    else
                                        throw exception;
                                }
                            }
                        }

                        if (resultProcessed)
                            break;
                    }
                }

                if (!resultProcessed)
                    // Execute normal business logic
                    result = stageInstance.callExecuteMethod(context);
            }

            // If execution was OK but no view was returned set the result as the default view
            if (result == null && returnDefaultView)
                result = stageInstance.getDefaultView();
        }
        catch (Exception exception)
        {
            exception.printStackTrace();
            String exceptionType = exception.getClass().getCanonicalName();

            // If an exception was raised check if there's a redirection to trigger...
            // ...current stage error view...
            if (stageInstance.getInjectedErrorViews().size() != 0 &&
                stageInstance.getInjectedErrorViews().containsKey(exceptionType))
            {
                context.addStageResult(IDispatcherErrorHandler.EXCEPTION, exception);
                result = stageInstance.getInjectedErrorViews().get(exceptionType);
            }
            // ...current stage error stage...
            else if (stageInstance.getInjectedErrorStages().size() != 0 &&
                     stageInstance.getInjectedErrorStages().containsKey(exceptionType))
            {
                context.addStageResult(IDispatcherErrorHandler.EXCEPTION, exception);
                context.redirectTo(stageInstance.getInjectedErrorStages().get(exceptionType));
            }
            // ...system wide exception handler stage...
            else if (ExceptionHandlers.hasHandler(exception))
            {
                ExceptionHandlers.handleException(context, exception);
            }
            else
                throw new RuntimeException(
                        "Unable to proceed! A business exception was raised and no error views or stages were" +
                        " defined for redirection. Aborting... ", new InternalFrameworkException(exception, context));
        }

        return result;
    }
}
