/**
 * 2007, Digitalis Informatica. All rights reserved. Distribuicao e Gestao de Informatica, Lda. Estrada de Paco de Arcos
 * num.9 - Piso -1 2780-666 Paco de Arcos Telefone: (351) 21 4408990 Fax: (351) 21 4408999 http://www.digitalis.pt
 */
package pt.digitalis.dif.codegen.util;

import pt.digitalis.dif.startup.DIFStartupConfiguration;
import pt.digitalis.dif.utils.IObjectFormatter;
import pt.digitalis.dif.utils.ObjectFormatter;
import pt.digitalis.dif.utils.ObjectFormatter.Format;
import pt.digitalis.log.LogLevel;
import pt.digitalis.utils.bytecode.exceptions.CodeGenerationException;
import pt.digitalis.utils.bytecode.holders.ClassHolder;
import pt.digitalis.utils.inspection.exception.ResourceNotFoundException;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * Defines a class enhancement object.
 *
 * @author Pedro Viegas <a href="mailto:pviegas@digitalis.pt">pviegas@digitalis.pt</a>
 * @author Rodrigo Gonalves <a href="mailto:rgoncalves@digitalis.pt">rgoncalves@digitalis.pt</a><br/>
 * @created Dec 10, 2007
 */
public class ClassEnhancements implements IObjectFormatter
{

    /** The class name for all entries. */
    private String className;

    /** List of incremental methods. */
    private List<String> incrementalMethods = new ArrayList<String>();

    /** The entries for all methods to enhance. */
    private Map<String, ClassMethodEnhancement> methodEnhancements = new HashMap<String, ClassMethodEnhancement>();

    /** The class holder for the enhanced object. */
    private ClassHolder theClassObject;

    /**
     * Builds a class enhancements object from the class name.
     *
     * @param clazz the class to be enhanced
     */
    public ClassEnhancements(ClassHolder clazz)
    {
        this.className = clazz.getFQName();
        this.theClassObject = clazz;
    }

    /**
     * Adds a processed enhancement to the class.
     *
     * @param enhancement the enhancement to add
     */
    public void addEnhancement(ClassMethodEnhancement enhancement)
    {
        methodEnhancements.put(enhancement.getMethodName(), enhancement);
    }

    /**
     * Adds a source to a given method. Takes care of the incremental logic according to the registered methods.
     *
     * @param methodName   the method name
     * @param methodSource the method source
     */
    public void addSource(String methodName, String methodSource)
    {
        ClassMethodEnhancement methodEnhancement = methodEnhancements.get(methodName);

        if (methodEnhancement == null)
            methodEnhancement = new ClassMethodEnhancement(methodName, incrementalMethods.contains(methodName));
        else
            methodSource = "\n" + methodSource;

        methodEnhancement.addSource(methodSource);
        methodEnhancements.put(methodName, methodEnhancement);
    }

    /**
     * Commits all the enhancements to a class file.
     *
     * @exception ResourceNotFoundException if the class can't be found
     * @exception CodeGenerationException   if the code can't be compiled
     */
    public void commitEnhancements() throws ResourceNotFoundException, CodeGenerationException
    {

        // Implement all enhancements
        for (String methodName : methodEnhancements.keySet())
        {
            String source = methodEnhancements.get(methodName).getSource();

            // In trace mode we see every CG line before it is executed.
            if (DIFStartupConfiguration.getLogLevel() == LogLevel.DEBUG ||
                DIFStartupConfiguration.getLogLevel() == LogLevel.TRACE)
                source = getSourceWithTraceInfo(methodName, source);

            if (!source.startsWith("{"))
                source = "{" + source + "}";

            this.theClassObject.updateMethodSource(methodName, source);
        }

        // Write class to disk
        this.theClassObject.writeClass();
    }

    /**
     * Returns the class name.
     *
     * @return the class name
     */
    public String getClassName()
    {
        return className;
    }

    /**
     * Returns the class object.
     *
     * @return the class object
     */
    public ClassHolder getClassObject()
    {
        return this.theClassObject;
    }

    /**
     * Returns the method enhancements map.
     *
     * @return the method enhancements map
     */
    public Map<String, ClassMethodEnhancement> getMethodEnhancements()
    {
        return methodEnhancements;
    }

    /**
     * Add trace info to system out for each line in the source code
     *
     * @param methodName the method name
     * @param source     the original source code without trace
     *
     * @return the source with trace added
     *
     * @exception ResourceNotFoundException if the class can't be found
     */
    private String getSourceWithTraceInfo(String methodName, String source) throws ResourceNotFoundException
    {
        String[] lines = source.split("\n");
        StringBuffer temp = new StringBuffer();

        for (String line : lines)
        {
            temp.append(
                    "System.out.println(\"" + this.theClassObject.getName() + " [" + methodName + "]: Executing - " +
                    line.replaceAll("\\\"", "\\\\\"").replaceAll("\\\\", "\\\\") + "\");\n");
            temp.append(line);
            temp.append("\n");
        }

        return temp.toString();
    }

    /**
     * Adds a method name to the list of incremental methods.
     *
     * @param methodName the name of the method to register
     */
    public void registerMethodAsIncremental(String methodName)
    {
        incrementalMethods.add(methodName);
    }

    /**
     * Adds a source to a given method. Takes care of the incremental logic according to the registered methods.
     *
     * @param methodName the method name
     * @param terminator the finalize code for the method
     */
    public void setTerminator(String methodName, String terminator)
    {
        ClassMethodEnhancement methodEnhancement = methodEnhancements.get(methodName);

        methodEnhancement.setTerminator(terminator);
        methodEnhancements.put(methodName, methodEnhancement);
    }

    @Override
    public ObjectFormatter toObjectFormatter(ObjectFormatter.Format format, List<Object> dumpedObjects)
    {
        ObjectFormatter formatter = new ObjectFormatter(format, dumpedObjects);

        formatter.addItem("Class Name", this.className);
        formatter.addItem("Incremental Methods", this.incrementalMethods);
        formatter.addItem("Method Enhancements", this.methodEnhancements);

        return formatter;
    }

    @Override
    public String toString()
    {
        return toObjectFormatter(Format.TEXT, null).getFormatedObject();
    }
}
