/**
 * - Digitalis Internal Framework v2.0 - (C) 2007, Digitalis Informatica. Distribuicao e Gestao de Informatica, Lda.
 * Estrada de Paco de Arcos num.9 - Piso -1 2780-666 Paco de Arcos Telefone: (351) 21 4408990 Fax: (351) 21 4408999
 * http://www.digitalis.pt
 */
package pt.digitalis.dif.codegen.util;

import com.google.inject.Inject;
import pt.digitalis.dif.codegen.CGAncillaries;
import pt.digitalis.dif.controller.ExceptionHandlers;
import pt.digitalis.dif.controller.security.managers.IAuthorizationManager;
import pt.digitalis.dif.controller.security.managers.IIdentityManager;
import pt.digitalis.dif.controller.security.objects.IDIFGroup;
import pt.digitalis.dif.controller.security.objects.IDIFUser;
import pt.digitalis.dif.dem.DEMAnnotationLogic;
import pt.digitalis.dif.dem.Entity;
import pt.digitalis.dif.dem.annotations.AnnotationMemberTags;
import pt.digitalis.dif.dem.annotations.AnnotationTags;
import pt.digitalis.dif.dem.annotations.Registrable;
import pt.digitalis.dif.dem.annotations.parameter.BindParameterIDs;
import pt.digitalis.dif.dem.annotations.parameter.FormConfigurable;
import pt.digitalis.dif.dem.annotations.parameter.Parameter;
import pt.digitalis.dif.dem.annotations.parameter.Persist;
import pt.digitalis.dif.dem.annotations.parameter.Rule;
import pt.digitalis.dif.dem.annotations.parameter.Rules;
import pt.digitalis.dif.dem.annotations.security.AccessControl;
import pt.digitalis.dif.dem.annotations.security.Group;
import pt.digitalis.dif.dem.annotations.security.Groups;
import pt.digitalis.dif.dem.annotations.security.User;
import pt.digitalis.dif.dem.annotations.security.Users;
import pt.digitalis.dif.dem.annotations.stage.ExceptionHandler;
import pt.digitalis.dif.dem.annotations.stage.LicenseEdition;
import pt.digitalis.dif.dem.managers.IParameterManager;
import pt.digitalis.dif.dem.managers.IRegistrationManager;
import pt.digitalis.dif.dem.managers.impl.UsageIssuesManagerImpl;
import pt.digitalis.dif.dem.objects.LicenseEditionType;
import pt.digitalis.dif.dem.objects.issues.IssueScope;
import pt.digitalis.dif.dem.objects.issues.IssueType;
import pt.digitalis.dif.dem.objects.issues.UsageIssue;
import pt.digitalis.dif.dem.objects.parameters.FormConfigurableDefinition;
import pt.digitalis.dif.dem.objects.parameters.IEditableParameter;
import pt.digitalis.dif.dem.objects.parameters.IParameter;
import pt.digitalis.dif.dem.objects.parameters.ParameterContext;
import pt.digitalis.dif.dem.objects.parameters.ParameterList;
import pt.digitalis.dif.dem.objects.parameters.ParameterScope;
import pt.digitalis.dif.dem.objects.parameters.constraints.impl.ParameterConstraintRegexImpl;
import pt.digitalis.dif.dem.objects.parameters.rules.IParameterRule;
import pt.digitalis.dif.dem.objects.parameters.rules.ParameterRuleAction;
import pt.digitalis.dif.exception.InternalFrameworkException;
import pt.digitalis.dif.exception.codegen.DIFCodeGenerationException;
import pt.digitalis.dif.exception.objects.ParameterException;
import pt.digitalis.dif.exception.security.AuthorizationManagerException;
import pt.digitalis.dif.exception.security.IdentityManagerException;
import pt.digitalis.dif.ioc.DIFIoCRegistry;
import pt.digitalis.dif.utils.logging.DIFLogger;
import pt.digitalis.utils.bytecode.exceptions.CodeGenerationException;
import pt.digitalis.utils.bytecode.holders.AnnotationHolder;
import pt.digitalis.utils.bytecode.holders.AnnotationMemberValueHolder;
import pt.digitalis.utils.bytecode.holders.AttributeHolder;
import pt.digitalis.utils.bytecode.holders.ClassHolder;
import pt.digitalis.utils.bytecode.holders.MethodHolder;
import pt.digitalis.utils.common.StringUtils;
import pt.digitalis.utils.config.ConfigurationException;
import pt.digitalis.utils.inspection.exception.ResourceNotFoundException;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;

/**
 * An implementation of the IClassEnhancer that uses Bytecode Utils to enhance the classes.
 *
 * @author Rodrigo Gon�alves <a href="mailto:rgoncalves@digitalis.pt">rgoncalves@digitalis.pt</a>
 * @author Pedro Viegas <a href="mailto:pviegas@digitalis.pt">pviegas@digitalis.pt</a>
 * @created Jul 5, 2007
 * @author Pedro Viegas <a href="mailto:pviegas@digitalis.pt">pviegas@digitalis.pt</a><br/>
 * @created 17 de Mar de 2011
 */

/**
 * @author Pedro Viegas <a href="mailto:pviegas@digitalis.pt">pviegas@digitalis.pt</a><br/>
 * @created 17 de Mar de 2011
 */

/**
 * @author Pedro Viegas <a href="mailto:pviegas@digitalis.pt">pviegas@digitalis.pt</a><br/>
 * @created 17 de Mar de 2011
 */
public class ClassEnhancerImpl implements IClassEnhancer
{

    /** Default password for new users */
    private static final String DEFAULT_PASSWORD = "@bcd1234";

    /** The groups processed. */
    static private List<String> groupsProcessed = Collections.synchronizedList(new ArrayList<String>());

    /** The users processed. */
    static private List<String> usersProcessed = Collections.synchronizedList(new ArrayList<String>());

    /** The authorization manager */
    private final IAuthorizationManager authorizationManager;

    /** The identity manager */
    private final IIdentityManager identityManager;

    /** The parameter manager */
    private final IParameterManager parameterManager;

    /** The registration manager */
    private final IRegistrationManager registrationManager;

    /** Map of all annotations and their corresponding Logic objects */
    private Map<String, DEMAnnotationLogic> annotationLogicMap = null;

    /**
     * Constructor.
     *
     * @param registrationManager
     *            the registration manager instance
     * @param parameterManager
     *            the parameter manager instance
     * @param identityManager
     *            the identity manager instance
     * @param authorizationManager
     *            the authorization manager instance
     * @throws CodeGenerationException
     *             if the needed resources for class enhancement could not be properly initialized
     */
    @Inject
    public ClassEnhancerImpl(IRegistrationManager registrationManager, IParameterManager parameterManager,
            IIdentityManager identityManager, IAuthorizationManager authorizationManager) throws CodeGenerationException
    {

        this.registrationManager = registrationManager;
        this.parameterManager = parameterManager;
        this.identityManager = identityManager;
        this.authorizationManager = authorizationManager;

        try
        {
            annotationLogicMap = DEMLoaderHelper.getAnnotationLogicMap();
        }
        catch (Exception exception)
        {
            throw new CodeGenerationException("Could not properly initialize resources ", exception);
        }
    }

    /**
     * Enhances the methods of a given class.
     *
     * @param classEnhancementContext
     *            the class enhancement context
     * @throws ResourceNotFoundException
     *             if class can not be loaded
     * @throws CodeGenerationException
     *             if class can not be enhanced
     * @throws DIFCodeGenerationException
     */
    private void enhaceMethods(ClassEnhancementContext classEnhancementContext)
            throws ResourceNotFoundException, CodeGenerationException, DIFCodeGenerationException
    {

        ClassHolder clazz = classEnhancementContext.getOriginalClassObject();

        for (MethodHolder method : clazz.getAllMethods().values())
        {

            Collection<AnnotationHolder> annotations = method.getAnnotations().values();

            // If there are annotations to process (method can have no
            // annotations!)
            if (annotations != null && annotations.size() > 0)
            {

                for (AnnotationHolder annotation : annotations)
                {

                    // ...get it's associated object...
                    DEMAnnotationLogic annotationLogic = annotationLogicMap.get(annotation.getName());

                    // ...if the object exists...
                    if (annotationLogic != null)
                    {
                        // if annotation usage is correct process it!
                        annotationLogic.validateUsage(method);
                        processAnnotation(classEnhancementContext, method, annotation, annotationLogic);
                    }
                    else
                        DIFLogger.getLogger().debug("Annotation @" + annotation.getName() +
                                                    " has no AnnotationLogicClass defined. Skipping...");
                }
            }
        }
    }

    /**
     * Reads and iterates over the class annotations, processing them according to their target scope.
     *
     * @see pt.digitalis.dif.codegen.util.IClassEnhancer#enhance(ClassHolder)
     */
    public ClassHolder enhance(ClassHolder clazz)
            throws ResourceNotFoundException, CodeGenerationException, DIFCodeGenerationException,
            IdentityManagerException, ConfigurationException
    {

        ClassEnhancementContext classEnhancementContext = new ClassEnhancementContext(clazz);

        enhanceClass(classEnhancementContext);
        enhanceAttributes(classEnhancementContext);
        enhaceMethods(classEnhancementContext);

        postProcessActions(classEnhancementContext);

        // Commit changes
        classEnhancementContext.commitEnhancements();

        return classEnhancementContext.getEntityClass();
    }

    /**
     * Enhances the attributes of a given class.
     *
     * @param classEnhancementContext
     *            the class enhancement context
     * @throws ResourceNotFoundException
     *             if class can not be loaded
     * @throws CodeGenerationException
     *             if class can not be enhanced
     * @throws DIFCodeGenerationException
     * @throws ConfigurationException
     */
    private void enhanceAttributes(ClassEnhancementContext classEnhancementContext)
            throws ResourceNotFoundException, CodeGenerationException, DIFCodeGenerationException,
            ConfigurationException
    {

        ClassHolder clazz = classEnhancementContext.getOriginalClassObject();

        // For each attribute...
        for (AttributeHolder attribute : clazz.getAttributes().values())
        {
            // For each annotation...
            for (AnnotationHolder annotation : attribute.getAnnotations().values())
            {
                // ...get it's associated object...
                DEMAnnotationLogic annotationLogic = annotationLogicMap.get(annotation.getName());

                // ...if the object exists...
                if (annotationLogic != null)
                {
                    // if annotation usage is correct process it!
                    annotationLogic.validateUsage(attribute);
                    processAnnotation(classEnhancementContext, attribute, annotation, annotationLogic);
                }
                else
                    DIFLogger.getLogger().debug("Annotation @" + annotation.getName() +
                                                " has no AnnotationLogicClass defined. Skipping...");
            }
        }
    }

    /**
     * Enhances a given class. Analyzes class annotations and implements the needed interfaces and methods according.
     *
     * @param classEnhancementContext
     *            the class enhancement context
     * @throws ResourceNotFoundException
     *             if class can not be loaded
     * @throws CodeGenerationException
     *             if class can not be enhanced
     * @throws DIFCodeGenerationException
     * @throws ConfigurationException
     * @throws IdentityManagerException
     */
    private void enhanceClass(ClassEnhancementContext classEnhancementContext)
            throws ResourceNotFoundException, CodeGenerationException, DIFCodeGenerationException,
            IdentityManagerException, ConfigurationException
    {

        ClassHolder clazz = classEnhancementContext.getOriginalClassObject();

        for (AnnotationHolder annotation : clazz.getAnnotations().values())
        {
            DEMAnnotationLogic annotationLogic = annotationLogicMap.get(annotation.getName());

            // ...if the object exists...
            if (annotationLogic != null)
            {
                // if annotation usage is correct process it!
                annotationLogic.validateUsage(clazz);
                processAnnotation(classEnhancementContext, annotation, annotationLogic);
            }
            else
                DIFLogger.getLogger().debug("Annotation @" + annotation.getName() +
                                            " has no AnnotationLogicClass defined. Skipping...");
        }
    }

    /**
     * Infers a given {@link Rule} annotation to a {@link IParameterRule} object
     *
     * @param parameterID
     *            the id of the associated stage parameter
     * @param ruleAnnotation
     *            the rule annotation
     * @return the converted {@link IParameterRule}
     * @throws ResourceNotFoundException
     */
    private IParameterRule<?> getRule(String parameterID, AnnotationHolder ruleAnnotation)
            throws ResourceNotFoundException
    {

        IParameterRule<?> paramRule = null;

        if (ruleAnnotation != null)
        {
            String ruleID = null;
            String ruleParameters = null;
            ParameterRuleAction ruleAction = ParameterRuleAction.HIDE;

            Map<String, AnnotationMemberValueHolder> ruleValues = ruleAnnotation.getMembers();

            AnnotationMemberValueHolder temp;

            // Override default from the declared parameters in @Persist
            temp = ruleValues.get(AnnotationMemberTags.RULE_ID);

            if (temp != null)
            {
                ruleID = temp.toString();

                temp = ruleValues.get(AnnotationMemberTags.RULE_PARAMETER_LIST);

                if (temp != null)
                {
                    ruleParameters = temp.toString();

                    temp = ruleValues.get(AnnotationMemberTags.RULE_ACTION);

                    if (temp != null && temp.enumValuetoString() != null)
                        ruleAction = ParameterRuleAction.valueOf(temp.enumValuetoString());

                    String value = ruleValues.get(AnnotationMemberTags.RULE_VALUE).toString();
                    String first = ruleValues.get(AnnotationMemberTags.RULE_FIRST).toString();
                    String last = ruleValues.get(AnnotationMemberTags.RULE_LAST).toString();

                    if (AnnotationTags.NONE.equals(value))
                        value = null;

                    if (AnnotationTags.NONE.equals(first))
                        first = null;

                    if (AnnotationTags.NONE.equals(last))
                        last = null;

                    paramRule = DIFIoCRegistry.getRegistry().getImplementation(IParameterRule.class, ruleID);

                    if (paramRule != null)
                        paramRule.init(parameterID, ruleParameters, ruleAction, value, first, last);
                }
            }
        }

        return paramRule;
    }

    /**
     * Post processing actions. Will analize the current DEm entity, with their processed annotations and perform
     * context analisys and code generation
     *
     * @param classEnhancementContext
     * @throws ResourceNotFoundException
     */
    private void postProcessActions(ClassEnhancementContext classEnhancementContext) throws ResourceNotFoundException
    {

        String entityID = EntityUtils.getEntityID(classEnhancementContext.getOriginalClassObject());
        Entity entityType = EntityUtils.getEntityType(classEnhancementContext.getOriginalClassObject());

        if (entityType != Entity.VALIDATOR)
        {
            ParameterList parameters = parameterManager.getParameters(entityType, entityID);

            // Parse all current entity parameters and each of their optional rules. for each set the dependent
            // parameter ReferencedInAnotherRule parameter
            if (parameters != null)
                for (IParameter<?> parameter : parameters.getParameters().values())
                {
                    for (IParameterRule<?> rule : parameter.getRules())
                    {
                        for (String dependentParameterId : rule.getParameters())
                        {

                            IParameter<?> param = parameters.getParameter(dependentParameterId.trim());

                            if (param == null)
                            {
                                UsageIssue issue = new UsageIssue();
                                issue.setIssueScope(IssueScope.LOADTIME);
                                issue.setIssueType(IssueType.ERROR);
                                issue.setIssueDescription("The parameter " + dependentParameterId.trim() +
                                                          " referenced in a @Rule of the parameter " +
                                                          parameter.getId() + " does not exist in the stage");
                                issue.setLocation(classEnhancementContext.getOriginalClassObject().getFQName());

                                UsageIssuesManagerImpl.getInstance().addIssue(issue);
                            }
                            else
                                param.setReferencedInARuleFromAnotherParameter(true);
                        }
                    }
                }
        }
    }

    /**
     * @see pt.digitalis.dif.codegen.util.IClassEnhancer#processAnnotation(pt.digitalis.dif.codegen.util.ClassEnhancementContext,
     *      pt.digitalis.utils.bytecode.holders.AnnotationHolder, pt.digitalis.dif.dem.DEMAnnotationLogic)
     */
    public void processAnnotation(ClassEnhancementContext classEnhancementContext, AnnotationHolder annotation,
            DEMAnnotationLogic annotationLogic)
            throws ResourceNotFoundException, CodeGenerationException, DIFCodeGenerationException,
            ConfigurationException, IdentityManagerException
    {

        String annotationName = annotation.getName();

        ClassHolder clazz = classEnhancementContext.getOriginalClassObject();

        // *** Process behavior-triggering annotations ***

        if (annotationName.equals(ExceptionHandler.class.getCanonicalName()))
        {
            String[] exceptions = CGAncillaries
                    .parse(annotation.getMembers().get(AnnotationMemberTags.ERROR_STAGE_EXCEPTIONS).toString(),
                            CGAncillaries.COMMA);

            for (int i = 0; i < exceptions.length; i++)
            {
                ExceptionHandlers.addExceptionHandler(exceptions[i], EntityUtils.getEntityID(clazz));
            }
        }
        else if (annotationName.equals(Registrable.class.getCanonicalName()))
        {

            Entity entityType = EntityUtils.getEntityType(clazz);
            String entityID = EntityUtils.getEntityID(clazz);
            String defaultName = annotation.getMembers().get("defaultRegisterName").toString();

            if (entityType != null)
            {
                // No default name means an unregistered but registrable registration.
                // Inaccessible until a valid registration occurs.

                // A default name will force a non-registrable so it can be accessed with the default name.
                if (defaultName.equals(AnnotationTags.NONE))
                    defaultName = "";

                registrationManager.addToRegistry(entityType, entityID, defaultName, "".equals(defaultName));
            }
        }
        else if (annotationName.equals(LicenseEdition.class.getCanonicalName()))
        {

            Entity entityType = EntityUtils.getEntityType(clazz);
            String entityID = EntityUtils.getEntityID(clazz);

            if (entityType != null)
            {
                String value = annotation.getMembers().get("value").enumValuetoString();
                registrationManager.addEntityLicenseEdition(entityType, entityID, LicenseEditionType.valueOf(value));
            }
        }
        else if (annotationName.equals(BindParameterIDs.class.getCanonicalName()))
        {
            // Parse String to array
            // String[] parametersIDs =
            // CGAncillaries.parse(annotation.getMembers().get("ids").toString(),
            // ",");

            // TODO: call the appropriate method on the ValidatorManager
            // Throw UnsupportedOpertationException to comply with Java
            // standards and warn user of unimplemented
            // feature.
            throw new UnsupportedOperationException(
                    annotationName + " not yet supported! Please remove it from class " + clazz.getFQName());
        }
        else if (annotationName.equals(Users.class.getCanonicalName()))
        {
            for (AnnotationHolder innerUser : annotation.getMembers().get(AnnotationMemberTags.VALUE)
                    .toAnnotationList(clazz))
            {
                processUserAnnotation(clazz, innerUser);
            }
        }
        else if (annotationName.equals(Groups.class.getCanonicalName()))
        {
            for (AnnotationHolder innerGroup : annotation.getMembers().get(AnnotationMemberTags.VALUE)
                    .toAnnotationList(clazz))
            {
                processGroupAnnotation(clazz, innerGroup);
            }
        }

        else if (annotationName.equals(User.class.getCanonicalName()))
        {
            processUserAnnotation(clazz, annotation);
        }
        else if (annotationName.equals(Group.class.getCanonicalName()))
        {
            processGroupAnnotation(clazz, annotation);
        }
        else if (annotationName.equals(AccessControl.class.getCanonicalName()))
        {

            // Get the parent info
            Entity entityType = EntityUtils.getEntityType(clazz);
            String entityID = EntityUtils.getEntityID(clazz);

            if (entityType.equals(Entity.APPLICATION) || entityType.equals(Entity.SERVICE) ||
                entityType.equals(Entity.STAGE))
            {
                // Get the parameters of the annotation
                String[] userIDs = CGAncillaries.parse(annotation.getMembers().get("users").toString(), ",");
                String[] groupIDs = CGAncillaries.parse(annotation.getMembers().get("groups").toString(), ",");
                boolean accessToNone = annotation.getMembers().get("none").toBoolean();

                if (userIDs.length == 0 || userIDs[0].equals(""))
                    userIDs = new String[0];
                if (groupIDs.length == 0 || groupIDs[0].equals(""))
                    groupIDs = new String[0];

                try
                {
                    for (String userID : userIDs)
                    {
                        // Parse the user id and profile, using the notation:
                        // "userID:profileID, userID2:profileID2..."
                        String[] userData = CGAncillaries.parse(userID, ":");

                        if (!usersProcessed.contains(userData[0]) && !identityManager.userExists(userData[0]))
                        {

                            if (userData.length < 2)
                            {
                                throw new DIFCodeGenerationException(
                                        "The annotation AccessControl must declare a user with a valid profile. Configuration: " +
                                        annotation.getMembers().get("users").toString());
                            }

                            if (!groupsProcessed.contains(userData[1]) && !identityManager.groupExists(userData[1]))
                            {
                                // Even if it fails will not try again. LDAP errors due to missing access on purpose are
                                // common and should be repeated at startup
                                groupsProcessed.add(userData[1]);

                                IDIFGroup group = DIFIoCRegistry.getRegistry().getImplementation(IDIFGroup.class);
                                group.setID(userData[1]);
                                group.setName(userData[1]);

                                identityManager.addGroup(group);
                            }

                            // Even if it fails will not try again. LDAP errors due to missing access on purpose are
                            // common and should be repeated at startup
                            usersProcessed.add(userData[0]);

                            IDIFUser user = DIFIoCRegistry.getRegistry().getImplementation(IDIFUser.class);
                            user.setID(userData[0]);
                            user.setName(userData[0]);
                            user.setNick(userData[0]);
                            user.setPassword(userData[0]);
                            user.setProfileID(userData[1]);

                            identityManager.addUser(user);
                        }

                        authorizationManager.grantDefaultAccessToUser(userData[0], entityType, entityID);
                    }

                    for (String groupID : groupIDs)
                    {
                        if (!groupsProcessed.contains(groupID) && !identityManager.groupExists(groupID))
                        {
                            // Even if it fails will not try again. LDAP errors due to missing access on purpose are
                            // common and should be repeated at startup
                            groupsProcessed.add(groupID);

                            IDIFGroup group = DIFIoCRegistry.getRegistry().getImplementation(IDIFGroup.class);
                            group.setID(groupID);
                            group.setName(groupID);

                            identityManager.addGroup(group);
                        }

                        authorizationManager.grantDefaultAccessToGroup(groupID, entityType, entityID);
                    }
                }
                catch (IdentityManagerException identityManagerException)
                {
                    if (identityManager.isReadOnly())
                        DIFLogger.getLogger().info("Identity Manager in Read Only mode.");
                    else
                        DIFLogger.getLogger()
                                .warn("Could not access the identity manager!" + identityManagerException.getMessage());
                }
                catch (AuthorizationManagerException authorizationManagerException)
                {
                    throw new ResourceNotFoundException("Could not access the authorization manager!",
                            authorizationManagerException);
                }

                if (accessToNone)
                    authorizationManager.revokeAccessFromPublic(entityType, entityID);
            }
        }
        // *** Process code generation annotations ***
        else
            // Get the source code for the annotation
            annotationLogic.addSourceCodeForAnnotation(annotation, classEnhancementContext);
    }

    /**
     * @see pt.digitalis.dif.codegen.util.IClassEnhancer#processAnnotation(pt.digitalis.dif.codegen.util.ClassEnhancementContext,
     *      pt.digitalis.utils.bytecode.holders.AttributeHolder, pt.digitalis.utils.bytecode.holders.AnnotationHolder,
     *      pt.digitalis.dif.dem.DEMAnnotationLogic)
     */
    @SuppressWarnings("unchecked")
    public void processAnnotation(ClassEnhancementContext classEnhancementContext, AttributeHolder attribute,
            AnnotationHolder annotation, DEMAnnotationLogic annotationLogic)
            throws ResourceNotFoundException, CodeGenerationException, DIFCodeGenerationException,
            ConfigurationException
    {

        // *** Process behavior-triggering annotations ***

        String annotationName = annotation.getName();
        boolean generateSourceCode = true;

        // The following section implements the following annotations:
        // Parameter, Persist, Validator
        if (annotationName.equals(Parameter.class.getCanonicalName()))
        {
            Map<String, AnnotationMemberValueHolder> paramValues = annotation.getMembers();

            String id = paramValues.get(AnnotationMemberTags.PARAMETER_ID).toString();
            String constraints = paramValues.get(AnnotationMemberTags.PARAMETER_CONSTRAINTS).toString();
            String defaultValue = paramValues.get(AnnotationMemberTags.PARAMETER_DEFAULT_VALUE).toString();
            String linkForm = paramValues.get(AnnotationMemberTags.PARAMETER_LINK_TO_FORM).toString();
            Boolean trusted = paramValues.get(AnnotationMemberTags.PARAMETER_TRUSTED).toBoolean();
            String[] regexp = paramValues.get(AnnotationMemberTags.PARAMETER_REGEXP_VALUE).toStringArray();

            ParameterScope scope = ParameterScope.REQUEST;
            String scopeStr = paramValues.get(AnnotationMemberTags.PERSIST_SCOPE).enumValuetoString();
            if (StringUtils.isNotBlank(scopeStr))
            {
                scope = ParameterScope.valueOf(scopeStr);
            }

            ParameterContext parameterContext = null;
            String contextStr = paramValues.get(AnnotationMemberTags.PERSIST_CONTEXT).enumValuetoString();
            if (StringUtils.isNotBlank(contextStr))
            {
                parameterContext = ParameterContext.valueOf(contextStr);
            }

            // No id given, use the attribute name to lowercase
            if (id.equals(AnnotationTags.GENERATE_ID))
                id = attribute.getName().toLowerCase();

            if (trusted)
            {
                classEnhancementContext.addEnhancement(CGAncillaries.STAGE_TRUSTED_PARAMETERS_BUILDER,
                        "trustedParameters.add(\"" + id + "\");");
            }

            // No constraints identifier, set to empty constraint string
            if (constraints.equals(AnnotationTags.NONE))
                constraints = "";

            if (regexp.length == 1)
            {
                if (constraints.length() > 0)
                    constraints += ",";
                constraints += "regex=" + regexp[0];
            }
            else if (regexp.length == 2)
            {
                if (constraints.length() > 0)
                    constraints += ",";

                constraints += "regex=" + ParameterConstraintRegexImpl.REGEXP_LIST_START +
                               regexp[0].replaceAll(",", ParameterConstraintRegexImpl.REGEXP_COMMA) +
                               ParameterConstraintRegexImpl.REGEXP_LIST_SEPARATOR +
                               regexp[1].replaceAll(",", ParameterConstraintRegexImpl.REGEXP_COMMA) +
                               ParameterConstraintRegexImpl.REGEXP_LIST_END;
            }
            else if (regexp.length > 2)
            {
                UsageIssuesManagerImpl.getInstance().addIssue(IssueType.ERROR, IssueScope.LOADTIME,
                        classEnhancementContext.getOriginalClassObject().getFQName(),
                        "Parameter \"" + annotation.getParentAttribute().getName() +
                        "\" has more than 2 regular expressions associated. " +
                        "Must have one if common for java and JavaScript or two, if diferent. " +
                        "More that two is a configuration error. All regular expressions will be ignored!\"", null);
            }

            // No default value, set to empty string
            if (defaultValue.equals(AnnotationTags.NONE))
                defaultValue = "";

            // No form to link, set to null
            if (linkForm.equals(AnnotationTags.NONE))
                linkForm = null;

            // Defaults - Persist
            boolean persistentToRepository = false;
            boolean allowAnonymous = false;

            // Check for the persist annotation
            AnnotationHolder persistAnnotation =
                    annotation.getParentAttribute().getAnnotations().get(Persist.class.getCanonicalName());

            if (persistAnnotation != null)
            {
                Map<String, AnnotationMemberValueHolder> persistValues = persistAnnotation.getMembers();

                // Override default from the declared parameters in @Persist
                scope = ParameterScope
                        .valueOf(persistValues.get(AnnotationMemberTags.PERSIST_SCOPE).enumValuetoString());
                persistentToRepository = persistValues.get(AnnotationMemberTags.PERSIST_REPOSITORY).toBoolean();
                allowAnonymous = persistValues.get(AnnotationMemberTags.PERSIST_ALLOW_ANONYMOUS).toBoolean();

                // TODO: Implement this. Wait for the User/Group Manager
                // String groupVisibility =
                // persistValues.get(AnnotationMemberTags.PERSIST_GROUP_VISIBILITY).toString();
            }

            // Check for the formConfigurable annotation
            FormConfigurableDefinition formConfigurableDef = new FormConfigurableDefinition();

            AnnotationHolder formConfigurableAnnotation =
                    annotation.getParentAttribute().getAnnotations().get(FormConfigurable.class.getCanonicalName());

            formConfigurableDef.setConfigurable(formConfigurableAnnotation != null);

            if (formConfigurableAnnotation != null && formConfigurableDef.isConfigurable())
            {
                if (formConfigurableAnnotation.getMembers()
                        .containsKey(AnnotationMemberTags.FORM_CONFIGURABLE_CAN_EDIT_MANDATORY))
                {
                    formConfigurableDef.setCanEditMandatory(formConfigurableAnnotation.getMembers()
                            .get(AnnotationMemberTags.FORM_CONFIGURABLE_CAN_EDIT_MANDATORY).toBoolean());
                }

                if (formConfigurableAnnotation.getMembers()
                        .containsKey(AnnotationMemberTags.FORM_CONFIGURABLE_CAN_EDIT_READONLY))
                {
                    formConfigurableDef.setCanEditReadOnly(formConfigurableAnnotation.getMembers()
                            .get(AnnotationMemberTags.FORM_CONFIGURABLE_CAN_EDIT_READONLY).toBoolean());
                }

                if (formConfigurableAnnotation.getMembers()
                        .containsKey(AnnotationMemberTags.FORM_CONFIGURABLE_CAN_EDIT_TEXT))
                {
                    formConfigurableDef.setCanEditText(formConfigurableAnnotation.getMembers()
                            .get(AnnotationMemberTags.FORM_CONFIGURABLE_CAN_EDIT_TEXT).toBoolean());
                }
            }

            // Defaults - Rule
            List<IParameterRule<?>> rules = new ArrayList<IParameterRule<?>>();
            boolean ruleWithError = false;

            // Check for the rule annotation
            AnnotationHolder ruleAnnotation =
                    annotation.getParentAttribute().getAnnotations().get(Rule.class.getCanonicalName());

            if (ruleAnnotation != null)
            {
                IParameterRule<?> rule = getRule(id, ruleAnnotation);
                if (rule != null)
                    rules.add(rule);
                else
                    ruleWithError = true;
            }

            // Check for the rules annotation
            AnnotationHolder rulesAnnotation =
                    annotation.getParentAttribute().getAnnotations().get(Rules.class.getCanonicalName());

            if (rulesAnnotation != null)
                for (AnnotationHolder innerRule : rulesAnnotation.getMembers().get(AnnotationMemberTags.VALUE)
                        .toAnnotationList(attribute))
                {
                    IParameterRule<?> rule = getRule(id, innerRule);
                    if (rule != null)
                        rules.add(rule);
                    else
                        ruleWithError = true;
                }

            // TODO: Implement @Validator

            @SuppressWarnings("rawtypes")
            IParameter param =
                    parameterManager.getParameterInstanceForType(annotation.getParentAttribute().getAttributeType());

            if (param == null)
            {
                generateSourceCode = false;
                UsageIssuesManagerImpl.getInstance().addIssue(IssueType.ERROR, IssueScope.LOADTIME,
                        classEnhancementContext.getOriginalClassObject().getFQName(),
                        "Parameter \"" + annotation.getParentAttribute().getName() + "\" has an unsupported type \"" +
                        annotation.getParentAttribute().getAttributeType() + "\"", null);
            }
            else
            {
                Entity parentType = EntityUtils.getEntityType(attribute.getParentClass());
                String parentID = EntityUtils.getEntityID(attribute.getParentClass());

                if (ruleWithError)
                {
                    UsageIssuesManagerImpl.getInstance().addIssue(IssueType.ERROR, IssueScope.LOADTIME,
                            classEnhancementContext.getOriginalClassObject().getFQName(),
                            "Parameter \"" + annotation.getParentAttribute().getName() + "\" has an unsupported rule",
                            null);
                }

                try
                {
                    param.initialize(id, parentType, parentID, formConfigurableDef, persistentToRepository,
                            allowAnonymous, scope, defaultValue, constraints, null, rules);
                    param.setFormLinked(linkForm);
                    ((IEditableParameter) param).setParameterContext(parameterContext);
                    parameterManager.registerParameter(param);
                }
                catch (ParameterException e)
                {
                    // Exception initializing the parameter. This parameter will be
                    // unavailable.
                    generateSourceCode = false;
                }

                // Only generate initialization code if it is a stage. If on another
                // entity it will only declare the
                // parameter and not inject it
                generateSourceCode = (parentType == Entity.STAGE);
            }
        }

        // If the code generation has not been overridden with a
        // behavior-triggering annotation generate it
        if (generateSourceCode)
            annotationLogic.addSourceCodeForAnnotation(classEnhancementContext, annotation, attribute);
    }

    /**
     * @see pt.digitalis.dif.codegen.util.IClassEnhancer#processAnnotation(pt.digitalis.dif.codegen.util.ClassEnhancementContext,
     *      pt.digitalis.utils.bytecode.holders.MethodHolder, pt.digitalis.utils.bytecode.holders.AnnotationHolder,
     *      pt.digitalis.dif.dem.DEMAnnotationLogic)
     */
    public void processAnnotation(ClassEnhancementContext classEnhancementContext, MethodHolder method,
            AnnotationHolder annotation, DEMAnnotationLogic annotationLogic)
            throws ResourceNotFoundException, CodeGenerationException, DIFCodeGenerationException
    {

        // Get the source code for the annotation
        annotationLogic.addSourceCodeForAnnotation(classEnhancementContext, annotation, method);
    }

    /**
     * Process the Group annotation
     *
     * @param clazz
     *            the class instance
     * @param annotation
     *            the annotation to process
     * @throws ResourceNotFoundException
     * @throws IdentityManagerException
     */
    private void processGroupAnnotation(ClassHolder clazz, AnnotationHolder annotation)
            throws ResourceNotFoundException, IdentityManagerException
    {
        // Get the parent info
        Entity entityType = EntityUtils.getEntityType(clazz);

        if (!identityManager.isReadOnly() && entityType.equals(Entity.APPLICATION))
        {
            String groupName = annotation.getMembers().get("groupName").toString();

            try
            {
                if (!groupsProcessed.contains(groupName) && !identityManager.groupExists(groupName))
                {
                    // Even if it fails will not try again. LDAP errors due to missing access on purpose are
                    // common and should be repeated at startup
                    groupsProcessed.add(groupName);

                    String fullName = annotation.getMembers().get("fullName").toString();
                    String groupParent = annotation.getMembers().get("groupParent").toString();

                    // Set dynamic default values...
                    if ("".equals(fullName))
                        fullName = groupName;

                    // Check for parent group...
                    if (!"".equals(groupParent) && !identityManager.groupExists(groupParent))
                    {
                        IDIFGroup group = DIFIoCRegistry.getRegistry().getImplementation(IDIFGroup.class);
                        group.setID(groupParent);
                        group.setName(groupParent);
                        group.setDescription(groupParent);
                        group.setDefault(true);

                        identityManager.addGroup(group);
                    }

                    // Create the group
                    IDIFGroup group = DIFIoCRegistry.getRegistry().getImplementation(IDIFGroup.class);
                    group.setID(groupName);
                    group.setName(groupName);
                    group.setDescription(fullName);

                    if (!"".equals(groupParent))
                        group.setParentGroupID(groupParent);

                    group.setDefault(true);

                    identityManager.addGroup(group);
                }
            }
            catch (IdentityManagerException identityManagerException)
            {
                if (identityManager.isReadOnly())
                    DIFLogger.getLogger().info("Identity Manager in Read Only mode.");
                else
                    DIFLogger.getLogger()
                            .warn("Could not access the identity manager!" + identityManagerException.getMessage());
            }
        }
    }

    /**
     * Process the User annotation
     *
     * @param clazz
     *            the class instance
     * @param annotation
     *            the annotation to process
     * @throws ResourceNotFoundException
     * @throws IdentityManagerException
     * @throws ConfigurationException
     */
    private void processUserAnnotation(ClassHolder clazz, AnnotationHolder annotation)
            throws ResourceNotFoundException, IdentityManagerException, ConfigurationException
    {
        // Get the parent info
        Entity entityType = EntityUtils.getEntityType(clazz);

        if (!identityManager.isReadOnly() && entityType.equals(Entity.APPLICATION))
        {
            String userName = annotation.getMembers().get("userName").toString();

            try
            {
                // If the user does not already exist...
                if (!usersProcessed.contains(userName) && !identityManager.userExists(userName))
                {
                    String password = annotation.getMembers().get("password").toString();
                    String fullName = annotation.getMembers().get("fullName").toString();
                    String nick = annotation.getMembers().get("nick").toString();
                    String email = annotation.getMembers().get("email").toString();
                    String profile = annotation.getMembers().get("profile").toString();

                    // Set dynamic default values...
                    if ("".equals(password))
                        password = DEFAULT_PASSWORD;

                    if ("".equals(fullName))
                        fullName = userName;

                    if ("".equals(nick))
                        nick = userName;

                    if ("".equals(email))
                        email = userName + "@domain.com";

                    String[] groups = CGAncillaries.parse(annotation.getMembers().get("groups").toString(), ",");
                    String[] attributes =
                            CGAncillaries.parse(annotation.getMembers().get("attributes").toString(), ",");

                    // Add the profile to the group list...
                    List<String> allGroups = new ArrayList<String>(Arrays.asList(groups));
                    allGroups.add(profile);

                    // Check all groups and create inexisting ones...
                    for (String groupName : allGroups)
                    {
                        if (!groupsProcessed.contains(groupName) && !identityManager.groupExists(groupName))
                        {
                            // Even if it fails will not try again. LDAP errors due to missing access on purpose are
                            // common and should be repeated at startup
                            groupsProcessed.add(groupName);

                            IDIFGroup group = DIFIoCRegistry.getRegistry().getImplementation(IDIFGroup.class);
                            group.setID(groupName);
                            group.setName(groupName);
                            group.setDescription(groupName);
                            group.setDefault(true);

                            identityManager.addGroup(group);
                        }
                    }

                    // Even if it fails will not try again. LDAP errors due to missing access on purpose are
                    // common and should be repeated at startup
                    usersProcessed.add(userName);

                    // Create the user...
                    IDIFUser user = DIFIoCRegistry.getRegistry().getImplementation(IDIFUser.class);
                    user.setID(userName);
                    user.setName(fullName);
                    user.setNick(fullName);
                    user.setPassword(password);
                    user.setEmail(email);
                    user.setProfileID(profile);
                    user.setDefault(true);

                    identityManager.addUser(user);

                    // Only after all groups and user have been created can we add the user to the groups
                    for (String groupName : allGroups)
                    {
                        identityManager.addUserToGroup(userName, groupName);
                    }

                    // Refresh the user, to pick up the repository bindings...
                    user = identityManager.getUser(userName);

                    // Parse and set all attributes
                    for (String attrPair : attributes)
                    {
                        String[] values = CGAncillaries.parse(attrPair, "=");

                        if (values.length == 2)
                            try
                            {
                                user.setAttribute(values[0], values[1]);
                            }
                            catch (InternalFrameworkException e)
                            {
                                DIFLogger.getLogger()
                                        .error("Could not add default attribute \"" + values[0] + " for user \"" +
                                               userName + "\"!");
                            }
                    }
                }
            }
            catch (IdentityManagerException identityManagerException)
            {
                if (identityManager.isReadOnly())
                    DIFLogger.getLogger().info("Identity Manager in Read Only mode.");
                else
                    DIFLogger.getLogger()
                            .warn("Could not access the identity manager!" + identityManagerException.getMessage());
            }
        }
    }
}
