/**
 * - Digitalis Internal Framework v2.0 - (C) 2007, Digitalis Informatica. Distribuicao e Gestao de Informatica, Lda.
 * Estrada de Paco de Arcos num.9 - Piso -1 2780-666 Paco de Arcos Telefone: (351) 21 4408990 Fax: (351) 21 4408999
 * http://www.digitalis.pt
 */
package pt.digitalis.dif.codegen.util;

import com.google.inject.Inject;
import pt.digitalis.dif.dem.DEMAnnotationLogic;
import pt.digitalis.utils.bytecode.holders.AnnotationHolder;
import pt.digitalis.utils.bytecode.holders.ClassHolder;
import pt.digitalis.utils.inspection.ResourceUtils;
import pt.digitalis.utils.inspection.exception.AuxiliaryOperationException;
import pt.digitalis.utils.inspection.exception.ResourceNotFoundException;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.ListIterator;
import java.util.Map;

/**
 * Holds the list of packages that will be searched for Entity classes by the DIFCodeGenerator. Provides methods to add
 * packages to the list of packages, and to access the list.
 *
 * @author Pedro Viegas <a href="mailto:pviegas@digitalis.pt">pviegas@digitalis.pt</a>
 * @author Rodrigo Gonalves <a href="mailto:rgoncalves@digitalis.pt">rgoncalves@digitalis.pt</a>
 * @created Jul 23, 2007
 */
final public class DEMLoaderHelper
{

    /** 'Applications' folder name by convention. */
    final static public String APPLICATIONS_DIR = "applications";

    /** 'Providers' folder name by convention. */
    final static public String PROVIDERS_DIR = "providers";

    /** 'Services' folder name by convention. */
    final static public String SERVICES_DIR = "services";

    /** 'Stages' folder name by convention. */
    final static public String STAGES_DIR = "stages";

    /** 'Validators' folder name by convention. */
    final static public String VALIDATORS_DIR = "validators";

    /** Class path for the DEM Annotations. */
    final static private String DEM_ANNOTATIONS_CLASS_PATH = "pt.digitalis.dif.dem.annotations";

    /** Class path for the DEM Entity Annotations. */
    final static private String DEM_ENTITIES_ANNOTATIONS_CLASS_PATH = DEM_ANNOTATIONS_CLASS_PATH + ".entities";

    /**
     * Defines the FQN of the <code>@AnnotationLogicClass</code> annotation.
     */
    final static private String FULL_PATH_ANNOTATION_LOGIC_CLASS =
            DEM_ANNOTATIONS_CLASS_PATH + ".metaannotations.AnnotationLogicClass";

    /** The annotationLogic Map */
    static private Map<String, DEMAnnotationLogic> annotationLogicMap = null;

    /** The DEM Entity annotations defined on the framework. */
    static private List<String> DEMEntitiesAnnontations = null;

    /** The list of packages to search. */
    static private List<String> packagesToSearch = new ArrayList<String>();

    /** Default constructor. Enforce noninstantiability */
    private DEMLoaderHelper()
    {
    }

    /**
     * Adds a package to the packageList after validating that this package does not already exist or is contained in an
     * existing parent package
     *
     * @param packageName the name of the package to add
     *
     * @return T if package was added, F otherwise
     */
    static public boolean addPackage(String packageName)
    {

        if (packagesToSearch == null)
            packagesToSearch = new ArrayList<String>();

        if (!packagesToSearch.contains(packageName))
        {
            if (!isPackageContained(packageName))
            {

                /*
                 * Remove from registry all eventual packages that will be contained by the new package <pre>
                 * IMPLEMENTATION NOTE: The usual "for" loop form (foreach) for collections uses an internal iterator of
                 * type Iterator<E> which doesn't support list modifying. Calling remove() on a list being iterated in
                 * such a loop raises a ConcurrentModificationException. To cope with this limitation a "for" loop with
                 * an explicit ListIterator<E> must be used and all modifying operations must be done through the
                 * iterator. </pre>
                 */

                for (ListIterator<String> iter = packagesToSearch.listIterator(); iter.hasNext(); )
                {
                    String registeredPackage = (String) iter.next();

                    if (registeredPackage.startsWith(packageName))
                        iter.remove();
                }

                // Add the new package to the registry
                packagesToSearch.add(packageName);

                return true;
            }
        }

        return false;
    }

    /**
     * Makes internal package list eligible for GC when it is no longer needed. Must be called explicitly since this is
     * a static class with static attributes for temporary data.
     */
    static public void cleanUp()
    {
        packagesToSearch = null;
        DEMEntitiesAnnontations = null;
    }

    /**
     * Builds a map of all DEM annotations with associated logic.
     *
     * @return the Map of annotations
     *
     * @exception ResourceNotFoundException   if DEM annotations classes can't be found
     * @exception AuxiliaryOperationException if a DEMAnnotationLogic object can't be created
     */
    static public Map<String, DEMAnnotationLogic> getAnnotationLogicMap()
            throws ResourceNotFoundException, AuxiliaryOperationException
    {

        if (annotationLogicMap == null)
        {

            List<String> demAnnotationsClasses = getDEMAnnotations();

            annotationLogicMap = new HashMap<String, DEMAnnotationLogic>();

            // Reuse vars
            ClassHolder tempAnnotationClass;
            AnnotationHolder annotationLogicClassAnnotation;

            // For each class...
            for (String annotationClassName : demAnnotationsClasses)
            {

                // ...build a holder...
                tempAnnotationClass = new ClassHolder(annotationClassName);

                // ..get the class that defines the annotation logic
                annotationLogicClassAnnotation =
                        tempAnnotationClass.getAnnotations().get(FULL_PATH_ANNOTATION_LOGIC_CLASS);

                // If annotation @AnnotationLogicClass is present...
                if (annotationLogicClassAnnotation != null)
                {

                    // Build an object of the appropriate type and put it on the map
                    annotationLogicMap.put(tempAnnotationClass.getFQName(), DEMAnnotationLogic
                            .makeObject(annotationLogicClassAnnotation.getMembers().get("value").toString(),
                                    tempAnnotationClass));
                }
                else
                {
                    // Request the construction of a default annotation logic
                    // object and put it on the map
                    annotationLogicMap.put(tempAnnotationClass.getFQName(), DEMAnnotationLogic
                            .makeObject(DEMAnnotationLogic.DEFAULT_DEM_ANNOTATION_LOGIC_CLASS, tempAnnotationClass));
                }
            }

            // Add the Google Guice inject annotation fot DiF to register injection as present
            annotationLogicMap.put(Inject.class.getCanonicalName(), DEMAnnotationLogic
                    .makeObject(DEMAnnotationLogic.DEFAULT_DEM_ANNOTATION_LOGIC_CLASS,
                            new ClassHolder(Inject.class.getCanonicalName())));
        }

        // Return map
        return annotationLogicMap;
    }

    /**
     * Returns a list with the names of all DEM annotations classes.
     *
     * @return the list with all DEM annotations class names
     *
     * @exception ResourceNotFoundException if some needed resource is not found
     */
    static public List<String> getDEMAnnotations() throws ResourceNotFoundException
    {
        return ResourceUtils.getClassesInDeepPackage(DEM_ANNOTATIONS_CLASS_PATH);
    }

    /**
     * Returns a list with the defined DEM Entity Annotation FQNs for the framework
     *
     * @return a list with the defined DEM Entity Annotation FQNs
     *
     * @exception ResourceNotFoundException if DEM annotations classes can't be found
     */
    public static List<String> getDEMEntitiesAnnontations() throws ResourceNotFoundException
    {
        // Lazy loading
        if (DEMEntitiesAnnontations == null)
            try
            {
                try
                {
                    // Fetches the the classes on the framework's DEM Entity
                    // conventioned package
                    DEMEntitiesAnnontations = ResourceUtils.getClassesInPackage(DEM_ENTITIES_ANNOTATIONS_CLASS_PATH);
                }
                catch (ResourceNotFoundException resourceNotFoundException)
                {
                    throw new ResourceNotFoundException(
                            "Could not found DEM annotations on " + DEM_ENTITIES_ANNOTATIONS_CLASS_PATH,
                            resourceNotFoundException);
                }

                // Treat the situation where no DEM entities are found...
                if (DEMEntitiesAnnontations.size() == 0 || DEMEntitiesAnnontations == null)
                    DEMEntitiesAnnontations = null;
            }
            catch (ResourceNotFoundException resourceNotFoundException)
            {
                // The caught exception is already of the correct type and
                // provides the needed information to spot the problem.
                // Therefore, wrap it up on another exception seems unnecessary.
                DEMEntitiesAnnontations = null;

                throw resourceNotFoundException;
            }

        return DEMEntitiesAnnontations;
    }

    /**
     * Searches a given package for DEM Entity annotated classes.
     *
     * @param packageName the package to search for DEM classes
     *
     * @return a List of CGClassHolder
     *
     * @exception ResourceNotFoundException if package is not found
     */
    static public List<ClassHolder> getDEMEntityClassesInPackage(String packageName) throws ResourceNotFoundException
    {

        // Get all classes for the given package
        List<String> packageClasses = ResourceUtils.getClassesInDeepPackage(packageName);

        // Initialize return object
        ArrayList<ClassHolder> entityClasses = new ArrayList<ClassHolder>();
        // Reuse var
        ClassHolder tempClass;

        // Add all DEM Entity classes to the return list
        for (String className : packageClasses)
        {

            // Build an holder for the class...
            try
            {
                tempClass = new ClassHolder(className);
            }
            catch (ResourceNotFoundException e)
            {
                tempClass = null;
            }

            if (tempClass != null)
            {
                // If it's a DEM Entity add holder to result
                if (isDEMAnnotatedClass(tempClass))
                    entityClasses.add(tempClass);
                else
                    // Else delete it from repository (it won't be needed for enhancement)
                    tempClass.deleteClassFromRepository();
            }
        }

        return entityClasses;
    }

    /**
     * Return the list of packages.
     *
     * @return the List of packages
     */
    static public List<String> getPackageList()
    {
        return packagesToSearch;
    }

    /**
     * Tests a class for DEM Entity annotations.
     *
     * @param clazz the class to search for annotations
     *
     * @return T if class is a DEM Entity, F otherwise.
     *
     * @exception ResourceNotFoundException if annotations can't be read
     */
    static private boolean isDEMAnnotatedClass(ClassHolder clazz) throws ResourceNotFoundException
    {
        boolean isDEMAnnotatedClass = false;

        // TODO: Viegas: WIP: Retirar com a subida de Java (incompatibilidade na leitura de classes em Java > 6)
        boolean discardClass =
                clazz.getFQName().startsWith("com.intellij.") || clazz.getFQName().startsWith("com.sun.") ||
                clazz.getFQName().startsWith("com.apple.");

        if (!discardClass)
        {
            // Check if any of the DEM Annotations is on the class's annotations
            for (String annotationName : getDEMEntitiesAnnontations())
            {
                // If at least one is present, return immediately
                if (clazz.containsAnnotation(annotationName))
                {
                    isDEMAnnotatedClass = true;
                    break;
                }
            }
        }

        return isDEMAnnotatedClass;
    }

    /**
     * Verifies if the given package is contained on a previously registered package.
     *
     * @param packageName the package name
     *
     * @return T if packageName is contained on any existing package
     */
    static private boolean isPackageContained(String packageName)
    {

        for (String registeredPackage : packagesToSearch)
        {
            if (packageName.startsWith(registeredPackage))
                return true;
        }
        return false;
    }
}