/**
 * 2007, Digitalis Informatica. All rights reserved. Distribuicao e Gestao de Informatica, Lda. Estrada de Paco de Arcos
 * num.9 - Piso -1 2780-666 Paco de Arcos Telefone: (351) 21 4408990 Fax: (351) 21 4408999 http://www.digitalis.pt
 */
package pt.digitalis.dif.controller;

import pt.digitalis.dif.controller.interfaces.IDIFRequest;
import pt.digitalis.dif.controller.interfaces.IDIFResponse;
import pt.digitalis.dif.controller.interfaces.IDispatcherErrorHandler;
import pt.digitalis.dif.controller.objects.DIFResponseStatus;
import pt.digitalis.dif.exception.controller.BusinessFlowException;
import pt.digitalis.dif.exception.controller.ControllerException;

/**
 * @author Pedro Viegas <a href="mailto:pviegas@digitalis.pt">pviegas@digitalis.pt</a>
 * @created Dec 21, 2007
 */
abstract public class AbstractControllerErrorHandler implements IDispatcherErrorHandler
{

    /**
     * Builds an error response from a stage id for error handling, with a defined handler stage (or the default one)
     *
     * @param originalrequest the original request
     * @param errorStageID    the error stage to redirect to. If null will redirect to the default error stage
     * @param exception       the exception launched
     *
     * @return the response
     */
    abstract protected IDIFResponse buildDynamicErrorResponse(IDIFRequest originalrequest, String errorStageID,
            Exception exception);

    /**
     * Builds an error response from a static resource for simple rendering for when the DIF has generated errors
     *
     * @param originalrequest the original request
     * @param exception       the exception launched
     *
     * @return the response
     */
    abstract protected IDIFResponse buildStaticErrorResponse(IDIFRequest originalrequest, Exception exception);

    /**
     * @see pt.digitalis.dif.controller.interfaces.IDispatcherErrorHandler#processException(pt.digitalis.dif.controller.interfaces.IDIFRequest,
     *         pt.digitalis.dif.exception.controller.BusinessFlowException)
     */
    public IDIFResponse processException(IDIFRequest originalRequest, BusinessFlowException businessFlowException)
    {

        return buildDynamicErrorResponse(originalRequest, ExceptionHandlers.getExceptionHandler(businessFlowException),
                businessFlowException);
    }

    /**
     * @see pt.digitalis.dif.controller.interfaces.IDispatcherErrorHandler#processException(pt.digitalis.dif.controller.interfaces.IDIFRequest,
     *         pt.digitalis.dif.exception.controller.ControllerException)
     */
    public IDIFResponse processException(IDIFRequest originalRequest, ControllerException controllerException)
    {
        if (controllerException.isGeneratedFromException())
            return buildStaticErrorResponse(originalRequest, controllerException);
        else
            return buildDynamicErrorResponse(originalRequest,
                    ExceptionHandlers.getExceptionHandler(controllerException), controllerException);
    }

    /**
     * @see pt.digitalis.dif.controller.interfaces.IDispatcherErrorHandler#processException(pt.digitalis.dif.controller.interfaces.IDIFRequest,
     *         java.lang.RuntimeException)
     */
    public IDIFResponse processException(IDIFRequest originalRequest, RuntimeException runtimeException)
    {
        return buildDynamicErrorResponse(originalRequest, ExceptionHandlers.getExceptionHandler(runtimeException),
                runtimeException);
    }

    /**
     * Creates a new error request base on the original request. Injects the original request in the error one for later
     * reference in the error stage.
     *
     * @param response        the response to add the context to
     * @param originalRequest the original request
     * @param exception       the exception launched
     *
     * @return the request with the context added
     */
    public IDIFResponse responseWithContext(IDIFResponse response, IDIFRequest originalRequest, Exception exception)
    {
        response.addStageResult(IDispatcherErrorHandler.ORIGINAL_REQUEST, originalRequest);
        response.addStageResult(IDispatcherErrorHandler.EXCEPTION, exception);

        if (response.getRequest() != null)
            response.getRequest().addAttribute(ExceptionHandlers.RAISED_EXCEPTION_ATTRIBUTE, exception);

        response.setResponseStatus(DIFResponseStatus.BAD_REQUEST);

        return response;
    }
}
