/**
 * - Digitalis Internal Framework v2.0 - (C) 2007, Digitalis Informatica. Distribuicao e Gestao de Informatica, Lda.
 * Estrada de Paco de Arcos num.9 - Piso -1 2780-666 Paco de Arcos Telefone: (351) 21 4408990 Fax: (351) 21 4408999
 * http://www.digitalis.pt
 */
package pt.digitalis.dif.controller;

import com.newrelic.api.agent.NewRelic;
import com.newrelic.api.agent.Trace;
import org.apache.commons.lang.exception.ExceptionUtils;
import pt.digitalis.dif.controller.interceptors.IDIFInterceptorAuthentication;
import pt.digitalis.dif.controller.interceptors.IDIFInterceptorBeforeDispatch;
import pt.digitalis.dif.controller.interceptors.IDIFInterpectorAfterExecute;
import pt.digitalis.dif.controller.interceptors.IDIFInterpectorBeforeExecute;
import pt.digitalis.dif.controller.interfaces.IControllerCleanupTask;
import pt.digitalis.dif.controller.interfaces.IDIFContext;
import pt.digitalis.dif.controller.interfaces.IDIFDispatcher;
import pt.digitalis.dif.controller.interfaces.IDIFRequest;
import pt.digitalis.dif.controller.interfaces.IDIFSession;
import pt.digitalis.dif.controller.interfaces.IPrivateDIFSession;
import pt.digitalis.dif.controller.objects.Constants;
import pt.digitalis.dif.controller.objects.ControllerExecutionStep;
import pt.digitalis.dif.controller.objects.DIFContext;
import pt.digitalis.dif.controller.objects.DIFResponse;
import pt.digitalis.dif.controller.objects.DIFResponseStatus;
import pt.digitalis.dif.controller.objects.DIFSession;
import pt.digitalis.dif.controller.objects.DispatcherAuthenticationResult;
import pt.digitalis.dif.controller.objects.SSOInfo;
import pt.digitalis.dif.controller.security.managers.IAuthenticationManager;
import pt.digitalis.dif.controller.security.managers.IAuthorizationManager;
import pt.digitalis.dif.controller.security.managers.IIdentityManager;
import pt.digitalis.dif.controller.security.managers.ISessionManager;
import pt.digitalis.dif.controller.security.objects.IDIFUser;
import pt.digitalis.dif.dem.DEMRegistryImpl;
import pt.digitalis.dif.dem.annotations.controller.Channel;
import pt.digitalis.dif.dem.interfaces.IStage;
import pt.digitalis.dif.dem.interfaces.IStageInstance;
import pt.digitalis.dif.dem.managers.IMessageManager;
import pt.digitalis.dif.dem.objects.ViewObject;
import pt.digitalis.dif.dem.objects.messages.MessageList;
import pt.digitalis.dif.exception.controller.BusinessFlowException;
import pt.digitalis.dif.exception.controller.ControllerException;
import pt.digitalis.dif.exception.controller.LicenseViolationException;
import pt.digitalis.dif.exception.controller.SessionTimeoutException;
import pt.digitalis.dif.exception.controller.UserHasNoAccessToStage;
import pt.digitalis.dif.exception.security.AuthenticationManagerException;
import pt.digitalis.dif.exception.security.IdentityManagerException;
import pt.digitalis.dif.flightrecorder.FlightRecorder;
import pt.digitalis.dif.ioc.DIFIoCRegistry;
import pt.digitalis.dif.startup.DIFInitializer;
import pt.digitalis.dif.startup.DIFStartupConfiguration;
import pt.digitalis.dif.utils.logging.DIFLogger;
import pt.digitalis.dif.utils.logging.performance.IPerformanceLogger;
import pt.digitalis.dif.utils.logging.performance.PerformanceLoggerConfiguration;
import pt.digitalis.utils.common.Chronometer;
import pt.digitalis.utils.common.StringUtils;
import pt.digitalis.utils.config.ConfigurationException;
import pt.digitalis.utils.config.IConfigurations;

import java.util.Map;
import java.util.Map.Entry;

/**
 * This class is the heart of the framework. Its responsibilities include receive and validate the incoming DIFRequest,
 * the request processing (if the request is valid) and the routing of the stage-generated DIFResponse.
 * AbstractDIFDispatcher implements the TEMPLATE METHOD design pattern (see GoF's "Design Patterns: Elements of Reusable
 * Object-Oriented Software"). As such, the <code> executeTaskSteps() </code> method is declared final. This method
 * defines the dispatching-cycle steps, and these steps must be always run in the same order to all the requests. There
 * is a set of protected methods that correspond to the aforementioned dispatching-cycle steps. These methods can be
 * overridden in derived classes if one wants to modify the behavior associated with a given step. ChALs should call a
 * particular controller so this class (the superclass) is declared <code>abstract</code> to inhibit instantiation.
 *
 * @author Rodrigo Goalves <a href="mailto:rgoncalves@digitalis.pt">rgoncalves@digitalis.pt</a>
 * @author Pedro Viegas <a href="mailto:pviegas@digitalis.pt">pviegas@digitalis.pt</a>
 * @created 2007/03/16
 */
abstract public class AbstractDIFDispatcher implements IDIFDispatcher
{

    /** The Constant BROWSER_VALIDATOR_STAGE_ID. */
    public static final String BROWSER_VALIDATOR_STAGE_ID = "BrowserValidator";

    /** The message for invalid stage requests. */
    final public static String INVALID_STAGE_ERROR_MESSAGE =
            "The stage is not registered in the DEM. Check the stage id.";

    /** The Constant MAX_REDIRECTS_BEFORE_CANCELATION. */
    final protected static int MAX_REDIRECTS_BEFORE_CANCELATION = 30;

    /** The message manager to get class messages. */
    static IMessageManager messageManager;

    /** Messages cache for all constraints. */
    static private MessageList messages;

    /** If an authentication error has occurred. */
    protected ControllerException authenticationException = null;

    /** The execution context. */
    protected IDIFContext difContext;

    /** The Authentication Manager. */
    protected IAuthenticationManager theAuthenticationManager;

    /** The authorization manager. */
    protected IAuthorizationManager theAuthorizationManager;

    /** The configuration manager. */
    protected IConfigurations theConfigurations;

    /** The identity manager. */
    protected IIdentityManager theIdentityManager;

    /** The session manager. */
    protected ISessionManager theSessionManager;

    /** The stage to be executed. */
    protected IStage theStage;

    /** the channel ID. */
    private String channelID = null;

    /**
     * Default constructor. This constructor must be inherited in the extended classes and injected with the instances
     * from the IoC
     *
     * @param theIdentityManager       the identity manager
     * @param theAuthenticationManager the authentication manager
     * @param theAuthorizationManager  the authorization manager
     * @param theSessionManager        the authentication manager
     * @param theConfigurationManager  the configurations manager
     */
    public AbstractDIFDispatcher(IIdentityManager theIdentityManager, IAuthenticationManager theAuthenticationManager,
            IAuthorizationManager theAuthorizationManager, ISessionManager theSessionManager,
            IConfigurations theConfigurationManager)
    {
        this.theIdentityManager = theIdentityManager;
        this.theAuthorizationManager = theAuthorizationManager;
        this.theSessionManager = theSessionManager;
        this.theConfigurations = theConfigurationManager;
        this.theAuthenticationManager = theAuthenticationManager;
    }

    /**
     * Get's the Message Manager from the IoC.
     *
     * @return the message manager instance
     */
    static private IMessageManager getMessageManager()
    {
        if (messageManager == null)
            messageManager = DIFIoCRegistry.getRegistry().getImplementation(IMessageManager.class);

        return messageManager;
    }

    /**
     * Perform any clean up tasks that have been contributed by ...
     *
     * @param context the context
     * @param success T if the stage was successfully executed
     *
     * @exception ControllerException the controller exception
     */
    static public final void performCleanup(IDIFContext context, boolean success) throws ControllerException
    {
        ControllerException controllerException = null;

        DIFLogger.getLogger().debug("Executing controller cycle clean up tasks...");

        for (IControllerCleanupTask task : DIFIoCRegistry.getRegistry()
                .getImplementations(IControllerCleanupTask.class))
        {
            try
            {
                DIFLogger.getLogger()
                        .debug("Executing controller cycle clean up tasks [" + task.getClass().getSimpleName() +
                               "]...");
                task.doTask(context, success);
            }
            catch (Exception e)
            {
                if (controllerException == null)
                    controllerException = new ControllerException(ControllerExecutionStep.DISPATCHER_CONCLUDE, e);
            }
        }

        DIFLogger.getLogger().debug("Executing controller cycle clean up tasks... done!");

        if (controllerException != null)
            throw controllerException;
    }

    /**
     * Authenticates a user on the framework. The method is defined with protected scope to allow overriding on the
     * subclasses.
     *
     * @exception ControllerException when any runtime exception is thrown or the authentication process has failed
     */
    @Trace(metricName = "DIF:Dispatcher:Authenticate", dispatcher = true)
    protected void authenticate() throws ControllerException
    {
        // 1. perform the necessary operations before authentication execution
        this.preAuthentication();
        // 2. execute the authentication
        DispatcherAuthenticationResult authenticationResult = this.performAuthentication();
        // 3. perform the post authentication operations
        this.postAuthentication(authenticationResult);
    }

    /**
     * Check the user permissions to run a given stage. The method is defined with protected scope to allow overriding
     * on the subclasses.
     *
     * @exception ControllerException when any runtime exception is thrown
     */
    @Trace(metricName = "DIF:Dispatcher:Authorize", dispatcher = true)
    protected void authorize() throws ControllerException
    {
        IDIFSession session = null;
        IDIFUser user = null;
        boolean authorized = false;
        String reason = "";
        Exception exception = null;

        try
        {
            session = this.getContext().getSession();

            if (session != null)
            {
                user = session.getUser();

                if (user != null)
                {
                    if (user.canAccess(this.getStage()))
                    {
                        authorized = true;
                    }
                    else
                    {
                        authorized = false;
                        reason = getMessages().get("noAccess")
                                .replace("${user}", "\"" + StringUtils.nvl(user.getID(), "NotAuthenticated") + "\"")
                                .replace("${name}", this.getStage().getName());

                        this.getContext().setResponseStatus(DIFResponseStatus.FORBIDDEN);
                    }
                }
                else
                {
                    if (!theAuthorizationManager.hasAccessPublic(this.getStage()))
                    {
                        authorized = false;
                        reason = getMessages().get("noAccess").replace("${user}", "NotAuthenticated")
                                .replace("${name}", this.getStage().getName());

                        this.getContext().setResponseStatus(DIFResponseStatus.UNAUTHORIZED);
                    }
                    else
                        authorized = true;
                }
            }
            else
            {
                // No session! Do not grant access! This is an error. Must
                // always exist a session.
                authorized = false;
                reason = getMessages().get("noSession").replace("${name}", this.getStage().getName());

                this.getContext().setResponseStatus(DIFResponseStatus.UNAUTHORIZED);
            }
        }
        catch (RuntimeException runtimeException)
        {
            authorized = false;
            exception = runtimeException;
        }

        if (!authorized)
        {
            UserHasNoAccessToStage controllerException =
                    new UserHasNoAccessToStage(ControllerExecutionStep.DISPATCHER_AUTHORIZATION, reason, exception);

            controllerException.addToExceptionContext("Context", this.getContext());

            throw controllerException;
        }
    }

    /**
     * This method checks if the license is valid.
     *
     * @return T if stage is registered, F otherwise
     */
    protected boolean checkLicense()
    {
        return this.getStage().isRegistered();
    }

    /**
     * Concludes the execution cycle. Touches the session to ensure it stays alive and returns any service objects to
     * the pool (not implemented as of version 0.0.1). If session is not active, it's not kept alive and thus on the
     * next request the access will be denied. 'Finalize' would be better name but it's forbidden due to a name-clash
     * with <code>java.lang.Object's finalize()</code> method.
     *
     * @exception ControllerException when any runtime exception is thrown
     */
    protected void conclude() throws ControllerException
    {
        try
        {
            ((IPrivateDIFSession) this.getContext().getRequest().getSession()).forceKeepAlive();
            // Keep session alive if it has not timed out
            theSessionManager.update(this.getContext().getSession());
        }
        catch (RuntimeException runtimeException)
        {
            ControllerException controllerException =
                    new ControllerException(ControllerExecutionStep.DISPATCHER_CONCLUDE, runtimeException);

            controllerException.addToExceptionContext("Context", this.getContext());

            throw controllerException;
        }
    }

    /**
     * Receives a BaseDIFRequest object as parameter and creates a service execution context for the request.
     *
     * @param difRequest the request received from AbstractChAL
     *
     * @return The request's context for service execution
     *
     * @exception ControllerException when any runtime exception is thrown
     */
    final private DIFContext createContext(IDIFRequest difRequest) throws ControllerException
    {
        DIFContext theContext = null;

        try
        {
            theContext = new DIFContext();

            theContext.setStage(difRequest.getStage());
            theContext.setRequest(difRequest);

            return theContext;
        }
        catch (RuntimeException runtimeException)
        {
            ControllerException controllerException =
                    new ControllerException(ControllerExecutionStep.DISPATCHER_CREATE_CONTEXT, runtimeException);

            controllerException.addToExceptionContext("Context", theContext);

            throw controllerException;
        }
    }

    /**
     * Generates the response that will be sent back to AbstractChAL from the request's execution context.
     *
     * @param stageInstance the executed stage instance
     *
     * @return The response to send back to AbstractChAL
     *
     * @exception ControllerException if an exception has occurred
     */
    protected DIFResponse createResponse(IStageInstance stageInstance) throws ControllerException
    {
        try
        {
            DIFResponse difResponse = new DIFResponse();

            difResponse.setView(this.getContext().getView());
            difResponse.setResultMessages(this.getContext().getResultMessages());
            difResponse.setStageResults(this.getContext().getStageResults());
            difResponse.setRequest(this.getContext().getRequest());
            difResponse.setStageInstance(stageInstance);
            difResponse.setHTTPHeaders(this.getContext().getHTTPHeaders());
            difResponse.setResponseStatus(this.getContext().getResponseStatus());

            return difResponse;
        }
        catch (RuntimeException runtimeException)
        {
            ControllerException controllerException =
                    new ControllerException(ControllerExecutionStep.DISPATCHER_CREATE_RESPONSE, runtimeException);

            controllerException.addToExceptionContext("Context", this.getContext());
            controllerException.addToExceptionContext("Stage Instance", stageInstance);

            throw controllerException;
        }
    }

    /**
     * @see pt.digitalis.dif.controller.interfaces.IDIFDispatcher#dispatch(pt.digitalis.dif.controller.interfaces.IDIFRequest)
     */
    @Override
    final public DIFResponse dispatch(IDIFRequest difRequest)
            throws BusinessFlowException, ControllerException, ConfigurationException
    {
        // Initialize framework
        DIFInitializer.initialize(true, true);

        // Create an execution context for the incoming request
        this.setContext(createContext(difRequest));

        // Create the response
        DIFResponse difResponse = null;
        IStageInstance stageInstance = null;
        boolean firstTime = true;

        for (IDIFInterceptorBeforeDispatch interceptor : DIFIoCRegistry.getRegistry()
                .getImplementations(IDIFInterceptorBeforeDispatch.class))
        {
            interceptor.executeLogic(this.getContext());
        }

        while (firstTime || this.getContext().hasRedirection())
        {
            FlightRecorder.reportRequest(this.getContext());

            if (!firstTime)
            {
                try
                {
                    this.getContext().handleRedirection();

                    if (stageInstance != null)
                    {
                        stageInstance.setContext(this.getContext());
                    }

                    if (this.getContext().getRedirectCount() > MAX_REDIRECTS_BEFORE_CANCELATION)
                        this.handleRedirectionError(stageInstance, this.getContext());
                }
                catch (RuntimeException runtimeException)
                {
                    ControllerException controllerException =
                            new ControllerException(ControllerExecutionStep.DISPATCHER_REDIRECTION, runtimeException);

                    controllerException.addToExceptionContext("Context", this.getContext());

                    throw controllerException;
                }
            }

            // Start an execution timer to measure each request
            Chronometer crono = new Chronometer();
            crono.start();

            try
            {
                try
                {
                    // Run the task steps
                    stageInstance = runDispatchingSteps(!firstTime);
                }
                catch (ControllerException exception)
                {

                    if (ControllerExecutionStep.DISPATCHER_AUTHORIZATION.equals(exception.getStep()))
                    {
                        // If the stage has no access we must determine what to
                        // do...
                        handleNonAuthorizedAccess(stageInstance, exception);
                    }
                    else
                        throw exception;
                }

                firstTime = false;

                logAccess(stageInstance, crono, null);
            }
            catch (BusinessFlowException exception)
            {
                logAccess(stageInstance, crono, exception);
                throw exception;
            }
            catch (ControllerException exception)
            {
                logAccess(stageInstance, crono, exception);
                throw exception;
            }
            catch (ConfigurationException exception)
            {
                logAccess(stageInstance, crono, exception);
                throw exception;
            }
            catch (Exception exception)
            {
                logAccess(stageInstance, crono, exception);
                throw new RuntimeException(exception);
            }
        }

        // Create a response to send back to AbstractChAL
        difResponse = createResponse(stageInstance);

        return difResponse;
    }

    /**
     * Actions to perform after a successful login.
     *
     * @param ssoInfo the single sign-on object
     */
    protected void doAfterLogin(SSOInfo ssoInfo)
    {
        // Nothing to do. This is here to facilitate SSOs integrations
    }

    /**
     * Actions to perform after a successful logout.
     *
     * @param ssoInfo the single sign-on object
     */
    protected void doAfterLogout(SSOInfo ssoInfo)
    {
        // Nothing to do. This is here to facilitate SSOs integrations
    }

    /**
     * Executes the appropriate tasks to serve the request.
     *
     * @return the executed stage instance
     *
     * @exception BusinessFlowException when an exception is thrown by the stage's init/execute steps
     * @exception ControllerException   when any runtime exception is thrown
     */
    @Trace(metricName = "DIF:Dispatcher:Execute", dispatcher = true)
    protected IStageInstance execute() throws BusinessFlowException, ControllerException
    {
        IStageInstance stage = null;
        ViewObject theView = null;

        try
        {
            stage = this.getStage().getInstance();
            this.getContext().setStageInstance(stage);

            // If the stage is configured for authentication error injection...
            if (stage.hasAuthenticationErrorInjection())
                stage.setAuthenticationError(getAuthenticationException());

            // Business initialization
            stage._CG_init(this.getContext());

            if (!stage.hasParameterErrorInjection() && stage.getParameterErrors().hasErrors())
                DIFLogger.getLogger().warn("There were errors in the parameter validation witch will be ignored since" +
                                           " there is no @InjectParameterErrors anotated attribute on the stage " +
                                           stage.getID());

            // Business logic execution
            theView = stage._CG_execute(this.getContext());

            // Business finalization
            stage._CG_finalize(this.getContext());

            // Set the View in context
            this.getContext().setView(theView);

            // Add the current stage to the navigation history object
            this.getContext().getSession().getNavigationHistory().addStage(stage);

            return stage;
        }
        catch (RuntimeException runtimeException)
        {
            ControllerException controllerException =
                    new ControllerException(ControllerExecutionStep.DISPATCHER_EXECUTE, runtimeException);

            controllerException.addToExceptionContext("Context", this.getContext());
            controllerException.addToExceptionContext("Stage", stage);
            controllerException.addToExceptionContext("View object to Render", theView);

            throw controllerException;
        }
    }

    /**
     * Executes custom logic after all stages execution step.
     *
     * @param stageInstance the stage instance
     */
    private void executeInterceptorsAfterExecution(IStageInstance stageInstance)
    {
        for (IDIFInterpectorAfterExecute interceptor : DIFIoCRegistry.getRegistry()
                .getImplementations(IDIFInterpectorAfterExecute.class))
        {
            interceptor.executeLogic(stageInstance);
        }
    }

    /**
     * Executes custom after before all stages execution step.
     *
     * @param context the current context
     *
     * @exception BusinessFlowException the business flow exception
     * @exception ControllerException   the controller exception
     */
    private void executeInterceptorsBeforeExecution(IDIFContext context)
            throws BusinessFlowException, ControllerException
    {
        for (IDIFInterpectorBeforeExecute interceptor : DIFIoCRegistry.getRegistry()
                .getImplementations(IDIFInterpectorBeforeExecute.class))
        {
            interceptor.executeLogic(context);
        }
    }

    /**
     * Gets the authentication exception.
     *
     * @return the authenticationException
     */
    public ControllerException getAuthenticationException()
    {
        return authenticationException;
    }

    /**
     * Sets the authentication exception.
     *
     * @param authenticationException the authenticationException to set
     */
    public void setAuthenticationException(ControllerException authenticationException)
    {
        this.authenticationException = authenticationException;
    }

    /**
     * @see pt.digitalis.dif.controller.interfaces.IDIFDispatcher#getChannelID()
     */
    @Override
    public String getChannelID()
    {
        if (channelID == null)
            channelID = this.getClass().getAnnotation(Channel.class).value();

        return channelID;
    }

    /**
     * Inspector for the execution context.
     *
     * @return the execution context
     */
    protected IDIFContext getContext()
    {
        return this.difContext;
    }

    /**
     * Modifier for the execution context.
     *
     * @param newContext the new execution context to set
     */
    private void setContext(DIFContext newContext)
    {
        this.difContext = newContext;
    }

    /**
     * Inspector for the identity manager.
     *
     * @return the defined identity manager
     */
    protected IIdentityManager getIdentityManager()
    {
        return this.theIdentityManager;
    }

    /**
     * Lazy loading getter of messages.
     *
     * @return the messages
     */
    protected Map<String, String> getMessages()
    {
        // If the messages have not yet been loaded do it now
        if (messages == null)
        {
            messages = getMessageManager().collectEntityMessagesFromRepository(AbstractDIFDispatcher.class);
        }

        return messages.getMessages(getContext().getLanguage());
    }

    /**
     * Used to inject the SSO Info for the dispatcher authentication process.
     *
     * @return the SSOInfo if available
     */
    protected SSOInfo getSSOInfo()
    {
        // Nothing to do. This is here to facilitate SSOs integrations
        return null;
    }

    /**
     * Inspector for the stage to execute.
     *
     * @return the stage to execute
     */
    private IStage getStage()
    {
        return this.theStage;
    }

    /**
     * Modifier for the stage to execute.
     *
     * @param newStage the new stage to execute
     */
    private void setStage(IStage newStage)
    {
        this.theStage = newStage;
    }

    /**
     * Handles the occurred exception to redirect to a given handler and perform cleanup actions.
     *
     * @param exception the raised exception
     *
     * @return T if the exception was handled and should not be re-throwned
     *
     * @exception BusinessFlowException the business flow exception
     * @exception ControllerException   the controller exception
     */
    protected boolean handleException(Exception exception) throws BusinessFlowException, ControllerException
    {

        performCleanup(this.getContext(), false);

        if (ExceptionHandlers.hasHandler(exception))
        {
            ExceptionHandlers.handleException(this.getContext(), exception);

            return true;
        }
        else
            return false;
    }

    /**
     * Will handle hot reload requests.
     */
    private void handleHotClassReload()
    {
        if (DIFStartupConfiguration.getDeveloperMode())
        {
            Object hotReloadParam = this.getContext().getRequest().getParameter(IDIFRequest.CLASS_RELOAD);
            boolean hotReloadRequested = hotReloadParam != null && hotReloadParam.toString().equalsIgnoreCase("true");

            if (hotReloadRequested)
            {
                // TODO: NOW: Clean DEM registries and managers (maybe rules also)
                // TODO: NOW: Trigger DIFInitialize in rebuild DEM mode
            }
        }
    }

    /**
     * Handles a non authorized access.
     *
     * @param stageInstance the stage instance requested
     * @param exception     the exception thrown
     *
     * @exception ControllerException    if the error was not a non authorized access
     * @exception ConfigurationException
     */
    abstract protected void handleNonAuthorizedAccess(IStageInstance stageInstance, ControllerException exception)
            throws ControllerException, ConfigurationException;

    /**
     * Handles a cyclic redirection.
     *
     * @param stageInstance the stage instance requested
     * @param difContext
     *
     * @exception ControllerException    if the error was not a non authorized access
     * @exception ConfigurationException
     */
    abstract protected void handleRedirectionError(IStageInstance stageInstance, IDIFContext difContext)
            throws ControllerException, ConfigurationException;

    /**
     * Helper method to log a request access.
     *
     * @param stageInstance the stage instance
     * @param crono         the crono
     * @param exception     the exception
     */
    private void logAccess(IStageInstance stageInstance, Chronometer crono, Exception exception)
    {
        try
        {
            // End execution timmer - All done!
            crono.end();

            if (crono.getTimePassedInSeconds() >=
                PerformanceLoggerConfiguration.getInstance().getLogSlowRequestsTimeInSecods())
            {
                StringBuffer logMessage = new StringBuffer();
                logMessage.append("\nSLOW STAGE EXECUTION:\n");
                logMessage.append("--------------------------------------------------------------------\n");
                logMessage.append("Stage: [" + this.getStage().getID() + "] " + this.getStage().getName() + "\n");
                logMessage.append("Execution time: " + crono.getTimePassedAsFormattedString() + "\n\n");

                DIFLogger.getLogger().warn(logMessage.toString());

                if (stageInstance != null)
                    DIFLogger.getLogger().debug(stageInstance.getContext().toString());
            }

            DIFIoCRegistry.getRegistry().getImplementation(IPerformanceLogger.class)
                    .logRequest(this.getContext().getRequest(), crono.getTimePassedInMilisecs(),
                            exception == null ? null : ExceptionUtils.getFullStackTrace(exception));
        }
        catch (Exception innerException)
        {
            innerException.printStackTrace();
        }
    }

    /**
     * Log in.
     *
     * @param userID           the user ID
     * @param suppliedPassword the supplied password
     *
     * @return the IDIF session
     *
     * @exception AuthenticationManagerException the authentication manager exception
     */
    protected IDIFSession logIn(String userID, String suppliedPassword) throws AuthenticationManagerException
    {
        return theSessionManager.logIn(getContext().getSession().getSessionID(), userID, suppliedPassword);
    }

    /**
     * Implements the authentication execution logic.
     *
     * @return the authentication result
     *
     * @exception ControllerException the controller exception
     */
    protected DispatcherAuthenticationResult performAuthentication() throws ControllerException
    {
        DispatcherAuthenticationResult dispatcherResult = DispatcherAuthenticationResult.NO_ACTION;

        if (this.getContext().getSession().isLogged() &&
            "true".equals(this.getContext().getRequest().getParameter(IDIFRequest.LOGOUT_PARAMETER_ID)))
        {

            String logoutRedirectUrl =
                    (String) this.getContext().getRequest().getParameter(IDIFRequest.LOGOUT_REDIRECT_URL_PARAMETER_ID);
            if (StringUtils.isBlank(logoutRedirectUrl))
            {
                logoutRedirectUrl = (String) this.getContext().getSession()
                        .getAttribute(IDIFRequest.LOGOUT_REDIRECT_URL_PARAMETER_ID);
                this.getContext().getSession().addAttribute(IDIFRequest.LOGOUT_REDIRECT_URL_PARAMETER_ID, null);
            }

            if (StringUtils.isNotBlank(logoutRedirectUrl))
            {
                this.getContext().getRequest().getParameters().put(IDIFRequest.REDIRECT_TO_URL, logoutRedirectUrl);
            }

            /*
             * The logout action must be performed before any of the other operations. There are two ways to invoke
             */

            String providerId = (String) this.getContext().getSession()
                    .getAttribute(DIFSession.REMOTE_AUTHENTICATION_PROVIDER_LOGOUT);
            if (providerId != null)
            {
                String title =
                        getMessages().get("remoteAuthenticationLogoutTitle").replace("${providerid}", providerId);
                String text = getMessages().get("remoteAuthenticationLogoutText").replace("${providerid}", providerId);

                this.getContext().addResultMessage("warn", title, text, true, true);
                this.getContext().getSession().addAttribute(DIFSession.REMOTE_AUTHENTICATION_PROVIDER_LOGOUT, null);
            }

            theSessionManager.logOut(this.getContext().getSession().getSessionID());
            dispatcherResult = DispatcherAuthenticationResult.LOGOUT;
        }

        /*
         * The stage requires authentication
         */
        else if (this.getStage().hasAuthentication())
        {
            Object userID = null;
            Object suppliedPassword = null;
            String reason = "";
            Exception exception = null;

            try
            {
                // ...get credentials from the request...
                userID = this.getContext().getRequest().getParameter(IDIFRequest.USER_PARAMETER_ID);
                suppliedPassword = this.getContext().getRequest().getParameter(IDIFRequest.PASSWORD_PARAMETER_ID);

                // The user is logged and is the same user, unnecessary login the user again
                if (this.getContext().getSession().isLogged() &&
                    this.getContext().getSession().getUser().getID().equals(userID))
                {

                    /* There is no need to authenticate the same user! */
                    userID = null;
                    suppliedPassword = null;
                }

                if ((userID == null || suppliedPassword == null) &&
                    dispatcherResult == DispatcherAuthenticationResult.NO_ACTION)
                    // No user logged and no authentication asked, so,
                    // authenticate does nothing but has passed
                    return dispatcherResult;
                else
                {

                    // If user's credentials are valid
                    if (userID != null && suppliedPassword != null &&
                        this.validateUserCredentials(userID.toString(), suppliedPassword.toString()))
                    {
                        // Log user
                        IDIFSession session = this.logIn(userID.toString(), suppliedPassword.toString());
                        this.difContext.setSession(session);

                        if (session.isLogged())
                        {
                            dispatcherResult = DispatcherAuthenticationResult.LOGIN;
                        }
                        else
                        {
                            dispatcherResult = DispatcherAuthenticationResult.FAILED;
                            // result = false;
                            reason = getMessages().get("failedWithGoodUser");
                        }
                    }
                    else
                    {
                        // A bad user/pass was given. Authentication has failed
                        // result = false;
                        dispatcherResult = DispatcherAuthenticationResult.FAILED;

                        if (userID != null && this.getIdentityManager().userExists(userID.toString()))
                            reason = getMessages().get("wrongPass");
                        else
                            reason = getMessages().get("noUser");
                    }
                }
            }
            catch (IdentityManagerException identityManagerException)
            {
                // result = false;
                dispatcherResult = DispatcherAuthenticationResult.FAILED;
                reason = "Could not access identity manager to validate the user's credentials! ";
                exception = identityManagerException;
            }
            catch (AuthenticationManagerException authenticationManagerException)
            {
                // result = false;
                dispatcherResult = DispatcherAuthenticationResult.FAILED;
                reason = "Could not access authentication manager to validate the user's credentials! ";
                exception = authenticationManagerException;
            }
            catch (RuntimeException runtimeException)
            {
                // result = false;
                dispatcherResult = DispatcherAuthenticationResult.FAILED;
                reason = null;
                exception = runtimeException;
            }
            catch (ControllerException controllerException)
            {
                dispatcherResult = DispatcherAuthenticationResult.FAILED;
                reason = controllerException.getMessage();
                exception = controllerException;
            }

            if (dispatcherResult != DispatcherAuthenticationResult.FAILED)
            {
                return dispatcherResult;
            }
            else
            {
                ControllerException controllerException;
                if (exception instanceof ControllerException)
                {
                    controllerException = (ControllerException) exception;
                }
                else
                {

                    controllerException =
                            new ControllerException(ControllerExecutionStep.DISPATCHER_AUTHENTICATION, reason,
                                    exception);

                    controllerException.addToExceptionContext("Context", this.getContext());
                    controllerException.addToExceptionContext("Supplied User ID", userID);

                    if (suppliedPassword != null)
                        controllerException.addToExceptionContext("Supplied User Password",
                                StringUtils.getRepeatedString("*", suppliedPassword.toString().length()));
                }
                if (reason != null)
                {

                    // A known error occurred.
                    if (this.getStage().hasAuthenticationErrorInjection())
                    {
                        // If the stage has asked authentication error injection...
                        // capture, set it and proceed with no
                        // errors
                        setAuthenticationException(controllerException);

                        return dispatcherResult;
                    }
                    else
                        // else proceed with normal error launch...
                        throw controllerException;
                }
                else
                {
                    throw controllerException;
                }
            }
        }

        return dispatcherResult;
    }

    /**
     * Post Authentication execution.
     *
     * @param result the DispatcherAuthenticationResult
     *
     * @exception ControllerException
     */
    protected void postAuthentication(DispatcherAuthenticationResult result) throws ControllerException
    {
        try
        {
            if (result != DispatcherAuthenticationResult.NO_ACTION)
            {
                // For each IAuthenticationSSO will execute the result action
                for (Entry<String, IDIFInterceptorAuthentication> entry : DIFIoCRegistry.getRegistry()
                        .getImplementationsMap(IDIFInterceptorAuthentication.class).entrySet())
                {
                    // The result action was a login, so will execute is after execution
                    if (result == DispatcherAuthenticationResult.LOGIN)
                        entry.getValue().doAfterLogin(this.getContext());
                        // The result action was a logout, so will execute is after execution
                    else if (result == DispatcherAuthenticationResult.LOGOUT)
                        entry.getValue().doAfterLogout(this.getContext());
                }
            }

            String providerId = (String) this.getContext().getSession()
                    .getAttribute(DIFSession.REMOTE_AUTHENTICATION_PROVIDER_LOGIN);

            /* The Remote Authentication Servlet publish that a authentication was performed by a external provider */
            if (providerId != null)
            {
                String title = getMessages().get("remoteAuthenticationLoginTitle").replace("${providerid}", providerId);
                String text = getMessages().get("remoteAuthenticationLoginText").replace("${providerid}", providerId);

                this.getContext().addResultMessage("warn", title, text, true, true);
                this.getContext().getSession().addAttribute(DIFSession.REMOTE_AUTHENTICATION_PROVIDER_LOGIN, null);
                this.getContext().getSession()
                        .addAttribute(DIFSession.REMOTE_AUTHENTICATION_PROVIDER_LOGOUT, providerId);
            }
        }
        catch (ConfigurationException e)
        {
            throw new ControllerException(ControllerExecutionStep.DISPATCHER_AUTHENTICATION, e);
        }
    }

    /**
     * Pre Authentication execution.
     */
    protected void preAuthentication()
    {
        // For each IAuthenticationSSO will execute is before login or before logout action
        for (Entry<String, IDIFInterceptorAuthentication> entry : DIFIoCRegistry.getRegistry()
                .getImplementationsMap(IDIFInterceptorAuthentication.class).entrySet())
        {
            // execute the before login sso login
            entry.getValue().doBeforeLogin(this.getContext());
            // execute the before logout logic
            entry.getValue().doBeforeLogout(this.getContext());
        }
    }

    /**
     * Execute the dispatching cycle steps according to established rules for the business object. This method defines
     * the dispatching cycle steps order of execution. This sequence should not change and thus method overriding on
     * subclasses is forbidden by declaring the method <code>final</code>.
     *
     * @param isRedirection the is redirection
     *
     * @return the executed stage instance
     *
     * @exception BusinessFlowException  when an exception is thrown by the stage's init/execute steps
     * @exception ControllerException    when any runtime exception is thrown
     * @exception ConfigurationException
     */
    @Trace(metricName = "DIF:Dispatcher", dispatcher = true)
    final private IStageInstance runDispatchingSteps(Boolean isRedirection)
            throws BusinessFlowException, ControllerException, ConfigurationException
    {
        // Hot reload of DEM classes
        this.handleHotClassReload();

        IStageInstance stageInstance = null;

        try
        {
            // Get an instance of the stage.
            this.setStage(DEMRegistryImpl.getRegistry().getStage(this.getContext().getStage()));

            // MonitorUtil.detailLogEntry("DIF:Dispatcher", "Run: " + this.getStage().getID());

            // 1st: Validates the session and context. Checks if the
            // application/service/stage is registered
            validate(isRedirection);

            // 2nd: Authenticate
            authenticate();

            // Report NewRelic if in test or develop mode...
            if (DIFStartupConfiguration.getTestingMode() || DIFStartupConfiguration.getDeveloperMode())
            {
                IDIFUser user = this.getContext().getSession().getUser();

                if (user != null)
                    NewRelic.addCustomParameter("User", user.getID() + " [" + user.getName() + "]");
            }

            // 3rd: Authorize
            if (this.getStage().hasAuthorization())
                authorize();

            // 4th: Before Execute business logic interceptors
            executeInterceptorsBeforeExecution(this.getContext());

            if (!this.getContext().hasRedirection())
            {
                // 5th: Execute business logic
                stageInstance = execute();

                // 6th: After Execute business logic interceptors
                executeInterceptorsAfterExecution(stageInstance);
            }

            // 7th: End execution
            conclude();

            performCleanup(this.getContext(), true);
        }
        catch (BusinessFlowException e)
        {
            if (!handleException(e))
                throw e;
        }
        catch (ControllerException e)
        {
            if (!handleException(e))
                throw e;
        }
        catch (Exception e)
        {
            if (!handleException(e))
                throw new ControllerException(ControllerExecutionStep.DISPATCHER_EXECUTE, e);
        }
        finally
        {
            // If the session isMarked for removal, it means that in the process of processing the current request (or
            // one of the redirection requests within, the session has timed out. Since the time out informs the user of
            // this process, we can get a new one.
            if (this.getContext().getSession().isMarkedForRemoval())
            {
                this.getContext()
                        .setSession(theSessionManager.createSession(this.getContext().getSession().getSessionID()));
                theSessionManager.logOut(this.getContext().getSession().getSessionID());
                postAuthentication(DispatcherAuthenticationResult.LOGOUT);
            }
        }

        return stageInstance;
    }

    /**
     * Validates the session and the stage. This method always runs. To avoid overriding on base classes it is declared
     * <code>final</code>. Session validation includes checking if the stage is active and hasn't timed out. Stage
     * validation includes checking if the stage is registrable and if so checks if the license is valid.
     *
     * @param isRedirection the is redirection
     *
     * @exception ControllerException when any runtime exception is thrown
     */
    @Trace(metricName = "DIF:Dispatcher:Validate", dispatcher = true)
    final protected void validate(Boolean isRedirection) throws ControllerException
    {
        try
        {
            if (this.getStage() == null)
            {
                this.getContext().setResponseStatus(DIFResponseStatus.NOT_FOUND);

                // the stage does not exist
                ControllerException exception =
                        new ControllerException(ControllerExecutionStep.DISPATCHER_VALIDATE_REQUEST,
                                INVALID_STAGE_ERROR_MESSAGE + "(" + this.getContext().getStage() + ")");
                exception.addToExceptionContext("Stage", getContext().getStage());

                throw exception;
            }

            if (!isRedirection && this.getContext().getSession().hasTimedOut() &&
                this.getContext().getSession().isLogged() &&
                !"true".equals(this.getContext().getRequest().getParameter(IDIFRequest.LOGOUT_PARAMETER_ID)) &&
                !this.getContext().getRequest().isTemplateMode())
            {
                SessionTimeoutException exception =
                        new SessionTimeoutException(ControllerExecutionStep.DISPATCHER_VALIDATE_REQUEST);
                exception.addToExceptionContext("Stage", getContext().getStage());

                ((IPrivateDIFSession) (this.getContext().getSession())).setMarkedForRemoval(true);

                throw exception;
            }

            Object invalidBrowserFlagAccepted =
                    this.getContext().getSession().getAttribute(Constants.INVALID_BROWSER_ACCEPTED);
            Object clientValidations =
                    this.getContext().getRequest().getAttribute(IDIFRequest.CLIENT_VALIDATIONS_ATTRIBUTE_ID);
            Boolean allowClientValidations = true;
            if (clientValidations != null)
            {
                allowClientValidations = (Boolean) clientValidations;
            }

            if (!this.getContext().getRequest().isComponentMode() && !this.getContext().getRequest().isTemplateMode() &&
                !this.getContext().getRequest().isAjaxMode() && invalidBrowserFlagAccepted == null &&
                !this.getContext().getRequest().getClient().getSupportedBrowser() &&
                this.getContext().getStage() != BROWSER_VALIDATOR_STAGE_ID && allowClientValidations)
                this.getContext().redirectTo(BROWSER_VALIDATOR_STAGE_ID);

            if (!checkLicense())
            {
                LicenseViolationException exception =
                        new LicenseViolationException(ControllerExecutionStep.DISPATCHER_VALIDATE_REQUEST);
                exception.addToExceptionContext("Stage", getContext().getStage());

                throw exception;
            }
        }
        catch (RuntimeException runtimeException)
        {
            ControllerException controllerException =
                    new ControllerException(ControllerExecutionStep.DISPATCHER_VALIDATE_REQUEST, runtimeException);

            controllerException.addToExceptionContext("Context", this.getContext());
            controllerException.addToExceptionContext("Stage", this.getContext());

            throw controllerException;
        }
    }

    /**
     * Validate the user credentials.
     *
     * @param userId   the userId
     * @param password the password
     *
     * @return validation result
     *
     * @exception IdentityManagerException if a IdentityManager Exception occurs.
     * @exception ControllerException      the controller exception
     */
    protected boolean validateUserCredentials(String userId, String password)
            throws IdentityManagerException, ControllerException
    {
        boolean result = this.getIdentityManager().isIdentityValid(userId, password);

        if (result)
        {
            // For each IAuthenticationPlugin will execute validateUser action
            for (Entry<String, IDIFInterceptorAuthentication> entry : DIFIoCRegistry.getRegistry()
                    .getImplementationsMap(IDIFInterceptorAuthentication.class).entrySet())
            {
                // all the authentication plugin's must return true, so that the authentication proceed
                result = entry.getValue().validateUser(this.getContext(), userId, password);
                if (!result)
                {
                    break;
                }
            }
        }

        return result;
    }
}