/**
 * 2010, Digitalis Informatica. All rights reserved. Distribuicao e Gestao de Informatica, Lda. Estrada de Paco de Arcos
 * num.9 - Piso -1 2780-666 Paco de Arcos Telefone: (351) 21 4408990 Fax: (351) 21 4408999 http://www.digitalis.pt
 */

package pt.digitalis.dif.controller;

import pt.digitalis.dif.controller.interfaces.IDIFContext;
import pt.digitalis.dif.controller.interfaces.IDispatcherErrorHandler;
import pt.digitalis.dif.controller.objects.ExceptionHandlerData;

import javax.servlet.ServletException;
import java.util.HashMap;
import java.util.Map;

/**
 * Exception Handler stages registry class. Allows registering of all stages that declare to handle specific exceptions.
 *
 * @author Pedro Viegas <a href="mailto:pviegas@digitalis.pt">pviegas@digitalis.pt</a><br/>
 * @created 2010/05/04
 */
public class ExceptionHandlers
{

    /** Attribute ID to save the raised exception caught by the exception handler */
    static public final String RAISED_EXCEPTION_ATTRIBUTE = "RaisedExceptionCaghtInHandler";

    /** Exception Handler Stages map */
    static private Map<String, ExceptionHandlerData> handlers = new HashMap<String, ExceptionHandlerData>();

    /**
     * Registers a new Exception handler
     *
     * @param exceptionClass the exception class to register
     * @param stageID        the ID for the handler stage
     */
    static public void addExceptionHandler(Class<Throwable> exceptionClass, String stageID)
    {
        handlers.put(exceptionClass.getCanonicalName(), new ExceptionHandlerData(exceptionClass, stageID));
    }

    /**
     * Registers a new Exception handler
     *
     * @param exceptionClassName the exception class name to register
     * @param stageID            the ID for the handler stage
     */
    @SuppressWarnings("unchecked")
    static public void addExceptionHandler(String exceptionClassName, String stageID)
    {
        try
        {
            Class<Throwable> clazz = (Class<Throwable>) Class.forName(exceptionClassName);
            addExceptionHandler(clazz, stageID);
        }
        catch (ClassNotFoundException e)
        {
            e.printStackTrace();
        }
    }

    /**
     * Gets the ID of the handler stage for the given exception
     *
     * @param exceptionClass the class of the exception
     *
     * @return the handler stage ID
     */
    static public String getExceptionHandler(Class<? extends Throwable> exceptionClass)
    {
        ExceptionHandlerData handler = handlers.get(exceptionClass.getCanonicalName());

        // No direct hit. Search for inheritance match
        if (handler == null)
        {
            for (ExceptionHandlerData currentHandler : handlers.values())
            {
                if (currentHandler.getExceptionClass().isAssignableFrom(exceptionClass))
                {
                    handler = currentHandler;
                    break;
                }
            }
        }

        if (handler == null)
            return null;
        else
            return handler.getHandlerStageID();
    }

    /**
     * Gets the ID of the handler stage for the given exception
     *
     * @param exceptionClassName the class name of the exception
     *
     * @return the handler stage ID
     */
    static public String getExceptionHandler(String exceptionClassName)
    {
        ExceptionHandlerData handler = handlers.get(exceptionClassName);

        // No direct hit. Search for inheritance match. Must have a class object
        if (handler == null)
        {
            try
            {
                @SuppressWarnings("unchecked")
                Class<Throwable> clazz = (Class<Throwable>) Class.forName(exceptionClassName);

                return getExceptionHandler(clazz);
            }
            catch (ClassNotFoundException e)
            {
                e.printStackTrace();
                return null;
            }
        }
        else
            return handler.getHandlerStageID();
    }

    /**
     * Gets the ID of the handler stage for the given exception
     *
     * @param exception the exception
     *
     * @return the handler stage ID
     */
    static public String getExceptionHandler(Throwable exception)
    {
        String result = null;
        result = getExceptionHandler(exception.getClass());

        while (result == null && exception != null)
        {
            if (exception.getCause() instanceof Exception)
                exception = exception.getCause();
            else if (exception instanceof ServletException)
                exception = ((ServletException) exception).getRootCause();
            else
                break;

            if (exception != null)
                result = getExceptionHandler(exception.getClass());
        }

        return result;
    }

    /**
     * Handles redirection actions for a given exsception. Will determine the correct handler, add the necessary
     * information to the request and redirect to the handler stage.
     *
     * @param context   the current running stage context
     * @param exception the exception raised to handle
     */
    static public void handleException(IDIFContext context, Exception exception)
    {
        String stageToRedirectTo = getExceptionHandler(exception);

        if (stageToRedirectTo != null)
        {
            context.addStageResult(IDispatcherErrorHandler.ORIGINAL_REQUEST, context.getRequest());
            context.addStageResult(IDispatcherErrorHandler.EXCEPTION, exception);
            context.getRequest().addAttribute(RAISED_EXCEPTION_ATTRIBUTE, exception);
            context.redirectTo(stageToRedirectTo, true);
        }
    }

    /**
     * Inspector for a given exception. Will determine of the given exception has a handler defined
     *
     * @param exceptionClass the exception class
     *
     * @return T if the exception has a registered handler
     */
    static public boolean hasHandler(Class<? extends Throwable> exceptionClass)
    {
        return (getExceptionHandler(exceptionClass) != null);
    }

    /**
     * Inspector for a given exception. Will determine of the given exception has a handler defined
     *
     * @param exception the exception
     *
     * @return T if the exception has a registered handler
     */
    static public boolean hasHandler(Exception exception)
    {
        return (getExceptionHandler(exception) != null);
    }

    /**
     * Inspector for a given exception. Will determine of the given exception has a handler defined
     *
     * @param exceptionClassName the exception class name
     *
     * @return T if the exception has a registered handler
     */
    static public boolean hasHandler(String exceptionClassName)
    {
        return (getExceptionHandler(exceptionClassName) != null);
    }

    /**
     * Removes a registered exception handler
     *
     * @param key
     */
    public static void removeExceptionHandler(String key)
    {
        handlers.remove(key);
    }
}
