package pt.digitalis.dif.controller.interfaces;

import pt.digitalis.dif.controller.objects.RESTAction;
import pt.digitalis.dif.exception.controller.ControllerException;
import pt.digitalis.utils.config.ConfigurationException;

/**
 * AbstractChAL stands for <b>Ch</b>annel <b>A</b>bstraction <b>L</b>ayer and as the name implies it's purpose is to
 * insulate the framework from the transport channel used by the original request. As can be implied from the statement
 * above, AbstractChAL should be tailored to take care of a specific transport channel details. As such, AbstractChAL
 * should be extended using inheritance and is the subclass that should be instantiated. To enforce the subclass
 * instantiation AbstractChAL is declared <code>abstract</code>. AbstractChAL is responsible for translating the
 * original request into a format that the framework can understand, then executing the appropriated Dispatcher and
 * finally publish the results.
 *
 * @param <RequestType>  the type of request of the specific AbstractChAL
 * @param <ResponseType> the type of response of the specific AbstractChAL
 *
 * @author Pedro Viegas <a href="mailto:pviegas@digitalis.pt">pviegas@digitalis.pt</a>
 */
public interface IChAL<RequestType, ResponseType>
{

    /**
     * This method is responsible for the response translation from the framework internal format into the channel's own
     * format. Each AbstractChAL should implement this method in a way adapted to the specific needs of the incoming
     * request. The original request type is passed to the method because some technologies (such as Servlets) don't
     * allow content writing on the response object.
     *
     * @param difResponse     the framework's response
     * @param originalRequest the original request
     * @param finalResponse   the response to send back to the Listener
     */
    public void publish(IDIFResponse difResponse, RequestType originalRequest, ResponseType finalResponse);

    /**
     * This method will be called by the client via Listener.
     *
     * @param originalRequest the original request made by the client
     * @param finalResponse   the response object passed by the Listener and that will be sent back
     * @param action          the REST action if specified
     *
     * @return the DIFResponse containing the framework's response
     *
     * @exception ConfigurationException
     */
    public IDIFResponse serve(RequestType originalRequest, ResponseType finalResponse, RESTAction action)
            throws ConfigurationException;

    /**
     * This method should receive the request from the Listener and translate it to a format understandable by the
     * framework. Each AbstractChAL should implement this method in a way adapted to the specific needs of the incoming
     * request.
     *
     * @param originalRequest The request received from the Listener
     *
     * @return a BaseDIFRequest object that will be sent to the Dispatcher
     *
     * @exception ControllerException when any runtime exception is thrown
     */
    public IDIFRequest translateRequest(RequestType originalRequest) throws ControllerException;
}
