package pt.digitalis.dif.controller.interfaces;

import pt.digitalis.dif.controller.objects.ClientDescriptor;
import pt.digitalis.dif.controller.objects.RESTAction;
import pt.digitalis.dif.dem.interfaces.IStage;
import pt.digitalis.dif.utils.IObjectFormatter;

import java.util.Map;

/**
 * Defines the expected behavior for a DIF request.
 *
 * @author Luis Pinto <a href="mailto:lpinto@digitalis.pt">lpinto@digitalis.pt</a>
 * @author Rodrigo Gonalves <a href="mailto:rgoncalves@digitalis.pt">rgoncalves@digitalis.pt</a>
 */
public interface IDIFRequest extends IObjectFormatter
{

    /** The id where requests indicates that want's hot class reload to be triggered */
    final static public String CLASS_RELOAD = "_classreload";

    /** The id where requests indicates that the chal client validations should or shouldn't be made */
    final static public String CLIENT_VALIDATIONS_ATTRIBUTE_ID = "clientValidationsAttributeId";

    /** The id where requests indicate that want's legacy ComboBoxes or not */
    final static public String LEGACY_COMBOBOXES = "_legacyCombos";

    /** The id where requests indicates that want's download the server log */
    final static public String LOG_PARAMETER_ID = "_log";

    /** The id where requests indicates that want's to login */
    final static public String LOGIN_PARAMETER_ID = "loginparam";

    /** The id where requests indicates that want's to logout */
    final static public String LOGOUT_PARAMETER_ID = "logoutparam";

    /** The id where requests indicates that when a logout occurs, a redirect must be made to the url */
    final static public String LOGOUT_REDIRECT_URL_PARAMETER_ID = "logoutRedirectUrl";

    /** The id where users pass the user password to authenticate */
    final static public String PASSWORD_PARAMETER_ID = "_pass";

    /** The exceptional redirect to parameter */
    final static public String REDIRECT_TO_URL = "redirectToUrl";

    /** The id where users pass the user id to authenticate */
    final static public String USER_PARAMETER_ID = "_user";

    /**
     * Adds a new attribute to the DIFRequest collection.
     *
     * @param attributeName  attribute identifier
     * @param attributeValue attribute value
     */
    public void addAttribute(String attributeName, Object attributeValue);

    /**
     * Adds a new parameter to the DIFRequest collection.
     *
     * @param parameterName  parameter identifier
     * @param parameterValue parameter value
     */
    public void addParameter(String parameterName, Object parameterValue);

    /**
     * Retrieves an attribute from the attribute collection
     *
     * @param name the name of the attribute to retrieve
     *
     * @return on Object with the attribute value
     */
    public Object getAttribute(String name);

    /**
     * Inspector for the request parameters.
     *
     * @return the request attributes
     */
    public Map<String, Object> getAttributes();

    /**
     * Modifier for the request attributes.
     *
     * @param attributes the attributes to set
     */
    public void setAttributes(Map<String, Object> attributes);

    /**
     * Inspector for the client descriptor.
     *
     * @return the client descriptor
     */
    public ClientDescriptor getClient();

    /**
     * @param client the client to set
     */
    public void setClient(ClientDescriptor client);

    /**
     * Format inspector.
     *
     * @return the format
     */
    public String getFormat();

    /**
     * Format modifier.
     *
     * @param format the format to set
     */
    public void setFormat(String format);

    /**
     * Retrieves a parameter from the parameter collection
     *
     * @param name the name of the parameter to retrieve
     *
     * @return on Object with the parameter value
     */
    public Object getParameter(String name);

    /**
     * Get the original value form the native channel request, unparsed by any security validator
     *
     * @param parameterID the parameter identifier
     *
     * @return the parameter raw value
     */
    public String getParameterRawValue(String parameterID);

    /**
     * Get the original value form the native channel request, unparsed by any security validator
     *
     * @param parameterID the parameter ID
     *
     * @return the parameter values
     */
    public String[] getParameterValues(String parameterID);

    /**
     * Inspector for the request parameters.
     *
     * @return the request parameters
     */
    public Map<String, Object> getParameters();

    /**
     * Modifier for the request parameters.
     *
     * @param parameters the parameters to set
     */
    public void setParameters(Map<String, Object> parameters);

    /**
     * @return the REST action executed, if specified.
     */
    public RESTAction getRestAction();

    /**
     * @param restAction sets the RES action executed.
     */
    public void setRestAction(RESTAction restAction);

    /**
     * Inspector for the session.
     *
     * @return the session
     */
    public IDIFSession getSession();

    /**
     * Modifier for the session
     *
     * @param session the session to set
     */
    public void setSession(IDIFSession session);

    /**
     * Stage inspector.
     *
     * @return the stage
     */
    public String getStage();

    /**
     * Stage modifier.
     *
     * @param newStage the stage to set
     */
    public void setStage(String newStage);

    /**
     * Stage inspector.
     *
     * @return the stage proxy
     */
    public IStage getStageProxy();

    /**
     * @return T if current stage has been requested in ajax mode
     */
    public boolean isAjaxMode();

    /**
     * @return T if current stage has been requested has a back request (go back to previous page)
     */
    public boolean isBackRequest();

    /**
     * @return T if current stage has been requested in component mode
     */
    public boolean isComponentMode();

    /**
     * @param componentMode the componentMode to set
     */
    public void setComponentMode(boolean componentMode);

    /**
     * Checks if is embedded mode.
     *
     * @return true, if is embedded mode
     */
    public boolean isEmbeddedMode();

    /**
     * Sets the embedded mode.
     *
     * @param embeddedMode the new embedded mode
     */
    public void setEmbeddedMode(boolean embeddedMode);

    /**
     * @return T if current stage has been requested in help mode
     */
    public boolean isHelpMode();

    /**
     * @param helpMode the helpMode to set
     */
    public void setHelpMode(boolean helpMode);

    /**
     * @return T if current stage has been requested in popup mode
     */
    public boolean isPopupMode();

    /**
     * @param popupMode the popupMode to set
     */
    public void setPopupMode(boolean popupMode);

    /**
     * @return T if current stage has been requested in REST mode
     */
    public boolean isRestCall();

    /**
     * @return T if current stage has been requested in template mode
     */
    public boolean isTemplateMode();

    /**
     * @param templateMode the templateMode to set
     */
    public void setTemplateMode(boolean templateMode);
}
