/**
 * - Digitalis Internal Framework v2.0 - (C) 2007, Digitalis Informatica. Distribuicao e Gestao de Informatica, Lda.
 * Estrada de Paco de Arcos num.9 - Piso -1 2780-666 Paco de Arcos Telefone: (351) 21 4408990 Fax: (351) 21 4408999
 * http://www.digitalis.pt
 */
package pt.digitalis.dif.controller.objects;

import pt.digitalis.dif.controller.http.HTTPConstants;
import pt.digitalis.dif.controller.interfaces.IDIFContext;
import pt.digitalis.dif.controller.interfaces.IDIFRequest;
import pt.digitalis.dif.controller.interfaces.IDIFSession;
import pt.digitalis.dif.dem.interfaces.IStage;
import pt.digitalis.dif.dem.interfaces.IStageInstance;
import pt.digitalis.dif.dem.managers.IDEMManager;
import pt.digitalis.dif.dem.managers.impl.UsageIssuesManagerImpl;
import pt.digitalis.dif.dem.objects.EventType;
import pt.digitalis.dif.dem.objects.ViewObject;
import pt.digitalis.dif.dem.objects.issues.IssueScope;
import pt.digitalis.dif.dem.objects.issues.IssueType;
import pt.digitalis.dif.dem.objects.issues.UsageIssue;
import pt.digitalis.dif.ioc.DIFIoCRegistry;
import pt.digitalis.dif.startup.DIFGeneralConfigurationParameters;
import pt.digitalis.dif.utils.ObjectFormatter;
import pt.digitalis.dif.utils.ObjectFormatter.Format;
import pt.digitalis.dif.utils.logging.DIFLogger;
import pt.digitalis.utils.common.CollectionUtils;
import pt.digitalis.utils.common.StringUtils;
import pt.digitalis.utils.common.SystemUtils;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * This class is the base class for the DIF execution context.
 *
 * @author Rodrigo Gonalves <a href="mailto:rgoncalves@digitalis.pt">rgoncalves@digitalis.pt</a>
 * @created 2007 /03/16
 */
public class DIFContext implements IDIFContext
{

    /** The redirection parameters. */
    public Map<String, Object> redirectionParameters = new HashMap<String, Object>();

    /**
     * The ViewObject.
     */
    protected ViewObject theView = new ViewObject();

    /** The DIFRequest. */
    private IDIFRequest difRequest = null;

    /** The dif response status. */
    private DIFResponseStatus difResponseStatus = null;

    /**
     * Marks the defined stage for redirection processing by the Dispatcher
     */
    private boolean hasRedirection = false;

    /**
     * Aditional headers we can put on response. This headers will prevail over the headers already added in framework
     */
    private Map<String, String> httpHeaders = new HashMap<String, String>();

    /** The redirect history. */
    private List<RedirectInfo> redirectHistory = new ArrayList<RedirectInfo>();

    /** the result message */
    private ArrayList<ResultMessage> resultMessages = new ArrayList<ResultMessage>();

    /**
     * Identifier of the Stage that should be executed.
     */
    private String stage = new String();

    /** The {@link IStageInstance} */
    private IStageInstance stageInstance;

    /** The processing result set. */
    private Map<String, Object> stageResults = new HashMap<String, Object>();

    /**
     * Identifier of the Stage to redirect after execution of the current Stage.
     */
    private String stageToRedirect = new String();

    /** A map of custom attributes for stage usage. These will have the same life span of this DIFContext instance */
    private Map<String, Object> temporaryAttributes = new HashMap<String, Object>();

    /**
     * Add http header.
     *
     * @param name  the name
     * @param value the value
     *
     * @see pt.digitalis.dif.controller.interfaces.IDIFContext#addHTTPHeader(java.lang.String, java.lang.String)
     *         pt.digitalis.dif.controller.interfaces.IDIFContext#addHTTPHeader(java.lang.String, java.lang.String)
     */
    @Override
    public void addHTTPHeader(String name, String value)
    {
        this.httpHeaders.put(name, value);
    }

    /**
     * Add result message.
     *
     * @param type        the type
     * @param title       the title
     * @param description the description
     *
     * @see pt.digitalis.dif.controller.interfaces.IDIFContext#addResultMessage(java.lang.String, java.lang.String,
     *         java.lang.String) pt.digitalis.dif.controller.interfaces.IDIFContext#addResultMessage(java.lang.String,
     *         java.lang.String,
     *         java.lang.String)
     */
    @Override
    public void addResultMessage(String type, String title, String description)
    {
        addResultMessage(type, title, description, null);
    }

    /**
     * Add result message.
     *
     * @param type        the type
     * @param title       the title
     * @param description the description
     * @param popupMode   the popup mode
     *
     * @see pt.digitalis.dif.controller.interfaces.IDIFContext#addResultMessage(java.lang.String, java.lang.String,
     *         java.lang.String, boolean) pt.digitalis.dif.controller.interfaces.IDIFContext#addResultMessage(java.lang.String,
     *         java.lang.String,
     *         java.lang.String, boolean)
     */
    @Override
    public void addResultMessage(String type, String title, String description, boolean popupMode)
    {
        resultMessages.add(new ResultMessage(type, title, description, popupMode, null));
    }

    /**
     * Add result message.
     *
     * @param type           the type
     * @param title          the title
     * @param description    the description
     * @param popupMode      the popup mode
     * @param requireConfirm the require confirm
     *
     * @see pt.digitalis.dif.controller.interfaces.IDIFContext#addResultMessage(java.lang.String, java.lang.String,
     *         java.lang.String, boolean, boolean) pt.digitalis.dif.controller.interfaces.IDIFContext#addResultMessage(java.lang.String,
     *         java.lang.String,
     *         java.lang.String, boolean, boolean)
     */
    @Override
    public void addResultMessage(String type, String title, String description, boolean popupMode,
            boolean requireConfirm)
    {
        resultMessages.add(new ResultMessage(type, title, description, popupMode, requireConfirm, null));
    }

    /**
     * Add result message.
     *
     * @param type        the type
     * @param title       the title
     * @param description the description
     * @param mode        the mode
     *
     * @see pt.digitalis.dif.controller.interfaces.IDIFContext#addResultMessage(java.lang.String, java.lang.String,
     *         java.lang.String, java.lang.String) pt.digitalis.dif.controller.interfaces.IDIFContext#addResultMessage(java.lang.String,
     *         java.lang.String,
     *         java.lang.String, java.lang.String)
     */
    @Override
    public void addResultMessage(String type, String title, String description, String mode)
    {
        resultMessages.add(new ResultMessage(type, title, description, false, mode));
    }

    /**
     * Add stage result.
     *
     * @param resultName  the result name
     * @param resultValue the result value
     *
     * @see pt.digitalis.dif.controller.interfaces.IDIFContext#addStageResult(String, Object)
     *         pt.digitalis.dif.controller.interfaces.IDIFContext#addStageResult(String, Object)
     */
    @Override
    public void addStageResult(String resultName, Object resultValue)
    {
        this.stageResults.put(resultName, resultValue);
    }

    /**
     * Gets http headers.
     *
     * @return the http headers
     *
     * @see pt.digitalis.dif.controller.interfaces.IDIFContext#getHTTPHeaders() pt.digitalis.dif.controller.interfaces.IDIFContext#getHTTPHeaders()
     */
    @Override
    public Map<String, String> getHTTPHeaders()
    {
        return this.httpHeaders;
    }

    /**
     * Gets language.
     *
     * @return the language
     *
     * @see pt.digitalis.dif.controller.interfaces.IDIFContext#getLanguage() pt.digitalis.dif.controller.interfaces.IDIFContext#getLanguage()
     */
    @Override
    public String getLanguage()
    {
        IDIFSession session = getSession();

        if (session == null)
            return DIFGeneralConfigurationParameters.getInstance().getDefaultLanguage();
        else
            return session.getLanguage();
    }

    /**
     * Gets redirect count.
     *
     * @return the redirect count
     *
     * @see pt.digitalis.dif.controller.interfaces.IDIFContext#getRedirectCount() pt.digitalis.dif.controller.interfaces.IDIFContext#getRedirectCount()
     */
    @Override
    public int getRedirectCount()
    {
        return this.redirectHistory.size();
    }

    /**
     * Gets redirect list.
     *
     * @return the redirect list
     *
     * @see pt.digitalis.dif.controller.interfaces.IDIFContext#getRedirectList() pt.digitalis.dif.controller.interfaces.IDIFContext#getRedirectList()
     */
    @Override
    public List<RedirectInfo> getRedirectList()
    {
        return this.redirectHistory;
    }

    /**
     * Gets redirection parameters.
     *
     * @return the redirection parameters
     *
     * @see pt.digitalis.dif.controller.interfaces.IDIFContext#getRedirectionParameters()
     *         pt.digitalis.dif.controller.interfaces.IDIFContext#getRedirectionParameters()
     */
    @Override
    public Map<String, Object> getRedirectionParameters()
    {
        return redirectionParameters;
    }

    /**
     * Gets request.
     *
     * @return the request
     *
     * @see pt.digitalis.dif.controller.interfaces.IDIFContext#getRequest() pt.digitalis.dif.controller.interfaces.IDIFContext#getRequest()
     */
    @Override
    public IDIFRequest getRequest()
    {
        return difRequest;
    }

    /**
     * Sets request.
     *
     * @param difRequest the dif request
     *
     * @see pt.digitalis.dif.controller.interfaces.IDIFContext#setRequest(pt.digitalis.dif.controller.interfaces.IDIFRequest)
     *         pt.digitalis.dif.controller.interfaces.IDIFContext#setRequest(pt.digitalis.dif.controller.interfaces.IDIFRequest)
     */
    @Override
    public void setRequest(IDIFRequest difRequest)
    {
        this.difRequest = difRequest;
    }

    /**
     * Gets response status.
     *
     * @return the response status
     *
     * @see pt.digitalis.dif.controller.interfaces.IDIFContext#getResponseStatus() pt.digitalis.dif.controller.interfaces.IDIFContext#getResponseStatus()
     */
    @Override
    public DIFResponseStatus getResponseStatus()
    {
        return difResponseStatus;
    }

    /**
     * Sets response status.
     *
     * @param status the status
     *
     * @see pt.digitalis.dif.controller.interfaces.IDIFContext#setResponseStatus(pt.digitalis.dif.controller.objects.DIFResponseStatus)
     *         pt.digitalis.dif.controller.interfaces.IDIFContext#setResponseStatus(pt.digitalis.dif.controller.objects.DIFResponseStatus)
     */
    @Override
    public void setResponseStatus(DIFResponseStatus status)
    {
        this.difResponseStatus = status;
    }

    /**
     * Gets response title.
     *
     * @return the response title
     *
     * @see pt.digitalis.dif.controller.interfaces.IDIFContext#getResponseTitle() pt.digitalis.dif.controller.interfaces.IDIFContext#getResponseTitle()
     */
    @Override
    public String getResponseTitle()
    {
        return StringUtils.toStringOrNull(this.getStageResults().get(OVERRIDE_PAGE_TITLE));
    }

    /**
     * Sets response title.
     *
     * @param title the title
     *
     * @see pt.digitalis.dif.controller.interfaces.IDIFContext#setResponseTitle(java.lang.String)
     *         pt.digitalis.dif.controller.interfaces.IDIFContext#setResponseTitle(java.lang.String)
     */
    @Override
    public void setResponseTitle(String title)
    {
        this.getStageResults().put(OVERRIDE_PAGE_TITLE, title);
    }

    /**
     * Gets result messages.
     *
     * @return the result messages
     *
     * @see pt.digitalis.dif.controller.interfaces.IDIFContext#getResultMessages() pt.digitalis.dif.controller.interfaces.IDIFContext#getResultMessages()
     */
    @Override
    public ArrayList<ResultMessage> getResultMessages()
    {
        return resultMessages;
    }

    /**
     * Gets session.
     *
     * @return the session
     *
     * @see pt.digitalis.dif.controller.interfaces.IDIFContext#getSession() pt.digitalis.dif.controller.interfaces.IDIFContext#getSession()
     */
    @Override
    public IDIFSession getSession()
    {
        if (this.difRequest != null)
        {
            return this.difRequest.getSession();
        }
        else
        {
            return null;
        }
    }

    /**
     * Sets session.
     *
     * @param session the session
     *
     * @see pt.digitalis.dif.controller.interfaces.IDIFContext#setSession(IDIFSession)
     *         pt.digitalis.dif.controller.interfaces.IDIFContext#setSession(IDIFSession)
     */
    @Override
    public void setSession(IDIFSession session)
    {
        this.difRequest.setSession(session);
    }

    /**
     * Gets stage.
     *
     * @return the stage
     *
     * @see pt.digitalis.dif.controller.interfaces.IDIFContext#getStage() pt.digitalis.dif.controller.interfaces.IDIFContext#getStage()
     */
    @Override
    public String getStage()
    {
        return this.stage;
    }

    /**
     * Sets stage.
     *
     * @param stage the stage
     *
     * @see pt.digitalis.dif.controller.interfaces.IDIFContext#setStage(String) pt.digitalis.dif.controller.interfaces.IDIFContext#setStage(String)
     */
    @Override
    public void setStage(String stage)
    {
        this.stage = stage;
    }

    /**
     * Gets stage instance.
     *
     * @return the stage instance
     *
     * @see pt.digitalis.dif.controller.interfaces.IDIFContext#getStageInstance() pt.digitalis.dif.controller.interfaces.IDIFContext#getStageInstance()
     */
    @Override
    public IStageInstance getStageInstance()
    {
        return stageInstance;
    }

    /**
     * Sets stage instance.
     *
     * @param stageInstance the stage instance
     *
     * @see pt.digitalis.dif.controller.interfaces.IDIFContext#setStageInstance(pt.digitalis.dif.dem.interfaces.IStageInstance)
     *         pt.digitalis.dif.controller.interfaces.IDIFContext#setStageInstance(pt.digitalis.dif.dem.interfaces.IStageInstance)
     */
    @Override
    public void setStageInstance(IStageInstance stageInstance)
    {
        this.stageInstance = stageInstance;
    }

    /**
     * Gets stage results.
     *
     * @return the stage results
     *
     * @see pt.digitalis.dif.controller.interfaces.IDIFContext#getStageResults() pt.digitalis.dif.controller.interfaces.IDIFContext#getStageResults()
     */
    @Override
    public final Map<String, Object> getStageResults()
    {
        return stageResults;
    }

    /**
     * Sets stage results.
     *
     * @param results the results
     *
     * @see pt.digitalis.dif.controller.interfaces.IDIFContext#setStageResults(Map)
     *         pt.digitalis.dif.controller.interfaces.IDIFContext#setStageResults(Map)
     */
    @Override
    public final void setStageResults(Map<String, Object> results)
    {
        this.stageResults = results;
    }

    /**
     * Gets temporary attributes.
     *
     * @return the temporary attributes
     *
     * @see pt.digitalis.dif.controller.interfaces.IDIFContext#getTemporaryAttributes()
     *         pt.digitalis.dif.controller.interfaces.IDIFContext#getTemporaryAttributes()
     */
    @Override
    public Map<String, Object> getTemporaryAttributes()
    {
        return temporaryAttributes;
    }

    /**
     * Sets temporary attributes.
     *
     * @param temporaryAttributes the temporary attributes
     *
     * @see pt.digitalis.dif.controller.interfaces.IDIFContext#setTemporaryAttributes(java.util.Map)
     *         pt.digitalis.dif.controller.interfaces.IDIFContext#setTemporaryAttributes(java.util.Map)
     */
    @Override
    public void setTemporaryAttributes(Map<String, Object> temporaryAttributes)
    {
        this.temporaryAttributes = temporaryAttributes;
    }

    /**
     * Gets view.
     *
     * @return the view
     *
     * @see pt.digitalis.dif.controller.interfaces.IDIFContext#getView() pt.digitalis.dif.controller.interfaces.IDIFContext#getView()
     */
    @Override
    public ViewObject getView()
    {
        return this.theView;
    }

    /**
     * Sets view.
     *
     * @param newView the new view
     *
     * @see pt.digitalis.dif.controller.interfaces.IDIFContext#setView(ViewObject) pt.digitalis.dif.controller.interfaces.IDIFContext#setView(ViewObject)
     */
    @Override
    public void setView(ViewObject newView)
    {
        this.theView = newView;
    }

    /**
     * This method sets the current stage equal to the redirection stage. The redirection stage is set to null and a the
     * redirection flag is set to T. The stage results are reset.
     *
     * @see pt.digitalis.dif.controller.interfaces.IDIFContext#handleRedirection() pt.digitalis.dif.controller.interfaces.IDIFContext#handleRedirection()
     */
    @Override
    public void handleRedirection()
    {
        this.stage = this.stageToRedirect;
        this.stageToRedirect = null;
        this.hasRedirection = false;
    }

    /**
     * Has redirection boolean.
     *
     * @return the boolean
     *
     * @see pt.digitalis.dif.controller.interfaces.IDIFContext#hasRedirection() pt.digitalis.dif.controller.interfaces.IDIFContext#hasRedirection()
     */
    @Override
    public boolean hasRedirection()
    {
        return this.hasRedirection;
    }

    /**
     * Redirect to.
     *
     * @param newStage the new stage
     *
     * @see pt.digitalis.dif.controller.interfaces.IDIFContext#redirectTo(java.lang.String)
     *         pt.digitalis.dif.controller.interfaces.IDIFContext#redirectTo(java.lang.String)
     */
    @Override
    public void redirectTo(String newStage)
    {
        this.redirectTo(newStage, "");
    }

    /**
     * Redirect to.
     *
     * @param newStage                 the new stage
     * @param allowRedirectToSameStage the allow redirect to same stage
     *
     * @see pt.digitalis.dif.controller.interfaces.IDIFContext#redirectTo(java.lang.String, boolean)
     *         pt.digitalis.dif.controller.interfaces.IDIFContext#redirectTo(java.lang.String, boolean)
     */
    @Override
    public void redirectTo(String newStage, boolean allowRedirectToSameStage)
    {
        this.redirectTo(newStage, "", allowRedirectToSameStage);
    }

    /**
     * Redirect to.
     *
     * @param newStage              the new stage
     * @param redirectionParameters the redirection parameters
     *
     * @see pt.digitalis.dif.controller.interfaces.IDIFContext#redirectTo(java.lang.String, java.util.Map)
     *         pt.digitalis.dif.controller.interfaces.IDIFContext#redirectTo(java.lang.String, java.util.Map)
     */
    @Override
    public void redirectTo(String newStage, Map<String, Object> redirectionParameters)
    {
        this.redirectTo(newStage, redirectionParameters, false);
    }

    /**
     * Redirect to.
     *
     * @param newStage                 the new stage
     * @param redirectionParameters    the redirection parameters
     * @param allowRedirectToSameStage the allow redirect to same stage
     *
     * @see pt.digitalis.dif.controller.interfaces.IDIFContext#redirectTo(java.lang.String, java.util.Map, boolean)
     *         pt.digitalis.dif.controller.interfaces.IDIFContext#redirectTo(java.lang.String, java.util.Map, boolean)
     */
    @Override
    public void redirectTo(String newStage, Map<String, Object> redirectionParameters, boolean allowRedirectToSameStage)
    {
        // Prevent redirect to the current stage and thus might allow indefinite cyclic redirects
        if (allowRedirectToSameStage || !this.getStage().equals(newStage))
        {
            this.stageToRedirect = newStage;
            this.hasRedirection = true;

            if (redirectionParameters != null && !redirectionParameters.isEmpty())
            {
                this.redirectionParameters = redirectionParameters;
                this.difRequest.setParameters(redirectionParameters);
            }

            // Register redirects for the current context
            this.redirectHistory.add(new RedirectInfo(this.stageToRedirect, redirectionParameters));
        }
        else
        {
            DIFLogger.getLogger().warn("Prevented a redirection to the same stage: \"" + newStage +
                                       "\" (cyclic redirection danger!).");
        }
    }

    /**
     * Redirect to.
     *
     * @param newStage              the new stage
     * @param redirectionParameters the redirection parameters
     *
     * @see pt.digitalis.dif.controller.interfaces.IDIFContext#redirectTo(java.lang.String, java.lang.String)
     *         pt.digitalis.dif.controller.interfaces.IDIFContext#redirectTo(java.lang.String, java.lang.String)
     */
    @Override
    public void redirectTo(String newStage, String redirectionParameters)
    {
        redirectTo(newStage, redirectionParameters, false);
    }

    /**
     * Redirect to.
     *
     * @param newStage                 the new stage
     * @param redirectionParameters    the redirection parameters
     * @param allowRedirectToSameStage the allow redirect to same stage
     *
     * @see pt.digitalis.dif.controller.interfaces.IDIFContext#redirectTo(java.lang.String, java.lang.String,
     *         boolean)
     *         pt.digitalis.dif.controller.interfaces.IDIFContext#redirectTo(java.lang.String, java.lang.String,
     *         boolean)
     */
    @Override
    public void redirectTo(String newStage, String redirectionParameters, boolean allowRedirectToSameStage)
    {
        Map<String, Object> parameterMap = new HashMap<String, Object>();

        if (redirectionParameters != null && !"".equals(redirectionParameters))
            parameterMap = CollectionUtils.keyValueStringToMapObject(redirectionParameters);

        redirectTo(newStage, parameterMap, allowRedirectToSameStage);
    }

    /**
     * Report issue usage issue.
     *
     * @param uID              the u id
     * @param type             the type
     * @param issueDescription the issue description
     *
     * @return the usage issue
     *
     * @see pt.digitalis.dif.controller.interfaces.IDIFContext#reportIssue(java.lang.String,
     *         pt.digitalis.dif.dem.objects.issues.IssueType, java.lang.String) pt.digitalis.dif.controller.interfaces.IDIFContext#reportIssue(java.lang.String,
     *         pt.digitalis.dif.dem.objects.issues.IssueType, java.lang.String)
     */
    @Override
    public UsageIssue reportIssue(String uID, IssueType type, String issueDescription)
    {
        return this.reportIssue(uID, type, issueDescription, null);
    }

    /**
     * Report issue usage issue.
     *
     * @param uID              the u id
     * @param type             the type
     * @param issueDescription the issue description
     * @param exception        the exception
     *
     * @return the usage issue
     *
     * @see pt.digitalis.dif.controller.interfaces.IDIFContext#reportIssue(java.lang.String,
     *         pt.digitalis.dif.dem.objects.issues.IssueType, java.lang.String, java.lang.Exception)
     *         pt.digitalis.dif.controller.interfaces.IDIFContext#reportIssue(java.lang.String,
     *         pt.digitalis.dif.dem.objects.issues.IssueType, java.lang.String, java.lang.Exception)
     */
    @Override
    public UsageIssue reportIssue(String uID, IssueType type, String issueDescription, Exception exception)
    {
        return this.reportIssue(uID, type, issueDescription, exception, true);
    }

    /**
     * Report issue usage issue.
     *
     * @param uID              the u id
     * @param type             the type
     * @param issueDescription the issue description
     * @param exception        the exception
     * @param showStackTrace   the show stack trace
     *
     * @return the usage issue
     *
     * @see pt.digitalis.dif.controller.interfaces.IDIFContext#reportIssue(java.lang.String,
     *         pt.digitalis.dif.dem.objects.issues.IssueType, java.lang.String, java.lang.Exception, boolean)
     *         pt.digitalis.dif.controller.interfaces.IDIFContext#reportIssue(java.lang.String,
     *         pt.digitalis.dif.dem.objects.issues.IssueType, java.lang.String, java.lang.Exception, boolean)
     */
    @Override
    public UsageIssue reportIssue(String uID, IssueType type, String issueDescription, Exception exception,
            boolean showStackTrace)
    {
        IStage stage = DIFIoCRegistry.getRegistry().getImplementation(IDEMManager.class).getStage(this.getStage());

        Object submitForm = this.getRequest().getParameter(HTTPConstants.FORM_SUBMIT_NAME);
        Object eventID = this.getRequest().getParameter(HTTPConstants.EVENT_ID);
        EventType eventType = null;
        String originalDescription = issueDescription;
        String requestSum = "";
        String traceContent = null;

        // If form submitted check if an event handler for it exists
        if (stage != null)
        {
            if (stage.getEventHandlers().get(EventType.FORM_SUBMIT).contains(submitForm))
                eventType = EventType.FORM_SUBMIT;
            else if (stage.getEventHandlers().get(EventType.FORM_SUBMIT_SAVE_ACTION).contains(submitForm))
                eventType = EventType.FORM_SUBMIT_SAVE_ACTION;
            else if (stage.getEventHandlers().get(EventType.FORM_SUBMIT_AJAX_REQUEST).contains(submitForm))
                eventType = EventType.FORM_SUBMIT_AJAX_REQUEST;
            else if (stage.getEventHandlers().get(EventType.AJAX_REQUEST).contains(eventID))
                eventType = EventType.AJAX_REQUEST;
            else if (stage.getEventHandlers().get(EventType.DOCUMENT_TYPE).contains(eventID))
                eventType = EventType.DOCUMENT_TYPE;
        }
        if (eventID != null)
            issueDescription += "<b>Event:</b> " + eventID + (eventType != null ? " [" + eventType + "]" : "");
        if (submitForm != null)
            issueDescription += "<b>Form:</b> " + submitForm;

        UsageIssue issue = new UsageIssue();
        issue.setUID(uID == null ? UsageIssuesManagerImpl.getInstance().generateUID() : this.getStage() + uID);

        if (showStackTrace)
        {
            Map<String, String> issueDetails = new HashMap<String, String>();

            if (exception == null)
            {
                StackTraceElement[] traceList = Thread.currentThread().getStackTrace();
                StackTraceElement[] parsedTraceList = new StackTraceElement[traceList.length - 1];
                int totalToRemove = 1;

                // remove the current call from the original stack
                while (traceList[totalToRemove].getMethodName().contains("reportIssue"))
                {
                    totalToRemove++;
                }
                System.arraycopy(traceList, totalToRemove, parsedTraceList, 0, parsedTraceList.length - totalToRemove);

                traceContent = SystemUtils.getStackTraceListHTML(parsedTraceList).toString();
            }
            else
                traceContent = SystemUtils.getStackTraceHTML(exception);

            if (stage != null)
                requestSum += "<b>Stage:</b> " + stage.getOriginalClassName() + "<br/>";
            if (eventID != null)
                requestSum += "<b>Event:</b> " + eventID + (eventType != null ? " [" + eventType + "]" : "") + "<br/>";
            if (submitForm != null)
                requestSum += "<b>Form:</b> " + submitForm + "<br/>";

            issueDetails.put("issueDescription", requestSum + "<br/>" + issueDescription);
            issueDetails.put("traceContent", traceContent == null ? "No stack trace available!" : traceContent);
            issueDetails.put("content", "<code>" + this.toString().replaceAll("\\n", "<br />") + "</code>");
            issue.setShowDetailsContent(issueDetails);
        }

        issue.setIssueScope(IssueScope.RUNTIME);
        issue.setIssueType(type);
        issue.setIssueDescription(issueDescription);
        issue.setIssueSmallDescription(originalDescription + "\n" + requestSum);
        issue.setIssuePlainDescription(originalDescription + "\n" + requestSum + "\n" +
                                       (StringUtils.isNotBlank(traceContent) ? traceContent : ""));
        if (stage != null)
        {
            issue.setLocation(stage.getOriginalClassName());
        }
        issue.setException(exception);
        issue.setContext(this);

        this.reportIssue(issue);

        return issue;
    }

    /**
     * Report issue usage issue.
     *
     * @param issue the issue
     *
     * @return the usage issue
     *
     * @see pt.digitalis.dif.controller.interfaces.IDIFContext#reportIssue(pt.digitalis.dif.dem.objects.issues.UsageIssue)
     *         pt.digitalis.dif.controller.interfaces.IDIFContext#reportIssue(pt.digitalis.dif.dem.objects.issues.UsageIssue)
     */
    @Override
    public UsageIssue reportIssue(UsageIssue issue)
    {
        UsageIssuesManagerImpl.getInstance().addIssue(issue);

        return issue;
    }

    /**
     * To object formatter object formatter.
     *
     * @param format        the format
     * @param dumpedObjects the dumped objects
     *
     * @return the object formatter
     *
     * @see pt.digitalis.dif.utils.IObjectFormatter#toObjectFormatter(pt.digitalis.dif.utils.ObjectFormatter.Format)
     *         pt.digitalis.dif.utils.IObjectFormatter#toObjectFormatter(pt.digitalis.dif.utils.ObjectFormatter.Format)
     */
    @Override
    public ObjectFormatter toObjectFormatter(ObjectFormatter.Format format, List<Object> dumpedObjects)
    {
        ObjectFormatter formatter = new ObjectFormatter(format, dumpedObjects);
        formatter.addItem("Request", difRequest);
        formatter.addItem("Has Redirection", hasRedirection);
        formatter.addItem("View", theView);
        formatter.addItem("Stage", stage);
        formatter.addItem("Stage Results", stageResults);
        formatter.addItemIfNotNull("Stage to redirect", stageToRedirect);

        return formatter;
    }

    /**
     * Prints the context information on an human-readable form. Overrides java.lang.Object#toString().
     *
     * @return the string
     *
     * @see java.lang.Object#toString() java.lang.Object#toString()
     */
    @Override
    public String toString()
    {
        return toObjectFormatter(Format.TEXT, null).getFormatedObject();
    }
}