/**
 * - Digitalis Internal Framework v2.0 - (C) 2007, Digitalis Informatica. Distribuicao e Gestao de Informatica, Lda.
 * Estrada de Paco de Arcos num.9 - Piso -1 2780-666 Paco de Arcos Telefone: (351) 21 4408990 Fax: (351) 21 4408999
 * http://www.digitalis.pt
 */
package pt.digitalis.dif.controller.objects;

import pt.digitalis.dif.controller.interfaces.INavigationHistory;
import pt.digitalis.dif.controller.interfaces.IPrivateDIFSession;
import pt.digitalis.dif.controller.security.managers.ISessionManager;
import pt.digitalis.dif.ioc.DIFIoCRegistry;
import pt.digitalis.dif.model.utils.AbstractBeanAttributes;
import pt.digitalis.dif.startup.DIFGeneralConfigurationParameters;
import pt.digitalis.dif.utils.ObjectFormatter;
import pt.digitalis.dif.utils.ObjectFormatter.Format;
import pt.digitalis.utils.common.BeanInspector;

import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;

// TODO: Auto-generated Javadoc

/**
 * Base implementation of a DIF Session. Should be extended by each specific channel session. (i.e. DIFSessionHTTPImpl)
 *
 * @author Rodrigo Gonalves <a href="mailto:rgoncalves@digitalis.pt">rgoncalves@digitalis.pt</a>
 * @author Pedro Viegas <a href="mailto:pviegas@digitalis.pt">pviegas@digitalis.pt</a>
 * @created Nov 29, 2007
 */
public class DIFSession extends AbstractBeanAttributes implements IPrivateDIFSession
{

    /** The session key indicating that authentication was performed by a remote provider. */
    public static final String REMOTE_AUTHENTICATION_PROVIDER_LOGIN = "remote_authentication_provider_login";

    /** The session key indicating that authentication was performed by a remote provider. */
    public static final String REMOTE_AUTHENTICATION_PROVIDER_LOGOUT = "remote_authentication_provider_logout";

    /** The session unique ID. */
    private final String sessionID;

    /** Session attributes. */
    private Map<String, Object> attributes = new HashMap<String, Object>();

    /** The client descriptor. */
    private ClientDescriptor clientDescriptor = null;

    /** Keeps record of session creation time. */
    private long firstAccessTime = 0;

    /** The current session navigation history. */
    private INavigationHistory history = null;

    /** The current session user language. */
    private String language = null;

    /** Keeps record of last session access. */
    private long lastAccessTime = 0;

    /** If the session been marked for removal. */
    private Boolean markedForRemoval = false;

    /** The number of requests. */
    private Long numberOfRequests = 0L;

    /** The defined time out value for the session. */
    private long sessionTimeOut = 0;

    /** The user associated to the session. */
    private DIFUserInSession user;

    /**
     * Default constructor. Sets the first access time.
     *
     * @param sessionID the session unique identifier
     */
    public DIFSession(String sessionID)
    {
        this.firstAccessTime = System.currentTimeMillis();
        this.lastAccessTime = System.currentTimeMillis();
        this.sessionID = sessionID;
    }

    /**
     * One-argument constructor that receives an user.
     *
     * @param sessionID the session unique identifier
     * @param user      a DIF user
     */
    public DIFSession(String sessionID, DIFUserInSession user)
    {
        this(sessionID);
        this.user = user;
    }

    /**
     * Adds the attribute.
     *
     * @param key   the key
     * @param value the value
     *
     * @see pt.digitalis.dif.controller.interfaces.IDIFSession#addAttribute(java.lang.String, java.lang.Object)
     */
    @Override
    public void addAttribute(String key, Object value)
    {
        attributes.put(key, value);
    }

    /**
     * Contains attribute.
     *
     * @param key the key
     *
     * @return true, if successful
     *
     * @see pt.digitalis.dif.controller.interfaces.IDIFSession#containsAttribute(java.lang.String)
     */
    @Override
    public boolean containsAttribute(String key)
    {
        return attributes.containsKey(key);
    }

    /**
     * Force keep alive.
     *
     * @see pt.digitalis.dif.controller.interfaces.IPrivateDIFSession#forceKeepAlive()
     */
    @Override
    public void forceKeepAlive()
    {
        this.setLastAccessTime(System.currentTimeMillis());
    }

    /**
     * Gets the attribute.
     *
     * @param key the key
     *
     * @return the attribute
     *
     * @see pt.digitalis.dif.controller.interfaces.IDIFSession#getAttribute(java.lang.String)
     */
    @Override
    public Object getAttribute(String key)
    {
        Object value = attributes.get(key);

        if (value == null && key != null)
        {
            // Inject IBeanAttributes logic for session characteristics on getAttribute
            if (key.contains("."))
            {
                String[] args = key.split("\\.", 2);

                return BeanInspector.getValue(getAttributeNoGraphNavigation(args[0].trim()), args[1].trim());
            }
            else
                return getAttributeNoGraphNavigation(key);
        }

        return value;
    }

    /**
     * Gets the attribute no graph navigation.
     *
     * @param attributeName the attribute name
     *
     * @return the attribute no graph navigation
     */
    @Override
    protected Object getAttributeNoGraphNavigation(String attributeName)
    {
        if ("attributes".equals(attributeName))
            return this.attributes;
        else if ("firstAccessTime".equals(attributeName))
            return this.firstAccessTime;
        else if ("history".equals(attributeName))
            return this.history;
        else if ("language".equals(attributeName))
            return this.language;
        else if ("lastAccessTime".equals(attributeName))
            return this.lastAccessTime;
        else if ("markedForRemoval".equals(attributeName))
            return this.markedForRemoval;
        else if ("sessionID".equals(attributeName))
            return this.sessionID;
        else if ("sessionTimeOut".equals(attributeName))
            return this.sessionTimeOut;
        else if ("user".equals(attributeName))
            return this.user;
        else if ("numberOfRequests".equals(attributeName))
            return this.numberOfRequests;
        else if ("clientDescriptor".equals(attributeName))
            return this.clientDescriptor;
        return null;
    }

    /**
     * Gets the attributes.
     *
     * @return the attributes
     *
     * @see pt.digitalis.dif.controller.interfaces.IDIFSession#getAttributes()
     */
    @Override
    public Map<String, Object> getAttributes()
    {
        return attributes;
    }

    /**
     * Sets the attributes.
     *
     * @param attributes the attributes
     *
     * @see pt.digitalis.dif.controller.interfaces.IDIFSession#setAttributes(Map)
     */
    @Override
    public void setAttributes(Map<String, Object> attributes)
    {
        this.attributes = attributes;
    }

    /**
     * Gets the client descriptor.
     *
     * @return the client descriptor
     *
     * @see pt.digitalis.dif.controller.interfaces.IDIFSession#getClientDescriptor()
     */
    @Override
    public ClientDescriptor getClientDescriptor()
    {
        return clientDescriptor;
    }

    /**
     * @see pt.digitalis.dif.controller.interfaces.IPrivateDIFSession#setClientDescriptor(pt.digitalis.dif.controller.objects.ClientDescriptor)
     */
    @Override
    public void setClientDescriptor(ClientDescriptor clientDescriptor)
    {
        this.clientDescriptor = clientDescriptor;

        DIFIoCRegistry.getRegistry().getImplementation(ISessionManager.class).reportNewClient(clientDescriptor);
    }

    /**
     * Inspector for the 'firstAccessTime' property.
     *
     * @return the 'firstAccessTime' value
     */
    public long getFirstAccessTime()
    {
        return this.firstAccessTime;
    }

    /**
     * Gets the language.
     *
     * @return the language
     *
     * @see pt.digitalis.dif.controller.interfaces.IDIFSession#getLanguage()
     */
    @Override
    public String getLanguage()
    {
        return language;
    }

    /**
     * @see pt.digitalis.dif.controller.interfaces.IDIFSession#setLanguage(java.lang.String)
     */
    @Override
    public void setLanguage(String language)
    {
        this.language = language;
    }

    /**
     * Inspector for the 'lastAccessTime' property.
     *
     * @return the 'lastAccessTime' value
     */
    public long getLastAccessTime()
    {
        return this.lastAccessTime;
    }

    /**
     * Modifier for the 'lastAccessTime' property.
     *
     * @param lastAccessTime the 'lastAccessTime' new value to set
     */
    public void setLastAccessTime(long lastAccessTime)
    {
        this.lastAccessTime = lastAccessTime;
    }

    /**
     * Gets the navigation history.
     *
     * @return the navigation history
     *
     * @see pt.digitalis.dif.controller.interfaces.IDIFSession#getNavigationHistory()
     */
    @Override
    public INavigationHistory getNavigationHistory()
    {
        if (history == null)
            history = DIFIoCRegistry.getRegistry().getImplementation(INavigationHistory.class);

        return history;
    }

    /**
     * @see pt.digitalis.dif.controller.interfaces.IDIFSession#getNumberOfRequests()
     */
    @Override
    public Long getNumberOfRequests()
    {
        return numberOfRequests;
    }

    /**
     * Gets the session ID.
     *
     * @return the session ID
     *
     * @see pt.digitalis.dif.controller.interfaces.IDIFSession#getSessionID()
     */
    @Override
    public String getSessionID()
    {
        return sessionID;
    }

    /**
     * Gets the session time out.
     *
     * @return the session time out
     *
     * @see pt.digitalis.dif.controller.interfaces.IDIFSession#getSessionTimeOut()
     */
    @Override
    public long getSessionTimeOut()
    {
        return this.sessionTimeOut;
    }

    /**
     * Sets the session time out.
     *
     * @param sessionTimeOut the new session time out
     *
     * @see pt.digitalis.dif.controller.interfaces.IDIFSession#setSessionTimeOut(long)
     */
    @Override
    public void setSessionTimeOut(long sessionTimeOut)
    {
        this.sessionTimeOut = sessionTimeOut;
    }

    /**
     * Gets the user.
     *
     * @return the user
     *
     * @see pt.digitalis.dif.controller.interfaces.IDIFSession#getUser()
     */
    @Override
    public DIFUserInSession getUser()
    {
        return user;
    }

    /**
     * Sets the user.
     *
     * @param user the new user
     *
     * @see pt.digitalis.dif.controller.interfaces.IDIFSession#setUser(pt.digitalis.dif.controller.objects.DIFUserInSession)
     */
    @Override
    public void setUser(DIFUserInSession user)
    {
        this.user = user;
    }

    /**
     * Checks for expired after time out.
     *
     * @return true, if successful
     *
     * @see pt.digitalis.dif.controller.interfaces.IPrivateDIFSession#hasExpiredAfterTimeOut()
     */
    @Override
    public boolean hasExpiredAfterTimeOut()
    {
        long currentTime = System.currentTimeMillis();
        Long timeAfterTimeOut = DIFGeneralConfigurationParameters.getInstance().getSessionExpirationTimeAfterTimeout();
        if (timeAfterTimeOut == null)
            timeAfterTimeOut = 0L;

        return (this.getLastAccessTime() + this.getSessionTimeOut() + timeAfterTimeOut < currentTime);
    }

    /**
     * The criteria for a timed out session is that the last access time plus the session time out must be higher than
     * the current time.
     *
     * @return true, if successful
     *
     * @see pt.digitalis.dif.controller.interfaces.IDIFSession#hasTimedOut()
     */
    @Override
    public boolean hasTimedOut()
    {

        long currentTime = System.currentTimeMillis();

        return (this.getLastAccessTime() + this.getSessionTimeOut() < currentTime);
    }

    /**
     * @see pt.digitalis.dif.controller.interfaces.IDIFSession#incrementRequests()
     */
    @Override
    public void incrementRequests()
    {
        numberOfRequests++;
    }

    /**
     * Checks if is logged.
     *
     * @return true, if is logged
     *
     * @see pt.digitalis.dif.controller.interfaces.IDIFSession#isLogged()
     */
    @Override
    public boolean isLogged()
    {
        return (user != null);
    }

    /**
     * Checks if is marked for removal.
     *
     * @return true, if is marked for removal
     *
     * @see pt.digitalis.dif.controller.interfaces.IDIFSession#isMarkedForRemoval()
     */
    @Override
    public boolean isMarkedForRemoval()
    {
        return markedForRemoval;
    }

    /**
     * Sets the marked for removal.
     *
     * @param markedForRemoval the new marked for removal
     */
    @Override
    public void setMarkedForRemoval(Boolean markedForRemoval)
    {
        this.markedForRemoval = markedForRemoval;
    }

    /**
     * Sets the 'lastAccesTime' property to "now".
     *
     * @see pt.digitalis.dif.controller.interfaces.IPrivateDIFSession#keepAlive()
     */
    @Override
    public void keepAlive()
    {
        if (!isMarkedForRemoval() && !hasTimedOut())
            forceKeepAlive();
    }

    /**
     * Sets the attribute.
     *
     * @param attributeName  the attribute name
     * @param attributeValue the attribute value
     *
     * @see pt.digitalis.utils.common.IBeanAttributes#setAttribute(java.lang.String, java.lang.Object)
     */
    @Override
    public void setAttribute(String attributeName, Object attributeValue)
    {
        if ("attributes".equals(attributeName))
            this.attributes = (Map<String, Object>) attributeValue;
        else if ("firstAccessTime".equals(attributeName))
            this.firstAccessTime = (Long) attributeValue;
        else if ("history".equals(attributeName))
            this.history = (INavigationHistory) attributeValue;
        else if ("language".equals(attributeName))
            this.language = (String) attributeValue;
        else if ("lastAccessTime".equals(attributeName))
            this.lastAccessTime = (Long) attributeValue;
        else if ("markedForRemoval".equals(attributeName))
            this.markedForRemoval = (Boolean) attributeValue;
        else if ("sessionTimeOut".equals(attributeName))
            this.sessionTimeOut = (Long) attributeValue;
        else if ("user".equals(attributeName))
            this.user = (DIFUserInSession) attributeValue;
        else if ("clientDescriptor".equals(attributeName))
            this.clientDescriptor = (ClientDescriptor) attributeValue;
        else if ("numberOfRequests".equals(attributeName))
            this.numberOfRequests = (Long) attributeValue;
    }

    /**
     * Sets the attribute from string.
     *
     * @param attributeName  the attribute name
     * @param attributeValue the attribute value
     *
     * @see pt.digitalis.utils.common.IBeanAttributes#setAttributeFromString(java.lang.String, java.lang.String)
     */
    @Override
    public void setAttributeFromString(String attributeName, String attributeValue)
    {
        if ("firstAccessTime".equals(attributeName))
            this.firstAccessTime = Long.parseLong(attributeValue);
        else if ("language".equals(attributeName))
            this.language = attributeValue;
        else if ("lastAccessTime".equals(attributeName))
            this.lastAccessTime = Long.parseLong(attributeName);
        else if ("markedForRemoval".equals(attributeName))
            this.markedForRemoval = Boolean.parseBoolean(attributeValue);
        else if ("sessionTimeOut".equals(attributeName))
            this.sessionTimeOut = Long.parseLong(attributeValue);
        else if ("numberOfRequests".equals(attributeName))
            this.numberOfRequests = Long.parseLong(attributeValue);
    }

    @Override
    public ObjectFormatter toObjectFormatter(Format format, List<Object> dumpedObjects)
    {
        ObjectFormatter formatter = new ObjectFormatter(format, dumpedObjects);

        formatter.addItem("Session ID", sessionID);
        formatter.addItem("First Access Time", firstAccessTime);
        formatter.addItem("Last Access Time", lastAccessTime);
        formatter.addItem("language", language);
        formatter.addItem("User in session", getUser());

        Set<Entry<String, Object>> attributesSetClone = new HashSet<Map.Entry<String, Object>>(attributes.entrySet());

        Map<String, Object> parsedAttributes = new HashMap<String, Object>();

        for (Entry<String, Object> entry : attributesSetClone)
        {
            if (!entry.getKey().startsWith("JSONCache:"))
                parsedAttributes.put(entry.getKey(), entry.getValue());
        }

        formatter.addItemIfNotNull("Attributes", parsedAttributes);

        return formatter;
    }

    @Override
    public String toString()
    {
        return toObjectFormatter(Format.TEXT, null).getFormatedObject();
    }
}
