/**
 * 2007, Digitalis Informatica. All rights reserved. Distribuicao e Gestao de Informatica, Lda. Estrada de Paco de Arcos
 * num.9 - Piso -1 2780-666 Paco de Arcos Telefone: (351) 21 4408990 Fax: (351) 21 4408999 http://www.digitalis.pt
 */

package pt.digitalis.dif.controller.security.managers;

import pt.digitalis.dif.controller.security.objects.ACLEntry;
import pt.digitalis.dif.controller.security.objects.IDIFGroup;
import pt.digitalis.dif.controller.security.objects.IDIFUser;
import pt.digitalis.dif.dem.Entity;
import pt.digitalis.dif.dem.interfaces.IApplication;
import pt.digitalis.dif.dem.interfaces.IService;
import pt.digitalis.dif.dem.interfaces.IStage;
import pt.digitalis.dif.exception.security.AuthorizationManagerException;

import java.util.List;

/**
 * Defines the behavior for an authorization manager.
 *
 * @author Rodrigo Gonalves <a href="mailto:rgoncalves@digitalis.pt">rgoncalves@digitalis.pt</a><br/>
 * @author Pedro Viegas <a href="mailto:pviegas@digitalis.pt">pviegas@digitalis.pt</a><br/>
 * @created 2007/12/03
 */
public interface IAuthorizationManager
{

    /**
     * Adds an ACL to the manager repository
     *
     * @param entry the entry
     *
     * @return T if the operation was successful
     *
     * @exception AuthorizationManagerException if a resource needed for adding authorization credentials can't be
     *                                          accessed
     */
    public boolean addACLEntry(ACLEntry entry) throws AuthorizationManagerException;

    /**
     * Returns the ACL by application.
     *
     * @param applicationID the application id
     *
     * @return the application ACL
     */
    public List<ACLEntry> findACLEntriesByApplication(String applicationID);

    /**
     * Returns the group ACL.
     *
     * @param groupID the group id
     *
     * @return the group ACL
     */
    public List<ACLEntry> findACLEntriesByGroup(String groupID);

    /**
     * Returns ACL by provider.
     *
     * @param provider the provider id
     *
     * @return the provider ACL
     */
    public List<ACLEntry> findACLEntriesByProvider(String provider);

    /**
     * Returns the ACL by service.
     *
     * @param serviceID the service id
     *
     * @return the service ACL
     */
    public List<ACLEntry> findACLEntriesByService(String serviceID);

    /**
     * Returns the ACL by stage.
     *
     * @param stageID the stage id
     *
     * @return the stage ACL
     */
    public List<ACLEntry> findACLEntriesByStage(String stageID);

    /**
     * Returns the user ACL.
     *
     * @param userID the user id
     *
     * @return the user ACL
     */
    public List<ACLEntry> findACLEntriesByUser(String userID);

    /**
     * Returns the user ACL, inheriting the ACL's from groups as well.
     *
     * @param userID the user id
     *
     * @return the user ACL
     *
     * @exception AuthorizationManagerException
     */
    public List<ACLEntry> findACLEntriesByUserInherited(String userID) throws AuthorizationManagerException;

    /**
     * Returns the public ACL.
     *
     * @return the public ACL
     */
    public List<ACLEntry> findPublicACLEntries();

    /**
     * Grants a group with access to a given stage.
     *
     * @param groupID    the id of the group to grant access
     * @param entityType the entity type
     * @param entityID   the entity ID
     *
     * @return T if access was granted, F otherwise.
     *
     * @exception AuthorizationManagerException if a resource needed for checking authorization credentials can't be
     *                                          accessed
     */
    public boolean grantAccessToGroup(String groupID, Entity entityType, String entityID)
            throws AuthorizationManagerException;

    /**
     * Grants the public with access to a given stage.
     *
     * @param entityType the entity type
     * @param entityID   the entity ID
     *
     * @return T if access was granted, F otherwise.
     *
     * @exception AuthorizationManagerException if a resource needed for checking authorization credentials can't be
     *                                          accessed
     */
    public boolean grantAccessToPublic(Entity entityType, String entityID) throws AuthorizationManagerException;

    /**
     * Grants a user with access to a given stage.
     *
     * @param userID     the id of the user to grant access
     * @param entityType the entity type
     * @param entityID   the entity ID
     *
     * @return T if access was granted, F otherwise.
     *
     * @exception AuthorizationManagerException if a resource needed for granting authorization credentials can't be
     *                                          accessed
     */
    public boolean grantAccessToUser(String userID, Entity entityType, String entityID)
            throws AuthorizationManagerException;

    /**
     * Grants a group with default access to a given stage.
     *
     * @param groupID    the id of the group to grant access
     * @param entityType the entity type
     * @param entityID   the entity ID
     *
     * @return T if access was granted, F otherwise.
     *
     * @exception AuthorizationManagerException if a resource needed for checking authorization credentials can't be
     *                                          accessed
     */
    public boolean grantDefaultAccessToGroup(String groupID, Entity entityType, String entityID)
            throws AuthorizationManagerException;

    /**
     * Grants a user with default access to a given stage.
     *
     * @param userID     the id of the user to grant access
     * @param entityType the entity type
     * @param entityID   the entity ID
     *
     * @return T if access was granted, F otherwise.
     *
     * @exception AuthorizationManagerException if a resource needed for granting authorization credentials can't be
     *                                          accessed
     */
    public boolean grantDefaultAccessToUser(String userID, Entity entityType, String entityID)
            throws AuthorizationManagerException;

    /**
     * Grants with default public access to a given stage.
     *
     * @param entityType the entity type
     * @param entityID   the entity ID
     *
     * @return T if the operation was successful
     *
     * @exception AuthorizationManagerException if a resource needed for granting authorization credentials can't be
     *                                          accessed
     */
    public boolean grantDefaultPublicAccess(Entity entityType, String entityID) throws AuthorizationManagerException;

    /**
     * Checks if a group can access a given stage.
     *
     * @param group      the group to grant access
     * @param entityType the entity type
     * @param entityID   the entity ID
     *
     * @return T if the group can access the stage, F otherwise
     *
     * @exception AuthorizationManagerException if a needed resource for authorization checking can't be found
     */
    public boolean hasAccessGroup(IDIFGroup group, Entity entityType, String entityID)
            throws AuthorizationManagerException;

    /**
     * Checks if the group can access a given application. Will parse the DEM for inherited grants
     *
     * @param group       the group to grant access
     * @param application the application to check
     *
     * @return T if the user can access the application, F otherwise
     *
     * @exception AuthorizationManagerException if a resource needed for checking authorization credentials can't be
     *                                          accessed
     */
    public boolean hasAccessGroup(IDIFGroup group, IApplication application) throws AuthorizationManagerException;

    /**
     * Checks if the group can access a given service. Will parse the DEM for inherited grants
     *
     * @param group   the group to grant access
     * @param service the service to check
     *
     * @return T if the user can access the service, F otherwise
     *
     * @exception AuthorizationManagerException if a resource needed for checking authorization credentials can't be
     *                                          accessed
     */
    public boolean hasAccessGroup(IDIFGroup group, IService service) throws AuthorizationManagerException;

    /**
     * Checks if a group can access to a given stage.
     *
     * @param group the group to grant access
     * @param stage the stage to check
     *
     * @return T if the group can access the stage, F otherwise
     *
     * @exception AuthorizationManagerException if a needed resource for authorization checking can't be found
     */
    public boolean hasAccessGroup(IDIFGroup group, IStage stage) throws AuthorizationManagerException;

    /**
     * Checks if the public can access to a given entity identified with a type and ID.
     *
     * @param entityType the entity type
     * @param entityID   the entity ID
     *
     * @return T if the user can access the stage, F otherwise
     */
    public boolean hasAccessPublic(Entity entityType, String entityID);

    /**
     * Checks if the public can access a given application.
     *
     * @param application the application to check
     *
     * @return T if the public can access the application, F otherwise
     */
    public boolean hasAccessPublic(IApplication application);

    /**
     * Checks if the public can access a given service.
     *
     * @param service the stage to check
     *
     * @return T if the public can access the service, F otherwise
     */
    public boolean hasAccessPublic(IService service);

    /**
     * Checks if the public can access a given stage.
     *
     * @param stage the stage to check
     *
     * @return T if the public can access the stage, F otherwise
     */
    public boolean hasAccessPublic(IStage stage);

    /**
     * Checks if a user can access to a given entity identified with a type and ID.
     *
     * @param user       the the user to check access
     * @param entityType the entity type
     * @param entityID   the entity ID
     *
     * @return T if the user can access the stage, F otherwise
     *
     * @exception AuthorizationManagerException if a resource needed for checking authorization credentials can't be
     *                                          accessed
     */
    public boolean hasAccessUser(IDIFUser user, Entity entityType, String entityID)
            throws AuthorizationManagerException;

    /**
     * Checks if the user can access a given application. Will parse the DEM for inherited grants
     *
     * @param user        the user to check access
     * @param application the application to check
     *
     * @return T if the user can access the application, F otherwise
     *
     * @exception AuthorizationManagerException if a resource needed for checking authorization credentials can't be
     *                                          accessed
     */
    public boolean hasAccessUser(IDIFUser user, IApplication application) throws AuthorizationManagerException;

    /**
     * Checks if the user can access a given service. Will parse the DEM for inherited grants
     *
     * @param user    the user to check access
     * @param service the service to check
     *
     * @return T if the user can access the service, F otherwise
     *
     * @exception AuthorizationManagerException if a resource needed for checking authorization credentials can't be
     *                                          accessed
     */
    public boolean hasAccessUser(IDIFUser user, IService service) throws AuthorizationManagerException;

    /**
     * Checks if the user can access a given stage. Will parse the DEM for inherited grants
     *
     * @param user  the user to check access
     * @param stage the stage to check
     *
     * @return T if the user can access the stage, F otherwise
     *
     * @exception AuthorizationManagerException if a resource needed for checking authorization credentials can't be
     *                                          accessed
     */
    public boolean hasAccessUser(IDIFUser user, IStage stage) throws AuthorizationManagerException;

    /**
     * Removes an ACL from the manager repository
     *
     * @param entry the entry
     *
     * @return T if the operation was successful
     */
    public boolean revokeACLEntry(ACLEntry entry);

    /**
     * Revokes the access of a group to a given stage.
     *
     * @param groupID    the id of the group to revoke access
     * @param entityType the entity type
     * @param entityID   the entity ID
     *
     * @return T if access was revoked, F otherwise.
     */
    public boolean revokeAccessFromGroup(String groupID, Entity entityType, String entityID);

    /**
     * Revokes the public access to a given stage. AccessControl
     *
     * @param entityType the entity type
     * @param entityID   the entity ID
     *
     * @return T if access was revoked, F otherwise.
     */
    public boolean revokeAccessFromPublic(Entity entityType, String entityID);

    /**
     * Revokes the access of a user to a given stage.
     *
     * @param userID     the id of the user to revoke access
     * @param entityType the entity type
     * @param entityID   the entity ID
     *
     * @return T if access was revoked, F otherwise.
     */
    public boolean revokeAccessFromUser(String userID, Entity entityType, String entityID);

    /**
     * Revokes the access of a group to all entities.
     *
     * @param groupID the id of the group to revoke access
     *
     * @return T if access was revoked, F otherwise.
     */
    public boolean revokeAllAccessFromGroup(String groupID);

    /**
     * Revokes the access of a user to all entities.
     *
     * @param userID the id of the user to revoke access
     *
     * @return T if access was revoked, F otherwise.
     */
    public boolean revokeAllAccessFromUser(String userID);
}
