/**
 * 2007, Digitalis Informatica. All rights reserved. Distribuicao e Gestao de Informatica, Lda. Estrada de Paco de Arcos
 * num.9 - Piso -1 2780-666 Paco de Arcos Telefone: (351) 21 4408990 Fax: (351) 21 4408999 http://www.digitalis.pt
 */

package pt.digitalis.dif.controller.security.managers.impl;

import com.google.inject.Inject;
import pt.digitalis.dif.controller.security.managers.IAuthorizationManager;
import pt.digitalis.dif.controller.security.managers.IIdentityManager;
import pt.digitalis.dif.controller.security.objects.ACLEntry;
import pt.digitalis.dif.controller.security.objects.IDIFGroup;
import pt.digitalis.dif.controller.security.objects.IDIFUser;
import pt.digitalis.dif.dem.Entity;
import pt.digitalis.dif.dem.interfaces.IApplication;
import pt.digitalis.dif.dem.interfaces.IService;
import pt.digitalis.dif.dem.interfaces.IStage;
import pt.digitalis.dif.dem.managers.IDEMManager;
import pt.digitalis.dif.exception.security.AuthorizationManagerException;
import pt.digitalis.dif.exception.security.IdentityManagerException;
import pt.digitalis.dif.utils.IObjectFormatter;
import pt.digitalis.dif.utils.ObjectFormatter;
import pt.digitalis.dif.utils.ObjectFormatter.Format;
import pt.digitalis.utils.common.StringUtils;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;

/**
 * Provides an abstract implementation for the authorization manager.
 *
 * @author Rodrigo Gonalves <a href="mailto:rgoncalves@digitalis.pt">rgoncalves@digitalis.pt</a><br/>
 * @created 2008/03/17
 */
abstract public class AbstractAuthorizationManagerImpl implements IAuthorizationManager, IObjectFormatter
{

    /** The DEM manager */
    protected IDEMManager demManager;

    /** The identity manager */
    protected IIdentityManager identityManager;

    /** T/** Maps the id of an group (K) to a set of ACL entries (V) granted to that group. */
    private Map<String, Set<ACLEntry>> groupAccessControlList = new ConcurrentHashMap<String, Set<ACLEntry>>();

    /** Maps the id of an entity (K) to an ACL entry (V). */
    private Map<String, ACLEntry> publicAccessControlList = new ConcurrentHashMap<String, ACLEntry>();

    /** Maps the id of an user (K) to a set of ACL entries (V) granted to that user. */
    private Map<String, Set<ACLEntry>> userAccessControlList = new ConcurrentHashMap<String, Set<ACLEntry>>();

    /**
     * Default constructor
     *
     * @param identityManager the identity manager
     * @param demManager      the DEM manager
     */
    @Inject
    public AbstractAuthorizationManagerImpl(IIdentityManager identityManager, IDEMManager demManager)
    {
        this.identityManager = identityManager;
        this.demManager = demManager;
    }

    /**
     * @see pt.digitalis.dif.controller.security.managers.IAuthorizationManager#addACLEntry(pt.digitalis.dif.controller.security.objects.ACLEntry)
     */
    @Override
    public boolean addACLEntry(ACLEntry entry) throws AuthorizationManagerException
    {

        if (entry.isUserACL())
            return addACLEntryToUser(entry);
        else if (entry.isGroupACL())
            return addACLEntryToGroup(entry, true);
        else if (entry.isPublicAccess())
            return addACLEntryToPublic(entry);
        else
            return false;
    }

    /**
     * Adds a new entry to the group ACL.
     *
     * @param entry              the entry to add
     * @param overrideIfExistant if T will override the existing ACL if existent. Only relevant for persistent ACL
     *                           repositories
     *
     * @return T if the operation succeeded, F otherwise
     *
     * @exception AuthorizationManagerException if a resource needed for checking authorization credentials can't be
     *                                          accessed
     */
    synchronized protected boolean addACLEntryToGroup(ACLEntry entry, boolean overrideIfExistant)
            throws AuthorizationManagerException
    {
        // Group exists
        if (!getGroupAccessControlList().containsKey(entry.getGroupID()))
        {
            // Create entry for the group
            getGroupAccessControlList().put(entry.getGroupID(), new HashSet<ACLEntry>());
        }

        ACLEntry previousEntry = null;
        for (ACLEntry acl : getGroupAccessControlList().get(entry.getGroupID()))
        {
            if (acl.equals(entry))
            {
                previousEntry = acl;
                break;
            }
        }

        if (previousEntry != null && overrideIfExistant)
            getGroupAccessControlList().get(entry.getGroupID()).remove(previousEntry);

        getGroupAccessControlList().get(entry.getGroupID()).add(entry);

        return getGroupAccessControlList().get(entry.getGroupID()).contains(entry);
    }

    /**
     * Adds a new entry to the group ACL.
     *
     * @param entry the entry to add
     *
     * @return T if the operation succeeded, F otherwise
     *
     * @exception AuthorizationManagerException if a resource needed for checking authorization credentials can't be
     *                                          accessed
     */
    protected boolean addACLEntryToPublic(ACLEntry entry) throws AuthorizationManagerException
    {
        if (StringUtils.isNotBlank(entry.getEntityID()))
            getPublicAccessControlList().put(entry.getEntityID(), entry);

        return true;
    }

    /**
     * Adds a new entry to the user ACL.
     *
     * @param entry the entry to add
     *
     * @return T if the operation succeeded, F otherwise
     *
     * @exception AuthorizationManagerException if a resource needed for checking authorization credentials can't be
     *                                          accessed
     */
    protected boolean addACLEntryToUser(ACLEntry entry) throws AuthorizationManagerException
    {
        try
        {
            // Validate the user and check for it's presence in the repository
            if (!getUserAccessControlList().containsKey(entry.getUserID()))
            {
                if (identityManager.userExists(entry.getUserID()))
                {

                    // Create the user entry in the ACL repository
                    getUserAccessControlList().put(entry.getUserID(), new HashSet<ACLEntry>());
                    getUserAccessControlList().get(entry.getUserID()).add(entry);
                    return true;
                }
                else
                    // Invalid user
                    return false;
            }
            else
            {
                if (checkAccessUser(identityManager.getUser(entry.getUserID()), entry.getEntityType(),
                        entry.getEntityID()))
                    // Already has access
                    return true;
                else
                {
                    return getUserAccessControlList().get(entry.getUserID()).add(entry);
                }
            }
        }
        catch (IdentityManagerException identityManagerException)
        {
            throw new AuthorizationManagerException("Could not access the identity manager to verify user existance!",
                    identityManagerException);
        }
    }

    /**
     * Check for access grants
     *
     * @param group      the group to check
     * @param entityType the entity type
     * @param entityID   the entity ID
     *
     * @return T if the user can access the stage, F otherwise
     *
     * @exception AuthorizationManagerException if a resource needed for checking authorization credentials can't be
     *                                          accessed
     */
    protected boolean checkAccessToGroup(IDIFGroup group, Entity entityType, String entityID)
            throws AuthorizationManagerException
    {

        boolean hasAccess = false;
        try
        {
            // Check if it's the entity has public access
            hasAccess = hasAccessPublic(entityType, entityID);

            if (!hasAccess)
            {
                if (group == null)
                    return false;

                else
                {

                    // Check direct access grant
                    hasAccess = checkGroupDirectAccess(group, entityType, entityID);
                    // If group hasn't direct access, check ancestors
                    if (!hasAccess)
                    {

                        // Get first-level ancestor (parent)
                        IDIFGroup ancestor = group.getParentGroup();

                        // If there are ancestors... (prevent ciclic ancestors)
                        if (ancestor != null && (!ancestor.getID().equalsIgnoreCase(group.getID())))
                        {
                            /*
                             * Check each one for access (grandparents and great-grandparents and the like are checked
                             * in this loop since the execution flow will pass here and pick the parents of each group
                             */
                            while (!hasAccess && ancestor != null)
                            {
                                if (hasAccessGroup(ancestor, entityType, entityID))
                                    hasAccess = true;
                                else
                                    ancestor = ancestor.getParentGroup();
                            }
                        }
                    }
                }
            }
        }
        catch (IdentityManagerException identityManagerException)
        {
            throw new AuthorizationManagerException(
                    "Could not access identity manager to check if group with ID " + group.getID() + " exists!",
                    identityManagerException);
        }

        return hasAccess;
    }

    /**
     * Check for access grants
     *
     * @param user       the user to check
     * @param entityType the entity type
     * @param entityID   the entity ID
     *
     * @return T if the user can access the stage, F otherwise
     *
     * @exception AuthorizationManagerException if a resource needed for checking authorization credentials can't be
     *                                          accessed
     */
    protected boolean checkAccessUser(IDIFUser user, Entity entityType, String entityID)
            throws AuthorizationManagerException
    {

        boolean hasAccess = false;

        try
        {
            // Check public access for the entity
            hasAccess = this.hasAccessPublic(entityType, entityID);

            if (!hasAccess)
            {
                if (user == null)
                    return false;

                else
                {

                    // Check direct grant
                    hasAccess = checkUserDirectAccess(user, entityType, entityID);

                    // If the user doesn't have direct access...
                    if (!hasAccess)
                    {
                        // Check user profile access...
                        if ((user.getProfileID() != null) && (hasAccessGroup(user.getProfile(), entityType, entityID)))
                            hasAccess = true;

                        // If profile doesn't have access check access in all user groups...
                        if (!hasAccess)
                        {
                            for (IDIFGroup group : user.getGroups().values())
                            {
                                if (hasAccessGroup(group, entityType, entityID))
                                {
                                    hasAccess = true;
                                    break;
                                }
                            }
                        }
                    }
                }
            }
        }
        catch (IdentityManagerException identityManagerException)
        {
            throw new AuthorizationManagerException(
                    "The identity manager can't be accessed to check user access privileges!",
                    identityManagerException);
        }

        return hasAccess;
    }

    /**
     * Check if a group has access granted to entity
     *
     * @param group
     *            the groupId
     * @param entityType
     *            the entity type
     * @param entityID
     *            the entity id
     * @return validation
     * @throws AuthorizationManagerException
     *             If a AuthorizationManager exception Occurrs
     */
    /**
     * Check if a group has access granted to entity
     *
     * @param group      the groupId
     * @param entityType the entity type
     * @param entityID   the entity id
     *
     * @return validation
     *
     * @exception AuthorizationManagerException If a AuthorizationManager exception Occurrs
     */
    protected boolean checkGroupDirectAccess(IDIFGroup group, Entity entityType, String entityID)
            throws AuthorizationManagerException
    {
        boolean hasAccess = false;
        if (getGroupAccessControlList().containsKey(group.getID()))
        {
            Set<ACLEntry> groupEntries = getGroupAccessControlList().get(group.getID());

            for (ACLEntry entry : groupEntries)
            {
                if (entry.getEntityType().equals(entityType) && entry.getEntityID().equals(entityID) &&
                    entry.isEnabled())
                {
                    hasAccess = true;
                    break;
                }
            }
        }
        return hasAccess;
    }

    /**
     * Implementation if a non-hierarchical check for direct access grants
     *
     * @param user       the user to check
     * @param entityType the entity type
     * @param entityID   the entity ID
     *
     * @return T if the user can access the stage, F otherwise
     *
     * @exception AuthorizationManagerException if a resource needed for checking authorization credentials can't be
     *                                          accessed
     */
    protected boolean checkUserDirectAccess(IDIFUser user, Entity entityType, String entityID)
            throws AuthorizationManagerException
    {

        boolean hasAccess = false;
        if (getUserAccessControlList().containsKey(user.getID()))
        {
            for (ACLEntry entry : getUserAccessControlList().get(user.getID()))
            {
                if (entityID.equals(entry.getEntityID()) && entityType == entry.getEntityType() && entry.isEnabled())
                {
                    hasAccess = true;
                    break;
                }
            }
        }
        return hasAccess;
    }

    /**
     * Searches a given ACL for a given entity ID and returns the list of entries that match the given entity ID. Method
     * overload for Map-based ACLs.
     *
     * @param entityID the entity ID to search for
     * @param acl      the acl to search
     *
     * @return the list of acl entries
     */
    private List<ACLEntry> collectACLEntriesForEntityID(String entityID, Map<String, Set<ACLEntry>> acl)
    {
        List<ACLEntry> result = new ArrayList<ACLEntry>();

        for (String key : acl.keySet())
        {
            result.addAll(collectACLEntriesForEntityID(entityID, acl.get(key)));
        }

        return result;
    }

    /**
     * Searches a given ACL for a given entity ID and returns the list of entries that match the given entity ID. Method
     * overload for Set-based ACLs.
     *
     * @param entityID the entity ID to search for
     * @param acl      the acl to search
     *
     * @return the list of acl entries
     */
    private List<ACLEntry> collectACLEntriesForEntityID(String entityID, Set<ACLEntry> acl)
    {
        List<ACLEntry> result = new ArrayList<ACLEntry>();
        entityID = StringUtils.lowerCase(entityID);

        ACLEntry entry = null;

        for (Iterator<ACLEntry> iterator = acl.iterator(); iterator.hasNext(); )
        {
            entry = iterator.next();
            if (entry.getEntityID().equals(entityID) && !result.contains(entry))
                result.add(entry);
        }

        return result;
    }

    /**
     * Returns a list with ACL entries that match a given entity ID.
     *
     * @param entityID the entity ID to search for
     * @param acl      the acl to search
     *
     * @return the list of acl entries
     */
    private List<ACLEntry> collectACLEntriesForEntityIDSimpleMap(String entityID, Map<String, ACLEntry> acl)
    {
        List<ACLEntry> result = new ArrayList<ACLEntry>();
        entityID = StringUtils.lowerCase(entityID);

        for (ACLEntry entry : acl.values())
        {
            if (entry.getEntityID() != null && entry.getEntityID().equals(entityID) && !result.contains(entry))
                result.add(entry);
        }

        return result;
    }

    /**
     * Creates an ACL entry for a group.
     *
     * @param groupID    the group ID
     * @param entityID   the entity ID
     * @param entityType the entity type
     *
     * @return the group entry
     */
    protected ACLEntry createGroupACLEntry(String groupID, String entityID, Entity entityType)
    {
        ACLEntry entry = new ACLEntry();
        entry.setEntityID(StringUtils.lowerCase(entityID));
        entry.setEntityType(entityType);
        entry.setGroupID(groupID);

        return entry;
    }

    /**
     * Creates a public ACL entry for a given entity.
     *
     * @param entityID   the entity ID
     * @param entityType the entity type
     *
     * @return the group entry
     */
    protected ACLEntry createPublicACLEntry(String entityID, Entity entityType)
    {
        ACLEntry entry = new ACLEntry();
        entry.setEntityID(entityID == null ? entityID : entityID.toLowerCase());
        entry.setEntityType(entityType);
        entry.setPublicAccess();

        return entry;
    }

    /**
     * Creates an ACL entry for a user.
     *
     * @param userID     the user ID
     * @param entityID   the entity ID
     * @param entityType the entity type
     *
     * @return the user entry
     */
    protected ACLEntry createUserACLEntry(String userID, String entityID, Entity entityType)
    {
        ACLEntry entry = new ACLEntry();
        entry.setEntityID(StringUtils.lowerCase(entityID));
        entry.setEntityType(entityType);
        entry.setUserID(userID);

        return entry;
    }

    /**
     * Delete group.
     *
     * @param groupID the group ID
     * @param entry   the entry
     *
     * @return true, if successful
     */
    protected boolean deleteGroup(String groupID, ACLEntry entry)
    {
        return getGroupAccessControlList().get(groupID).remove(entry);
    }

    /**
     * Disable group.
     *
     * @param groupID        the group ID
     * @param existentEntry  the existent entry
     * @param entryToDisable the entry to disable
     *
     * @return true, if successful
     */
    protected boolean disableGroup(String groupID, ACLEntry existentEntry, ACLEntry entryToDisable)
    {
        // Remove existing entry
        getGroupAccessControlList().get(groupID).remove(existentEntry);
        // Disable entry
        entryToDisable.setEnabled(false);
        // Add disabled entry
        return getGroupAccessControlList().get(groupID).add(entryToDisable);
    }

    /**
     * Implementation for finding inherited ACL entries for a user
     *
     * @param userID
     * @param group
     *
     * @return a
     *
     * @exception IdentityManagerException
     */
    protected List<ACLEntry> doFindACLEntriesByUserInherited(String userID, IDIFGroup group)
            throws IdentityManagerException
    {
        List<ACLEntry> result = new ArrayList<ACLEntry>();
        if (group != null)
        {
            Collection<IDIFGroup> subGroups = this.identityManager.getGroupGroups(group.getID()).values();
            for (IDIFGroup subGrp : subGroups)
            {
                result.addAll(doFindACLEntriesByUserInherited(userID, subGrp));
            }
            result.addAll(findACLEntriesByGroup(group.getID()));
        }

        return result;
    }

    /**
     * Implementation if a non-hierarchical check for access grants
     *
     * @param entityType the entity type
     * @param entityID   the entity ID
     *
     * @return T if the user can access the stage, F otherwise
     */
    protected boolean doHasAccessPublic(Entity entityType, String entityID)
    {

        if (getPublicAccessControlList().containsKey(entityID))
        {
            if (entityID.equals(getPublicAccessControlList().get(entityID).getEntityID()) &&
                entityType == getPublicAccessControlList().get(entityID).getEntityType() &&
                getPublicAccessControlList().get(entityID).isEnabled())
                // Found it!
                return true;
        }
        // No access!
        return false;
    }

    /**
     * @see pt.digitalis.dif.controller.security.managers.IAuthorizationManager#findACLEntriesByApplication(java.lang.String)
     */
    @Override
    public List<ACLEntry> findACLEntriesByApplication(String applicationID)
    {
        return findACLEntryByEntity(applicationID);
    }

    /**
     * @see pt.digitalis.dif.controller.security.managers.IAuthorizationManager#findACLEntriesByGroup(java.lang.String)
     */
    @Override
    public List<ACLEntry> findACLEntriesByGroup(String groupID)
    {
        Set<ACLEntry> results = getGroupAccessControlList().get(groupID);

        if (results != null)
            return new ArrayList<ACLEntry>(results);
        else
            return new ArrayList<ACLEntry>();
    }

    /**
     * @see pt.digitalis.dif.controller.security.managers.IAuthorizationManager#findACLEntriesByProvider(java.lang.String)
     */
    @Override
    public List<ACLEntry> findACLEntriesByProvider(String providerID)
    {
        return findACLEntryByEntity(providerID);
    }

    /**
     * @see pt.digitalis.dif.controller.security.managers.IAuthorizationManager#findACLEntriesByService(java.lang.String)
     */
    @Override
    public List<ACLEntry> findACLEntriesByService(String serviceID)
    {
        return findACLEntryByEntity(serviceID);
    }

    /**
     * @see pt.digitalis.dif.controller.security.managers.IAuthorizationManager#findACLEntriesByStage(java.lang.String)
     */
    @Override
    public List<ACLEntry> findACLEntriesByStage(String stageID)
    {
        return findACLEntryByEntity(stageID);
    }

    /**
     * @see pt.digitalis.dif.controller.security.managers.IAuthorizationManager#findACLEntriesByUser(java.lang.String)
     */
    @Override
    public List<ACLEntry> findACLEntriesByUser(String userID)
    {
        List<ACLEntry> result = new ArrayList<ACLEntry>();

        if (getUserAccessControlList().containsKey(userID))
            result.addAll(getUserAccessControlList().get(userID));

        return result;
    }

    /**
     * @see pt.digitalis.dif.controller.security.managers.IAuthorizationManager#findACLEntriesByUserInherited(java.lang.String)
     */
    @Override
    public List<ACLEntry> findACLEntriesByUserInherited(String userID) throws AuthorizationManagerException
    {
        List<ACLEntry> result = new ArrayList<ACLEntry>();
        // TODO fsouto check this exception
        try
        {
            Map<String, IDIFGroup> userGroups = this.identityManager.getUserGroups(userID);
            for (IDIFGroup grp : userGroups.values())
            {
                result.addAll(this.doFindACLEntriesByUserInherited(userID, grp));
            }

            // add all the remaining
            result.addAll(this.findACLEntriesByUser(userID));
        }
        catch (IdentityManagerException e)
        {
            throw new AuthorizationManagerException(e);
        }
        return result;
    }

    /**
     * Helper method that finds all the ACL entries for a given entity. Searches on the three available ACLs.
     *
     * @param entityID the entity ID
     *
     * @return the list of entries for the given entity
     */
    private List<ACLEntry> findACLEntryByEntity(String entityID)
    {
        entityID = StringUtils.lowerCase(entityID);

        // User List
        List<ACLEntry> result = collectACLEntriesForEntityID(entityID, getUserAccessControlList());

        // Group List
        result.addAll(collectACLEntriesForEntityID(entityID, getGroupAccessControlList()));

        // Public List
        result.addAll(collectACLEntriesForEntityIDSimpleMap(entityID, getPublicAccessControlList()));

        return result;
    }

    /**
     * @see pt.digitalis.dif.controller.security.managers.IAuthorizationManager#findPublicACLEntries()
     */
    @Override
    public List<ACLEntry> findPublicACLEntries()
    {
        List<ACLEntry> result = new ArrayList<ACLEntry>();

        for (ACLEntry entry : getPublicAccessControlList().values())
        {
            result.add(entry);
        }

        return result;
    }

    /**
     * Inspector for the 'groupAccessControlList' attribute.
     *
     * @return the groupAccessControlList value
     */
    public Map<String, Set<ACLEntry>> getGroupAccessControlList()
    {
        return groupAccessControlList;
    }

    /**
     * Inspector for the 'publicAccessControlList' attribute.
     *
     * @return the publicAccessControlList value
     */
    public Map<String, ACLEntry> getPublicAccessControlList()
    {
        return publicAccessControlList;
    }

    /**
     * Inspector for the 'userAccessControlList' attribute.
     *
     * @return the userAccessControlList value
     */
    public Map<String, Set<ACLEntry>> getUserAccessControlList()
    {
        return userAccessControlList;
    }

    /**
     * @see pt.digitalis.dif.controller.security.managers.IAuthorizationManager#grantAccessToGroup(java.lang.String,
     *         pt.digitalis.dif.dem.Entity, java.lang.String)
     */
    @Override
    public boolean grantAccessToGroup(String groupID, Entity entityType, String entityID)
            throws AuthorizationManagerException
    {
        ACLEntry entry = createGroupACLEntry(groupID, entityID, entityType);

        if (getGroupAccessControlList().get(groupID) != null)
        {
            for (ACLEntry existentEntry : getGroupAccessControlList().get(groupID))
                /*
                 * IMPLEMENTATION NOTE - see note on revokeAccessFromUser(String, Entity, String) method.
                 */
            {
                if ((existentEntry.getGroupID()).equals(entry.getGroupID()) &&
                    existentEntry.getEntityType().equals(entry.getEntityType()) &&
                    existentEntry.getEntityID().equals(entry.getEntityID()))
                {
                    entry = existentEntry;
                    if (entry.isEnabled())
                    {
                        return true;
                    }
                    else
                    {
                        entry.setEnabled(true);
                    }
                }
            }
        }
        return addACLEntryToGroup(entry, true);
    }

    /**
     * @see pt.digitalis.dif.controller.security.managers.IAuthorizationManager#grantAccessToPublic(pt.digitalis.dif.dem.Entity,
     *         java.lang.String)
     */
    @Override
    public boolean grantAccessToPublic(Entity entityType, String entityID) throws AuthorizationManagerException
    {
        return addACLEntryToPublic(createPublicACLEntry(entityID, entityType));
    }

    /**
     * @see pt.digitalis.dif.controller.security.managers.IAuthorizationManager#grantAccessToUser(java.lang.String,
     *         pt.digitalis.dif.dem.Entity, java.lang.String)
     */
    @Override
    public boolean grantAccessToUser(String userID, Entity entityType, String entityID)
            throws AuthorizationManagerException
    {
        return addACLEntryToUser(createUserACLEntry(userID, entityID, entityType));
    }

    /**
     * @see pt.digitalis.dif.controller.security.managers.IAuthorizationManager#grantDefaultAccessToGroup(java.lang.String,
     *         pt.digitalis.dif.dem.Entity, java.lang.String)
     */
    @Override
    public boolean grantDefaultAccessToGroup(String groupID, Entity entityType, String entityID)
            throws AuthorizationManagerException
    {
        ACLEntry entry = createGroupACLEntry(groupID, entityID, entityType);
        entry.setDefault(true);
        return addACLEntryToGroup(entry, false);
    }

    /**
     * @see pt.digitalis.dif.controller.security.managers.IAuthorizationManager#grantDefaultAccessToUser(java.lang.String,
     *         pt.digitalis.dif.dem.Entity, java.lang.String)
     */
    @Override
    public boolean grantDefaultAccessToUser(String userID, Entity entityType, String entityID)
            throws AuthorizationManagerException
    {
        ACLEntry entry = createUserACLEntry(userID, entityID, entityType);
        entry.setDefault(true);
        return addACLEntryToUser(entry);
    }

    /**
     * @see pt.digitalis.dif.controller.security.managers.IAuthorizationManager#grantDefaultPublicAccess(pt.digitalis.dif.dem.Entity,
     *         java.lang.String)
     */
    @Override
    public boolean grantDefaultPublicAccess(Entity entityType, String entityID) throws AuthorizationManagerException
    {
        if (hasAccessPublic(entityType, entityID))
            // Already has access
            return true;
        else
        {
            // Grant access
            ACLEntry entry = createPublicACLEntry(entityID, entityType);
            entry.setDefault(true);
            return addACLEntryToPublic(entry);
        }
    }

    /**
     * @see pt.digitalis.dif.controller.security.managers.IAuthorizationManager#hasAccessGroup(pt.digitalis.dif.controller.security.objects.IDIFGroup,
     *         pt.digitalis.dif.dem.Entity, java.lang.String)
     */
    @Override
    public final boolean hasAccessGroup(IDIFGroup group, Entity entityType, String entityID)
            throws AuthorizationManagerException
    {
        if (Entity.APPLICATION.equals(entityType))
            return hasAccessGroup(group, demManager.getApplication(entityID));

        else if (Entity.SERVICE.equals(entityType))
            return hasAccessGroup(group, demManager.getService(entityID));

        else if (Entity.STAGE.equals(entityType))
            return hasAccessGroup(group, demManager.getStage(entityID));

        else
            return false;
    }

    /**
     * @see pt.digitalis.dif.controller.security.managers.IAuthorizationManager#hasAccessGroup(pt.digitalis.dif.controller.security.objects.IDIFGroup,
     *         pt.digitalis.dif.dem.interfaces.IApplication)
     */
    @Override
    public boolean hasAccessGroup(IDIFGroup group, IApplication application) throws AuthorizationManagerException
    {
        if (application == null)
            return false;

        return checkAccessToGroup(group, Entity.APPLICATION, application.getID());
    }

    /**
     * @see pt.digitalis.dif.controller.security.managers.IAuthorizationManager#hasAccessGroup(pt.digitalis.dif.controller.security.objects.IDIFGroup,
     *         pt.digitalis.dif.dem.interfaces.IService)
     */
    @Override
    public boolean hasAccessGroup(IDIFGroup group, IService service) throws AuthorizationManagerException
    {
        if (service == null)
            return false;

        boolean hasPublicAccess = doHasAccessPublic(Entity.SERVICE, service.getID());

        return (!hasPublicAccess && checkAccessToGroup(group, Entity.SERVICE, service.getID())) ||
               (hasPublicAccess && hasAccessGroup(group, service.getApplication()));
    }

    /**
     * @see pt.digitalis.dif.controller.security.managers.IAuthorizationManager#hasAccessGroup(pt.digitalis.dif.controller.security.objects.IDIFGroup,
     *         pt.digitalis.dif.dem.interfaces.IStage)
     */
    @Override
    public final boolean hasAccessGroup(IDIFGroup group, IStage stage) throws AuthorizationManagerException
    {
        if (stage == null)
            return false;

        boolean hasPublicAccess = doHasAccessPublic(Entity.STAGE, stage.getID());

        return (!hasPublicAccess && checkAccessToGroup(group, Entity.STAGE, stage.getID())) ||
               (hasPublicAccess && hasAccessGroup(group, stage.getService()));
    }

    /**
     * @see pt.digitalis.dif.controller.security.managers.IAuthorizationManager#hasAccessPublic(pt.digitalis.dif.dem.Entity,
     *         java.lang.String)
     */
    @Override
    public final boolean hasAccessPublic(Entity entityType, String entityID)
    {
        if (Entity.APPLICATION.equals(entityType))
            return hasAccessPublic(demManager.getApplication(entityID));

        else if (Entity.SERVICE.equals(entityType))
            return hasAccessPublic(demManager.getService(entityID));

        else if (Entity.STAGE.equals(entityType))
            return hasAccessPublic(demManager.getStage(entityID));

        else
            return false;
    }

    /**
     * @see pt.digitalis.dif.controller.security.managers.IAuthorizationManager#hasAccessPublic(pt.digitalis.dif.dem.interfaces.IApplication)
     */
    @Override
    public boolean hasAccessPublic(IApplication application)
    {
        if (application == null)
            return false;

        return doHasAccessPublic(Entity.APPLICATION, application.getID());
    }

    /**
     * @see pt.digitalis.dif.controller.security.managers.IAuthorizationManager#hasAccessPublic(pt.digitalis.dif.dem.interfaces.IService)
     */
    @Override
    public boolean hasAccessPublic(IService service)
    {
        if (service == null)
            return false;

        return doHasAccessPublic(Entity.SERVICE, service.getID()) &&
               doHasAccessPublic(Entity.APPLICATION, service.getApplication().getID());
    }

    /**
     * @see pt.digitalis.dif.controller.security.managers.IAuthorizationManager#hasAccessPublic(pt.digitalis.dif.dem.interfaces.IStage)
     */
    @Override
    public final boolean hasAccessPublic(IStage stage)
    {
        if (stage == null)
            return false;

        return doHasAccessPublic(Entity.STAGE, stage.getID()) &&
               doHasAccessPublic(Entity.SERVICE, stage.getService().getID()) &&
               doHasAccessPublic(Entity.APPLICATION, stage.getService().getApplication().getID());
    }

    /**
     * @see pt.digitalis.dif.controller.security.managers.IAuthorizationManager#hasAccessUser(pt.digitalis.dif.controller.security.objects.IDIFUser,
     *         pt.digitalis.dif.dem.Entity, java.lang.String)
     */
    @Override
    public final boolean hasAccessUser(IDIFUser user, Entity entityType, String entityID)
            throws AuthorizationManagerException
    {
        if (Entity.APPLICATION.equals(entityType))
            return hasAccessUser(user, demManager.getApplication(entityID));

        else if (Entity.SERVICE.equals(entityType))
            return hasAccessUser(user, demManager.getService(entityID));

        else if (Entity.STAGE.equals(entityType))
            return hasAccessUser(user, demManager.getStage(entityID));

        else
            return false;
    }

    /**
     * @see pt.digitalis.dif.controller.security.managers.IAuthorizationManager#hasAccessUser(pt.digitalis.dif.controller.security.objects.IDIFUser,
     *         pt.digitalis.dif.dem.interfaces.IApplication)
     */
    @Override
    public boolean hasAccessUser(IDIFUser user, IApplication application) throws AuthorizationManagerException
    {
        if (application == null)
            return false;

        return checkAccessUser(user, Entity.APPLICATION, application.getID());
    }

    /**
     * @see pt.digitalis.dif.controller.security.managers.IAuthorizationManager#hasAccessUser(pt.digitalis.dif.controller.security.objects.IDIFUser,
     *         pt.digitalis.dif.dem.interfaces.IService)
     */
    @Override
    public boolean hasAccessUser(IDIFUser user, IService service) throws AuthorizationManagerException
    {
        if (service == null)
            return false;

        boolean hasPublicAccess = doHasAccessPublic(Entity.SERVICE, service.getID());

        return (!hasPublicAccess && checkAccessUser(user, Entity.SERVICE, service.getID())) ||
               (hasPublicAccess && hasAccessUser(user, service.getApplication()));
    }

    /**
     * @see pt.digitalis.dif.controller.security.managers.IAuthorizationManager#hasAccessUser(pt.digitalis.dif.controller.security.objects.IDIFUser,
     *         pt.digitalis.dif.dem.interfaces.IStage)
     */
    @Override
    public final boolean hasAccessUser(IDIFUser user, IStage stage) throws AuthorizationManagerException
    {
        if (stage == null)
            return false;

        boolean hasPublicAccess = doHasAccessPublic(Entity.STAGE, stage.getID());

        return (!hasPublicAccess && checkAccessUser(user, Entity.STAGE, stage.getID())) ||
               (hasPublicAccess && hasAccessUser(user, stage.getService()));
    }

    /**
     * @see pt.digitalis.dif.controller.security.managers.IAuthorizationManager#revokeACLEntry(pt.digitalis.dif.controller.security.objects.ACLEntry)
     */
    @Override
    public boolean revokeACLEntry(ACLEntry entry)
    {
        if (entry.isUserACL())
            return revokeAccessFromUser(entry.getUserID(), entry.getEntityType(), entry.getEntityID());
        else if (entry.isGroupACL())
            return revokeAccessFromGroup(entry.getGroupID(), entry.getEntityType(), entry.getEntityID());
        else if (entry.isPublicAccess())
            return revokeAccessFromPublic(entry.getEntityType(), entry.getEntityID());
        else
            return false;
    }

    /**
     * @see pt.digitalis.dif.controller.security.managers.IAuthorizationManager#revokeAccessFromGroup(java.lang.String,
     *         pt.digitalis.dif.dem.Entity, java.lang.String)
     */
    @Override
    public boolean revokeAccessFromGroup(String groupID, Entity entityType, String entityID)
    {

        // Group exists
        if (getGroupAccessControlList().containsKey(groupID))
        {
            ACLEntry entry = createGroupACLEntry(groupID, entityID, entityType);

            for (ACLEntry existentEntry : getGroupAccessControlList().get(groupID))
                /*
                 * IMPLEMENTATION NOTE - see note on revokeAccessFromUser(String, Entity, String) method.
                 */
            {
                if ((existentEntry.getGroupID()).equals(entry.getGroupID()) &&
                    existentEntry.getEntityType().equals(entry.getEntityType()) &&
                    existentEntry.getEntityID().equals(entry.getEntityID()))
                {
                    // If it's not default, remove it!
                    if (!existentEntry.isDefault())
                        return deleteGroup(groupID, existentEntry);
                        // If it's default, disable it
                    else
                    {
                        // Get the ACL for user
                        Set<ACLEntry> entries = getGroupAccessControlList().get(groupID);

                        // Search for the entry to disable
                        for (ACLEntry entryToDisable : entries)
                        {
                            if (existentEntry.equals(entryToDisable))
                            {
                                // Disabled entry
                                return disableGroup(groupID, existentEntry, entryToDisable);
                            }
                        }
                    }
                }
            }

            // Not found!
            return false;
        }
        else
        {
            return false;
        }
    }

    /**
     * @see pt.digitalis.dif.controller.security.managers.IAuthorizationManager#revokeAccessFromPublic(pt.digitalis.dif.dem.Entity,
     *         java.lang.String)
     */
    @Override
    public boolean revokeAccessFromPublic(Entity entityType, String entityID)
    {
        entityID = StringUtils.lowerCase(entityID);

        if (getPublicAccessControlList().containsKey(entityID))
        {

            // Can't delete a "default" entry!!
            if (getPublicAccessControlList().get(entityID).getEntityID().equals(entityID) &&
                getPublicAccessControlList().get(entityID).getEntityType().equals(entityType))
            {
                if (!getPublicAccessControlList().get(entityID).isDefault())
                    getPublicAccessControlList().remove(entityID);
                else
                {
                    // Disable existing entry
                    getPublicAccessControlList().get(entityID).setEnabled(false);
                }
            }
            return true;
        }

        // Not found!
        return false;
    }

    /**
     * @see pt.digitalis.dif.controller.security.managers.IAuthorizationManager#revokeAccessFromUser(java.lang.String,
     *         pt.digitalis.dif.dem.Entity, java.lang.String)
     */
    @Override
    public boolean revokeAccessFromUser(String userID, Entity entityType, String entityID)
    {

        // User exists
        if (getUserAccessControlList().containsKey(userID))
        {
            ACLEntry entry = createUserACLEntry(userID, entityID, entityType);

            for (ACLEntry existentEntry : getUserAccessControlList().get(userID))

            {
                if ((existentEntry.getUserID()).equals(entry.getUserID()) &&
                    existentEntry.getEntityType().equals(entry.getEntityType()) &&
                    existentEntry.getEntityID().equals(entry.getEntityID()))
                {
                    // If it's not default, remove it!
                    if (getUserAccessControlList().get(userID).contains(existentEntry) && !existentEntry.isDefault())
                    {
                        boolean status = getUserAccessControlList().get(userID).remove(existentEntry);

                        // If there are no more entries associated to the user remove the user
                        if (getUserAccessControlList().get(userID).size() == 0)
                        {
                            getUserAccessControlList().remove(userID);
                            status = true;
                        }

                        return status;
                    }
                    // If it's default, disable it
                    else
                    {
                        // Get the ACL for user
                        Set<ACLEntry> entries = getUserAccessControlList().get(userID);

                        // Search for the entry to disable
                        for (ACLEntry entryToDisable : entries)
                        {
                            if (existentEntry.equals(entryToDisable))
                            {
                                // Remove existing entry
                                getUserAccessControlList().get(userID).remove(existentEntry);
                                // Disable entry
                                entryToDisable.setEnabled(false);
                                // Add disabled entry
                                return getUserAccessControlList().get(userID).add(entryToDisable);
                            }
                        }
                    }
                }
            }

            // Not found!
            return false;
        }
        else
        {
            return false;
        }
    }

    /**
     * @see pt.digitalis.dif.controller.security.managers.IAuthorizationManager#revokeAllAccessFromGroup(java.lang.String)
     */
    @Override
    public boolean revokeAllAccessFromGroup(String groupID)
    {
        if (getGroupAccessControlList().containsKey(groupID))
        {
            getGroupAccessControlList().remove(groupID);
            return true;
        }
        else
            return false;
    }

    /**
     * @see pt.digitalis.dif.controller.security.managers.IAuthorizationManager#revokeAllAccessFromUser(java.lang.String)
     */
    @Override
    public boolean revokeAllAccessFromUser(String userID)
    {
        if (getUserAccessControlList().containsKey(userID))
        {
            getUserAccessControlList().remove(userID);
            return true;
        }
        else
            return false;
    }

    @Override
    public ObjectFormatter toObjectFormatter(Format format, List<Object> dumpedObjects)
    {
        ObjectFormatter formatter = new ObjectFormatter(format, dumpedObjects);
        formatter.addItem("User ACLs", userAccessControlList);
        formatter.addItem("Group ACLs", groupAccessControlList);

        return formatter;
    }

    @Override
    public String toString()
    {
        return toObjectFormatter(Format.TEXT, null).getFormatedObject();
    }
}
