/**
 * 2008, Digitalis Informatica. All rights reserved. Distribuicao e Gestao de Informatica, Lda. Estrada de Paco de Arcos
 * num.9 - Piso -1 2780-666 Paco de Arcos Telefone: (351) 21 4408990 Fax: (351) 21 4408999 http://www.digitalis.pt
 */

package pt.digitalis.dif.controller.security.managers.impl;

import pt.digitalis.dif.controller.security.managers.IIdentityManagerPrivate;
import pt.digitalis.dif.controller.security.objects.IDIFGroup;
import pt.digitalis.dif.exception.security.IdentityManagerException;

import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

/**
 * @author Rodrigo Gonalves <a href="mailto:rgoncalves@digitalis.pt">rgoncalves@digitalis.pt</a><br/>
 * @created 2008/10/01
 */
abstract public class AbstractIdentityManager implements IIdentityManagerPrivate
{

    /** The 'N/A' String literal. */
    final static public String N_A = "N/A";

    /** The managed attributes */
    private List<String> managedAttributes;

    /** User parameters. Will be used to store the users parameters */
    private Map<String, Map<String, Object>> parameters = new HashMap<String, Map<String, Object>>();

    /**
     * @see pt.digitalis.dif.controller.security.managers.IIdentityManager#addManagedAttribute(java.lang.String)
     */
    public void addManagedAttribute(String attribute) throws IdentityManagerException
    {
        getManagedAttributes().add(attribute);
    }

    /**
     * @see pt.digitalis.dif.controller.security.managers.IIdentityManager#addManagedAttributes(java.lang.String)
     */
    public void addManagedAttributes(String attributes) throws IdentityManagerException
    {
        String[] separatedAttributes = attributes.split(",\\s*");
        for (String attribute : separatedAttributes)
        {
            this.addManagedAttribute(attribute);
        }
    }

    /**
     * @see pt.digitalis.dif.controller.security.managers.IIdentityManager#containsUserParameter(java.lang.String,
     *         java.lang.String)
     */
    public boolean containsUserParameter(String userID, String parameterID)
    {
        return this.parameters.containsKey(userID) ? this.parameters.get(userID).containsKey(parameterID) : false;
    }

    /**
     * @see pt.digitalis.dif.controller.security.managers.IIdentityManager#countAllGroupsOfUser(java.lang.String,
     *         boolean)
     */
    public int countAllGroupsOfUser(String userId, boolean parseGroupHierarchy) throws IdentityManagerException
    {
        int result = 0;
        if (parseGroupHierarchy)
        {
            Set<String> groups = new HashSet<String>();
            for (IDIFGroup group : this.getUserGroups(userId).values())
            {
                groups.add(group.getID());
                IDIFGroup parentGroup = group.getParentGroup();
                while (parentGroup != null)
                {
                    groups.add(parentGroup.getID());
                    parentGroup = parentGroup.getParentGroup();
                }
            }
            result = groups.size();
        }
        else
        {
            result = countAllGroupsOfUser(userId);
        }
        return result;
    }

    /**
     * @see pt.digitalis.dif.controller.security.managers.IIdentityManager#gatherManagedAttributes()
     */
    abstract public List<String> gatherManagedAttributes() throws IdentityManagerException;

    /**
     * @see pt.digitalis.dif.controller.security.managers.IIdentityManager#getManagedAttributes()
     */
    public List<String> getManagedAttributes() throws IdentityManagerException
    {
        if (managedAttributes == null)
            managedAttributes = gatherManagedAttributes();

        return managedAttributes;
    }

    /**
     * @see pt.digitalis.dif.controller.security.managers.IIdentityManager#getUserGroups(java.lang.String, boolean)
     */
    public Map<String, IDIFGroup> getUserGroups(String userID, boolean parseGroupHierarchy)
            throws IdentityManagerException
    {
        Map<String, IDIFGroup> result = this.getUserGroups(userID);
        if (parseGroupHierarchy)
        {
            Map<String, IDIFGroup> newGroups = new HashMap<String, IDIFGroup>();
            for (IDIFGroup group : result.values())
            {
                IDIFGroup parentGroup = group.getParentGroup();
                while (parentGroup != null)
                {
                    newGroups.put(parentGroup.getID(), parentGroup);
                    parentGroup = parentGroup.getParentGroup();
                }
            }
            result.putAll(newGroups);
        }

        return result;
    }

    /**
     * @see pt.digitalis.dif.controller.security.managers.IIdentityManager#getUserGroupsIDs(java.lang.String,
     *         boolean)
     */
    public Set<String> getUserGroupsIDs(String userID, boolean parseGroupHierarchy) throws IdentityManagerException
    {

        Set<String> result = new HashSet<String>();

        if (parseGroupHierarchy)
        {
            result.addAll(getUserGroups(userID, parseGroupHierarchy).keySet());
        }
        else
        {
            result.addAll(this.getUserGroupsIDs(userID));
        }

        return result;
    }

    /**
     * @see pt.digitalis.dif.controller.security.managers.IIdentityManager#getUserParameter(java.lang.String,
     *         java.lang.String)
     */
    public Object getUserParameter(String userID, String parameterID)
    {
        // Check that user exists
        if (this.parameters.containsKey(userID) && this.parameters.get(userID).containsKey(parameterID))
        {
            return this.parameters.get(userID).get(parameterID);
        }
        else
            return null;
    }

    /**
     * @see pt.digitalis.dif.controller.security.managers.IIdentityManager#getUserParameters(java.lang.String)
     */
    public Map<String, Object> getUserParameters(String userID)
    {
        // Check that user exists
        if (this.parameters.containsKey(userID))
        {
            return this.parameters.get(userID);
        }
        else
            return null;
    }

    /**
     * @see pt.digitalis.dif.controller.security.managers.IIdentityManager#isUserInGroup(java.lang.String,
     *         java.lang.String, boolean)
     */
    public boolean isUserInGroup(String userID, String groupID, boolean parseGroupHierarchy)
            throws IdentityManagerException
    {
        boolean result;
        if (parseGroupHierarchy)
        {
            result = getUserGroupsIDs(userID, true).contains(groupID);
        }
        else
        {
            result = isUserInGroup(userID, groupID);
        }
        return result;
    }

    /**
     * @see pt.digitalis.dif.controller.security.managers.IIdentityManager#removeUserParameter(java.lang.String,
     *         java.lang.String)
     */
    public void removeUserParameter(String userID, String parameterID)
    {
        this.parameters.get(userID).remove(parameterID);
    }

    /**
     * @see pt.digitalis.dif.controller.security.managers.IIdentityManager#setUserParameter(java.lang.String,
     *         java.lang.String, java.lang.Object)
     */
    public void setUserParameter(String userID, String parameterID, Object parameterValue)
    {

        // Initialize object if needed
        if (this.parameters.get(userID) == null)
            this.parameters.put(userID, new HashMap<String, Object>());

        // Add parameter
        this.parameters.get(userID).put(parameterID, parameterValue);
    }

    /**
     * @see pt.digitalis.dif.controller.security.managers.IIdentityManager#setUserParameters(java.lang.String,
     *         java.util.Map)
     */
    public void setUserParameters(String userID, Map<String, Object> parametersMap)
    {
        // Add parameter map
        this.parameters.put(userID, parametersMap);
    }
}
