/**
 * 2007, Digitalis Informatica. All rights reserved. Distribuicao e Gestao de Informatica, Lda. Estrada de Paco de Arcos
 * num.9 - Piso -1 2780-666 Paco de Arcos Telefone: (351) 21 4408990 Fax: (351) 21 4408999 http://www.digitalis.pt
 */

package pt.digitalis.dif.controller.security.managers.impl;

import com.google.inject.Inject;
import com.newrelic.api.agent.Trace;
import pt.digitalis.dif.controller.interfaces.IDIFSession;
import pt.digitalis.dif.controller.interfaces.IPrivateDIFSession;
import pt.digitalis.dif.controller.objects.ClientDescriptor;
import pt.digitalis.dif.controller.objects.Constants;
import pt.digitalis.dif.controller.objects.DIFSession;
import pt.digitalis.dif.controller.objects.DIFUserInSession;
import pt.digitalis.dif.controller.security.managers.IAuthenticationManager;
import pt.digitalis.dif.controller.security.managers.IIdentityManager;
import pt.digitalis.dif.controller.security.managers.ISessionManager;
import pt.digitalis.dif.controller.security.managers.ISessionManagerInternal;
import pt.digitalis.dif.controller.security.objects.IDIFUser;
import pt.digitalis.dif.exception.security.AuthenticationManagerException;
import pt.digitalis.dif.exception.security.IdentityManagerException;
import pt.digitalis.dif.startup.DIFGeneralConfigurationParameters;
import pt.digitalis.dif.utils.IObjectFormatter;
import pt.digitalis.dif.utils.ObjectFormatter;
import pt.digitalis.dif.utils.ObjectFormatter.Format;
import pt.digitalis.utils.common.StringUtils;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * DIF's default implementation of a session manager. REFACTOR: Should extract a superclass from this as a base
 * implementation for this and other sessionManager impl's
 *
 * @author Pedro Viegas <a href="mailto:pviegas@digitalis.pt">pviegas@digitalis.pt</a><br/>
 * @author Rodrigo Gonalves <a href="mailto:rgoncalves@digitalis.pt">rgoncalves@digitalis.pt</a><br/>
 * @created Dec 11, 2007
 */
public class SessionManagerImpl implements ISessionManager, ISessionManagerInternal, IObjectFormatter
{

    /**
     * The authentication manager.
     */
    @Inject
    protected IAuthenticationManager theAuthenticationManager;

    /**
     * The identity manager.
     */
    @Inject
    protected IIdentityManager theIDManager;

    /**
     * The Sessions by hardware.
     */
    private Map<String, Long> sessionsByHardware = new HashMap<String, Long>();

    /**
     * The Sessions by software.
     */
    private Map<String, Long> sessionsBySoftware = new HashMap<String, Long>();

    /**
     * The Total session since startup.
     */
    private Long totalSessionSinceStartup = 0L;

    /**
     * List of logged sessions.
     */
    private Map<String, IPrivateDIFSession> loggedSessions = new HashMap<String, IPrivateDIFSession>();

    /**
     * Instantiates a new Session manager.
     */
    public SessionManagerImpl()
    {
        // Launches the inactive session monitoring thread
        SessionGarbageCollector collector = new SessionGarbageCollector(this);
        collector.start();
    }

    /**
     * Create session idif session.
     *
     * @param sessionID the session id
     *
     * @return the idif session
     *
     * @see pt.digitalis.dif.controller.security.managers.ISessionManager#createSession(java.lang.String)
     *         pt.digitalis.dif.controller.security.managers.ISessionManager#createSession(java.lang.String)
     *         pt.digitalis.dif.controller.security.managers.ISessionManager#createSession(java.lang.String)
     *         pt.digitalis.dif.controller.security.managers.ISessionManager#createSession(java.lang.String)
     */
    @Override
    public IDIFSession createSession(String sessionID)
    {
        IPrivateDIFSession session = loggedSessions.get(sessionID);

        if (session == null || session.isMarkedForRemoval())
        {
            totalSessionSinceStartup++;
            session = new DIFSession(sessionID);
            session.setSessionTimeOut(DIFGeneralConfigurationParameters.getInstance().getSessionTimeout());

            loggedSessions.put(sessionID, session);
        }
        else
            update(session);

        return session;
    }

    /**
     * Gets logged sessions.
     *
     * @return the loggedSessions
     */
    public Map<String, IPrivateDIFSession> getLoggedSessions()
    {
        return loggedSessions;
    }

    /**
     * Gets session.
     *
     * @param sessionID the session id
     *
     * @return the session
     *
     * @see pt.digitalis.dif.controller.security.managers.ISessionManager#getSession(java.lang.String)
     *         pt.digitalis.dif.controller.security.managers.ISessionManager#getSession(java.lang.String)
     *         pt.digitalis.dif.controller.security.managers.ISessionManager#getSession(java.lang.String)
     *         pt.digitalis.dif.controller.security.managers.ISessionManager#getSession(java.lang.String)
     */
    @Override
    public IDIFSession getSession(String sessionID)
    {
        return loggedSessions.get(sessionID);
    }

    /**
     * Gets sessions.
     *
     * @return the sessions
     *
     * @see pt.digitalis.dif.controller.security.managers.ISessionManager#getSessions()
     *         pt.digitalis.dif.controller.security.managers.ISessionManager#getSessions()
     *         pt.digitalis.dif.controller.security.managers.ISessionManager#getSessions()
     *         pt.digitalis.dif.controller.security.managers.ISessionManager#getSessions()
     */
    @Override
    public List<IDIFSession> getSessions()
    {
        return new ArrayList<IDIFSession>(loggedSessions.values());
    }

    /**
     * Gets sessions by hardware.
     *
     * @return the sessions by hardware
     */
    @Override
    public Map<String, Long> getSessionsByHardware()
    {
        return sessionsByHardware;
    }

    /**
     * Gets sessions by software.
     *
     * @return the sessions by software
     */
    @Override
    public Map<String, Long> getSessionsBySoftware()
    {
        return sessionsBySoftware;
    }

    /**
     * Gets total session since startup.
     *
     * @return the total session since startup
     */
    @Override
    public Long getTotalSessionSinceStartup()
    {
        return totalSessionSinceStartup;
    }

    /**
     * Is session present boolean.
     *
     * @param sessionID the session id
     *
     * @return the boolean
     *
     * @see pt.digitalis.dif.controller.security.managers.ISessionManager#isSessionPresent(java.lang.String)
     *         pt.digitalis.dif.controller.security.managers.ISessionManager#isSessionPresent(java.lang.String)
     *         pt.digitalis.dif.controller.security.managers.ISessionManager#isSessionPresent(java.lang.String)
     *         pt.digitalis.dif.controller.security.managers.ISessionManager#isSessionPresent(java.lang.String)
     */
    @Override
    public boolean isSessionPresent(String sessionID)
    {
        return loggedSessions.containsKey(sessionID);
    }

    /**
     * Log in idif session.
     *
     * @param sessionID the session id
     * @param userID    the user id
     * @param password  the password
     *
     * @return the idif session
     *
     * @exception AuthenticationManagerException the authentication manager exception
     * @see pt.digitalis.dif.controller.security.managers.ISessionManager#logIn(java.lang.String, java.lang.String,
     *         java.lang.String) pt.digitalis.dif.controller.security.managers.ISessionManager#logIn(java.lang.String,
     *         java.lang.String,
     *         java.lang.String)pt.digitalis.dif.controller.security.managers.ISessionManager#logIn(java.lang.String,
     *         java.lang.String, java.lang.String)pt.digitalis.dif.controller.security.managers.ISessionManager#logIn(java.lang.String,
     *         java.lang.String, java.lang.String)
     */
    @Override
    @Trace(metricName = "DIF:SessionManager:Login", dispatcher = true)
    public IDIFSession logIn(String sessionID, String userID, String password) throws AuthenticationManagerException
    {
        IDIFUser loggedUser = theAuthenticationManager.logIn(sessionID, userID, password);

        if (loggedUser != null)
            return this.logInNoPasswordValidation(sessionID, loggedUser, password);
        else
            return this.createSession(sessionID);
    }

    /**
     * Log in no password validation.
     *
     * @param sessionID  the session ID
     * @param loggedUser the logged user
     * @param password   the password
     *
     * @return the IDIF session
     *
     * @exception AuthenticationManagerException the authentication manager exception
     */
    private IDIFSession logInNoPasswordValidation(String sessionID, IDIFUser loggedUser, String password)
            throws AuthenticationManagerException
    {

        IDIFSession session = createSession(sessionID);

        if (session.getUser() != null)
        {
            session = this.logOut(sessionID);
        }

        if (loggedUser != null)
        {
            session.setUser(new DIFUserInSession(loggedUser, password));
            update(session);
        }

        return session;
    }

    /**
     * Log in no password validation idif session.
     *
     * @param sessionID the session id
     * @param userID    the user id
     * @param password  the password
     *
     * @return the idif session
     *
     * @exception AuthenticationManagerException the authentication manager exception
     * @see pt.digitalis.dif.controller.security.managers.ISessionManagerInternal#logInNoPasswordValidation(java.lang.String,
     *         java.lang.String, java.lang.String) pt.digitalis.dif.controller.security.managers.ISessionManagerInternal#logInNoPasswordValidation(java.lang.String,
     *         java.lang.String, java.lang.String)pt.digitalis.dif.controller.security.managers.ISessionManagerInternal#logInNoPasswordValidation(java.lang.String,
     *         java.lang.String, java.lang.String)pt.digitalis.dif.controller.security.managers.ISessionManagerInternal#logInNoPasswordValidation(java.lang.String,
     *         java.lang.String, java.lang.String)
     */
    @Override
    public IDIFSession logInNoPasswordValidation(String sessionID, String userID, String password)
            throws AuthenticationManagerException
    {
        try
        {
            IDIFUser loggedUser = theIDManager.getUser(userID);
            return logInNoPasswordValidation(sessionID, loggedUser, password);
        }
        catch (IdentityManagerException e)
        {
            throw new AuthenticationManagerException("Problem getting the user from identity manager", e);
        }
    }

    /**
     * Log out idif session.
     *
     * @param sessionID the session id
     *
     * @return the idif session
     *
     * @see pt.digitalis.dif.controller.security.managers.ISessionManager#logOut(java.lang.String)
     *         pt.digitalis.dif.controller.security.managers.ISessionManager#logOut(java.lang.String)
     *         pt.digitalis.dif.controller.security.managers.ISessionManager#logOut(java.lang.String)
     *         pt.digitalis.dif.controller.security.managers.ISessionManager#logOut(java.lang.String)
     */
    @Override
    @Trace(metricName = "DIF:SessionManager:Logout", dispatcher = true)
    public IDIFSession logOut(String sessionID)
    {
        IDIFSession session = loggedSessions.get(sessionID);

        if (session != null)
        {
            session.setUser(null);
            theAuthenticationManager.logOut(sessionID);
            session.getNavigationHistory().cleanUpAfterLogout(session);
            update(session);
            reinitializeSession(sessionID);
        }

        return session;
    }

    /**
     * Cleans up an existant session, after logout or clean session request
     *
     * @param sessionID the session ID to clean up
     */
    private void reinitializeSession(String sessionID)
    {
        IPrivateDIFSession session = loggedSessions.get(sessionID);

        if (session != null)
        {
            Object sessionInvalidBrowser = session.getAttribute(Constants.INVALID_BROWSER_ACCEPTED);
            session.setAttributes(new HashMap<String, Object>());
            session.addAttribute(Constants.INVALID_BROWSER_ACCEPTED, sessionInvalidBrowser);
            loggedSessions.put(sessionID, session);
        }
    }

    /**
     * Removes a session
     *
     * @param sessionID the session id
     */
    synchronized public void removeSession(String sessionID)
    {
        loggedSessions.remove(sessionID);

        // Instructs the authenticated manager that this client has disconnected from DIF. It will decide if there is an
        // authentication record present what to do with it. Keep or remove it.
        theAuthenticationManager.disconnectClient(sessionID);
    }

    /**
     * Report new client.
     *
     * @param clientDescriptor the client descriptor
     */
    @Override
    public void reportNewClient(ClientDescriptor clientDescriptor)
    {
        Pattern pattern = Pattern.compile(".*(\\s[0-9].*)");
        Matcher matcher;

        String hardware = pattern.matcher(StringUtils.nvl(clientDescriptor.getHardware(), "")).replaceFirst("");
        String software = pattern.matcher(StringUtils.nvl(clientDescriptor.getSoftware(), "")).replaceFirst("");

        this.sessionsByHardware.put(hardware,
                this.sessionsByHardware.containsKey(hardware) ? this.sessionsByHardware.get(hardware) + 1 : 1);
        this.sessionsBySoftware.put(software,
                this.sessionsBySoftware.containsKey(software) ? this.sessionsBySoftware.get(software) + 1 : 1);
    }

    @Override
    public ObjectFormatter toObjectFormatter(Format format, List<Object> dumpedObjects)
    {
        ObjectFormatter formatter = new ObjectFormatter(format, dumpedObjects);
        formatter.addItem("Sessions", loggedSessions);

        return formatter;
    }

    @Override
    public String toString()
    {
        return toObjectFormatter(Format.TEXT, null).getFormatedObject();
    }

    /**
     * Update boolean.
     *
     * @param session the session
     *
     * @return the boolean
     *
     * @see pt.digitalis.dif.controller.security.managers.ISessionManager#update(pt.digitalis.dif.controller.interfaces.IDIFSession)
     *         pt.digitalis.dif.controller.security.managers.ISessionManager#update(pt.digitalis.dif.controller.interfaces.IDIFSession)
     *         pt.digitalis.dif.controller.security.managers.ISessionManager#update(pt.digitalis.dif.controller.interfaces.IDIFSession)
     *         pt.digitalis.dif.controller.security.managers.ISessionManager#update(pt.digitalis.dif.controller.interfaces.IDIFSession)
     */
    @Override
    public boolean update(IDIFSession session)
    {
        boolean exists = isSessionPresent(session.getSessionID());

        if (exists)
        {
            IPrivateDIFSession privSession = (IPrivateDIFSession) session;

            privSession.keepAlive();
            loggedSessions.put(privSession.getSessionID(), privSession);
        }

        return exists;
    }
}
