/**
 * 2007, Digitalis Informatica. All rights reserved. Distribuicao e Gestao de Informatica, Lda. Estrada de Paco de Arcos
 * num.9 - Piso -1 2780-666 Paco de Arcos Telefone: (351) 21 4408990 Fax: (351) 21 4408999 http://www.digitalis.pt
 */
package pt.digitalis.dif.controller.security.objects;

import pt.digitalis.dif.controller.security.managers.IAuthorizationManager;
import pt.digitalis.dif.controller.security.managers.IIdentityManager;
import pt.digitalis.dif.controller.security.managers.IIdentityManagerPrivate;
import pt.digitalis.dif.controller.security.managers.IUserPreferencesManager;
import pt.digitalis.dif.dem.Entity;
import pt.digitalis.dif.dem.interfaces.IStage;
import pt.digitalis.dif.exception.InternalFrameworkException;
import pt.digitalis.dif.exception.security.AuthorizationManagerException;
import pt.digitalis.dif.exception.security.IdentityManagerException;
import pt.digitalis.dif.ioc.DIFIoCRegistry;
import pt.digitalis.dif.utils.IObjectFormatter;
import pt.digitalis.dif.utils.ObjectFormatter;
import pt.digitalis.dif.utils.ObjectFormatter.Format;
import pt.digitalis.utils.common.StringUtils;
import pt.digitalis.utils.common.collections.CaseInsensitiveHashMap;
import pt.digitalis.utils.config.ConfigurationException;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;

/**
 * Default implementation VALIDATE: Check the Cloneable usage good/bad practice!
 *
 * @author Rodrigo Gon�alves <a href="mailto:rgoncalves@digitalis.pt">rgoncalves@digitalis.pt</a>
 * @author Pedro Viegas <a href="mailto:pviegas@digitalis.pt">pviegas@digitalis.pt</a>
 * @created Dec 3, 2007
 */
public class DIFUserImpl implements IDIFClonableUser, Cloneable, IObjectFormatter
{

    /** The authorization manager implementation */
    static private IAuthorizationManager authorizationManager =
            DIFIoCRegistry.getRegistry().getImplementation(IAuthorizationManager.class);

    /** The identity manager implementation */
    static private IIdentityManager identityManager =
            DIFIoCRegistry.getRegistry().getImplementation(IIdentityManager.class);

    /** The user preferences manager. */
    static private IUserPreferencesManager userPreferencesManager =
            DIFIoCRegistry.getRegistry().getImplementation(IUserPreferencesManager.class);

    /** Groups to remove temporarly from user without affecting the identity manager persistance */
    private final Set<String> groupsExcludeTemporarily = new HashSet<String>();

    /** Temporary groups to include into the final user groups. This groups must have the session life expectancy */
    private final Map<String, IDIFGroup> temporaryGroups = new HashMap<String, IDIFGroup>();

    /** User attributes. */
    private CaseInsensitiveHashMap<Object> attributes = new CaseInsensitiveHashMap<Object>();

    /** User attributes to remove. */
    private List<String> attributesToRemove = new ArrayList<String>();

    /** The email of the user */
    private String email;

    /** If the user is enabled */
    private boolean enabled;

    /** Local cache for user groups */
    private Map<String, IDIFGroup> groupsCache = null;

    /** Local cache for user Groups Hierarchy */
    private Map<String, IDIFGroup> groupsHierarchyCache = null;

    /** The unique id for the user */
    private String id;

    /** Is "default user"? */
    private boolean isDefault;

    /** The full name for the user */
    private String name;

    /** The nick for the user */
    private String nick;

    /** The password of the user */
    private String password;

    /** the user profile group identifier */
    private String profileID;

    /** The user preferences. */
    private CaseInsensitiveHashMap<Object> userPreferences = null;

    /**
     * Default constructor
     */
    public DIFUserImpl()
    {
        this.cleanCache();
    }

    /**
     * Constructor from a IDIFUser object
     *
     * @param user     the IDIFUser to use as a base for the new user to create
     * @param password the password to set
     *
     * @exception IdentityManagerException if the profile can't be accessed
     */
    public DIFUserImpl(IDIFUser user, String password) throws IdentityManagerException
    {
        setProps(user);

        this.password = password;
        this.cleanCache();
    }

    /**
     * @see pt.digitalis.dif.controller.security.objects.IDIFUser#addGroupToExcludeTemporarily(java.lang.String)
     */
    @Override
    public void addGroupToExcludeTemporarily(String groupId)
    {
        groupsExcludeTemporarily.add(groupId);
    }

    /**
     * @see pt.digitalis.dif.controller.security.objects.IDIFUser#addTempGroup(java.lang.String)
     */
    @Override
    public void addTempGroup(String groupId)
    {
        try
        {
            IDIFGroup group = identityManager.getGroup(groupId);
            if (group != null)
            {
                temporaryGroups.put(group.getID(), group);
            }
        }
        catch (IdentityManagerException e)
        {
            // Do nothing
        }
    }

    /**
     * @see pt.digitalis.dif.controller.security.objects.IUserAuthorization#canAccess(pt.digitalis.dif.dem.Entity,
     *         java.lang.String)
     */
    @Override
    public boolean canAccess(Entity resourceType, String resourceId)
    {
        try
        {
            return authorizationManager.hasAccessUser(this, resourceType, resourceId) ||
                   authorizationManager.hasAccessPublic(resourceType, resourceId);
        }
        catch (AuthorizationManagerException authorizationManagerException)
        {
            return false;
        }
    }

    /**
     * @see pt.digitalis.dif.controller.security.objects.IUserAuthorization#canAccess(pt.digitalis.dif.dem.interfaces.IStage)
     */
    @Override
    public boolean canAccess(IStage stage)
    {
        if (stage == null)
            return false;

        return this.canAccess(Entity.STAGE, stage.getID());
    }

    /**
     * @see pt.digitalis.dif.controller.security.objects.IDIFUser#cleanCache()
     */
    @Override
    public void cleanCache()
    {
        groupsCache = null;
        this.userPreferences = null;
    }

    /**
     * @see java.lang.Object#clone()
     */
    @Override
    protected Object clone() throws CloneNotSupportedException
    {
        try
        {
            DIFUserImpl clone = new DIFUserImpl(this, this.password);

            return clone;
        }
        catch (IdentityManagerException identityManagerException)
        {
            throw new RuntimeException(
                    "Could not clone object because the identity manager raised an excpetion an vital data could not be fetched!",
                    identityManagerException);
        }
    }

    /**
     * @see pt.digitalis.dif.controller.security.objects.IDIFClonableUser#cloneUser()
     */
    @Override
    public IDIFClonableUser cloneUser()
    {
        try
        {
            return (IDIFClonableUser) this.clone();
        }
        catch (CloneNotSupportedException e)
        {
            return null;
        }
    }

    /**
     * @see pt.digitalis.dif.controller.security.objects.IDIFUser#containsAttribute(java.lang.String)
     */
    @Override
    public boolean containsAttribute(String id)
    {
        return attributes.containsKey(id);
    }

    /**
     * @see pt.digitalis.dif.controller.security.objects.IDIFUser#containsParameter(java.lang.String)
     */
    @Override
    public boolean containsParameter(String id)
    {
        return DIFIoCRegistry.getRegistry().getImplementation(IIdentityManager.class)
                .containsUserParameter(this.getID(), id);
    }

    /**
     * @see java.lang.Object#equals(java.lang.Object)
     */
    @Override
    public boolean equals(Object obj)
    {
        if (this == obj)
            return true;
        if (obj == null)
            return false;
        if (getClass() != obj.getClass())
            return false;
        DIFUserImpl other = (DIFUserImpl) obj;
        if (id == null)
        {
            if (other.id != null)
                return false;
        }
        else if (!id.equals(other.id))
            return false;
        return true;
    }

    /**
     * @see pt.digitalis.dif.controller.security.objects.IDIFUser#getAttribute(java.lang.String)
     */
    @Override
    public Object getAttribute(String id)
    {
        return attributes.get(id);
    }

    /**
     * @see pt.digitalis.dif.controller.security.objects.IDIFUser#getAttributes()
     */
    @Override
    public CaseInsensitiveHashMap<Object> getAttributes()
    {
        return attributes;
    }

    /**
     * @see pt.digitalis.dif.controller.security.objects.IDIFUser#setAttributes(java.util.Map)
     */
    @Override
    public void setAttributes(Map<String, Object> attributes) throws InternalFrameworkException
    {
        // Store old values
        CaseInsensitiveHashMap<Object> formerAttributeValues = this.attributes;

        try
        {
            CaseInsensitiveHashMap<Object> attributesToChange = new CaseInsensitiveHashMap<Object>();

            if (attributes != null)
            {
                for (Entry<String, Object> entry : attributes.entrySet())
                {
                    boolean attributeExistentInTheUser =
                            StringUtils.isNotBlank(StringUtils.toStringOrNull(this.attributes.get(entry.getKey())));

                    if (attributeExistentInTheUser)
                    {
                        boolean valueChanged = !this.attributes.get(entry.getKey()).equals(entry.getValue());

                        if (valueChanged && !identityManager.isSaveParameterValuesOnlyIfEmpty())
                        {
                            attributesToChange.put(entry.getKey(), entry.getValue());
                        }
                    }
                    else
                    {
                        attributesToChange.put(entry.getKey(), entry.getValue());
                    }
                }
            }

            // Clean all attributes to ensure new passed data
            this.attributes = new CaseInsensitiveHashMap<Object>();
            // Set new values on the object
            if (attributes != null)
                this.attributes.putAll(attributes);

            if (!attributesToChange.isEmpty())
            {
                // Set values on Id Manager
                ((IIdentityManagerPrivate) DIFIoCRegistry.getRegistry().getImplementation(IIdentityManager.class))
                        .persistUserAttributes(this.getID(), attributesToChange);
            }
        }
        catch (IdentityManagerException identityManagerException)
        {
            // Roll back attribute values
            this.attributes = formerAttributeValues;
            // Issue warning
            throw new InternalFrameworkException("Could not update the attribute values on the Identity Manager!",
                    identityManagerException, null);
        }
        catch (ConfigurationException e)
        {
            this.attributes = formerAttributeValues;
            throw new InternalFrameworkException(
                    "Could not update the attribute values on the Identity Manager! Error reading Configuration", e,
                    null);
        }
    }

    /**
     * @see pt.digitalis.dif.controller.security.objects.IDIFUser#getAttributesToRemove()
     */
    @Override
    public List<String> getAttributesToRemove()
    {
        return attributesToRemove;
    }

    /**
     * @see pt.digitalis.dif.controller.security.objects.IDIFUser#getEmail()
     */
    @Override
    public String getEmail()
    {
        return email;
    }

    /**
     * @see pt.digitalis.dif.controller.security.objects.IDIFUser#setEmail(java.lang.String)
     */
    @Override
    public void setEmail(String email)
    {
        this.email = email;
    }

    /**
     * @see pt.digitalis.dif.controller.security.objects.IDIFUser#getGroupIDs()
     */
    @Override
    public Set<String> getGroupIDs() throws IdentityManagerException
    {
        return getGroups().keySet();
    }

    /**
     * @see pt.digitalis.dif.controller.security.objects.IDIFUser#getGroups()
     */
    @Override
    public Map<String, IDIFGroup> getGroups() throws IdentityManagerException
    {
        if (groupsCache == null)
        {
            groupsCache = identityManager.getUserGroups(id);
            // Add temporari groups without affecting identity manager persistance
            groupsCache.putAll(this.temporaryGroups);

            // Exclude a groups temporarily from the user withot affecting the identity manager persistance
            for (String groupID : this.groupsExcludeTemporarily)
            {
                groupsCache.remove(groupID);
            }
        }

        return groupsCache;
    }

    /**
     * @see pt.digitalis.dif.controller.security.objects.IDIFUser#getGroupsHierarchy()
     */
    @Override
    public Map<String, IDIFGroup> getGroupsHierarchy() throws IdentityManagerException
    {
        if (groupsHierarchyCache == null)
        {
            groupsHierarchyCache = identityManager.getUserGroups(id, true);
            // Add temporari groups without affecting identity manager persistance
            groupsHierarchyCache.putAll(this.temporaryGroups);

            // Exclude a groups temporarily from the user withot affecting the identity manager persistance
            for (String groupID : this.groupsExcludeTemporarily)
            {
                groupsHierarchyCache.remove(groupID);
            }
        }

        return groupsHierarchyCache;
    }

    /**
     * @see pt.digitalis.dif.controller.security.objects.IDIFUser#getID()
     */
    @Override
    public String getID()
    {
        return id;
    }

    /**
     * @see pt.digitalis.dif.controller.security.objects.IDIFUser#setID(java.lang.String)
     */
    @Override
    public void setID(String id)
    {
        this.id = id;
    }

    /**
     * @see pt.digitalis.dif.controller.security.objects.IDIFUser#getName()
     */
    @Override
    public String getName()
    {
        return name;
    }

    /**
     * @see pt.digitalis.dif.controller.security.objects.IDIFUser#setName(java.lang.String)
     */
    @Override
    public void setName(String name)
    {
        this.name = name;
    }

    /**
     * @see pt.digitalis.dif.controller.security.objects.IDIFUser#getNick()
     */
    @Override
    public String getNick()
    {
        return nick;
    }

    /**
     * @see pt.digitalis.dif.controller.security.objects.IDIFUser#setNick(java.lang.String)
     */
    @Override
    public void setNick(String nick)
    {
        this.nick = nick;
    }

    /**
     * @see pt.digitalis.dif.controller.security.objects.IDIFUser#getParameter(String)
     */
    @Override
    public Object getParameter(String id)
    {
        return DIFIoCRegistry.getRegistry().getImplementation(IIdentityManager.class)
                .getUserParameter(this.getID(), id);
    }

    /**
     * @see pt.digitalis.dif.controller.security.objects.IDIFUser#getParameters()
     */
    @Override
    public Map<String, Object> getParameters()
    {
        return DIFIoCRegistry.getRegistry().getImplementation(IIdentityManager.class).getUserParameters(this.getID());
    }

    /**
     * @see pt.digitalis.dif.controller.security.objects.IDIFUser#setParameters(java.util.Map)
     */
    @Override
    public void setParameters(Map<String, Object> parameters)
    {
        DIFIoCRegistry.getRegistry().getImplementation(IIdentityManager.class)
                .setUserParameters(this.getID(), parameters);
    }

    /**
     * @return the user password
     */
    @Override
    public String getPassword()
    {
        return password;
    }

    /**
     * @see pt.digitalis.dif.controller.security.objects.IDIFUser#setPassword(java.lang.String)
     */
    @Override
    public void setPassword(String password)
    {
        this.password = password;
    }

    /**
     * @see pt.digitalis.dif.controller.security.objects.IDIFUser#getProfile()
     */
    @Override
    public IDIFGroup getProfile() throws IdentityManagerException
    {
        if (profileID == null)
            return null;

        return getGroups().get(profileID);
    }

    /**
     * @see pt.digitalis.dif.controller.security.objects.IDIFUser#getProfileID()
     */
    @Override
    public String getProfileID()
    {
        return this.profileID;
    }

    /**
     * @see pt.digitalis.dif.controller.security.objects.IDIFUser#setProfileID(java.lang.String)
     */
    @Override
    public void setProfileID(String profileGroupID)
    {
        this.profileID = profileGroupID;

        cleanCache();
    }

    /**
     * @see pt.digitalis.dif.controller.security.objects.IDIFUser#getUserPreference(java.lang.String)
     */
    @Override
    public Object getUserPreference(String id) throws InternalFrameworkException
    {
        return this.getUserPreferences().get(id);
    }

    /**
     * @see pt.digitalis.dif.controller.security.objects.IDIFUser#getUserPreferences()
     */
    @Override
    public CaseInsensitiveHashMap<Object> getUserPreferences() throws InternalFrameworkException
    {
        if (this.userPreferences == null)
        {
            this.userPreferences = userPreferencesManager.getUserPreferences(this.getID());
            if (this.userPreferences == null)
            {
                this.userPreferences = new CaseInsensitiveHashMap<Object>();
            }
        }

        return this.userPreferences;
    }

    /**
     * @see java.lang.Object#hashCode()
     */
    @Override
    public int hashCode()
    {
        final int prime = 31;
        int result = 1;
        result = prime * result + ((id == null) ? 0 : id.hashCode());
        return result;
    }

    /**
     * @see pt.digitalis.dif.controller.security.objects.IDIFUser#initializeAttributes(java.util.Map)
     */
    @Override
    public void initializeAttributes(Map<String, Object> attrs)
    {
        attributes = new CaseInsensitiveHashMap<Object>();
        attributes.putAll(attrs);
    }

    /**
     * @see pt.digitalis.dif.controller.security.objects.IDIFUser#isDefault()
     */
    @Override
    public boolean isDefault()
    {
        return this.isDefault;
    }

    /**
     * @see pt.digitalis.dif.controller.security.objects.IDIFUser#setDefault(boolean)
     */
    @Override
    public void setDefault(boolean isDefault)
    {
        this.isDefault = isDefault;
    }

    /**
     * @see pt.digitalis.dif.controller.security.objects.IDIFUser#isEnabled()
     */
    @Override
    public boolean isEnabled()
    {
        return enabled;
    }

    /**
     * @see pt.digitalis.dif.controller.security.objects.IDIFUser#setEnabled(boolean)
     */
    @Override
    public void setEnabled(boolean enabled)
    {
        this.enabled = enabled;
    }

    /**
     * @see pt.digitalis.dif.controller.security.objects.IDIFUser#refresh()
     */
    @Override
    public void refresh()
    {
        IDIFUser updatedUser;
        try
        {
            updatedUser = identityManager.getUser(id);
            if (updatedUser != null)
            {
                setProps(updatedUser);
            }
        }
        catch (IdentityManagerException e)
        {
            // Nothing. Exception will be logged and refresh will only apply to cache vars...
        }

        cleanCache();
    }

    /**
     * @see pt.digitalis.dif.controller.security.objects.IDIFUser#removeAttribute(java.lang.String)
     */
    @Override
    public void removeAttribute(String id)
    {
        attributes.remove(id);
        attributesToRemove.add(id);
    }

    /**
     * @see pt.digitalis.dif.controller.security.objects.IDIFUser#removeGroupToExcludeTemporarily(java.lang.String)
     */
    @Override
    public void removeGroupToExcludeTemporarily(String groupId)
    {
        groupsExcludeTemporarily.remove(groupId);
    }

    /**
     * @see pt.digitalis.dif.controller.security.objects.IDIFUser#removeParameter(java.lang.String)
     */
    @Override
    public void removeParameter(String id)
    {
        DIFIoCRegistry.getRegistry().getImplementation(IIdentityManager.class).removeUserParameter(this.getID(), id);
    }

    /**
     * @see pt.digitalis.dif.controller.security.objects.IDIFUser#removeTempGroup(java.lang.String)
     */
    @Override
    public void removeTempGroup(String groupId)
    {
        temporaryGroups.remove(groupId);
    }

    /**
     * @see pt.digitalis.dif.controller.security.objects.IDIFUser#setAttribute(java.lang.String, java.lang.Object)
     */
    @Override
    public void setAttribute(String id, Object attribute) throws InternalFrameworkException
    {
        // Store old value
        Object formerAttributeValue = this.attributes.get(id);

        try
        {
            boolean persistValue = true;
            boolean attributeExistentInTheUser =
                    StringUtils.isNotBlank(StringUtils.toStringOrNull(this.attributes.get(id)));

            if (attributeExistentInTheUser)
            {
                boolean valueChanged = !this.attributes.get(id).equals(attribute);
                persistValue = valueChanged && !identityManager.isSaveParameterValuesOnlyIfEmpty();
            }

            // Set new attribute value on object
            this.attributes.put(id, attribute);

            if (persistValue)
            {
                // Set new attribute on Id Manager
                ((IIdentityManagerPrivate) DIFIoCRegistry.getRegistry().getImplementation(IIdentityManager.class))
                        .persistUserAttribute(this.getID(), id, attribute);
            }
        }
        catch (IdentityManagerException identityManagerException)
        {
            // Roll back attribute value
            this.attributes.put(id, formerAttributeValue);
            // Issue warning
            throw new InternalFrameworkException("Could not update the attribute value on the Identity Manager!",
                    identityManagerException, null);
        }
        catch (ConfigurationException configurationException)
        {
            throw new InternalFrameworkException("Error reading configuration!", configurationException, null);
        }
    }

    /**
     * @see pt.digitalis.dif.controller.security.objects.IDIFUser#setParameter(java.lang.String, java.lang.Object)
     */
    @Override
    public void setParameter(String id, Object parameter)
    {
        DIFIoCRegistry.getRegistry().getImplementation(IIdentityManager.class)
                .setUserParameter(this.getID(), id, parameter);
    }

    /**
     * Sets the internal user properties with the given user ones
     *
     * @param user the user to get the updated properties from
     *
     * @exception IdentityManagerException
     */
    private void setProps(IDIFUser user) throws IdentityManagerException
    {
        this.id = user.getID();
        this.nick = user.getNick();
        this.name = user.getName();
        this.email = user.getEmail();
        this.enabled = user.isEnabled();
        this.attributes = user.getAttributes();
        this.profileID = user.getProfileID();
    }

    /**
     * @see pt.digitalis.dif.controller.security.objects.IDIFUser#setUserPreference(java.lang.String,
     *         java.lang.Object)
     */
    @Override
    public void setUserPreference(String id, Object attribute) throws InternalFrameworkException
    {
        userPreferencesManager.setUserPreferences(this.getID(), id, attribute);
        this.getUserPreferences().put(id, attribute);
    }

    @Override
    public ObjectFormatter toObjectFormatter(ObjectFormatter.Format format, List<Object> dumpedObjects)
    {
        ObjectFormatter formatter = new ObjectFormatter(format, dumpedObjects);

        formatter.addItem("ID", getID());
        formatter.addItemIfNotNull("Name", getName());
        formatter.addItemIfNotNull("Password", getPassword());
        formatter.addItemIfNotNull("Nick", getNick());
        formatter.addItemIfNotNull("Email", getEmail());
        formatter.addItem("Enabled", isEnabled());
        formatter.addItemIfNotNull("Attributes", attributes);
        formatter.addItemIfNotNull("Parameters", this.getParameters());

        try
        {
            IDIFGroup profile = getProfile();
            formatter.addItemIfNotNull("Profile", profile);
        }
        catch (IdentityManagerException identityManagerException)
        {
            throw new RuntimeException("Could not access the user's profile on the identity manager!",
                    identityManagerException);
        }

        try
        {
            Map<String, IDIFGroup> groups = getGroups();
            formatter.addItemIfNotNull("Groups", groups);
        }
        catch (IdentityManagerException identityManagerException)
        {
            throw new RuntimeException("Could not access the user's groups on the identity manager!",
                    identityManagerException);
        }

        return formatter;
    }

    @Override
    public String toString()
    {
        return this.toObjectFormatter(Format.TEXT, null).getFormatedObject();
    }
}