/**
 * 2007, Digitalis Informatica. All rights reserved. Distribuicao e Gestao de Informatica, Lda. Estrada de Paco de Arcos
 * num.9 - Piso -1 2780-666 Paco de Arcos Telefone: (351) 21 4408990 Fax: (351) 21 4408999 http://www.digitalis.pt
 */
package pt.digitalis.dif.controller.security.objects;

import pt.digitalis.dif.exception.InternalFrameworkException;
import pt.digitalis.dif.exception.security.IdentityManagerException;
import pt.digitalis.dif.utils.IObjectFormatter;
import pt.digitalis.utils.common.collections.CaseInsensitiveHashMap;
import pt.digitalis.utils.config.ConfigurationException;

import java.util.List;
import java.util.Map;
import java.util.Set;

/**
 * Defines the DIF user features interface
 *
 * @author Pedro Viegas <a href="mailto:pviegas@digitalis.pt">pviegas@digitalis.pt</a>
 * @author Rodrigo Gon�alves <a href="mailto:rgoncalves@digitalis.pt">rgoncalves@digitalis.pt</a><br/>
 * @created Dec 3, 2007
 */
public interface IDIFUser extends IUserAuthorization, IObjectFormatter
{

    /**
     * Add a group to exclude temporarily from the user.
     *
     * @param groupId the group id
     */
    public void addGroupToExcludeTemporarily(String groupId);

    /**
     * Adds the temporary group. This user group has the session life expectancy
     *
     * @param groupId the group id
     */
    public void addTempGroup(String groupId);

    /**
     * Clears any existing cache
     */
    public void cleanCache();

    /**
     * Searches for a given attribute
     *
     * @param id the id of the attribute to search
     *
     * @return T if the attribute is present
     */
    public boolean containsAttribute(String id);

    /**
     * Searches for a given parameter
     *
     * @param id the id of the parameter to search
     *
     * @return T if the parameter is present
     */
    public boolean containsParameter(String id);

    /**
     * Gets the user attribute.
     *
     * @param id the id of the attribute
     *
     * @return the user attribute
     */
    public Object getAttribute(String id);

    /**
     * Gets the user attribute list.
     *
     * @return the user attribute list
     */
    public CaseInsensitiveHashMap<Object> getAttributes();

    /**
     * Sets the attributes.
     *
     * @param attributes the attributes to set
     *
     * @exception InternalFrameworkException if the attribute can't be set
     */
    public void setAttributes(Map<String, Object> attributes) throws InternalFrameworkException;

    /**
     * Inspector for the 'attributesToRemove' attribute.
     *
     * @return the attributesToRemove value
     */
    public List<String> getAttributesToRemove();

    /**
     * Returns the user e-mail address.
     *
     * @return the user e-mail
     */
    public String getEmail();

    /**
     * Sets the user e-mail address.
     *
     * @param email the email to set
     */
    public void setEmail(String email);

    /**
     * Retrieves the list of group ids associated with this user
     *
     * @return the groups
     *
     * @exception IdentityManagerException if the users can't be found on the identity manager
     */
    public Set<String> getGroupIDs() throws IdentityManagerException;

    /**
     * Retrieves the list of groups associated with this user
     *
     * @return the groups
     *
     * @exception IdentityManagerException if the groups can't be found on the identity manager
     */
    public Map<String, IDIFGroup> getGroups() throws IdentityManagerException;

    /**
     * Retrieves the list of groups associated with this user, parsing the group hierarchy
     *
     * @return the groups
     *
     * @exception IdentityManagerException if the groups can't be found on the identity manager
     */
    public Map<String, IDIFGroup> getGroupsHierarchy() throws IdentityManagerException;

    /**
     * Returns the user id.
     *
     * @return the user id
     */
    public String getID();

    /**
     * Sets the user id.
     *
     * @param id the id to set
     */
    public void setID(String id);

    /**
     * Returns the user full name.
     *
     * @return the user's full name
     */
    public String getName();

    /**
     * Sets the user full name.
     *
     * @param name the name to set
     */
    public void setName(String name);

    /**
     * Returns the user's nick name.
     *
     * @return the user's nick name
     */
    public String getNick();

    /**
     * Sets the user nick name.
     *
     * @param nick the nick name to set
     */
    public void setNick(String nick);

    /**
     * Gets a user parameter.
     *
     * @param id the id of the parameter
     *
     * @return the user parameter
     */
    public Object getParameter(String id);

    /**
     * Gets the user parameter list.
     *
     * @return the user parameter list
     */
    public Map<String, Object> getParameters();

    /**
     * Sets the parameters.
     *
     * @param parameters the parameters to set
     */
    public void setParameters(Map<String, Object> parameters);

    /**
     * Retrieves the user profile or default group
     *
     * @return the profile group
     *
     * @exception IdentityManagerException if the profile can't be found on the identity manager
     */
    public IDIFGroup getProfile() throws IdentityManagerException;

    /**
     * Retrieves the user profile or default group ID
     *
     * @return the profile group
     *
     * @exception IdentityManagerException if the profile can't be accessed
     */
    public String getProfileID() throws IdentityManagerException;

    /**
     * Sets the profile ID or default group of the user
     *
     * @param profileGroupID the group to set as the profile
     */
    public void setProfileID(String profileGroupID);

    /**
     * Gets the user preference.
     *
     * @param id the id
     *
     * @return the user preference
     *
     * @exception InternalFrameworkException the internal framework exception
     */
    public Object getUserPreference(String id) throws InternalFrameworkException;

    /**
     * Gets the user preferences.
     *
     * @return the user preferences
     *
     * @exception InternalFrameworkException the internal framework exception
     */
    public CaseInsensitiveHashMap<Object> getUserPreferences() throws InternalFrameworkException;

    /**
     * @param attrs the new attributes to set
     */
    public void initializeAttributes(Map<String, Object> attrs);

    /**
     * A default user is a user that is created by the default configuration of the application. As such it cannot be
     * deleted. Only disabled.
     *
     * @return T if the user is a default user
     */
    public boolean isDefault();

    /**
     * Sets the user as a default user.
     *
     * @param isDefault
     */
    public void setDefault(boolean isDefault);

    /**
     * A disabled user will not be able to log on
     *
     * @return T if the user is enabled
     */
    public boolean isEnabled();

    /**
     * @param enabled the enabled to set
     */
    public void setEnabled(boolean enabled);

    /**
     * Refreshes the current user from the persistent layer
     */
    public void refresh();

    /**
     * Removes the user attribute.
     *
     * @param id the id of the attribute
     */
    public void removeAttribute(String id);

    /**
     * Removes a group to exclude temporarily from the user.
     *
     * @param groupId the group id
     */
    public void removeGroupToExcludeTemporarily(String groupId);

    /**
     * Removes the user parameter.
     *
     * @param id the id of the parameter
     */
    public void removeParameter(String id);

    /**
     * Removes the temporary group.
     *
     * @param groupId the group id
     */
    public void removeTempGroup(String groupId);

    /**
     * Sets the user attribute.
     *
     * @param id        the id of the attribute
     * @param attribute the attribute to set
     *
     * @exception InternalFrameworkException if the attribute can't be set
     * @exception ConfigurationException
     */
    public void setAttribute(String id, Object attribute) throws InternalFrameworkException, ConfigurationException;

    /**
     * Sets the user parameters.
     *
     * @param id        the id of the parameter
     * @param parameter the parameter to set
     */
    public void setParameter(String id, Object parameter);

    /**
     * Sets the user password.
     *
     * @param password the password to set
     */
    public void setPassword(String password);

    /**
     * Sets the user preference.
     *
     * @param id        the id
     * @param attribute the attribute
     *
     * @exception InternalFrameworkException the internal framework exception
     */
    public void setUserPreference(String id, Object attribute) throws InternalFrameworkException;
}