/**
 * - Digitalis Internal Framework v2.0 - (C) 2007, Digitalis Informatica. Distribuicao e Gestao de Informatica, Lda.
 * Estrada de Paco de Arcos num.9 - Piso -1 2780-666 Paco de Arcos Telefone: (351) 21 4408990 Fax: (351) 21 4408999
 * http://www.digitalis.pt
 */
package pt.digitalis.dif.dem;

import pt.digitalis.dif.dem.interfaces.IApplication;
import pt.digitalis.dif.dem.interfaces.IDEMRegistry;
import pt.digitalis.dif.dem.interfaces.IEntity;
import pt.digitalis.dif.dem.interfaces.IProvider;
import pt.digitalis.dif.dem.interfaces.IService;
import pt.digitalis.dif.dem.interfaces.IStage;
import pt.digitalis.dif.utils.logging.DIFLogger;
import pt.digitalis.utils.bytecode.holders.ClassHolder;
import pt.digitalis.utils.common.collections.CaseInsensitiveHashMap;
import pt.digitalis.utils.inspection.exception.ResourceNotFoundException;

import java.util.HashSet;
import java.util.Map;
import java.util.Set;

/**
 * DIF entity model (DEM) Registry class. This class is populated by the code generation module. This module reads the
 * DEM annotations in classpath's classes, generates the appropriate code and populates the Registry. Since only one DEM
 * Registry is needed, the class implements the Singleton pattern. <b>Note to class maintainers:</b> all the attribute
 * accesses - even for setting/removing values - should be made through the supplied inspectors
 * <code>getProviders()</code>, <code>getApplications()</code>, <code>getServices()</code> and <code>getStages()</code>.
 * Accessing directly to the attributes does not guarantee that they are properly initialized. This need is due to the
 * fact that the code generation module inserts the proper attribute initialization code in the aforementioned getters,
 * so accessing through them is the only way to guarantee that they contain the proper data.
 *
 * @author Rodrigo Gonalves <a href="mailto:rgoncalves@digitalis.pt">rgoncalves@digitalis.pt</a>
 * @author Pedro Viegas <a href="mailto:pviegas@digitalis.pt">pviegas@digitalis.pt</a>
 * @created 2007/04/24
 */
public class DEMRegistryImpl implements IDEMRegistry
{

    /** The registered applications container (the key is the ID). */
    static private Map<String, IApplication> applications = new CaseInsensitiveHashMap<IApplication>();

    /** The default provider. */
    static private String defaultProvider;

    /** The overridden registered stages container. */
    static private Set<String> overridenStages = new HashSet<String>();

    /** The registered providers container (the key is the ID). */
    static private Map<String, IProvider> providers = new CaseInsensitiveHashMap<IProvider>();

    /** The registry object. */
    static private DEMRegistryImpl registry = new DEMRegistryImpl();

    /** The registered services container (the key is the ID). */
    static private Map<String, IService> services = new CaseInsensitiveHashMap<IService>();

    /** The registered stages container (the key is the ID). */
    static private Map<String, IStage> stages = new CaseInsensitiveHashMap<IStage>();

    /**
     * Private constructor to prevent multiple instantiation (Singleton pattern).
     */
    private DEMRegistryImpl()
    {
    }

    /**
     * Registers an application.
     *
     * @param clazz the application class to register
     *
     * @return the added application
     */
    static public IApplication addApplication(ClassHolder clazz)
    {
        IApplication application;

        try
        {
            application = (IApplication) clazz.getClassInstance();
            applications.put(application.getID(), application);

            return application;
        }
        catch (ClassNotFoundException e)
        {
            e.printStackTrace();

            return null;
        }
        catch (InstantiationException e)
        {
            e.printStackTrace();

            return null;
        }
        catch (IllegalAccessException e)
        {
            e.printStackTrace();

            return null;
        }
        catch (ResourceNotFoundException e)
        {
            e.printStackTrace();

            return null;
        }
    }

    // --- Maintenance methods for DIF's private use ---

    /**
     * Registers a provider.
     *
     * @param entityType the entity Type to be added
     * @param clazz      the provider class to register
     *
     * @return the added entity
     */
    static public IEntity addEntity(Entity entityType, ClassHolder clazz)
    {

        if (entityType.equals(Entity.PROVIDER))
            return addProvider(clazz);
        if (entityType.equals(Entity.APPLICATION))
            return addApplication(clazz);
        if (entityType.equals(Entity.SERVICE))
            return addService(clazz);
        if (entityType.equals(Entity.STAGE))
            return addStage(clazz);

        return null;
    }

    /**
     * Registers a provider.
     *
     * @param clazz the provider class to register
     *
     * @return the added provider
     */
    static public IProvider addProvider(ClassHolder clazz)
    {
        IProvider provider;

        try
        {
            provider = (IProvider) clazz.getClassInstance();
            providers.put(provider.getID(), provider);

            return provider;
        }
        catch (ClassNotFoundException e)
        {
            e.printStackTrace();

            return null;
        }
        catch (InstantiationException e)
        {
            e.printStackTrace();

            return null;
        }
        catch (IllegalAccessException e)
        {
            e.printStackTrace();

            return null;
        }
        catch (ResourceNotFoundException e)
        {
            e.printStackTrace();

            return null;
        }
    }

    /**
     * Registers a service.
     *
     * @param clazz the service class to register
     *
     * @return the added service
     */
    static public IService addService(ClassHolder clazz)
    {
        IService service;

        try
        {
            service = (IService) clazz.getClassInstance();
            services.put(service.getID(), service);

            return service;
        }
        catch (ClassNotFoundException e)
        {
            e.printStackTrace();

            return null;
        }
        catch (InstantiationException e)
        {
            e.printStackTrace();

            return null;
        }
        catch (IllegalAccessException e)
        {
            e.printStackTrace();

            return null;
        }
        catch (ResourceNotFoundException e)
        {
            e.printStackTrace();

            return null;
        }
    }

    /**
     * Registers a stage.
     *
     * @param clazz the stage class to register
     *
     * @return the added stage
     */
    static public IStage addStage(ClassHolder clazz)
    {
        IStage stage;

        try
        {
            stage = (IStage) clazz.getClassInstance();

            // Stage override logic: Once a stage has been overridden it it not added. This must be done this way since
            // we cannot be sure of the order in which the stages are added to the registry
            if (!overridenStages.contains(stage.getID()))
                stages.put(stage.getID(), stage);

            // If the stage should override another one...
            String overrideStageList = stage.getOverridesStageID();
            if (overrideStageList != null)
            {
                String[] overrideStageArray = overrideStageList.split(",");

                for (String overrideStage : overrideStageArray)
                {
                    overrideStage = overrideStage.trim().toLowerCase();

                    DIFLogger.getLogger()
                            .debug("Stage \"" + stage.getName() + " (" + stage.getID() + ")\" has overriden stage \"" +
                                   overrideStage + "\"");

                    stages.put(overrideStage, stage);
                    overridenStages.add(overrideStage);
                }
            }

            return stage;
        }
        catch (ClassNotFoundException e)
        {
            e.printStackTrace();

            return null;
        }
        catch (InstantiationException e)
        {
            e.printStackTrace();

            return null;
        }
        catch (IllegalAccessException e)
        {
            e.printStackTrace();

            return null;
        }
        catch (ResourceNotFoundException e)
        {
            e.printStackTrace();

            return null;
        }
    }

    /**
     * Returns the map of all registered applications.
     *
     * @return the map of all registered applications
     */
    public static Map<String, IApplication> getApplications()
    {
        return applications;
    }

    /**
     * Returns the default provider.
     *
     * @return the default provider
     */
    static public IProvider getDefaultProvider()
    {
        return providers.get(getDefaultProviderID());
    }

    /**
     * Returns the default provider.
     *
     * @param providerID the new default provider
     */
    static public void setDefaultProvider(String providerID)
    {
        defaultProvider = providerID;
    }

    /**
     * Returns the default provider id.
     *
     * @return the default provider id
     */
    static public String getDefaultProviderID()
    {
        if (defaultProvider == null)
            // If no default provider has been provided use the first existent one
            defaultProvider = providers.keySet().iterator().next();

        return defaultProvider;
    }

    /**
     * Returns the map of all registered providers.
     *
     * @return the map of all registered providers
     */
    public static Map<String, IProvider> getProviders()
    {
        return providers;
    }

    // --- General use public methods (accessible through IDEMRegistry) ---

    /**
     * Returns the DEMRegistryImpl instance (Singleton pattern).
     *
     * @return the registry object
     */
    static public DEMRegistryImpl getRegistry()
    {
        return registry;
    }

    /**
     * Returns the map of all services registered.
     *
     * @return the map of all services registered
     */
    public static Map<String, IService> getServices()
    {
        return services;
    }

    /**
     * Returns the map of all stages registered.
     *
     * @return the map of all stages registered
     */
    static public Map<String, IStage> getStages()
    {
        return stages;
    }

    /** @see pt.digitalis.dif.dem.interfaces.IDEMRegistry#getApplication(java.lang.String) */
    @Override
    public IApplication getApplication(String applicationID)
    {
        if (applicationID == null)
            return null;
        else
            return applications.get(applicationID.toLowerCase());
    }

    /** @see pt.digitalis.dif.dem.interfaces.IDEMRegistry#getProvider(java.lang.String) */
    @Override
    public IProvider getProvider(String providerID)
    {
        if (providerID == null)
            return null;
        else
            return providers.get((providerID.toLowerCase()));
    }

    /** @see pt.digitalis.dif.dem.interfaces.IDEMRegistry#getService(java.lang.String) */
    @Override
    public IService getService(String serviceID)
    {
        if (serviceID == null)
            return null;
        else
            return services.get(serviceID.toLowerCase());
    }

    /** @see pt.digitalis.dif.dem.interfaces.IDEMRegistry#getStage(java.lang.String) */
    @Override
    public IStage getStage(String stageID)
    {
        if (stageID == null)
            return null;
        else
            return stages.get(stageID.toLowerCase());
    }
}
