/**
 * - Digitalis Internal Framework v2.0 - (C) 2007, Digitalis Informatica. Distribuicao e Gestao de Informatica, Lda.
 * Estrada de Paco de Arcos num.9 - Piso -1 2780-666 Paco de Arcos Telefone: (351) 21 4408990 Fax: (351) 21 4408999
 * http://www.digitalis.pt
 */
package pt.digitalis.dif.dem.interfaces;

import pt.digitalis.dif.dem.objects.EventType;
import pt.digitalis.dif.dem.objects.ViewObject;
import pt.digitalis.dif.dem.objects.parameters.IParameters;

import java.util.List;
import java.util.Map;

/**
 * This interface represents a Stage entity on the DEM.
 * <p>
 * A Stage is related to a Service.
 *
 * @author Pedro Viegas <a href="mailto:pviegas@digitalis.pt">pviegas@digitalis.pt</a>
 * @author Rodrigo Gonalves <a href="mailto:rgoncalves@digitalis.pt">rgoncalves@digitalis.pt</a>
 * @created 2007 /07/12
 */
public interface IStage extends IEntity, ICallback, IRegistrable
{

    /**
     * Allow access without valid consents.
     *
     * @return true, if successful
     */
    public boolean allowAccessWithoutValidConsents();

    /**
     * Returns the default error Stage defined for this Stage
     *
     * @return the default error Stage
     */
    public IStage getDefaultErrorStage();

    /**
     * Returns the default error view defined for this Stage.
     *
     * @return the default error view
     */
    public ViewObject getDefaultErrorView();

    /**
     * Returns the default view.
     *
     * @return the default view
     */
    public ViewObject getDefaultView();

    /**
     * Returns the list of all the Stages submit handlers.
     *
     * @return the list
     */
    public Map<EventType, List<String>> getEventHandlers();

    /**
     * Gets the included message stages.
     *
     * @return the included message stages
     */
    public List<String> getIncludedMessageStages();

    /**
     * Returns the list of all the error Stages injected in this Stage.
     *
     * @return the list of all the injected Stages
     */
    public Map<String, String> getInjectedErrorStages();

    /**
     * Returns the list of all the error views injected in this Stage.
     *
     * @return the list of all the injected ViewObject
     */
    public Map<String, ViewObject> getInjectedErrorViews();

    /**
     * Returns the list of all the Stages injected in this Stage.
     *
     * @return the list of all the injected Stages
     */
    public List<String> getInjectedStages();

    /**
     * Returns the list of all the views injected in this Stage.
     *
     * @return the list of all the injected ViewObject
     */
    public List<ViewObject> getInjectedViews();

    /**
     * Return an instance of the stage for execution
     *
     * @return the stage instance
     */
    public IStageInstance getInstance();

    /**
     * Gets java script head freemarker template.
     *
     * @param view the view
     *
     * @return the java script head template
     */
    public String getJavaScriptHeadFreeMarkerTemplate(ViewObject view);

    /**
     * Gets java script head template.
     *
     * @param view the view
     *
     * @return the java script head template
     */
    public String getJavaScriptHeadTemplate(ViewObject view);

    /**
     * Gets java script on load freemarker template.
     *
     * @param view the view
     *
     * @return the java script on load template
     */
    public String getJavaScriptOnLoadFreeMarkerTemplate(ViewObject view);

    /**
     * Gets java script on load template.
     *
     * @param view the view
     *
     * @return the java script on load template
     */
    public String getJavaScriptOnLoadTemplate(ViewObject view);

    /**
     * Gets java script templates.
     *
     * @return the java script templates
     */
    public Map<String, Map<String, String>> getJavaScriptTemplates();

    /**
     * Returns a message associated with a given ID
     *
     * @param language  the language to get the messages in
     * @param messageID the message ID
     *
     * @return the message with the given id
     */
    public String getMessageForLanguage(String language, String messageID);

    /**
     * Returns all messages associated with the Entity.
     *
     * @param language the language to get the messages in
     *
     * @return a map with all the messages related to the Entity
     */
    public Map<String, String> getMessagesForLanguage(String language);

    /**
     * Gets overrides stage id.
     *
     * @return the stage id of the stage that the present one overrides. Null otherwise.
     */
    public String getOverridesStageID();

    /**
     * Returns the parameters for the current stage
     *
     * @return the parameters
     */
    public IParameters getParameters();

    /**
     * Returns the Stage service.
     *
     * @return the Stage service
     */
    public IService getService();

    /**
     * Obtains the original stage class name
     *
     * @return the class name
     */
    public String getStageInstanceClassName();

    /**
     * Returns the list of trusted parameters that will not be sanitized if this feature is active
     *
     * @return the list of trusted parameters
     */
    public List<String> getTrustedParameters();

    /**
     * Defines if the stage has authentication.
     *
     * @return T if the stage has , F otherwise
     */
    public boolean hasAuthentication();

    /**
     * Registers if the stage has injected the authentication errors. If so the dispatcher will not fail on
     * authentication errors and instead inject them so the stage may process them.
     *
     * @return T if the stage has injected the authentication errors onto an attribute
     */
    public boolean hasAuthenticationErrorInjection();

    /**
     * Defines if the stage has authorization.
     *
     * @return T if the stage has , F otherwise
     */
    public boolean hasAuthorization();

    /**
     * Has injected contributions boolean.
     *
     * @return T if any attribute or method in annotated with {@link com.google.inject.Inject}
     */
    public boolean hasInjectedContributions();

    /**
     * Has java script head freemarker template.
     *
     * @param view the view
     *
     * @return the boolean
     */
    public boolean hasJavaScriptHeadFreeMarkerTemplate(ViewObject view);

    /**
     * Has java script head template.
     *
     * @param view the view
     *
     * @return the boolean
     */
    public boolean hasJavaScriptHeadTemplate(ViewObject view);

    /**
     * Has java script on load freemarker template.
     *
     * @param view the view
     *
     * @return the boolean
     */
    public boolean hasJavaScriptOnLoadFreeMarkerTemplate(ViewObject view);

    /**
     * Has java script on load template.
     *
     * @param view the view
     *
     * @return the boolean
     */
    public boolean hasJavaScriptOnLoadTemplate(ViewObject view);

    /**
     * Registers if the stage has injected the parameter errors. If not the framework must provide extra reporting of
     * the errors. Else it is delegated to the business stage implementation.
     *
     * @return T if the stage has injected the errors onto an attribute
     */
    public boolean hasParameterErrorInjection();

    /**
     * Checks if the current form has a validation logic handler for the given form
     *
     * @param formName the form name
     *
     * @return T if present
     */
    public boolean hasValidationLogicForForm(String formName);
}
