/**
 * - Digitalis Internal Framework v2.0 - (C) 2007, Digitalis Informatica. Distribuicao e Gestao de Informatica, Lda.
 * Estrada de Paco de Arcos num.9 - Piso -1 2780-666 Paco de Arcos Telefone: (351) 21 4408990 Fax: (351) 21 4408999
 * http://www.digitalis.pt
 */
package pt.digitalis.dif.dem.managers;

import pt.digitalis.dif.dem.Entity;
import pt.digitalis.dif.dem.interfaces.IApplication;
import pt.digitalis.dif.dem.interfaces.IEntity;
import pt.digitalis.dif.dem.interfaces.IProvider;
import pt.digitalis.dif.dem.interfaces.IService;
import pt.digitalis.dif.dem.interfaces.IStage;
import pt.digitalis.dif.dem.objects.messages.MessageList;
import pt.digitalis.dif.dem.objects.messages.MessageTranslations;
import pt.digitalis.utils.inspection.exception.ResourceNotFoundException;

import java.util.Map;
import java.util.Set;

/**
 * Manages the DEM messages, providing operations for access, pooling and persistence.
 *
 * @author Pedro Viegas <a href="mailto:pviegas@digitalis.pt">pviegas@digitalis.pt</a>
 * @author Rodrigo Gonalves <a href="mailto:rgoncalves@digitalis.pt">rgoncalves@digitalis.pt</a>
 * @created 2007/06/01
 */
public interface IMessageManager
{

    /**
     * Adds a new supported language to the manager.
     *
     * @param language the new supported language to add
     */
    public void addSupportedLanguage(String language);

    /**
     * Collect entity messages from resource LanguageID, MessageID, theMessageObject
     *
     * @param clazz the class to read the messages from
     *
     * @return the collected messages
     */
    public MessageList collectEntityMessagesFromRepository(Class<?> clazz);

    /**
     * Collect entity messages from resource LanguageID, MessageID, theMessageObject
     *
     * @param type     the type of the entity or null if it's not a DEM entity
     * @param instance the entity instance
     *
     * @return the collected messages
     */
    public MessageList collectEntityMessagesFromRepository(Entity type, Object instance);

    /**
     * Collect entity messages from resource LanguageID, MessageID, theMessageObject
     *
     * @param type      the type of the entity or null if it's not a DEM entity
     * @param id        the entity id
     * @param className the class name
     * @param parentID  the entity's parent ID
     *
     * @return the collected messages
     *
     * @exception ResourceNotFoundException if a needed resource for message collection can't be found
     */
    public MessageList collectEntityMessagesFromRepository(Entity type, String id, String className, String parentID)
            throws ResourceNotFoundException;

    /**
     * Collect entity messages from resource LanguageID, MessageID, theMessageObject
     *
     * @param messagePath the messagePath for the message file (without the ".messages.xx" suffix)
     *
     * @return the collected messages
     */
    public MessageList collectEntityMessagesFromRepository(String messagePath);

    /**
     * Gets the messages from the repository and collects them on first use
     *
     * @param type   the entity type
     * @param entity the entity object
     *
     * @return the messages collected
     */
    public MessageList getMessageList(Entity type, IEntity entity);

    /**
     * Gets the messages of a given Provider with all translations.
     *
     * @param application the application for which the messages should be retrieved
     *
     * @return the application messages
     */
    public MessageList getMessageList(IApplication application);

    /**
     * Gets the messages of a given Provider with all translations.
     *
     * @param provider the provider for which the messages should be retrieved
     *
     * @return the provider messages
     */
    public MessageList getMessageList(IProvider provider);

    /**
     * Gets the messages of a given service with all translations.
     *
     * @param service the service for which the messages should be retrieved
     *
     * @return the service messages
     */
    public MessageList getMessageList(IService service);

    /**
     * Gets the messages of a given stage with all translations.
     *
     * @param stage the stage for which the messages should be retrieved
     *
     * @return the stage messages
     */
    public MessageList getMessageList(IStage stage);

    /**
     * Gets the message list.
     *
     * @param id the id
     *
     * @return the message list
     */
    public MessageList getMessageList(String id);

    /**
     * Gets the messages of a given Application.
     *
     * @param application the application for which the messages should be retrieved
     * @param language    the language to filter
     *
     * @return the application messages
     */
    public Map<String, String> getMessages(IApplication application, String language);

    /**
     * Gets the messages of a given Provider.
     *
     * @param provider the provider for which the messages should be retrieved
     * @param language the language to search
     *
     * @return the provider messages
     */
    public Map<String, String> getMessages(IProvider provider, String language);

    /**
     * Gets the messages of a given Service.
     *
     * @param service  the service for which the messages should be retrieved
     * @param language the language to filter
     *
     * @return the service messages
     */
    public Map<String, String> getMessages(IService service, String language);

    /**
     * Gets the messages of a given Stage.
     *
     * @param stage    the stage for which the messages should be retrieved
     * @param language the language to filter
     *
     * @return the stage messages
     */
    public Map<String, String> getMessages(IStage stage, String language);

    /**
     * Returns the list of supported languages.
     *
     * @return the list of supported languages
     */
    public Set<String> getSupportedLanguages();

    /** Invalidates the messages, forcing their reloading. */
    public void invalidateMessages();

    /**
     * Checks if a given language is supported.
     *
     * @param language the language to check
     *
     * @return T if the passed language is supported, F otherwise
     */
    public boolean isLanguageSupported(String language);

    /**
     * @return T this implementation supports saving messages to a persistent media. If F changes will be lost after a
     *         server restart
     */
    public boolean isPersistent();

    /**
     * Refresh message cache
     */
    public void refreshCache();

    /**
     * Updates a message with a given ID on a given class.
     *
     * @param clazz     the class
     * @param messageID the ID of the message to update
     * @param message   the message (and it's translations) to update
     */
    public void updateMessage(Class<?> clazz, String messageID, MessageTranslations message);

    /**
     * Updates a message with a given ID on a given entity.
     *
     * @param entity    the entity
     * @param messageID the ID of the message to update
     * @param message   the message (and it's translations) to update
     */
    public void updateMessage(IEntity entity, String messageID, MessageTranslations message);

    /**
     * Updates a message with a given ID.
     *
     * @param id        the id
     * @param messageID the ID of the message to update
     * @param message   the message (and it's translations) to update
     */
    public void updateMessage(String id, String messageID, MessageTranslations message);
}
