/**
 * - Digitalis Internal Framework v2.0 - (C) 2007, Digitalis Informatica. Distribuicao e Gestao de Informatica, Lda.
 * Estrada de Paco de Arcos num.9 - Piso -1 2780-666 Paco de Arcos Telefone: (351) 21 4408990 Fax: (351) 21 4408999
 * http://www.digitalis.pt
 */
package pt.digitalis.dif.dem.managers;

import pt.digitalis.dif.dem.Entity;
import pt.digitalis.dif.dem.interfaces.IStage;
import pt.digitalis.dif.dem.objects.LicenseEditionType;
import pt.digitalis.dif.exception.manager.RegistrationManagerException;
import pt.digitalis.utils.config.ConfigurationException;

/**
 * This interface manages the DEM entities registration process.
 *
 * @author Pedro Viegas <a href="mailto:pviegas@digitalis.pt">pviegas@digitalis.pt</a>
 * @created Oct 3, 2007
 */
public interface IRegistrationManager
{

    /**
     * Activate application registration.
     *
     * @param applicationID the application id
     *
     * @exception ConfigurationException       the configuration exception
     * @exception RegistrationManagerException the registration manager exception
     */
    public void activateApplicationRegistration(String applicationID)
            throws ConfigurationException, RegistrationManagerException;

    void activateEntityRegistration(Entity type, String entityID)
            throws RegistrationManagerException, ConfigurationException;

    /**
     * Activate provider registration.
     *
     * @param providerID the provider id
     *
     * @exception ConfigurationException       the configuration exception
     * @exception RegistrationManagerException the registration manager exception
     */
    public void activateProviderRegistration(String providerID)
            throws ConfigurationException, RegistrationManagerException;

    /**
     * Activate service registration.
     *
     * @param serviceID the service id
     *
     * @exception ConfigurationException       the configuration exception
     * @exception RegistrationManagerException the registration manager exception
     */
    public void activateServiceRegistration(String serviceID)
            throws ConfigurationException, RegistrationManagerException;

    /**
     * Activate stage registration.
     *
     * @param stageID the stage id
     *
     * @exception ConfigurationException       the configuration exception
     * @exception RegistrationManagerException the registration manager exception
     */
    public void activateStageRegistration(String stageID) throws ConfigurationException, RegistrationManagerException;

    /**
     * Adds the entity license edition.
     *
     * @param type               the type
     * @param entityID           the entity ID
     * @param licenseEditionType the license edition type
     */
    public void addEntityLicenseEdition(Entity type, String entityID, LicenseEditionType licenseEditionType);

    /**
     * Adds a new record to the registry. User registrations will be automatically picked up by the Registry
     *
     * @param type        the Entity type
     * @param entityID    the entity ID
     * @param name        the name to register to (null if not registered by default)
     * @param registrable if the Entity is registrable
     *
     * @exception ConfigurationException the configuration exception
     */
    public void addToRegistry(Entity type, String entityID, String name, boolean registrable)
            throws ConfigurationException;

    /**
     * Deactivate application registration.
     *
     * @param applicationID the application id
     *
     * @exception ConfigurationException       the configuration exception
     * @exception RegistrationManagerException the registration manager exception
     */
    public void deactivateApplicationRegistration(String applicationID)
            throws ConfigurationException, RegistrationManagerException;

    /**
     * Deactivate entity registration.
     *
     * @param type     the entity type
     * @param entityID the service id
     */
    public void deactivateEntityRegistration(Entity type, String entityID)
            throws ConfigurationException, RegistrationManagerException;

    /**
     * Deactivate provider registration.
     *
     * @param providerID the provider id
     *
     * @exception ConfigurationException       the configuration exception
     * @exception RegistrationManagerException the registration manager exception
     */
    public void deactivateProviderRegistration(String providerID)
            throws ConfigurationException, RegistrationManagerException;

    /**
     * Deactivate service registration.
     *
     * @param serviceID the service id
     *
     * @exception ConfigurationException       the configuration exception
     * @exception RegistrationManagerException the registration manager exception
     */
    public void deactivateServiceRegistration(String serviceID)
            throws ConfigurationException, RegistrationManagerException;

    /**
     * Deactivate stage registration.
     *
     * @param stageID the stage id
     *
     * @exception ConfigurationException       the configuration exception
     * @exception RegistrationManagerException the registration manager exception
     */
    public void deactivateStageRegistration(String stageID) throws ConfigurationException, RegistrationManagerException;

    /**
     * Inspects the Application Edition
     *
     * @param applicationID the application ID
     *
     * @return {@link LicenseEditionType}
     */
    public LicenseEditionType getApplicationEdition(String applicationID);

    /**
     * Inspects the Provider Edition
     *
     * @param providerID the provider ID
     *
     * @return {@link LicenseEditionType}
     */
    public LicenseEditionType getProviderEdition(String providerID);

    /**
     * Inspects the Service Edition
     *
     * @param serviceID the service ID
     *
     * @return {@link LicenseEditionType}
     */
    public LicenseEditionType getServiceEdition(String serviceID);

    /**
     * Inspects the Stage Client Name
     *
     * @param stageID the stage ID
     *
     * @return the stage client name
     */
    public String getStageClientName(IStage stageID);

    /**
     * Inspects the Stage Edition
     *
     * @param stageID the stage ID
     *
     * @return {@link LicenseEditionType}
     */
    public LicenseEditionType getStageEdition(String stageID);

    /**
     * Inspects the registration state of a given application
     *
     * @param applicationID the provider ID
     *
     * @return T if the application is registered, F otherwise
     */
    public boolean isApplicationRegistered(String applicationID);

    /**
     * Inspects the registration possibility of a given application
     *
     * @param applicationID the provider ID
     *
     * @return T if the application is registrable, F otherwise
     */
    public boolean isApplicationRegistrable(String applicationID);

    /**
     * Inspects the registration state of a given provider
     *
     * @param providerID the provider ID
     *
     * @return T if the provider is registered, F otherwise
     */
    public boolean isProviderRegistered(String providerID);

    /**
     * Inspects the registration possibility of a given provider
     *
     * @param providerID the provider ID
     *
     * @return T if the provider is registrable, F otherwise
     */
    public boolean isProviderRegistrable(String providerID);

    /**
     * Is registration active boolean.
     *
     * @param entityType the entity type
     * @param entityID   the entity id
     *
     * @return the boolean
     */
    public boolean isRegistrationActive(Entity entityType, String entityID);

    /**
     * Inspects the registration state of a given service
     *
     * @param serviceID the service ID
     *
     * @return T if the service is registered, F otherwise
     */
    public boolean isServiceRegistered(String serviceID);

    /**
     * Inspects the registration possibility of a given service
     *
     * @param serviceID the service ID
     *
     * @return T if the service is registrable, F otherwise
     */
    public boolean isServiceRegistrable(String serviceID);

    /**
     * Inspects the registration state of a given stage
     *
     * @param stageId the stage ID
     *
     * @return T if the stage is registered, F otherwise
     */
    public boolean isStageRegistered(String stageId);

    /**
     * Inspects the registration possibility of a given stage
     *
     * @param stageId the stage ID
     *
     * @return T if the stage is registrable, F otherwise
     */
    public boolean isStageRegistrable(String stageId);

    /**
     * Registers a given application if it's not already registered.
     *
     * @param applicationID the application ID
     * @param name          the name to register to
     * @param key           the key to register
     *
     * @return T if the registration was successful
     *
     * @exception RegistrationManagerException if a error occurrs in registration manager
     * @exception ConfigurationException       the configuration exception
     */
    public boolean registerApplication(String applicationID, String name, String key)
            throws RegistrationManagerException, ConfigurationException;

    /**
     * Registers an entity
     *
     * @param type     the type of the entity
     * @param entityID the id of the entity
     * @param name     the name to register to
     * @param key      the key to register
     *
     * @return T if the registration was successful
     *
     * @exception RegistrationManagerException if a error occurs in registration manager
     * @exception ConfigurationException       the configuration exception
     */
    public boolean registerEntity(Entity type, String entityID, String name, String key)
            throws RegistrationManagerException, ConfigurationException;

    /**
     * Registers a given provider if it's not already registered.
     *
     * @param providerID the provider ID
     * @param name       the name to register to
     * @param key        the key to register
     *
     * @return T if the registration was successful
     *
     * @exception RegistrationManagerException if a error occurrs in registration manager
     * @exception ConfigurationException       the configuration exception
     */
    public boolean registerProvider(String providerID, String name, String key)
            throws RegistrationManagerException, ConfigurationException;

    /**
     * Registers a given service if it's not already registered.
     *
     * @param serviceID the service ID
     * @param name      the name to register to
     * @param key       the key to register
     *
     * @return T if the registration was successful
     *
     * @exception RegistrationManagerException if a error occurrs in registration manager
     * @exception ConfigurationException       the configuration exception
     */
    public boolean registerService(String serviceID, String name, String key)
            throws RegistrationManagerException, ConfigurationException;

    /**
     * Registers a given stage if it's not already registered.
     *
     * @param stageId the stage ID
     * @param name    the name to register to
     * @param key     the key to register
     *
     * @return T if the registration was successful
     *
     * @exception RegistrationManagerException if a error occurrs in registration manager
     * @exception ConfigurationException       the configuration exception
     */
    public boolean registerStage(String stageId, String name, String key)
            throws RegistrationManagerException, ConfigurationException;

    /**
     * Unregisters a given application if it's registered.
     *
     * @param applicationID the application ID
     *
     * @exception ConfigurationException the configuration exception
     */
    public void unregisterApplication(String applicationID) throws ConfigurationException;

    /**
     * Unregisters a given provider if it's registered.
     *
     * @param providerID the provider ID
     *
     * @exception ConfigurationException the configuration exception
     */
    public void unregisterProvider(String providerID) throws ConfigurationException;

    /**
     * Unregisters a given service if it's registered.
     *
     * @param serviceID the service ID
     *
     * @exception ConfigurationException the configuration exception
     */
    public void unregisterService(String serviceID) throws ConfigurationException;

    /**
     * Unregisters a given stage if it's registered.
     *
     * @param stageId the stage ID
     *
     * @exception ConfigurationException the configuration exception
     */
    public void unregisterStage(String stageId) throws ConfigurationException;
}
