/**
 * 2014, Digitalis Informatica. All rights reserved. Distribuicao e Gestao de Informatica, Lda. Estrada de Paco de Arcos
 * num.9 - Piso -1 2780-666 Paco de Arcos Telefone: (351) 21 4408990 Fax: (351) 21 4408999 http://www.digitalis.pt
 */

package pt.digitalis.dif.dem.managers.impl;

import pt.digitalis.dif.dem.annotations.CustomTemplate;
import pt.digitalis.dif.dem.managers.ITemplateManager;
import pt.digitalis.dif.ioc.DIFIoCRegistry;
import pt.digitalis.dif.startup.DIFGeneralConfigurationParameters;
import pt.digitalis.utils.common.StringUtils;
import pt.digitalis.utils.config.AbstractConfigurationsImpl;
import pt.digitalis.utils.config.ConfigItem;
import pt.digitalis.utils.config.IConfigurations;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * @author Luis Pinto <a href="mailto:lpinto@digitalis.pt">lpinto@digitalis.pt</a><br/>
 * @created 14/07/2016
 */
public abstract class AbstractTemplateManager implements ITemplateManager
{

    /** The resources cache. */
    protected static Map<String, Map<String, StringBuffer>> resourcesCache =
            new HashMap<String, Map<String, StringBuffer>>();

    /**
     * Gets the original template content.
     *
     * @param resource the resource
     * @param language the language
     *
     * @return the original template content
     *
     * @exception Exception
     */
    protected abstract StringBuffer doGetTemplateContent(String resource, String language) throws Exception;

    /**
     * @see pt.digitalis.dif.dem.managers.ITemplateManager#getTemplateContent(java.lang.String, java.lang.String)
     */
    @Override
    public final StringBuffer getTemplateContent(String resource, String language) throws Exception
    {
        if (!resourcesCache.containsKey(resource) || !resourcesCache.get(resource).containsKey(language))
        {
            StringBuffer content = doGetTemplateContent(resource, language);

            if (content == null)
            {
                if (language != null &&
                    !language.equalsIgnoreCase(DIFGeneralConfigurationParameters.getInstance().getDefaultLanguage()))
                {
                    /* Content not found in the requested language, so let's try to find it in the default language */
                    content = doGetTemplateContent(resource,
                            DIFGeneralConfigurationParameters.getInstance().getDefaultLanguage());
                }
                if (content == null)
                {
                    /* content not found in any language, so let's try to find it without language */
                    content = doGetTemplateContent(resource, null);
                }
            }

            if (!resourcesCache.containsKey(resource))
            {
                resourcesCache.put(resource, new HashMap<String, StringBuffer>());
            }
            resourcesCache.get(resource).put(language, content);
        }

        return resourcesCache.get(resource).get(language);
    }

    /**
     * @see pt.digitalis.dif.dem.managers.ITemplateManager#getTemplateList()
     */
    @Override
    public List<HTMLTemplateItem> getTemplateList() throws Exception
    {
        List<HTMLTemplateItem> templates = new ArrayList<HTMLTemplateItem>();

        // Parse configuration points
        AbstractConfigurationsImpl configs =
                (AbstractConfigurationsImpl) DIFIoCRegistry.getRegistry().getImplementation(IConfigurations.class);
        configs.readConfigurationsPointsForPackage("pt");

        for (Class<?> clazz : configs.getCacheConfigurationPoints().values())
        {
            List<ConfigItem> configItems = configs.getConfigItemsMap(clazz);
            Object configInstance = null;

            for (ConfigItem configItem : configItems)
            {
                CustomTemplate templateAnnotation = configItem.getGetter().getAnnotation(CustomTemplate.class);

                if (templateAnnotation != null)
                {
                    if (configInstance == null)
                        configInstance = configs.readConfiguration(clazz);

                    HTMLTemplateItem templateItem = new HTMLTemplateItem();
                    templateItem.setName(StringUtils.toStringOrNull(configItem.getGetter().invoke(configInstance)));
                    templateItem.setDescription(templateAnnotation.value());

                    templates.add(templateItem);
                }
            }
        }

        return templates;
    }

    /**
     * Invalidates the cache
     *
     * @param resource the resource
     *
     * @exception Exception
     */
    protected void invalidateCache(String resource) throws Exception
    {
        resourcesCache.remove(resource);
    }
}
