/**
 * - Digitalis Internal Framework v2.0 - (C) 2007, Digitalis Informatica. Distribuicao e Gestao de Informatica, Lda.
 * Estrada de Paco de Arcos num.9 - Piso -1 2780-666 Paco de Arcos Telefone: (351) 21 4408990 Fax: (351) 21 4408999
 * http://www.digitalis.pt
 */
package pt.digitalis.dif.dem.managers.impl;

import pt.digitalis.dif.codegen.util.DEMLoaderEntityRegistry;
import pt.digitalis.dif.dem.Entity;
import pt.digitalis.dif.dem.annotations.entities.ApplicationDefinition;
import pt.digitalis.dif.dem.annotations.entities.ServiceDefinition;
import pt.digitalis.dif.dem.interfaces.IApplication;
import pt.digitalis.dif.dem.interfaces.IEntity;
import pt.digitalis.dif.dem.interfaces.IProvider;
import pt.digitalis.dif.dem.interfaces.IService;
import pt.digitalis.dif.dem.interfaces.IStage;
import pt.digitalis.dif.dem.objects.messages.Message;
import pt.digitalis.dif.dem.objects.messages.MessageList;
import pt.digitalis.dif.dem.objects.messages.MessagesLocation;
import pt.digitalis.dif.startup.DIFGeneralConfigurationParameters;
import pt.digitalis.dif.startup.DIFStartupConfiguration;
import pt.digitalis.dif.utils.logging.DIFLogger;
import pt.digitalis.utils.bytecode.holders.ClassHolder;
import pt.digitalis.utils.inspection.ResourceUtils;
import pt.digitalis.utils.inspection.exception.ResourceNotFoundException;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.net.URL;
import java.net.URLDecoder;
import java.util.Enumeration;
import java.util.Map.Entry;
import java.util.Properties;

/**
 * Manages the DEM messages, providing operations for access, pooling and persistence.
 *
 * @author Pedro Viegas <a href="mailto:pviegas@digitalis.pt">pviegas@digitalis.pt</a>
 * @author Rodrigo Gonalves <a href="mailto:rgoncalves@digitalis.pt">rgoncalves@digitalis.pt</a>
 * @created 2007/06/01
 */
public class MessageManagerImpl extends AbstractMessageManager
{

    /**
     * @see pt.digitalis.dif.dem.managers.IMessageManager#collectEntityMessagesFromRepository(java.lang.Class)
     */
    public MessageList collectEntityMessagesFromRepository(Class<?> clazz)
    {

        String fileName;
        String name = clazz.getSimpleName();

        // Add the conventioned location
        if (DIFStartupConfiguration.getMessagesLocation().equals(MessagesLocation.MESSAGES_FOLDER))
            fileName = MESSAGES_FOLDER + clazz.getSimpleName() + AbstractMessageManager.DEFAULT_MESSAGE_FILE_EXTENSION;
        else
            fileName = clazz.getName().replace(".", "/") + AbstractMessageManager.DEFAULT_MESSAGE_FILE_EXTENSION;

        return readMessageFile(fileName, name);
    }

    /**
     * @see pt.digitalis.dif.dem.managers.IMessageManager#collectEntityMessagesFromRepository(pt.digitalis.dif.dem.Entity,
     *         java.lang.Object)
     */
    public MessageList collectEntityMessagesFromRepository(Entity type, Object instance)
    {

        // If there is no type let's treat this like any other normal class
        if (type == null)
            return collectEntityMessagesFromRepository(instance.getClass());

        // Otherwise let's build a logical message file location based on the DEM hierarchy
        String messageFile;
        String name;

        IEntity entity = (IEntity) instance;
        name = type.toString() + " " + AbstractMessageManager.NUMBER_SIGN + entity.getID();

        if (DIFStartupConfiguration.getMessagesLocation().equals(MessagesLocation.MESSAGES_FOLDER))
        {
            messageFile = entity.getOriginalClassName();
            messageFile = messageFile.substring(messageFile.lastIndexOf(".") + 1) +
                          AbstractMessageManager.DEFAULT_MESSAGE_FILE_EXTENSION;

            if (type.equals(Entity.STAGE))
            {
                IStage stage = (IStage) instance;

                messageFile = stage.getService().getApplication().getProvider().getID() + "/" +
                              stage.getService().getApplication().getID() + "/" + stage.getService().getID() + "/" +
                              messageFile;
            }
            if (type.equals(Entity.SERVICE))
            {
                IService service = (IService) instance;

                messageFile =
                        service.getApplication().getProvider().getID() + "/" + service.getApplication().getID() + "/" +
                        service.getID() + "/" + messageFile;
            }
            if (type.equals(Entity.APPLICATION))
            {
                IApplication application = (IApplication) instance;

                messageFile = application.getProvider().getID() + "/" + application.getID() + "/" + messageFile;
            }
            if (type.equals(Entity.PROVIDER))
            {
                IProvider provider = (IProvider) instance;

                messageFile = provider.getID() + "/" + messageFile;
            }

            // Add the conventioned prefix
            messageFile = MESSAGES_FOLDER + messageFile;
        }
        else
        {
            messageFile = entity.getOriginalClassName();
            messageFile = messageFile.replace(".", "/") + AbstractMessageManager.DEFAULT_MESSAGE_FILE_EXTENSION;
        }

        return readMessageFile(messageFile, name);
    }

    /**
     * @see pt.digitalis.dif.dem.managers.IMessageManager#collectEntityMessagesFromRepository(pt.digitalis.dif.dem.Entity,
     *         java.lang.String, java.lang.String, java.lang.String)
     */
    public MessageList collectEntityMessagesFromRepository(Entity type, String id, String className, String parentID)
            throws ResourceNotFoundException
    {

        final String PROVIDER_ATTRIBUTE_NAME = "provider";
        final String APPLICATION_ATTRIBUTE_NAME = "application";

        // Otherwise let's build a logical message file location based on the DEM hierarchy
        String messageFile;
        String name;

        name = type.toString() + " " + AbstractMessageManager.NUMBER_SIGN + id;

        ClassHolder classHolder = null;
        String applicationID = null;
        String providerID = null;
        String serviceID = null;

        if (DIFStartupConfiguration.getMessagesLocation().equals(MessagesLocation.MESSAGES_FOLDER))
        {
            messageFile = className.substring(className.lastIndexOf(".") + 1);
            messageFile += AbstractMessageManager.DEFAULT_MESSAGE_FILE_EXTENSION;

            if (type.equals(Entity.STAGE))
            {

                // Stage parent is a service, use the parentID (service) to find the application, which is the really
                // needed one
                classHolder = DEMLoaderEntityRegistry.getService(parentID.toLowerCase());
                if (classHolder != null)
                {
                    serviceID = classHolder.generateID();
                    applicationID =
                            (classHolder.getAnnotations().get(ServiceDefinition.class.getCanonicalName()).getMembers()
                                     .get(APPLICATION_ATTRIBUTE_NAME).toString()).toLowerCase();
                    classHolder = DEMLoaderEntityRegistry.getApplication(applicationID);
                    // Now use the application to fetch the provider
                    providerID = (classHolder.getAnnotations().get(ApplicationDefinition.class.getCanonicalName())
                                          .getMembers().get(PROVIDER_ATTRIBUTE_NAME).toString()).toLowerCase();

                    // id is stageID, parentID is service ID
                    messageFile = providerID + "/" + applicationID + "/" + serviceID + "/" + id + "/" + messageFile;
                }
            }
            else if (type.equals(Entity.SERVICE))
            {

                // Service parent is an application
                classHolder = DEMLoaderEntityRegistry.getApplication(parentID.toLowerCase());

                if (classHolder != null)
                {
                    applicationID = classHolder.generateID();
                    // Now use the application to fetch the provider
                    providerID = classHolder.getAnnotations().get(ApplicationDefinition.class.getCanonicalName())
                            .getMembers().get(PROVIDER_ATTRIBUTE_NAME).toString();
                    classHolder = DEMLoaderEntityRegistry.getProvider(providerID.toLowerCase());
                    providerID = classHolder.generateID();

                    // id is serviceID
                    messageFile = providerID + "/" + parentID.toLowerCase() + "/" + id + "/" + messageFile;
                }

                messageFile = parentID.toLowerCase() + "/" + id + "/" + messageFile;
            }
            else if (type.equals(Entity.PROVIDER))
            {
                messageFile = id + "/" + messageFile;
            }

            // Add the conventioned prefix
            messageFile = MESSAGES_FOLDER + messageFile;
        }
        else
        {
            messageFile = className;
            messageFile = messageFile.replace(".", "/") + AbstractMessageManager.DEFAULT_MESSAGE_FILE_EXTENSION;
        }

        if (classHolder != null || !(type.equals(Entity.SERVICE) || type.equals(Entity.STAGE)))
            return readMessageFile(messageFile, name);
        else
            return null;
    }

    /**
     * @see pt.digitalis.dif.dem.managers.IMessageManager#collectEntityMessagesFromRepository(java.lang.String)
     */
    public MessageList collectEntityMessagesFromRepository(String messagePath)
    {

        String fileName;

        // Add the conventioned location
        if (DIFStartupConfiguration.getMessagesLocation().equals(MessagesLocation.MESSAGES_FOLDER))
            fileName = MESSAGES_FOLDER + messagePath + AbstractMessageManager.DEFAULT_MESSAGE_FILE_EXTENSION;
        else
            fileName = messagePath.replace(".", "/") + AbstractMessageManager.DEFAULT_MESSAGE_FILE_EXTENSION;

        return readMessageFile(fileName, messagePath);
    }

    /**
     * @see pt.digitalis.dif.dem.managers.IMessageManager#isPersistent()
     */
    public boolean isPersistent()
    {
        return false;
    }

    /**
     * Does the actual reading of the resource file
     *
     * @param fileName   the name of the message resource file to read
     * @param entityName the name of the entity to witch we are loading the messages
     *
     * @return the read messages
     */
    protected MessageList readMessageFile(String fileName, String entityName)
    {

        MessageList messages = new MessageList();
        boolean error = false;
        String fileNameBaseDir;

        fileName = fileName.replace('/', File.separatorChar);
        fileName = fileName.replace('\\', File.separatorChar);

        if (fileName.lastIndexOf(File.separatorChar) == -1)
            error = true;

        // If there are files to process...
        if (!error)
        {
            fileNameBaseDir = fileName.substring(0, fileName.lastIndexOf(File.separator));

            try
            {
                Properties messageFileProps = new Properties();
                String language = null;

                // GetResources must always receive paths with the "/" separator.
                // For files on disk both work but for files inside JARs only "/" will work
                fileNameBaseDir = fileNameBaseDir.replace('\\', '/');

                // Get the URLs for the message files based on the default "xxx.message"
                Enumeration<URL> messageFileDirs =
                        Thread.currentThread().getContextClassLoader().getResources(fileNameBaseDir);

                if (!messageFileDirs.hasMoreElements())
                    messageFileDirs = Thread.currentThread().getContextClassLoader()
                            .getResources(fileNameBaseDir.replace('\\', '/'));

                // For each file...
                while (messageFileDirs.hasMoreElements())
                {
                    URL messageDir = messageFileDirs.nextElement();

                    try
                    {
                        for (String messageFileName : ResourceUtils.getFilesInDir(messageDir, true))
                        {

                            String messageFullFileName;
                            boolean isCorrectFile = true;

                            if (messageDir.getProtocol().equalsIgnoreCase("FILE"))
                            {
                                messageFullFileName = fileNameBaseDir + File.separator + messageFileName;
                                isCorrectFile = messageFullFileName.contains(fileName);
                            }
                            else
                            {
                                messageFullFileName = messageFileName;
                                isCorrectFile = messageFullFileName.contains(fileName.replaceAll("\\\\", "/"));
                            }

                            // If the file is for the current class. Many files will exist on the given directory
                            if (isCorrectFile)
                            {
                                if (messageDir.getProtocol().equalsIgnoreCase("FILE"))
                                {
                                    String filePath = messageDir.getPath();

                                    if (!filePath.endsWith(File.separator))
                                        filePath += File.separator;

                                    filePath += messageFileName;

                                    messageFileProps.load(new FileInputStream(
                                            new File(URLDecoder.decode(messageFileName, "UTF-8"))));
                                }
                                else
                                {
                                    // ...load the messages to the 'properties' object...
                                    InputStream is = Thread.currentThread().getContextClassLoader()
                                            .getResourceAsStream(messageFileName);
                                    if (is != null)
                                    {
                                        messageFileProps.load(is);
                                    }
                                }
                                // Infer the language from the file name
                                language = messageFileName.substring(messageFileName.lastIndexOf(".")).toLowerCase();

                                // Process the language (check if it's the default)
                                if (AbstractMessageManager.DEFAULT_MESSAGE_FILE_EXTENSION.equals(language))
                                    language = DIFGeneralConfigurationParameters.getInstance().getDefaultLanguage();
                                else
                                {
                                    language = language.substring(1);
                                    addSupportedLanguage(language);
                                }

                                // For each entry on the 'properties'....
                                for (Entry<Object, Object> entry : messageFileProps.entrySet())
                                {
                                    // ...add it as a message translation
                                    messages.addMessageTranslation(entry.getKey().toString(), language,
                                            new Message(entry.getValue().toString()));
                                }
                                messageFileProps = new Properties();
                            }
                        }
                    }
                    catch (ResourceNotFoundException e)
                    {
                        error = true;
                    }
                }
            }
            catch (IOException e)
            {
                /*
                 * Untestable in the sense that one can't have a way to explicitly raise an IOException due to
                 * InputStream errors.
                 */
                error = true;
            }
        }

        // Inform the user about message processing
        if (error)
            DIFLogger.getLogger().debug(entityName + ": No message file present.");
        else
            DIFLogger.getLogger().debug(entityName + ": message file " + fileName + " loaded");

        // Return the message list (might be empty!!)
        return messages;
    }
}