/**
 * - Digitalis Internal Framework v2.0 - (C) 2007, Digitalis Informatica. Distribuicao e Gestao de Informatica, Lda.
 * Estrada de Paco de Arcos num.9 - Piso -1 2780-666 Paco de Arcos Telefone: (351) 21 4408990 Fax: (351) 21 4408999
 * http://www.digitalis.pt
 */
package pt.digitalis.dif.dem.managers.impl;

import pt.digitalis.dif.dem.Entity;
import pt.digitalis.dif.dem.interfaces.IApplication;
import pt.digitalis.dif.dem.interfaces.IProvider;
import pt.digitalis.dif.dem.interfaces.IService;
import pt.digitalis.dif.dem.interfaces.IStage;
import pt.digitalis.dif.dem.managers.IParameterManager;
import pt.digitalis.dif.dem.objects.parameters.IParameter;
import pt.digitalis.dif.dem.objects.parameters.ParameterList;
import pt.digitalis.dif.ioc.DIFIoCRegistry;
import pt.digitalis.dif.utils.logging.DIFLogger;

import java.util.HashMap;
import java.util.Map;

/**
 * Manages the DEM parameters, providing operations for access, pooling and persistence.
 *
 * @author Pedro Viegas <a href="mailto:pviegas@digitalis.pt">pviegas@digitalis.pt</a>
 * @author Rodrigo Gonalves <a href="mailto:rgoncalves@digitalis.pt">rgoncalves@digitalis.pt</a>
 * @created 2007/06/01
 */
public class ParameterManagerImpl implements IParameterManager
{

    /** Map of application parameter lists */
    private Map<String, ParameterList> applicationParameters = new HashMap<String, ParameterList>();

    /** Map of provider parameter lists */
    private Map<String, ParameterList> providerParameters = new HashMap<String, ParameterList>();

    /** Map of service parameter lists */
    private Map<String, ParameterList> serviceParameters = new HashMap<String, ParameterList>();

    /** Map of stage parameter lists */
    private Map<String, ParameterList> stageParameters = new HashMap<String, ParameterList>();

    /** The supported class types for parameter attributes */
    private Map<String, String> supportedParameterTypes = null;

    /** Builds the map of supported parameter types and their corresponding implementations */
    private synchronized void buildSupportedParameterTypesMap()
    {
        @SuppressWarnings("rawtypes")
        Map<String, IParameter> impls = DIFIoCRegistry.getRegistry().getImplementationsMap(IParameter.class);

        supportedParameterTypes = new HashMap<String, String>();

        for (String id : impls.keySet())
        {
            IParameter<?> parameter = impls.get(id);

            for (Object className : parameter.getSupportedClasses())
            {
                supportedParameterTypes.put(className.toString(), id);
            }
        }
    }

    /**
     * @see pt.digitalis.dif.dem.managers.IParameterManager#getParameter(pt.digitalis.dif.dem.Entity,
     *         java.lang.String,
     *         java.lang.String)
     */
    public IParameter<?> getParameter(Entity entityType, String entityID, String parameterID)
    {

        ParameterList list = getParameters(entityType, entityID);

        if (list == null)
            return null;
        else
            return list.getParameter(parameterID);
    }

    /**
     * @see pt.digitalis.dif.dem.managers.IParameterManager#getParameterInstanceForType(java.lang.String)
     */
    public IParameter<?> getParameterInstanceForType(String typeClassName)
    {

        if (isParameterTypeSupported(typeClassName))
            return DIFIoCRegistry.getRegistry()
                    .getImplementation(IParameter.class, supportedParameterTypes.get(typeClassName));
        else
        {
            DIFLogger.getLogger().error("Bad usage: Parameter type \"" + typeClassName + "\" not supported: ");

            return null;
        }
    }

    /**
     * @see pt.digitalis.dif.dem.managers.IParameterManager#getParameters(pt.digitalis.dif.dem.Entity,
     *         java.lang.String)
     */
    public ParameterList getParameters(Entity entityType, String entityID)
    {
        if (entityType.equals(Entity.PROVIDER))
            return providerParameters.get(entityID);
        else if (entityType.equals(Entity.APPLICATION))
            return applicationParameters.get(entityID);
        else if (entityType.equals(Entity.SERVICE))
            return serviceParameters.get(entityID);
        else if (entityType.equals(Entity.STAGE))
            return stageParameters.get(entityID);
        else
            return null;
    }

    /**
     * @see pt.digitalis.dif.dem.managers.IParameterManager#getParameters(pt.digitalis.dif.dem.interfaces.IApplication)
     */
    public ParameterList getParameters(IApplication application)
    {
        ParameterList list = applicationParameters.get(application.getID());

        if (list == null)
        {
            list = new ParameterList();
            applicationParameters.put(application.getID(), list);
        }

        return list;
    }

    /**
     * @see pt.digitalis.dif.dem.managers.IParameterManager#getParameters(pt.digitalis.dif.dem.interfaces.IProvider)
     */
    public ParameterList getParameters(IProvider provider)
    {
        ParameterList list = providerParameters.get(provider.getID());

        if (list == null)
        {
            list = new ParameterList();
            providerParameters.put(provider.getID(), list);
        }

        return list;
    }

    /**
     * @see pt.digitalis.dif.dem.managers.IParameterManager#getParameters(pt.digitalis.dif.dem.interfaces.IService)
     */
    public ParameterList getParameters(IService service)
    {
        ParameterList list = serviceParameters.get(service.getID());

        if (list == null)
        {
            list = new ParameterList();
            serviceParameters.put(service.getID(), list);
        }

        return list;
    }

    /**
     * @see pt.digitalis.dif.dem.managers.IParameterManager#getParameters(IStage)
     */
    public ParameterList getParameters(IStage stage)
    {
        ParameterList list = stageParameters.get(stage.getID());

        if (list == null)
        {
            list = new ParameterList();
            stageParameters.put(stage.getID(), list);
        }

        return list;
    }

    /**
     * @see pt.digitalis.dif.dem.managers.IParameterManager#isParameterTypeSupported(java.lang.String)
     */
    public boolean isParameterTypeSupported(String typeClassName)
    {

        if (supportedParameterTypes == null)
            buildSupportedParameterTypesMap();

        return supportedParameterTypes.containsKey(typeClassName);
    }

    /**
     * Registers an application parameter.
     *
     * @param parameter the parameter to register
     */
    private synchronized void registerApplicationParameter(IParameter<?> parameter)
    {

        if (!applicationParameters.containsKey(parameter.getParentID()))
        {
            ParameterList parameters = new ParameterList();
            parameters.addParameter(parameter);
            applicationParameters.put(parameter.getParentID(), parameters);
        }
        else
        {
            applicationParameters.get(parameter.getParentID()).addParameter(parameter);
        }
    }

    /**
     * @see pt.digitalis.dif.dem.managers.IParameterManager#registerParameter(pt.digitalis.dif.dem.objects.parameters.IParameter)
     */
    public synchronized void registerParameter(IParameter<?> parameter)
    {

        if (parameter.getParentType() == Entity.PROVIDER)
        {
            registerProviderParameter(parameter);
        }
        else if (parameter.getParentType() == Entity.APPLICATION)
        {
            registerApplicationParameter(parameter);
        }
        else if (parameter.getParentType() == Entity.SERVICE)
        {
            registerServiceParameter(parameter);
        }
        else if (parameter.getParentType() == Entity.STAGE)
        {
            registerStageParameter(parameter);
        }

        // Map<String, ParameterList> parameterCache = null;
        //
        // // Determine the correct entity type and corresponding cache map
        // if (parameter.getParentType() == Entity.PROVIDER)
        // parameterCache = providerParameters;
        //
        // else if (parameter.getParentType() == Entity.APPLICATION)
        // parameterCache = applicationParameters;
        //
        // else if (parameter.getParentType() == Entity.SERVICE)
        // parameterCache = serviceParameters;
        //
        // else if (parameter.getParentType() == Entity.STAGE)
        // parameterCache = stageParameters;
        //
        //
        // // Get the parameter list object
        // ParameterList list = parameterCache.get(parameter.getParentID());
        //
        // // If the parameter list object is yet to be instantiated create it
        // if (list == null) {
        // list = new ParameterList();
        //
        // // Add it to the cache map
        // parameterCache.put(parameter.getParentID(), list);
        // }
        //
        // // Add the parameter to the obtained list
        // list.addParameter(parameter);
    }

    /**
     * Registers a provider parameter.
     *
     * @param parameter the parameter to register
     */
    private synchronized void registerProviderParameter(IParameter<?> parameter)
    {

        if (!providerParameters.containsKey(parameter.getParentID()))
        {
            ParameterList parameters = new ParameterList();
            parameters.addParameter(parameter);
            providerParameters.put(parameter.getParentID(), parameters);
        }
        else
        {
            providerParameters.get(parameter.getParentID()).addParameter(parameter);
        }
    }

    /**
     * Registers a service parameter.
     *
     * @param parameter the parameter to register
     */
    private synchronized void registerServiceParameter(IParameter<?> parameter)
    {

        if (!serviceParameters.containsKey(parameter.getParentID()))
        {
            ParameterList parameters = new ParameterList();
            parameters.addParameter(parameter);
            serviceParameters.put(parameter.getParentID(), parameters);
        }
        else
        {
            serviceParameters.get(parameter.getParentID()).addParameter(parameter);
        }
    }

    /**
     * Registers a stage parameter.
     *
     * @param parameter the parameter to register
     */
    private synchronized void registerStageParameter(IParameter<?> parameter)
    {

        if (!stageParameters.containsKey(parameter.getParentID()))
        {
            ParameterList parameters = new ParameterList();
            parameters.addParameter(parameter);
            stageParameters.put(parameter.getParentID(), parameters);
        }
        else
        {
            stageParameters.get(parameter.getParentID()).addParameter(parameter);
        }
    }
}
