/**
 * - Digitalis Internal Framework v2.0 - (C) 2007, Digitalis Informatica. Distribuicao e Gestao de Informatica, Lda.
 * Estrada de Paco de Arcos num.9 - Piso -1 2780-666 Paco de Arcos Telefone: (351) 21 4408990 Fax: (351) 21 4408999
 * http://www.digitalis.pt
 */
package pt.digitalis.dif.dem.managers.impl;

import com.google.inject.Inject;
import pt.digitalis.dif.dem.Entity;
import pt.digitalis.dif.dem.interfaces.IStage;
import pt.digitalis.dif.dem.managers.IRegistrationManager;
import pt.digitalis.dif.dem.objects.ILicense;
import pt.digitalis.dif.dem.objects.LicenseEditionType;
import pt.digitalis.dif.exception.manager.RegistrationManagerException;
import pt.digitalis.dif.ioc.DIFIoCRegistry;
import pt.digitalis.dif.startup.DIFGeneralConfigurationParameters;
import pt.digitalis.utils.common.StringUtils;
import pt.digitalis.utils.config.ConfigurationException;
import pt.digitalis.utils.config.IConfigurations;

import java.util.HashMap;
import java.util.Map;
import java.util.Properties;

/**
 * Manages the DEM Entities registration process.
 *
 * @author Pedro Viegas <a href="mailto:pviegas@digitalis.pt">pviegas@digitalis.pt</a>
 * @author Rodrigo Gonalves <a href="mailto:rgoncalves@digitalis.pt">rgoncalves@digitalis.pt</a>
 * @created 2007 /06/01
 */
public class RegistrationManagerImpl implements IRegistrationManager
{

    /**
     * ID of the config area where the Repository will manage the registrations
     */
    final static String REPOSITORY_CONFIG_ID = "dif2";

    /**
     * ID that identifies the key
     */
    final static String REPOSITORY_KEY_ID = "key";

    /**
     * The constant REPOSITORY_DISABLE_ID.
     */
    final static String REPOSITORY_DISABLE_ID = "disabled";

    /**
     * ID that identifies the name
     */
    final static String REPOSITORY_NAME_ID = "name";

    /**
     * ID of the section where the Repository will manage the registrations
     */
    final static String REPOSITORY_SECTION_ID = "Registrations";

    /**
     * Map of entities license editions
     */
    static private Map<String, LicenseEditionType> entitiesLicenseEditions = new HashMap<String, LicenseEditionType>();

    /**
     * Missing registrable information tag
     */
    static private String MISSING_REGISTRABLE_INFO = "DIF auto-generated: Missing registrable information!";

    /**
     * Map of registrations
     */
    static private Map<String, ILicense> registrations = new HashMap<String, ILicense>();

    /**
     * The registration persistence repository.
     */
    @Inject
    IConfigurations configRepository;

    @Override
    public void activateApplicationRegistration(String applicationID)
            throws ConfigurationException, RegistrationManagerException
    {
        activateEntityRegistration(Entity.APPLICATION, applicationID);
    }

    @Override
    public void activateEntityRegistration(Entity type, String entityID)
            throws RegistrationManagerException, ConfigurationException
    {
        String id = Entity.getID(type, entityID);
        ILicense license = getRegistrationRecord(type, entityID);

        if (!license.isRegistrable())
            throw new RegistrationManagerException(
                    "You are trying to activate the registration of an unregistrable entity");

        license.setActive(true);

        // Updates the Map with the registered record
        registrations.put(id, license);

        // Persists to the Repository
        persistRegistration(id, license);
    }

    @Override
    public void activateProviderRegistration(String providerID)
            throws ConfigurationException, RegistrationManagerException
    {
        activateEntityRegistration(Entity.PROVIDER, providerID);
    }

    @Override
    public void activateServiceRegistration(String serviceID)
            throws ConfigurationException, RegistrationManagerException
    {
        activateEntityRegistration(Entity.SERVICE, serviceID);
    }

    @Override
    public void activateStageRegistration(String stageID) throws ConfigurationException, RegistrationManagerException
    {
        activateEntityRegistration(Entity.STAGE, stageID);
    }

    /**
     * @see pt.digitalis.dif.dem.managers.IRegistrationManager#addEntityLicenseEdition(pt.digitalis.dif.dem.Entity,
     *         java.lang.String, pt.digitalis.dif.dem.objects.LicenseEditionType)
     */
    @Override
    public void addEntityLicenseEdition(Entity type, String entityID, LicenseEditionType licenseEditionType)
    {
        String id = Entity.getID(type, entityID.toLowerCase());
        entitiesLicenseEditions.put(id, licenseEditionType);
    }

    /**
     * @see pt.digitalis.dif.dem.managers.IRegistrationManager#addToRegistry(pt.digitalis.dif.dem.Entity,
     *         java.lang.String, java.lang.String, boolean)
     */
    @Override
    public void addToRegistry(Entity type, String entityID, String name, boolean registrable)
            throws ConfigurationException
    {

        String id = Entity.getID(type, entityID.toLowerCase());

        ILicense license = DIFIoCRegistry.getRegistry().getImplementation(ILicense.class);
        license.setName(name);
        license.setRegistrable(registrable);

        // Checks if there are persistent configurations in the repository
        license = checkRepositoryForPersistentRegistration(id, license);

        registrations.put(id, license);
    }

    /**
     * Checks the persistent repository for saved registrations.
     *
     * @param id      the entity id
     * @param license the default registration record
     *
     * @return the updated registration record
     *
     * @exception ConfigurationException the configuration exception
     */
    private ILicense checkRepositoryForPersistentRegistration(String id, ILicense license) throws ConfigurationException
    {

        Properties props = configRepository.readConfiguration(REPOSITORY_CONFIG_ID, REPOSITORY_SECTION_ID);

        // Tries to get the registrations from the persistence configuration
        Object savedKey = props.get(id + "." + REPOSITORY_KEY_ID);

        if (savedKey != null)
            license.register(savedKey.toString(), id);

        String persistantActive = StringUtils.toStringOrNull(props.get(id + "." + REPOSITORY_DISABLE_ID));

        if (StringUtils.isNotBlank(persistantActive))
        {
            boolean isActive = !"true".equalsIgnoreCase(persistantActive);
            license.setActive(isActive);
        }

        return license;
    }

    @Override
    public void deactivateApplicationRegistration(String applicationID)
            throws ConfigurationException, RegistrationManagerException
    {
        deactivateEntityRegistration(Entity.APPLICATION, applicationID);
    }

    @Override
    public void deactivateEntityRegistration(Entity type, String entityID)
            throws ConfigurationException, RegistrationManagerException
    {
        String id = Entity.getID(type, entityID);
        ILicense license = getRegistrationRecord(type, entityID);

        if (!license.isRegistrable())
            throw new RegistrationManagerException(
                    "You are trying to deactivate the registration of an unregistrable entity");

        license.setActive(false);

        // Updates the Map with the registered record
        registrations.put(id, license);

        // Persists to the Repository
        persistRegistration(id, license);
    }

    @Override
    public void deactivateProviderRegistration(String providerID)
            throws ConfigurationException, RegistrationManagerException
    {
        deactivateEntityRegistration(Entity.PROVIDER, providerID);
    }

    @Override
    public void deactivateServiceRegistration(String serviceID)
            throws ConfigurationException, RegistrationManagerException
    {
        deactivateEntityRegistration(Entity.SERVICE, serviceID);
    }

    @Override
    public void deactivateStageRegistration(String stageID) throws ConfigurationException, RegistrationManagerException
    {
        deactivateEntityRegistration(Entity.STAGE, stageID);
    }

    /**
     * @see pt.digitalis.dif.dem.managers.IRegistrationManager#getApplicationEdition(java.lang.String)
     */
    @Override
    public LicenseEditionType getApplicationEdition(String applicationID)
    {
        return getRegistrationRecord(Entity.APPLICATION, applicationID).getEdition();
    }

    /**
     * @see pt.digitalis.dif.dem.managers.IRegistrationManager#getProviderEdition(java.lang.String)
     */
    @Override
    public LicenseEditionType getProviderEdition(String providerID)
    {
        return getRegistrationRecord(Entity.PROVIDER, providerID).getEdition();
    }

    /**
     * Utility accessor for the registration object for any entity. It provides the search and initialization for first
     * time access. Private use only since this is a contained helper class that encapsulates all specifics about entity
     * registration.
     *
     * @param type the entity type to get
     * @param id   the id of the Entity
     *
     * @return the registration record for the entity
     */
    private ILicense getRegistrationRecord(Entity type, String id)
    {
        String entityID = Entity.getID(type, id);
        ILicense license = registrations.get(entityID);

        if (license == null)
        {
            // The DIF DEM rules state that missing registration information translates to a public registration.
            // If the CodeGen and DEM parsing utility has not provided any registration information then this is a
            // unregistrable entity. Create a new license accordingly.

            license = DIFIoCRegistry.getRegistry().getImplementation(ILicense.class);
            license.setName(MISSING_REGISTRABLE_INFO);
            license.setRegistrable(false);

            // Updates the Map with the registered record. This will only persist in memory until a register/unregister
            // command is issued. Only then it will be considered a true registration license. This one is temporary.
            registrations.put(entityID, license);
        }

        return license;
    }

    /**
     * @see pt.digitalis.dif.dem.managers.IRegistrationManager#getServiceEdition(java.lang.String)
     */
    @Override
    public LicenseEditionType getServiceEdition(String serviceID)
    {
        return getRegistrationRecord(Entity.SERVICE, serviceID).getEdition();
    }

    /**
     * @see pt.digitalis.dif.dem.managers.IRegistrationManager#getStageClientName(pt.digitalis.dif.dem.interfaces.IStage)
     */
    @Override
    public String getStageClientName(IStage stage)
    {
        String result = getRegistrationRecord(Entity.STAGE, stage.getID()).getName();
        if (MISSING_REGISTRABLE_INFO.equals(result))
        {
            result = getRegistrationRecord(Entity.SERVICE, stage.getService().getID()).getName();
            if (MISSING_REGISTRABLE_INFO.equals(result))
            {
                result = getRegistrationRecord(Entity.APPLICATION, stage.getService().getApplication().getID())
                        .getName();
                if (MISSING_REGISTRABLE_INFO.equals(result))
                {
                    result = getRegistrationRecord(Entity.PROVIDER,
                            stage.getService().getApplication().getProvider().getID()).getName();
                    if (MISSING_REGISTRABLE_INFO.equals(result))
                    {
                        result = DIFGeneralConfigurationParameters.getInstance().getClient();
                    }
                }
            }
        }
        return result;
    }

    /**
     * @see pt.digitalis.dif.dem.managers.IRegistrationManager#getStageEdition(java.lang.String)
     */
    @Override
    public LicenseEditionType getStageEdition(String stageID)
    {
        return getRegistrationRecord(Entity.STAGE, stageID).getEdition();
    }

    /**
     * @see pt.digitalis.dif.dem.managers.IRegistrationManager#isApplicationRegistered(java.lang.String)
     */
    @Override
    public boolean isApplicationRegistered(String applicationID)
    {
        ILicense license = getRegistrationRecord(Entity.APPLICATION, applicationID);

        return license.isRegistered() && license.isActive() &&
               isEntityEditionRegistered(Entity.APPLICATION, applicationID);
    }

    /**
     * @see pt.digitalis.dif.dem.managers.IRegistrationManager#isApplicationRegistrable(java.lang.String)
     */
    @Override
    public boolean isApplicationRegistrable(String applicationID)
    {
        return getRegistrationRecord(Entity.APPLICATION, applicationID).isRegistrable();
    }

    /**
     * Checks if the entity license edition is registered.
     *
     * @param type the type
     * @param id   the id
     *
     * @return true, if is entity license edition registered
     */
    private boolean isEntityEditionRegistered(Entity type, String id)
    {
        boolean result = false;
        LicenseEditionType entityEdition = LicenseEditionType.STANDARD;
        String entityID = Entity.getID(type, id);

        if (entitiesLicenseEditions.containsKey(entityID))
        {
            entityEdition = entitiesLicenseEditions.get(entityID);
        }

        if (entityEdition.equals(LicenseEditionType.STANDARD))
        {
            result = true;
        }
        else
        {
            result = entityEdition.equals(getRegistrationRecord(type, id).getEdition());
        }

        return result;
    }

    /**
     * @see pt.digitalis.dif.dem.managers.IRegistrationManager#isProviderRegistered(java.lang.String)
     */
    @Override
    public boolean isProviderRegistered(String providerID)
    {
        ILicense license = getRegistrationRecord(Entity.PROVIDER, providerID);

        return license.isRegistered() && license.isActive() && isEntityEditionRegistered(Entity.PROVIDER, providerID);
    }

    /**
     * @see pt.digitalis.dif.dem.managers.IRegistrationManager#isProviderRegistrable(java.lang.String)
     */
    @Override
    public boolean isProviderRegistrable(String providerID)
    {
        return getRegistrationRecord(Entity.PROVIDER, providerID).isRegistrable();
    }

    @Override
    public boolean isRegistrationActive(Entity entityType, String entityID)
    {
        return this.getRegistrationRecord(entityType, entityID).isActive();
    }

    /**
     * @see pt.digitalis.dif.dem.managers.IRegistrationManager#isServiceRegistered(java.lang.String)
     */
    @Override
    public boolean isServiceRegistered(String serviceID)
    {
        ILicense license = getRegistrationRecord(Entity.SERVICE, serviceID);

        return license.isRegistered() && license.isActive() && isEntityEditionRegistered(Entity.SERVICE, serviceID);
    }

    /**
     * @see pt.digitalis.dif.dem.managers.IRegistrationManager#isServiceRegistrable(java.lang.String)
     */
    @Override
    public boolean isServiceRegistrable(String serviceID)
    {
        return getRegistrationRecord(Entity.SERVICE, serviceID).isRegistrable();
    }

    /**
     * @see pt.digitalis.dif.dem.managers.IRegistrationManager#isStageRegistered(java.lang.String)
     */
    @Override
    public boolean isStageRegistered(String stageId)
    {
        ILicense license = getRegistrationRecord(Entity.STAGE, stageId);

        return license.isRegistered() && license.isActive() && isEntityEditionRegistered(Entity.STAGE, stageId);
    }

    /**
     * @see pt.digitalis.dif.dem.managers.IRegistrationManager#isStageRegistrable(java.lang.String)
     */
    @Override
    public boolean isStageRegistrable(String stageId)
    {
        return getRegistrationRecord(Entity.STAGE, stageId).isRegistrable();
    }

    /**
     * Persists the given registration to the repository
     *
     * @param id      the id of the Entity
     * @param license the registration license
     *
     * @return T if the operation was successful
     *
     * @exception ConfigurationException the configuration exception
     */
    private boolean persistRegistration(String id, ILicense license) throws ConfigurationException
    {

        Properties props = new Properties();

        props.put(id + "." + REPOSITORY_NAME_ID, license.getName() != null ? license.getName() : "");

        if (license.getKey() == null)
            props.put(id + "." + REPOSITORY_KEY_ID, "");
        else
            props.put(id + "." + REPOSITORY_KEY_ID, license.getKey());

        props.put(id + "." + REPOSITORY_DISABLE_ID, Boolean.toString(!license.isActive()));

        return configRepository.writeConfiguration(REPOSITORY_CONFIG_ID, REPOSITORY_SECTION_ID, props);
    }

    /**
     * @see pt.digitalis.dif.dem.managers.IRegistrationManager#registerApplication(java.lang.String,
     *         java.lang.String,
     *         java.lang.String)
     */
    @Override
    public boolean registerApplication(String applicationID, String name, String key)
            throws RegistrationManagerException, ConfigurationException
    {
        return registerEntity(Entity.APPLICATION, applicationID, name, key);
    }

    @Override
    public boolean registerEntity(Entity type, String entityID, String name, String key)
            throws RegistrationManagerException, ConfigurationException
    {
        String id = Entity.getID(type, entityID);
        ILicense license = getRegistrationRecord(type, entityID);

        if (!license.isRegistrable())
            throw new RegistrationManagerException("You are trying to register a unregistrable entity");

        license.setName(name);
        license.register(key, id);

        // Updates the Map with the registered record
        registrations.put(id, license);

        // Persists to the Repository
        persistRegistration(id, license);

        return license.isRegistered();
    }

    /**
     * @see pt.digitalis.dif.dem.managers.IRegistrationManager#registerProvider(java.lang.String, java.lang.String,
     *         java.lang.String)
     */
    @Override
    public boolean registerProvider(String providerID, String name, String key)
            throws RegistrationManagerException, ConfigurationException
    {
        return registerEntity(Entity.PROVIDER, providerID, name, key);
    }

    /**
     * @see pt.digitalis.dif.dem.managers.IRegistrationManager#registerService(java.lang.String, java.lang.String,
     *         java.lang.String)
     */
    @Override
    public boolean registerService(String serviceID, String name, String key)
            throws RegistrationManagerException, ConfigurationException
    {
        return registerEntity(Entity.SERVICE, serviceID, name, key);
    }

    /**
     * @see pt.digitalis.dif.dem.managers.IRegistrationManager#registerStage(java.lang.String, java.lang.String,
     *         java.lang.String)
     */
    @Override
    public boolean registerStage(String stageId, String name, String key)
            throws RegistrationManagerException, ConfigurationException
    {
        return registerEntity(Entity.STAGE, stageId, name, key);
    }

    /**
     * @see pt.digitalis.dif.dem.managers.IRegistrationManager#unregisterApplication(java.lang.String)
     */
    @Override
    public void unregisterApplication(String applicationID) throws ConfigurationException
    {
        unregisterEntity(Entity.APPLICATION, applicationID);
    }

    /**
     * Unregisters an entity
     *
     * @param type     the type of the entity
     * @param entityID the id of the entity
     *
     * @exception ConfigurationException the configuration exception
     */
    private void unregisterEntity(Entity type, String entityID) throws ConfigurationException
    {
        String id = Entity.getID(type, entityID);
        ILicense license = getRegistrationRecord(type, entityID);

        license.unregister();

        // Updates the Map with the registered record
        registrations.put(id, license);

        // Persists to the Repository
        persistRegistration(id, license);
    }

    /**
     * @see pt.digitalis.dif.dem.managers.IRegistrationManager#unregisterProvider(java.lang.String)
     */
    @Override
    public void unregisterProvider(String providerID) throws ConfigurationException
    {
        unregisterEntity(Entity.PROVIDER, providerID);
    }

    /**
     * @see pt.digitalis.dif.dem.managers.IRegistrationManager#unregisterService(java.lang.String)
     */
    @Override
    public void unregisterService(String serviceID) throws ConfigurationException
    {
        unregisterEntity(Entity.SERVICE, serviceID);
    }

    /**
     * @see pt.digitalis.dif.dem.managers.IRegistrationManager#unregisterStage(java.lang.String)
     */
    @Override
    public void unregisterStage(String stageId) throws ConfigurationException
    {
        unregisterEntity(Entity.STAGE, stageId);
    }
}
