/**
 * 2008, Digitalis Informatica. All rights reserved. Distribuicao e Gestao de Informatica, Lda. Estrada de Paco de Arcos
 * num.9 - Piso -1 2780-666 Paco de Arcos Telefone: (351) 21 4408990 Fax: (351) 21 4408999 http://www.digitalis.pt
 */

package pt.digitalis.dif.dem.managers.impl;

import pt.digitalis.dif.dem.managers.IUsageIssuesManager;
import pt.digitalis.dif.dem.objects.issues.IssueScope;
import pt.digitalis.dif.dem.objects.issues.IssueType;
import pt.digitalis.dif.dem.objects.issues.UsageIssue;
import pt.digitalis.dif.utils.logging.DIFLogger;
import pt.digitalis.utils.common.StringUtils;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * The user issue manager implementation. Singleton pattern.
 *
 * @author Rodrigo Gonalves <a href="mailto:rgoncalves@digitalis.pt">rgoncalves@digitalis.pt</a><br/>
 * @created 2008/11/13
 */
public class UsageIssuesManagerImpl implements IUsageIssuesManager
{

    /** The singleton instance. */
    final static private IUsageIssuesManager INSTANCE = new UsageIssuesManagerImpl();

    /** Flag that controls if there are issues */
    private boolean hasIssues = false;

    /** Flag that controls thr total issue count */
    private Long issueCounter = 0L;

    /** The issues map. <K=location, V=issue set> */
    private Map<String, List<UsageIssue>> issues = new HashMap<String, List<UsageIssue>>();

    /** the unique sequence counter for generating UIDs */
    private Long uidSequenceCounter = 0L;

    /** Private ctor. */
    private UsageIssuesManagerImpl()
    {
    }

    /**
     * Singleton pattern instance accessor.
     *
     * @return the instance
     */
    static public IUsageIssuesManager getInstance()
    {
        return INSTANCE;
    }

    /**
     * @see pt.digitalis.dif.dem.managers.IUsageIssuesManager#addIssue(pt.digitalis.dif.dem.objects.issues.IssueType,
     *         pt.digitalis.dif.dem.objects.issues.IssueScope, java.lang.String, java.lang.String, java.lang.Exception)
     */
    @Override
    public void addIssue(IssueType issueType, IssueScope issueScope, String location, String issueDescription,
            Exception exception)
    {

        UsageIssue issue = new UsageIssue();

        if (issueType != null)
            issue.setIssueType(issueType);

        if (issueScope != null)
            issue.setIssueScope(issueScope);

        if (location != null)
            issue.setLocation(location);

        if (issueDescription != null)
            issue.setIssueDescription(issueDescription);

        if (exception != null)
            issue.setException(exception);

        this.addIssue(issue);
    }

    /**
     * @see pt.digitalis.dif.dem.managers.IUsageIssuesManager#addIssue(pt.digitalis.dif.dem.objects.issues.UsageIssue)
     */
    @Override
    public void addIssue(UsageIssue usageIssue)
    {
        // Check if the given UID exists, if not create it
        if (!getIssues().containsKey(usageIssue.getLocation()))
            getIssues().put(usageIssue.getLocation(), new ArrayList<UsageIssue>());

        // Add the issue
        boolean addIssue = true;

        // Validate issue UID if defined. Prevent duplicate reportings...
        if (StringUtils.isNotBlank(usageIssue.getUID()))
        {
            for (UsageIssue existingIssue : getIssues().get(usageIssue.getLocation()))
            {
                if (usageIssue.getUID().equals(existingIssue.getUID()))
                {
                    addIssue = false;
                    existingIssue.incrementTotalCount();
                    break;
                }
            }
        }
        else
        {
            usageIssue.setUID(getInstance().generateUID());
        }

        if (addIssue)
            getIssues().get(usageIssue.getLocation()).add(usageIssue);

        if (!hasIssues)
            hasIssues = true;

        issueCounter++;
    }

    /**
     * @see pt.digitalis.dif.dem.managers.IUsageIssuesManager#generateUID()
     */
    @Override
    synchronized public String generateUID()
    {
        return "usageIssue" + uidSequenceCounter++;
    }

    /**
     * @see pt.digitalis.dif.dem.managers.IUsageIssuesManager#getIssueCounter()
     */
    @Override
    public Long getIssueCounter()
    {
        return issueCounter;
    }

    /**
     * @see pt.digitalis.dif.dem.managers.IUsageIssuesManager#getIssues()
     */
    @Override
    public Map<String, List<UsageIssue>> getIssues()
    {
        return this.issues;
    }

    /**
     * @see pt.digitalis.dif.dem.managers.IUsageIssuesManager#getIssues(java.lang.String)
     */
    @Override
    public List<UsageIssue> getIssues(String entityUID)
    {
        if (getIssues().containsKey(entityUID))
            return getIssues().get(entityUID);
        else
            return null;
    }

    /**
     * @see pt.digitalis.dif.dem.managers.IUsageIssuesManager#getIssuesByUID(java.lang.String)
     */
    @Override
    public List<UsageIssue> getIssuesByUID(String issueUID)
    {
        List<UsageIssue> result = new ArrayList<UsageIssue>();

        for (List<UsageIssue> issueList : this.getIssues().values())
        {
            for (UsageIssue issue : issueList)
            {
                if (issue.getUID() != null && issue.getUID().equals(issueUID))
                    result.add(issue);
            }
        }

        return result;
    }

    /**
     * @see pt.digitalis.dif.dem.managers.IUsageIssuesManager#hasIssues()
     */
    @Override
    public boolean hasIssues()
    {
        return this.hasIssues;
    }

    /**
     * @see pt.digitalis.dif.dem.managers.IUsageIssuesManager#hasIssues(boolean)
     */
    @Override
    public boolean hasIssues(boolean ignoreDiscarded)
    {
        if (!ignoreDiscarded)
            return this.hasIssues();
        else
        {
            for (List<UsageIssue> issueList : this.getIssues().values())
            {
                for (UsageIssue issue : issueList)
                {
                    if (!issue.isDiscarded())
                        return true;
                }
            }

            return false;
        }
    }

    /**
     * @see pt.digitalis.dif.dem.managers.IUsageIssuesManager#hasIssues(java.lang.String)
     */
    @Override
    public boolean hasIssues(String location)
    {
        if (hasIssues())
        {
            // One issue is "issues"
            if (this.getIssues().get(location) != null && this.getIssues().get(location).size() > 0)
                return true;
        }
        return false;
    }

    /**
     * @see pt.digitalis.dif.dem.managers.IUsageIssuesManager#logAllIssuesByEntity()
     */
    @Override
    public void logAllIssuesByEntity()
    {
        // Log only if there's data
        if (getIssues().size() > 0)
        {
            for (String uid : getIssues().keySet())
            {
                logIssues(uid);
            }
        }
    }

    /**
     * @see pt.digitalis.dif.dem.managers.IUsageIssuesManager#logAllIssuesByType()
     */
    @Override
    public void logAllIssuesByType()
    {
        if (getIssues().size() > 0)
        {
            List<UsageIssue> errorIssues = new ArrayList<UsageIssue>();
            List<UsageIssue> warnIssues = new ArrayList<UsageIssue>();
            List<UsageIssue> infoIssues = new ArrayList<UsageIssue>();

            for (String uid : getIssues().keySet())
            {
                for (UsageIssue issue : getIssues(uid))
                {
                    if (issue.getIssueType().equals(IssueType.ERROR))
                        errorIssues.add(issue);
                    else if (issue.getIssueType().equals(IssueType.WARN))
                        warnIssues.add(issue);
                    else if (issue.getIssueType().equals(IssueType.INFO))
                        infoIssues.add(issue);
                }
            }

            if (errorIssues.size() > 0)
                for (UsageIssue issue : errorIssues)
                {
                    DIFLogger.getLogger().error(issue.toString());
                }

            if (warnIssues.size() > 0)
                for (UsageIssue issue : warnIssues)
                {
                    DIFLogger.getLogger().warn(issue.toString());
                }

            if (infoIssues.size() > 0)
                for (UsageIssue issue : infoIssues)
                {
                    DIFLogger.getLogger().info(issue.toString());
                }
        }
    }

    /**
     * @see pt.digitalis.dif.dem.managers.IUsageIssuesManager#logIssues(java.lang.String)
     */
    @Override
    public void logIssues(String entityUID)
    {
        // Log only if there's data
        if (getIssues(entityUID).size() > 0)
        {
            for (UsageIssue issue : getIssues(entityUID))
            {
                if (issue.getIssueType().equals(IssueType.ERROR))
                {
                    DIFLogger.getLogger().error(issue.toString());
                }
                else if (issue.getIssueType().equals(IssueType.WARN))
                {
                    DIFLogger.getLogger().warn(issue.toString());
                }
                else if (issue.getIssueType().equals(IssueType.INFO))
                {
                    DIFLogger.getLogger().info(issue.toString());
                }
            }
        }
    }
}
