/**
 * 2007, Digitalis Informatica. All rights reserved. Distribuicao e Gestao de Informatica, Lda. Estrada de Paco de Arcos
 * num.9 - Piso -1 2780-666 Paco de Arcos Telefone: (351) 21 4408990 Fax: (351) 21 4408999 http://www.digitalis.pt
 */
package pt.digitalis.dif.dem.objects.parameters;

import pt.digitalis.dif.controller.interfaces.IDIFContext;
import pt.digitalis.dif.dem.Entity;
import pt.digitalis.dif.dem.interfaces.IStageInstance;
import pt.digitalis.dif.dem.objects.parameters.constraints.IParameterConstraint;
import pt.digitalis.dif.dem.objects.parameters.errors.ParameterErrorList;
import pt.digitalis.dif.dem.objects.parameters.rules.IParameterRule;
import pt.digitalis.dif.dem.objects.parameters.validators.IParameterValidator;
import pt.digitalis.dif.exception.objects.ParameterException;
import pt.digitalis.dif.utils.extensions.document.DocumentRepositoryEntry;
import pt.digitalis.utils.config.ConfigurationException;

import java.math.BigDecimal;
import java.util.Date;
import java.util.List;
import java.util.Map;

// TODO: Auto-generated Javadoc

/**
 * This class will define a Parameter.
 * <p>
 * It will hold information relative to the Parameter value, ID key and validation constraints.
 *
 * @param <T> generic type of the parameter
 *
 * @author Pedro Viegas <a href="mailto:pviegas@digitalis.pt">pviegas@digitalis.pt</a>
 * @created Nov 23, 2007
 */
public interface IParameter<T> extends Cloneable
{

    /**
     * Creates a new instance of the current object.
     *
     * @return {@link IParameter} clone
     *
     * @exception CloneNotSupportedException the clone not supported exception
     */
    public IParameter<T> clone() throws CloneNotSupportedException;

    /**
     * Parameter Initializer. Since IoC provides the construction this should be called after TODO: Refactor this
     * method!! This is a bad coding practice! Methods should have between 3 and 7 arguments. Nine is too much. Use
     * encapsulation.
     *
     * @param allowAnonymous                        if a USER scoped parameter should allow anonymous access
     * @param constraints                           the string that defines the constraints to apply
     * @param defaultValue                          the parameter default value
     * @param formLinked                            the form name of null if not set
     * @param id                                    the parameter id
     * @param parameterScope                        the scope of the parameter
     * @param parameterContext                      the parameter context
     * @param parentID                              the id of the parent entity
     * @param parentType                            the parent entity type
     * @param persistToRepository                   if the parameter value should be persisted to the repository
     * @param referencedInARuleFromAnotherParameter the new referencedInARuleFromAnotherParameter value to set
     * @param required                              the required
     * @param formConfigurableType                  the form configurable type
     * @param rules                                 the parameter rules
     * @param validators                            a validator list to validate the parameter values
     * @param value                                 the value
     */
    public void forceInitialize(boolean allowAnonymous, Map<String, IParameterConstraint> constraints, T defaultValue,
            String formLinked, String id, ParameterScope parameterScope, ParameterContext parameterContext,
            String parentID, Entity parentType, boolean persistToRepository,
            boolean referencedInARuleFromAnotherParameter, boolean required,
            FormConfigurableDefinition formConfigurableType, List<IParameterRule<T>> rules,
            Map<String, IParameterValidator> validators, T value);

    /**
     * Gets the constraints.
     *
     * @return the constraints
     */
    public Map<String, IParameterConstraint> getConstraints();

    /**
     * Gets the form configurable definition.
     *
     * @return the form configurable definition
     */
    public FormConfigurableDefinition getFormConfigurableDefinition();

    /**
     * Gets the form linked.
     *
     * @return the form name of null if not set
     */
    public String getFormLinked();

    /**
     * Sets the form linked.
     *
     * @param formLinked the new form linked
     */
    public void setFormLinked(String formLinked);

    /**
     * Gets the id.
     *
     * @return the id
     */
    public String getId();

    /**
     * Gets the parameter scope.
     *
     * @return the persistScope
     */
    public ParameterScope getParameterScope();

    /**
     * Gets the parent ID.
     *
     * @return the parentID
     */
    public String getParentID();

    /**
     * Gets the parent type.
     *
     * @return the parentType
     */
    public Entity getParentType();

    /**
     * Inspector for the 'rules' attribute.
     *
     * @return the rules value
     */
    public List<IParameterRule<T>> getRules();

    /**
     * Gets the supported classes.
     *
     * @return the list of supported class types for the parameter value
     */
    public List<String> getSupportedClasses();

    /**
     * Gets the validators.
     *
     * @return the validators
     */
    public Map<String, IParameterValidator> getValidators();

    /**
     * Returns the parameter value. This method will check a number of possible sources for the value of the parameter
     * depending on it's scope and attributes.
     *
     * @param context the DIF context to read if needed
     *
     * @return the value
     *
     * @exception ParameterException if the parameter could not be read
     */
    public T getValue(IDIFContext context) throws ParameterException;

    /**
     * Gets the value as big decimal.
     *
     * @param context the DIF context to read if needed
     *
     * @return the value of the parameter as a Double
     *
     * @exception ParameterException if a conversion error occurs
     */
    public BigDecimal getValueAsBigDecimal(IDIFContext context) throws ParameterException;

    /**
     * Gets the value as boolean.
     *
     * @param context the DIF context to read if needed
     *
     * @return the value of the parameter as a boolean
     *
     * @exception ParameterException if the parameter value cannot be read
     */
    public boolean getValueAsBoolean(IDIFContext context) throws ParameterException;

    /**
     * Gets the value as date.
     *
     * @param context the DIF context to read if needed
     *
     * @return the value of the parameter as a Date
     *
     * @exception ParameterException if a conversion error occurs
     */
    public Date getValueAsDate(IDIFContext context) throws ParameterException;

    /**
     * Returns the parameter value as a Document Note: Only Document types parameters will correctly respond to this
     * method. All others will report an invalid usage {@link ParameterException}
     *
     * @param context the DIF context to read if needed
     *
     * @return the string representation of the parameter value
     *
     * @exception ParameterException if the parameter value cannot be read
     */
    public DocumentRepositoryEntry getValueAsDocument(IDIFContext context) throws ParameterException;

    /**
     * Gets the value as double.
     *
     * @param context the DIF context to read if needed
     *
     * @return the value of the parameter as a Double
     *
     * @exception ParameterException if a conversion error occurs
     */
    public Double getValueAsDouble(IDIFContext context) throws ParameterException;

    /**
     * Gets the value as long.
     *
     * @param context the DIF context to read if needed
     *
     * @return the value of the parameter as an Long
     *
     * @exception ParameterException if a conversion error occurs
     */
    public Long getValueAsLong(IDIFContext context) throws ParameterException;

    /**
     * Returns the parameter value as a String.
     *
     * @param context the DIF context to read if needed
     *
     * @return the string representation of the parameter value
     *
     * @exception ParameterException if the parameter value cannot be read
     */
    public String getValueAsString(IDIFContext context) throws ParameterException;

    /**
     * Parameter Initializer. Since IoC provides the construction this should be called after TODO: Refactor this
     * method!! This is a bad coding practice! Methods should have between 3 and 7 arguments. Nine is too much. Use
     * encapsulation.
     *
     * @param id                   the parameter id
     * @param parentType           the parent entity type
     * @param parentID             the id of the parent entity
     * @param formConfigurableType the form configurable type
     * @param persistToRepository  if the parameter value should be persisted to the repository
     * @param allowAnonymousAccess if a USER scoped parameter should allow anonymous access
     * @param parameterScope       the scope of the parameter
     * @param defaultValue         the parameter default value
     * @param constraintDefinition the string that defines the constraints to apply
     * @param validators           a validator list to validate the parameter values
     * @param rules                the parameter rules
     *
     * @exception ParameterException     if the default value can't be converted to the inner parameter type
     * @exception ConfigurationException
     */
    public void initialize(String id, Entity parentType, String parentID,
            FormConfigurableDefinition formConfigurableType, boolean persistToRepository, boolean allowAnonymousAccess,
            ParameterScope parameterScope, String defaultValue, String constraintDefinition,
            Map<String, IParameterValidator> validators, List<IParameterRule<T>> rules)
            throws ParameterException, ConfigurationException;

    /**
     * Checks if is allow anonymous.
     *
     * @return the allowAnonymous
     */
    public boolean isAllowAnonymous();

    /**
     * Inspector for the 'formConfigurable' attribute.
     *
     * @return the formConfigurable value
     */
    public boolean isFormConfigurable();

    /**
     * Sets the form configurable.
     *
     * @param formConfigurableType the new form configurable
     */
    public void setFormConfigurable(FormConfigurableDefinition formConfigurableType);

    /**
     * Checks if is numeric.
     *
     * @return T if the current parameter is of a numeric type
     */
    public boolean isNumeric();

    /**
     * Checks if is persist to repository.
     *
     * @return the persistToRepository
     */
    public boolean isPersistToRepository();

    /**
     * Inspector for the 'referencedInARuleFromAnotherParameter' attribute.
     *
     * @return the referencedInARuleFromAnotherParameter value
     */
    public boolean isReferencedInARuleFromAnotherParameter();

    /**
     * Modifier for the 'referencedInARuleFromAnotherParameter' attribute.
     *
     * @param referencedInARuleFromAnotherParameter the new referencedInARuleFromAnotherParameter value to set
     */
    public void setReferencedInARuleFromAnotherParameter(boolean referencedInARuleFromAnotherParameter);

    /**
     * Checks if is required.
     *
     * @return the required
     */
    public boolean isRequired();

    /**
     * Checks if is string setter supported.
     *
     * @return T if we can set the parameter with a String value representation
     */
    public boolean isStringSetterSupported();

    /**
     * Updates the value of the parameter from the relevant sources. This method will check a number of possible sources
     * for the value of the parameter depending on it's scope and attributes.
     *
     * @param stageInstance the DIF stage to read if needed
     *
     * @return return the error message if the validation fails
     *
     * @exception ConfigurationException
     */
    public ParameterErrorList refreshParameterValue(IStageInstance stageInstance) throws ConfigurationException;

    /**
     * Sets the value of the parameter to its default and runs all constraints and validations.
     *
     * @param stageInstance the requested stage
     *
     * @return return the error messages if the validation fails
     *
     * @exception ConfigurationException
     */
    public ParameterErrorList setDefaultValue(IStageInstance stageInstance) throws ConfigurationException;

    /**
     * Sets the value of the parameter to its default and runs all constraints and validations.
     *
     * @param stageInstance            the requested stage
     * @param initializationInProgress T if called within the dif parameter initialization
     *
     * @return return the error messages if the validation fails
     *
     * @exception ConfigurationException
     */
    public ParameterErrorList setDefaultValue(IStageInstance stageInstance, boolean initializationInProgress)
            throws ConfigurationException;

    /**
     * Sets the value of the parameter and runs all constraints and validations.
     *
     * @param value         the value to set
     * @param stageInstance the DIF stage to read if needed. WARNING: If null is interpreted like an initialization and
     *                      no scope
     *                      repositories will be accessed/updated.
     *
     * @return return the error messages if the validation fails
     *
     * @exception ConfigurationException
     */
    public ParameterErrorList setValue(T value, IStageInstance stageInstance) throws ConfigurationException;

    /**
     * Sets the value of the parameter and runs all constraints and validations.
     *
     * @param value                    the value to set
     * @param stageInstance            the DIF stage to read if needed. WARNING: If null is interpreted like an
     *                                 initialization and no scope
     *                                 repositories will be accessed/updated.
     * @param initializationInProgress T if called within the dif parameter initialization
     *
     * @return return the error messages if the validation fails
     *
     * @exception ConfigurationException
     */
    public ParameterErrorList setValue(T value, IStageInstance stageInstance, boolean initializationInProgress)
            throws ConfigurationException;

    /**
     * Sets the value of the parameter converting if from a string representation and runs all constraints and
     * validations.
     *
     * @param value         the value to set
     * @param stageInstance the DIF stage to read if needed.
     *
     * @return return the error messages if the validation fails
     *
     * @exception ConfigurationException
     */
    public ParameterErrorList setValueFromString(String value, IStageInstance stageInstance)
            throws ConfigurationException;

    /**
     * Sets the value of the parameter converting if from a string representation and runs all constraints and
     * validations.
     *
     * @param value                    the value to set
     * @param stageInstance            the DIF stage to read if needed.
     * @param initializationInProgress T if called within the dif parameter initialization
     *
     * @return return the error messages if the validation fails
     *
     * @exception ConfigurationException
     */
    public ParameterErrorList setValueFromString(String value, IStageInstance stageInstance,
            boolean initializationInProgress) throws ConfigurationException;
}
