/**
 * 2007, Digitalis Informatica. All rights reserved. Distribuicao e Gestao de Informatica, Lda. Estrada de Paco de Arcos
 * num.9 - Piso -1 2780-666 Paco de Arcos Telefone: (351) 21 4408990 Fax: (351) 21 4408999 http://www.digitalis.pt
 */
package pt.digitalis.dif.dem.objects.parameters.constraints.impl.date;

import pt.digitalis.dif.dem.interfaces.IStageInstance;
import pt.digitalis.dif.dem.objects.parameters.constraints.AbstractParameterConstraint;
import pt.digitalis.utils.common.DateUtils;

import java.text.ParseException;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * Date parameter constraint implementation
 *
 * @author Pedro Viegas <a href="mailto:pviegas@digitalis.pt">pviegas@digitalis.pt</a>
 * @created Nov 19, 2007
 */
public class ParameterConstraintDateImpl extends AbstractParameterConstraint
{

    /** Supported classes by numeric constraints */
    @SuppressWarnings("serial")
    final static private List<String> suportedClasses = new ArrayList<String>()
    {

        {
            add(Date.class.getCanonicalName());
        }
    };

    /** Last tested value */
    String lastTestedValue;

    /**
     * @see pt.digitalis.dif.dem.objects.parameters.constraints.IParameterConstraint#configureConstraint(java.lang.String)
     */
    public void configureConstraint(String configurationString)
    {
        // Nothing to do. No parameters to parse.
    }

    /**
     * @see pt.digitalis.dif.dem.objects.parameters.constraints.AbstractParameterConstraint#getErrorMessageValues(boolean)
     */
    @Override
    protected Map<String, String> getErrorMessageValues(boolean parseValues)
    {
        Map<String, String> substitutions = new HashMap<String, String>();

        if (parseValues)
        {
            substitutions.put("value", lastTestedValue);
        }

        return substitutions;
    }

    /**
     * @see pt.digitalis.dif.dem.objects.parameters.constraints.IParameterConstraint#getJavaScriptValidationCondition()
     */
    public String getJavaScriptValidationCondition()
    {

        // Applied the d/m/Y H with the + ' 12' for the value. This is because a bug in the javascript
        // specification causing for example a date '02/04/1978' not being recognized. The result will be
        // 02/04/1978 01:00 when the system expects 02/04/1978 00:00. This occurrs when the daylight hour changes
        // from western time to summer time in some dates, not all
        return "Ext.Date.parse(value + ' 12','d/m/Y H', true) != undefined";
    }

    /**
     * @see pt.digitalis.dif.dem.objects.parameters.constraints.IParameterConstraint#isSupportedClass(java.lang.Class)
     */
    public boolean isSupportedClass(Class<?> clazz)
    {
        return suportedClasses.contains(clazz.getCanonicalName());
    }

    /**
     * @see pt.digitalis.dif.dem.objects.parameters.constraints.AbstractParameterConstraint#validateConstraint(java.lang.Object,
     *         pt.digitalis.dif.dem.interfaces.IStageInstance)
     */
    @Override
    public boolean validateConstraint(Object value, IStageInstance stageInstance)
    {

        if (value == null)
            return this.validateConstraint((String) null, stageInstance);
        else if (value instanceof Date)
            return true;
        else if (value instanceof String)
            return this.validateConstraint(value.toString(), stageInstance);
        else
            return false;
    }

    /**
     * @see pt.digitalis.dif.dem.objects.parameters.constraints.IParameterConstraint#validateConstraint(java.lang.String,
     *         pt.digitalis.dif.dem.interfaces.IStageInstance)
     */
    public boolean validateConstraint(String value, IStageInstance stageInstance)
    {
        lastTestedValue = value;

        if (value == null)
            return true;
        else
            try
            {
                DateUtils.stringToSimpleDate(value);
                return true;
            }
            catch (ParseException e)
            {
                return false;
            }
            catch (NumberFormatException e)
            {
                return false;
            }
    }
}
