/**
 * 2007, Digitalis Informatica. All rights reserved. Distribuicao e Gestao de Informatica, Lda. Estrada de Paco de Arcos
 * num.9 - Piso -1 2780-666 Paco de Arcos Telefone: (351) 21 4408990 Fax: (351) 21 4408999 http://www.digitalis.pt
 */
package pt.digitalis.dif.dem.objects.parameters.errors;

import pt.digitalis.dif.controller.http.HTTPConstants;
import pt.digitalis.dif.controller.interfaces.IDIFContext;
import pt.digitalis.dif.dem.interfaces.ICustomFormDefinition;
import pt.digitalis.dif.dem.interfaces.IStageInstance;
import pt.digitalis.dif.dem.managers.ICustomFormManager;
import pt.digitalis.dif.dem.managers.IParameterManager;
import pt.digitalis.dif.dem.objects.FeatureState;
import pt.digitalis.dif.dem.objects.FormFieldCustomization;
import pt.digitalis.dif.dem.objects.parameters.IEditableParameter;
import pt.digitalis.dif.dem.objects.parameters.IParameter;
import pt.digitalis.dif.dem.objects.parameters.types.AbstractParameter;
import pt.digitalis.dif.exception.objects.ParameterException;
import pt.digitalis.dif.ioc.DIFIoCRegistry;
import pt.digitalis.dif.utils.IObjectFormatter;
import pt.digitalis.dif.utils.ObjectFormatter;
import pt.digitalis.dif.utils.ObjectFormatter.Format;
import pt.digitalis.dif.utils.extensions.ICaptcha;
import pt.digitalis.utils.common.StringUtils;
import pt.digitalis.utils.config.ConfigurationException;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;

/**
 * A report object of any parameter errors that have occurred in the initialization of a stage. These will be injected
 * into a stage attribute for evaluation by the stage as the developer sees fit.
 *
 * @author Pedro Viegas <a href="mailto:pviegas@digitalis.pt">pviegas@digitalis.pt</a>
 * @author Rodrigo Gon�alves <a href="mailto:rgoncalves@digitalis.pt">rgoncalves@digitalis.pt</a>
 */
public class ParameterErrors implements IObjectFormatter
{

    /** the captcha implementation */
    private static ICaptcha captcha = null;

    /** A Map of all errors by parameter name. */
    private final Map<String, ParameterErrorList> errors = new HashMap<String, ParameterErrorList>();

    /** the current stage instance. */
    private final IStageInstance stageInstance;

    /**
     * Liswt of parameters that the errors must be ignored, since a form customization (or any other feature that so
     * desires) has instructed the form to ignore these fields
     */
    List<String> parametersTodiscardErrors = new ArrayList<String>();

    /**
     * keeps track of all refreshed parameter called by the refreshParameter method. this is used to know what
     * parameters must still be initialized for declared custom parameters
     */
    private List<String> refreshedParameters = new ArrayList<String>();

    /**
     * Constructor.
     *
     * @param stageInstance the current stage
     */
    public ParameterErrors(IStageInstance stageInstance)
    {
        this.stageInstance = stageInstance;
    }

    /**
     * Adds a new parameter to.
     *
     * @param parameterId the parameter to add the error to
     * @param error       the error
     *
     * @exception ParameterException if the parameter does not exist
     */
    public void addParameterError(String parameterId, ParameterError error) throws ParameterException
    {
        // Must ignore errors for parameters that the form configuration has determined as invisible!
        if (!parametersTodiscardErrors.contains(parameterId))
        {
            if (parameterId != null)
                parameterId = parameterId.toLowerCase();

            ParameterErrorList errorsForParameter = getErrorsForParameter(parameterId);

            if (errorsForParameter == null)
            {
                IParameter<?> parameter = null;
                Object parameterValue = null;

                if (parameterId != null && !"".equals(parameterId))
                {
                    // Error for a specific parameter
                    parameter = DIFIoCRegistry.getRegistry().getImplementation(IParameterManager.class)
                            .getParameters(stageInstance).getParameter(parameterId);

                    if (parameter == null)
                        // Changed so that custom stage instance parameters are picked up here...
                        parameter = stageInstance.getParameters().getStageParameters().getParameter(parameterId);

                    if (parameter == null)
                        throw new ParameterException(
                                "Parameter \"" + parameterId + "\" does not exist in the stage \"" +
                                stageInstance.getID() + "\" context.", parameter);

                    try
                    {
                        parameterValue = parameter.getValue(stageInstance.getContext());
                    }
                    catch (ParameterException e)
                    {
                        // Nothing to do!
                    }
                }

                errorsForParameter = new ParameterErrorList(parameter, parameterValue);
                errors.put(parameterId, errorsForParameter);
            }

            errorsForParameter.addError(error);
        }
    }

    /**
     * Discards all errors.
     */
    public void discardAllErrors()
    {
        errors.clear();
        stageInstance.setParameterErrors(this);
    }

    /**
     * Discards all errors for a specific parameter.
     *
     * @param parameterId the parameter to discard errors
     */
    public void discardErrors(String parameterId)
    {
        errors.remove(parameterId.toLowerCase());
        stageInstance.setParameterErrors(this);
    }

    /**
     * Discards all errors for a specific parameter.
     *
     * @param type the parameter type of errors to discard
     */
    public void discardErrorsOfType(ParameterErrorType type)
    {
        for (Entry<String, ParameterErrorList> listEntry : errors.entrySet())
        {

            ParameterErrorList value = listEntry.getValue();
            List<ParameterError> previousErrorList = value.getErrorList();
            List<ParameterError> newErrorList = new ArrayList<ParameterError>();

            for (ParameterError error : previousErrorList)
            {
                if (!error.getErrorType().equals(type))
                    newErrorList.add(error);
            }

            value.setErrorList(newErrorList);
            errors.put(listEntry.getKey(), value);
        }

        stageInstance.setParameterErrors(this);
    }

    /**
     * Get all errors for all parameters.
     *
     * @return the errors
     */
    public Map<String, ParameterErrorList> getAllParameterErrors()
    {
        return errors;
    }

    /**
     * Retrieves the errors for a given parameter ID.
     *
     * @param parameterID the desired parameter id
     *
     * @return the parameter list
     */
    public ParameterErrorList getErrorsForParameter(String parameterID)
    {
        if (parameterID == null)
            return null;
        else
            return errors.get(parameterID.toLowerCase());
    }

    /**
     * Checks if errors exist.
     *
     * @return T if there are errors
     */
    public boolean hasErrors()
    {
        for (ParameterErrorList list : errors.values())
        {
            if (list.getErrorList().size() > 0)
                return true;
        }

        return false;
    }

    /**
     * Checks if errors exist.
     *
     * @param formName the form name
     *
     * @return T if there are errors
     */
    public boolean hasErrorsForForm(String formName)
    {
        for (ParameterErrorList list : errors.values())
        {
            if (list.getParameter() != null && formName.equalsIgnoreCase(list.getParameter().getFormLinked()) &&
                list.getErrorList() != null && list.getErrorList().size() > 0)
                return true;
        }

        return false;
    }

    /**
     * Checks if errors exist. Will ignore all errors related with partial submit of a form. Required fields not
     * submited. But validating errors in the fields that were filled.
     *
     * @return T if there are errors
     */
    public boolean hasErrorsIgnoreParcialSubmitErrors()
    {
        for (ParameterErrorList list : errors.values())
        {
            for (ParameterError error : list.getErrorList())

            // If a non MISSING error is found, there are non-partioal-submit errors
            {
                if (!ParameterErrorType.MISSING.equals(error.getErrorType()))
                    return true;
            }
        }

        return false;
    }

    /**
     * Refreshes the passed parameter and adds errors to the list if they are detected.
     *
     * @param param         the parameter to refresh
     * @param stageInstance the stage of the refresh request
     *
     * @return the refreshed value
     *
     * @exception ConfigurationException
     */
    public Object refreshParameter(IParameter<?> param, IStageInstance stageInstance) throws ConfigurationException
    {
        Object result = null;
        this.refreshedParameters.add(param.getId().toLowerCase());

        if (stageInstance != null)
        {
            IDIFContext context = stageInstance.getContext();

            // Get custom form configuration if present
            ICustomFormDefinition customFormDef = null;
            FormFieldCustomization customFormFieldDef = null;
            Object currentFormNameObj = context.getRequest().getParameter(HTTPConstants.FORM_SUBMIT_STAGE);
            boolean stageContainsForm = context.getStage()
                    .equalsIgnoreCase(currentFormNameObj == null ? null : currentFormNameObj.toString());

            boolean parameterIsFromForm = false;

            String paramFormName = param.getFormLinked();
            String formSubmitName = (String) context.getRequest().getParameter(HTTPConstants.FORM_SUBMIT_NAME);
            String formConfigID =
                    (String) context.getRequest().getParameter(HTTPConstants.FORM_SUBMIT__CONFIG_BUSINESS_ID);

            if (paramFormName != null && formSubmitName != null)
                parameterIsFromForm = paramFormName.equals(formSubmitName);

            // Form parameter that has a configuration active (as submited by the form)
            if (parameterIsFromForm && param.isFormConfigurable())
            {
                customFormDef = DIFIoCRegistry.getRegistry().getImplementation(ICustomFormManager.class)
                        .getConfiguration(stageInstance, formConfigID);

                if (customFormDef != null)
                {
                    customFormFieldDef = customFormDef.getCustomizedParameters().get(param.getId().toLowerCase());

                    if (customFormFieldDef != null && customFormFieldDef.getMandatory() != FeatureState.DEFAULT)
                        ((IEditableParameter) param).setRequired(customFormFieldDef.getMandatory() == FeatureState.ON);
                }
            }

            ParameterErrorList list;

            // Refresh the value and collect any errors that may occur
            list = param.refreshParameterValue(stageInstance);

            try
            {
                result = param.getValue(context);
            }
            catch (ParameterException e)
            {
                ParameterError error = new ParameterError(e.getMessage(), ParameterErrorType.OTHER);
                list.addError(error);
            }

            // Form parameter that has a configuration active (as submited by the form)
            if (customFormDef != null && customFormDef.getExcludedParameters().contains(param.getId()))
            {
                list.getErrorList().clear();
                parametersTodiscardErrors.add(param.getId());
            }

            if (list.getErrorList().size() > 0)
            {
                // If the parameter is not link to any form or is linked to form that was submitted, add errors if they
                // exist
                if (param.getFormLinked() == null || stageContainsForm && parameterIsFromForm)
                {
                    errors.put(param.getId(), list);
                }
                else
                {
                    // Search reported error to see which errors force reporting
                    ParameterErrorList forcedErrors =
                            new ParameterErrorList(list.getParameter(), list.getAttemptedValue());

                    for (ParameterError error : list.getErrorList())
                    {
                        if (error.isForceShowError())
                            forcedErrors.addError(error);
                    }

                    if (forcedErrors.getErrorList().size() > 0)
                    {
                        errors.put(param.getId(), forcedErrors);
                    }
                }
            }
        }

        return result;
    }

    /**
     * Refreshes all remaining parameters and adds errors to the list if they are detected.
     *
     * @param stageInstance the stage of the refresh request
     *
     * @exception ParameterException
     * @exception ConfigurationException
     */
    public void refreshParameters(IStageInstance stageInstance) throws ParameterException, ConfigurationException
    {
        for (IParameter<?> parameter : stageInstance.getParameters().getStageParameters().getParameters().values())
        {
            if (!this.refreshedParameters.contains(parameter.getId().toLowerCase()))
                this.refreshParameter(parameter, stageInstance);
        }

        if (stageInstance.isFeatureEnabled(ICaptcha.CAPTCHA_PRESENT))
        {
            // Has captcha, must validate it for the given form is any was submited
            Object submitStage = stageInstance.getContext().getRequest().getParameter(HTTPConstants.FORM_SUBMIT_STAGE);
            Object submitForm = stageInstance.getContext().getRequest().getParameter(HTTPConstants.FORM_SUBMIT_NAME);

            if (submitStage != null && stageInstance.getID().equalsIgnoreCase(submitStage.toString()) &&
                submitForm != null &&
                stageInstance.isFeatureEnabled(ICaptcha.CAPTCHA_PRESENT + ":" + submitForm.toString()))
            {
                if (captcha == null)
                    captcha = DIFIoCRegistry.getRegistry().getImplementation(ICaptcha.class);

                if (!captcha.isCaptchaValueValid(stageInstance.getContext()))
                {

                    if (StringUtils.isBlank(captcha.getCaptchaSubmitedValue(stageInstance.getContext())))
                    {
                        if (this.getErrorsForParameter(ICaptcha.CAPTCHA_INPUT_ID).getErrorList().isEmpty())
                            this.addParameterError(ICaptcha.CAPTCHA_INPUT_ID, new ParameterError(
                                    AbstractParameter.getMessages()
                                            .getMessages(stageInstance.getContext().getLanguage()).get("required"),
                                    ParameterErrorType.MISSING));
                    }
                    else
                        this.addParameterError(ICaptcha.CAPTCHA_INPUT_ID, new ParameterError(
                                AbstractParameter.getMessages().getMessages(stageInstance.getContext().getLanguage())
                                        .get("captchaInvalid"), ParameterErrorType.OTHER));
                }
            }
        }
    }

    @Override
    public ObjectFormatter toObjectFormatter(Format format, List<Object> dumpedObjects)
    {
        ObjectFormatter formatter = new ObjectFormatter(format, dumpedObjects);
        formatter.addItem("Errors", errors);

        return formatter;
    }

    @Override
    public String toString()
    {
        return toObjectFormatter(Format.TEXT, null).getFormatedObject();
    }
}
