/**
 * 2008, Digitalis Informatica. All rights reserved. Distribuicao e Gestao de Informatica, Lda. Estrada de Paco de Arcos
 * num.9 - Piso -1 2780-666 Paco de Arcos Telefone: (351) 21 4408990 Fax: (351) 21 4408999 http://www.digitalis.pt
 */

package pt.digitalis.dif.dem.objects.parameters.rules;

import pt.digitalis.dif.controller.http.HTTPConstants;
import pt.digitalis.dif.dem.annotations.AnnotationTags;
import pt.digitalis.dif.dem.interfaces.ICustomFormDefinition;
import pt.digitalis.dif.dem.interfaces.IStageInstance;
import pt.digitalis.dif.dem.managers.ICustomFormManager;
import pt.digitalis.dif.dem.managers.IMessageManager;
import pt.digitalis.dif.dem.objects.FeatureState;
import pt.digitalis.dif.dem.objects.FormFieldCustomization;
import pt.digitalis.dif.dem.objects.parameters.IEditableParameter;
import pt.digitalis.dif.dem.objects.parameters.IParameter;
import pt.digitalis.dif.exception.objects.ParameterException;
import pt.digitalis.dif.ioc.DIFIoCRegistry;
import pt.digitalis.dif.utils.IObjectFormatter;
import pt.digitalis.dif.utils.ObjectFormatter;
import pt.digitalis.dif.utils.ObjectFormatter.Format;
import pt.digitalis.utils.common.StringUtils;
import pt.digitalis.utils.config.ConfigurationException;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Map;

/**
 * A base implementation for all parameter rules
 *
 * @param <T> the parameter base type
 *
 * @author Pedro Viegas <a href="mailto:pviegas@digitalis.pt">pviegas@digitalis.pt</a><br/>
 * @created Mar 31, 2009
 */
public abstract class AbstractParameterRule<T> implements IParameterRule<T>, IObjectFormatter
{

    /** The parameter that this rule is associated to */
    protected String parameterID;

    /** The rule action */
    private ParameterRuleAction action;

    /**
     *
     */
    private FormFieldCustomization customFormFieldDef = null;

    /** The description for end value */
    private String descriptionEndValue = null;

    /** The description for start value */
    private String descriptionStartValue = null;

    /** The description value */
    private String descriptionValue = null;

    /** The end value for validations */
    private String endValue = null;

    /** the rule parameters */
    private List<String> parameters = new ArrayList<String>();

    /** The start value for validations */
    private String startValue = null;

    /** The value for validation */
    private String value = null;

    /**
     * @see pt.digitalis.dif.dem.objects.parameters.rules.IParameterRule#getAction()
     */
    @Override
    public ParameterRuleAction getAction()
    {
        return action;
    }

    /**
     * @see pt.digitalis.dif.dem.objects.parameters.rules.IParameterRule#getDescriptionEndValue()
     */
    @Override
    public String getDescriptionEndValue()
    {
        return descriptionEndValue;
    }

    /**
     * @see pt.digitalis.dif.dem.objects.parameters.rules.IParameterRule#setDescriptionEndValue(java.lang.String)
     */
    @Override
    public void setDescriptionEndValue(String descriptionEndValue)
    {
        this.descriptionEndValue = descriptionEndValue;
    }

    /**
     * @see pt.digitalis.dif.dem.objects.parameters.rules.IParameterRule#getDescriptionStartValue()
     */
    @Override
    public String getDescriptionStartValue()
    {
        return descriptionStartValue;
    }

    /**
     * @see pt.digitalis.dif.dem.objects.parameters.rules.IParameterRule#setDescriptionStartValue(java.lang.String)
     */
    @Override
    public void setDescriptionStartValue(String descriptionStartValue)
    {
        this.descriptionStartValue = descriptionStartValue;
    }

    /**
     * @see pt.digitalis.dif.dem.objects.parameters.rules.IParameterRule#getDescriptionValue()
     */
    @Override
    public String getDescriptionValue()
    {
        return descriptionValue;
    }

    /**
     * @see pt.digitalis.dif.dem.objects.parameters.rules.IParameterRule#setDescriptionValue(java.lang.String)
     */
    @Override
    public void setDescriptionValue(String descriptionValue)
    {
        this.descriptionValue = descriptionValue;
    }

    /**
     * @see pt.digitalis.dif.dem.objects.parameters.rules.IParameterRule#getEndValue()
     */
    @Override
    public String getEndValue()
    {
        return endValue;
    }

    /**
     * @see pt.digitalis.dif.dem.objects.parameters.rules.IParameterRule#setEndValue(java.lang.String)
     */
    @Override
    public AbstractParameterRule<T> setEndValue(String endValue)
    {
        this.endValue = endValue;

        return this;
    }

    /**
     * Returns the list of messages available for the parameter rule
     *
     * @param stage the stage instance that the parameter is associated to
     *
     * @return the parameter rule Messages
     */
    protected Map<String, String> getMessages(IStageInstance stage)
    {
        return DIFIoCRegistry.getRegistry().getImplementation(IMessageManager.class)
                .collectEntityMessagesFromRepository(this.getClass()).getMessages(stage.getContext().getLanguage());
    }

    /**
     * Inspector for the 'parameter' attribute.
     *
     * @param stage the stage instance that the parameter is associated to
     *
     * @return the parameter value
     *
     * @exception ParameterException if the parameter does not exist
     */
    public IParameter<T> getParameter(IStageInstance stage) throws ParameterException
    {
        return (IParameter<T>) stage.getParameters().getAllAvailableParameters().getParameter(parameterID);
    }

    /**
     * Get's the name of a given parameter from the stage messages. When missing the parameter ID is returned (camel
     * case interpreted)
     *
     * @param stage       the parameter's current stage
     * @param parameterID the parameter ID to fetch the name
     *
     * @return the parameter name
     */
    public String getParameterName(IStageInstance stage, String parameterID)
    {
        String message = stage.getMessage(parameterID);

        if (message == null)
            message = StringUtils.camelCaseToString(parameterID);

        return message;
    }

    /**
     * @see pt.digitalis.dif.dem.objects.parameters.rules.IParameterRule#getParameters()
     */
    @Override
    public List<String> getParameters()
    {
        return parameters;
    }

    /**
     * @see pt.digitalis.dif.dem.objects.parameters.rules.IParameterRule#getStartValue()
     */
    @Override
    public String getStartValue()
    {
        return startValue;
    }

    /**
     * @see pt.digitalis.dif.dem.objects.parameters.rules.IParameterRule#setStartValue(java.lang.String)
     */
    @Override
    public AbstractParameterRule<T> setStartValue(String startValue)
    {
        this.startValue = startValue;

        return this;
    }

    /**
     * @see pt.digitalis.dif.dem.objects.parameters.rules.IParameterRule#getValue()
     */
    @Override
    public String getValue()
    {
        return value;
    }

    /**
     * @see pt.digitalis.dif.dem.objects.parameters.rules.IParameterRule#setValue(java.lang.String)
     */
    @Override
    public AbstractParameterRule<T> setValue(String value)
    {
        this.value = value;

        return this;
    }

    /**
     * @see pt.digitalis.dif.dem.objects.parameters.rules.IParameterRule#init(java.lang.String, java.lang.String,
     *         pt.digitalis.dif.dem.objects.parameters.rules.ParameterRuleAction, java.lang.String, java.lang.String,
     *         java.lang.String)
     */
    @Override
    public AbstractParameterRule<T> init(String parameterID, String parameters, ParameterRuleAction action,
            String value, String first, String last)
    {
        this.parameterID = parameterID;
        this.action = action;

        if (parameters != null && !"".equals(parameters))
        {
            List<String> parametersPassed = Arrays.asList(parameters.split(","));

            for (String parameterPassed : parametersPassed)
            {
                this.parameters.add(parameterPassed.trim().toLowerCase());
            }
        }

        if (value != null && !value.contains(AnnotationTags.NONE.toString()) && value.contains(":"))
        {
            String[] splitCommaValues = value.split(",");
            for (String v : splitCommaValues)
            {
                String[] splitValue = v.split(":");
                if (this.value != null)
                {
                    this.value += ",";
                }
                else
                {
                    this.value = "";
                }
                this.value += splitValue[0];

                if (this.descriptionValue != null)
                {
                    this.descriptionValue += ",";
                }
                else
                {
                    this.descriptionValue = "";
                }
                this.descriptionValue += splitValue[1];
            }
        }
        else
        {
            this.value = value;
        }

        if (first != null && !first.contains(AnnotationTags.NONE.toString()) && first.contains(":"))
        {
            String[] splitValue = first.split(":");
            this.startValue = splitValue[0];
            this.descriptionStartValue = splitValue[1];
        }
        else
        {
            this.startValue = first;
        }

        if (last != null && !last.contains(AnnotationTags.NONE.toString()) && last.contains(":"))
        {
            String[] splitValue = last.split(":");
            this.endValue = splitValue[0];
            this.descriptionEndValue = splitValue[1];
        }
        else
        {
            this.endValue = last;
        }

        return this;
    }

    /**
     * Refreshes a given associated parameter if needed (depended on possible custom form configurations and if the
     * initialization is in progress
     *
     * @param stage                    the current parameter state
     * @param param                    the associated parameter
     * @param initializationInProgress
     *
     * @return T if this parameter's initialization should be skipped (possible since the form customization can have it
     *         configured and excluded)
     *
     * @exception ParameterException
     * @exception ConfigurationException
     */
    protected boolean refreshAssociatedParam(IStageInstance stage, IParameter param, boolean initializationInProgress)
            throws ParameterException, ConfigurationException
    {
        // Get custom form configuration if present
        ICustomFormDefinition customFormDef = null;
        String formSubmitName = (String) stage.getContext().getRequest().getParameter(HTTPConstants.FORM_SUBMIT_NAME);
        String formConfigID =
                (String) stage.getContext().getRequest().getParameter(HTTPConstants.FORM_SUBMIT__CONFIG_BUSINESS_ID);
        boolean hasCheckedForCustomForm = false;
        boolean skipValidation = false;

        if (param == null)
            throw new ParameterException("Parameter does not exist: " + parameterID, null, null);

        boolean parameterIsFromForm = false;
        String paramFormName = param.getFormLinked();

        if (paramFormName != null && formSubmitName != null)
            parameterIsFromForm = paramFormName.equals(formSubmitName);

        // Form parameter that has a configuration active (as submitted by the form)
        if (parameterIsFromForm && param.isFormConfigurable())
        {
            if (!hasCheckedForCustomForm)
            {
                customFormDef = DIFIoCRegistry.getRegistry().getImplementation(ICustomFormManager.class)
                        .getConfiguration(stage, formConfigID);
                hasCheckedForCustomForm = true;
            }

            if (customFormDef != null)
            {
                customFormFieldDef = customFormDef.getCustomizedParameters().get(param.getId().toLowerCase());

                if (customFormFieldDef != null)
                {
                    if (customFormFieldDef.getMandatory() != FeatureState.DEFAULT)
                        ((IEditableParameter) param).setRequired(customFormFieldDef.getMandatory() == FeatureState.ON);

                    if (customFormDef.getExcludedParameters().contains(param.getId()))
                        skipValidation = true;
                }
            }
        }

        if (!skipValidation)
        {
            if (initializationInProgress)
                param.refreshParameterValue(stage);
        }

        return skipValidation;
    }

    @Override
    public ObjectFormatter toObjectFormatter(Format format, List<Object> dumpedObjects)
    {
        ObjectFormatter formatter = new ObjectFormatter(format, dumpedObjects);
        formatter.addItem("Action", this.action);
        formatter.addItem("End Value", this.endValue);
        formatter.addItem("Parameter ID", this.parameterID);
        formatter.addItem("Parameters", this.parameters);
        formatter.addItem("Start Value", this.startValue);
        formatter.addItem("Value", this.value);
        formatter.addItem("Description Value", this.descriptionValue);
        formatter.addItem("Description Start Value", this.descriptionStartValue);
        formatter.addItem("Description End Value", this.descriptionEndValue);

        return formatter;
    }

    @Override
    public String toString()
    {
        return toObjectFormatter(Format.TEXT, null).getFormatedObject();
    }
}
