/**
 * - Digitalis Internal Framework v2.0 - (C) 2007, Digitalis Informatica. Distribuicao e Gestao de Informatica, Lda.
 * Estrada de Paco de Arcos num.9 - Piso -1 2780-666 Paco de Arcos Telefone: (351) 21 4408990 Fax: (351) 21 4408999
 * http://www.digitalis.pt
 */
package pt.digitalis.dif.dem.objects.parameters.types;

import pt.digitalis.dif.controller.http.HTTPConstants;
import pt.digitalis.dif.controller.interfaces.IDIFContext;
import pt.digitalis.dif.controller.interfaces.IDIFSession;
import pt.digitalis.dif.controller.security.objects.IDIFUser;
import pt.digitalis.dif.dem.Entity;
import pt.digitalis.dif.dem.annotations.parameter.Rule;
import pt.digitalis.dif.dem.interfaces.IStageInstance;
import pt.digitalis.dif.dem.managers.IMessageManager;
import pt.digitalis.dif.dem.objects.messages.MessageList;
import pt.digitalis.dif.dem.objects.parameters.FormConfigurableDefinition;
import pt.digitalis.dif.dem.objects.parameters.IEditableParameter;
import pt.digitalis.dif.dem.objects.parameters.IParameter;
import pt.digitalis.dif.dem.objects.parameters.ParameterContext;
import pt.digitalis.dif.dem.objects.parameters.ParameterScope;
import pt.digitalis.dif.dem.objects.parameters.constraints.IParameterConstraint;
import pt.digitalis.dif.dem.objects.parameters.constraints.ParameterConstraintResult;
import pt.digitalis.dif.dem.objects.parameters.errors.ParameterError;
import pt.digitalis.dif.dem.objects.parameters.errors.ParameterErrorList;
import pt.digitalis.dif.dem.objects.parameters.errors.ParameterErrorType;
import pt.digitalis.dif.dem.objects.parameters.rules.IParameterRule;
import pt.digitalis.dif.dem.objects.parameters.rules.ParameterRuleResult;
import pt.digitalis.dif.dem.objects.parameters.validators.IParameterValidator;
import pt.digitalis.dif.exception.objects.ParameterException;
import pt.digitalis.dif.ioc.DIFIoCRegistry;
import pt.digitalis.dif.startup.DIFGeneralConfigurationParameters;
import pt.digitalis.dif.utils.IObjectFormatter;
import pt.digitalis.dif.utils.ObjectFormatter;
import pt.digitalis.dif.utils.ObjectFormatter.Format;
import pt.digitalis.dif.utils.extensions.document.DocumentRepositoryEntry;
import pt.digitalis.dif.utils.logging.DIFLogger;
import pt.digitalis.log.ILogWrapper;
import pt.digitalis.utils.common.StringUtils;
import pt.digitalis.utils.config.ConfigurationException;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;

/**
 * This class will define a Parameter.
 * <p>
 * It will hold information relative to the Parameter value, ID key and validation constraints.
 *
 * @param <T> generic type of the parameter
 *
 * @author Pedro Viegas <a href="mailto:pviegas@digitalis.pt">pviegas@digitalis.pt</a><br/>
 * @author Rodrigo Gonalves <a href="mailto:rgoncalves@digitalis.pt">rgoncalves@digitalis.pt</a>
 * @created 2007 /05/04
 */
abstract public class AbstractParameter<T> implements IParameter<T>, IEditableParameter, IObjectFormatter
{

    /** The Constant DEFAULT_PARAMETER_CONTEXT. */
    private final static ParameterContext DEFAULT_PARAMETER_CONTEXT = ParameterContext.SERVICE;

    /** The message manager to get class messages */
    static IMessageManager messageManager;

    /** Logger for specific logging needs */
    static private ILogWrapper logger = DIFLogger.getLogger();

    /** Messages cache for all constraints */
    static private MessageList messages;

    /** If the USER scoped parameter can be accessed by anonymous access without reporting an error */
    private boolean allowAnonymous = false;

    /** A list of constraints to validate on parameter setting */
    private Map<String, IParameterConstraint> constraints = new HashMap<String, IParameterConstraint>();

    /** The parameter default value */
    private T defaultValue;

    /**
     * used to determine if it is the first run for static parameters first time only initialization (default value
     * setting)
     */
    private Boolean firstInitialization = true;

    /** if T allows the form parameter to be configured (thus declaring the form as configurable) */
    private FormConfigurableDefinition formConfigurableDef = null;

    /** The form the parameter is linked to if available */
    private String formLinked = null;

    /** The id of the parameter */
    private String id;

    /** The parameter context. */
    private ParameterContext parameterContext;

    /** The parameter scope */
    private ParameterScope parameterScope;

    /** The id of the parent entity that owns the parameter */
    private String parentID;

    /** The entity type of the parent entity that owns the parameter */
    private Entity parentType;

    /** Persist to the persistent repository. Persist between JVM restarts */
    private Boolean persistToRepository = false;

    /** If this parameter is referenced in a {@link Rule} used in another {@link IParameter} */
    private Boolean referencedInARuleFromAnotherParameter = false;

    /** If the parameter must always be passed a value. Only for REQUEST scoped parameters */
    private Boolean required = false;

    /** The parameter rules */
    private List<IParameterRule<T>> rules = new ArrayList<IParameterRule<T>>();

    /** A list of associated validators to validate on parameter setting */
    private Map<String, IParameterValidator> validators = new HashMap<String, IParameterValidator>();

    /** The parameter current value */
    private T value;

    /**
     * Get's the Message Manager from the IoC
     *
     * @return the message manager instance
     */
    static private IMessageManager getMessageManager()
    {
        if (messageManager == null)
            messageManager = DIFIoCRegistry.getRegistry().getImplementation(IMessageManager.class);

        return messageManager;
    }

    /**
     * Lazy loading getter of messages
     *
     * @return the messages
     */
    public static MessageList getMessages()
    {
        // If the messages have not yet been loaded do it now
        if (messages == null)
        {
            messages = getMessageManager().collectEntityMessagesFromRepository(AbstractParameter.class);
        }

        return messages;
    }

    /**
     * Modifier for the 'rules' attribute.
     *
     * @param rule the new rule to add
     */
    public void addRule(IParameterRule<T> rule)
    {
        this.rules.add(rule);
    }

    /**
     * To override in each parameter type class as needed
     *
     * @return the list of automatic constraints that each parameter type adds
     */
    protected String automaticConstraints()
    {
        return "";
    }

    /**
     * Clone parameter.
     *
     * @return the parameter
     *
     * @exception CloneNotSupportedException the clone not supported exception
     * @see java.lang.Object#clone() java.lang.Object#clone()
     */
    @Override
    public IParameter<T> clone() throws CloneNotSupportedException
    {
        try
        {
            IParameter<T> parameter;

            parameter = this.getClass().newInstance();
            parameter
                    .forceInitialize(this.allowAnonymous, this.constraints, this.defaultValue, this.formLinked, this.id,
                            this.parameterScope, this.parameterContext, this.parentID, this.parentType,
                            this.persistToRepository, this.referencedInARuleFromAnotherParameter, this.required,
                            this.formConfigurableDef, this.rules, this.validators, this.value);

            return parameter;
        }
        catch (InstantiationException e)
        {
            throw new CloneNotSupportedException(e.getMessage());
        }
        catch (IllegalAccessException e)
        {
            throw new CloneNotSupportedException(e.getMessage());
        }
    }

    /**
     * Converts an object to strong format, according to any rules necessary in conversion of the given parameter type
     *
     * @param obj the object to convert
     *
     * @return the object in a string representation
     */
    protected String convertObjectToString(Object obj)
    {
        return StringUtils.toStringOrNull(obj);
    }

    /**
     * Force initialize.
     *
     * @param allowAnonymous                        the allow anonymous
     * @param constraints                           the constraints
     * @param defaultValue                          the default value
     * @param formLinked                            the form linked
     * @param id                                    the id
     * @param parameterScope                        the parameter scope
     * @param parameterContext                      the parameter context
     * @param parentID                              the parent id
     * @param parentType                            the parent type
     * @param persistToRepository                   the persist to repository
     * @param referencedInARuleFromAnotherParameter the referenced in a rule from another parameter
     * @param required                              the required
     * @param formConfigurableType                  the form configurable type
     * @param rules                                 the rules
     * @param validators                            the validators
     * @param value                                 the value
     *
     * @see pt.digitalis.dif.dem.objects.parameters.IParameter#forceInitialize(boolean, java.util.Map,
     *         java.lang.Object,
     *         java.lang.String, java.lang.String, pt.digitalis.dif.dem.objects.parameters.ParameterScope,
     *         pt.digitalis.dif.dem.objects.parameters.ParameterContext, java.lang.String, pt.digitalis.dif.dem.Entity,
     *         boolean, boolean, boolean, pt.digitalis.dif.dem.objects.parameters.FormConfigurableDefinition,
     *         java.util.List, java.util.Map, java.lang.Object) pt.digitalis.dif.dem.objects.parameters.IParameter#forceInitialize(boolean,
     *         java.util.Map, java.lang.Object,
     *         java.lang.String, java.lang.String, pt.digitalis.dif.dem.objects.parameters.ParameterScope,
     *         pt.digitalis.dif.dem.objects.parameters.ParameterContext, java.lang.String, pt.digitalis.dif.dem.Entity,
     *         boolean, boolean, boolean, pt.digitalis.dif.dem.objects.parameters.FormConfigurableDefinition,
     *         java.util.List, java.util.Map, java.lang.Object)
     */
    @Override
    public void forceInitialize(boolean allowAnonymous, Map<String, IParameterConstraint> constraints, T defaultValue,
            String formLinked, String id, ParameterScope parameterScope, ParameterContext parameterContext,
            String parentID, Entity parentType, boolean persistToRepository,
            boolean referencedInARuleFromAnotherParameter, boolean required,
            FormConfigurableDefinition formConfigurableType, List<IParameterRule<T>> rules,
            Map<String, IParameterValidator> validators, T value)
    {
        this.allowAnonymous = allowAnonymous;
        this.defaultValue = defaultValue;
        this.formLinked = formLinked;
        this.id = id;
        this.parameterScope = parameterScope;
        this.parameterContext = parameterContext;
        this.parentID = parentID;
        this.parentType = parentType;
        this.persistToRepository = persistToRepository;
        this.referencedInARuleFromAnotherParameter = referencedInARuleFromAnotherParameter;
        this.required = required;
        this.formConfigurableDef = formConfigurableType;

        this.constraints = new HashMap<String, IParameterConstraint>();
        this.constraints.putAll(constraints);

        this.rules = new ArrayList<IParameterRule<T>>();
        this.rules.addAll(rules);

        this.validators = new HashMap<String, IParameterValidator>();
        this.validators.putAll(validators);

        this.value = value;
    }

    /**
     * Gets constraints.
     *
     * @return the constraints
     */
    @Override
    public Map<String, IParameterConstraint> getConstraints()
    {
        return constraints;
    }

    /**
     * Return a common debug prefix
     *
     * @return the prefix
     */
    private String getDebugPrefix()
    {
        return "[" + getParameterScope().toString() + " parameter \"" + getId() + "\"] ";
    }

    /**
     * Gets form configurable definition.
     *
     * @return the form configurable definition
     *
     * @see pt.digitalis.dif.dem.objects.parameters.IParameter#getFormConfigurableDefinition()
     *         pt.digitalis.dif.dem.objects.parameters.IParameter#getFormConfigurableDefinition()
     */
    @Override
    public FormConfigurableDefinition getFormConfigurableDefinition()
    {
        return formConfigurableDef;
    }

    /**
     * Gets form linked.
     *
     * @return the form linked
     *
     * @see pt.digitalis.dif.dem.objects.parameters.IParameter#getFormLinked() pt.digitalis.dif.dem.objects.parameters.IParameter#getFormLinked()
     */
    @Override
    public String getFormLinked()
    {
        return formLinked;
    }

    /**
     * Sets form linked.
     *
     * @param formLinked the form linked
     *
     * @see pt.digitalis.dif.dem.objects.parameters.IParameter#setFormLinked(java.lang.String)
     *         pt.digitalis.dif.dem.objects.parameters.IParameter#setFormLinked(java.lang.String)
     */
    @Override
    public void setFormLinked(String formLinked)
    {
        this.formLinked = formLinked;
    }

    /**
     * Gets id.
     *
     * @return the id
     */
    @Override
    public String getId()
    {
        return id;
    }

    /**
     * Return the language from the context of the default if not available
     *
     * @param context the context
     *
     * @return the language
     */
    protected String getLanguage(IDIFContext context)
    {
        // If a session is available get the current language, else, the default will be used
        if (context != null && context.getSession() != null && context.getSession().getLanguage() != null)
        {
            return context.getSession().getLanguage().toLowerCase();
        }
        else
            return DIFGeneralConfigurationParameters.getInstance().getDefaultLanguage();
    }

    /**
     * Inspector for the 'parameterContext' attribute.
     *
     * @return the parameterContext value
     */
    public ParameterContext getParameterContext()
    {
        return parameterContext;
    }

    /**
     * Modifier for the 'parameterContext' attribute.
     *
     * @param parameterContext the new parameterContext value to set
     */
    @Override
    public void setParameterContext(ParameterContext parameterContext)
    {
        this.parameterContext = parameterContext;
    }

    /**
     * Gets parameter scope.
     *
     * @return the parameterScope
     */
    @Override
    public ParameterScope getParameterScope()
    {
        return parameterScope;
    }

    /**
     * Sets parameter scope.
     *
     * @param parameterScope the parameter scope
     *
     * @see pt.digitalis.dif.dem.objects.parameters.IEditableParameter#setParameterScope(pt.digitalis.dif.dem.objects.parameters.ParameterScope)
     *         pt.digitalis.dif.dem.objects.parameters.IEditableParameter#setParameterScope(pt.digitalis.dif.dem.objects.parameters.ParameterScope)
     */
    @Override
    public void setParameterScope(ParameterScope parameterScope)
    {
        this.parameterScope = parameterScope;
    }

    /**
     * Gets the parameter session id.
     *
     * @param parameterContext the parameter context
     * @param stageInstance    the stage instance
     *
     * @return the parameter session id
     */
    private String getParameterSessionId(ParameterContext parameterContext, IStageInstance stageInstance)
    {
        String result = "ParameterSessionId_" + this.getId();
        ParameterContext paramCtx = parameterContext;

        if (paramCtx == null)
        {
            paramCtx = DEFAULT_PARAMETER_CONTEXT;
        }

        if (!ParameterContext.GLOBAL.equals(paramCtx))
        {
            if (ParameterContext.APPLICATION.equals(paramCtx))
            {
                result = stageInstance.getService().getApplication().getID() + "_" + result;
            }
            else if (ParameterContext.SERVICE.equals(paramCtx))
            {
                result = stageInstance.getService().getID() + "_" + result;
            }
            else if (ParameterContext.STAGE.equals(paramCtx))
            {
                result = stageInstance.getID() + "_" + result;
            }
        }

        result = stageInstance.callParameterContextPrefix(stageInstance.getContext().getRequest(), this, result);

        return result;
    }

    /**
     * Gets parent id.
     *
     * @return the parentID
     */
    @Override
    public String getParentID()
    {
        return parentID;
    }

    /**
     * Gets parent type.
     *
     * @return the parentType
     */
    @Override
    public Entity getParentType()
    {
        return parentType;
    }

    /**
     * Gets rules.
     *
     * @return the rules
     *
     * @see pt.digitalis.dif.dem.objects.parameters.IParameter#getRules() pt.digitalis.dif.dem.objects.parameters.IParameter#getRules()
     */
    @Override
    public List<IParameterRule<T>> getRules()
    {
        return rules;
    }

    /**
     * Gets validators.
     *
     * @return the validators
     */
    @Override
    public Map<String, IParameterValidator> getValidators()
    {
        return validators;
    }

    /**
     * Gets value.
     *
     * @param context the context
     *
     * @return the value
     *
     * @exception ParameterException the parameter exception
     * @see pt.digitalis.dif.dem.objects.parameters.IParameter#getValue(IDIFContext)
     *         pt.digitalis.dif.dem.objects.parameters.IParameter#getValue(IDIFContext)
     */
    @Override
    public T getValue(IDIFContext context) throws ParameterException
    {

        if (parameterScope == ParameterScope.SESSION)
        {
            if (context == null || context.getSession() == null)
            {
                String message;

                if (context == null)
                    message = getMessages().getMessages().get("noSession");
                else
                    message = getMessages().getMessages(context.getLanguage()).get("noSession");

                throw new ParameterException(getDebugPrefix() + message, this);
            }

            return (T) context.getSession()
                    .getAttribute(this.getParameterSessionId(parameterContext, context.getStageInstance()));
        }
        else if (parameterScope == ParameterScope.USER)
        {
            if (context == null || context.getSession() == null || context.getSession().getUser() == null)
                if (allowAnonymous)
                {
                    logWarning(
                            "Can't process a USER scoped parameter without a user authenticated in session. Returning null.");
                    return null;
                }
                else
                {
                    String language =
                            (context == null ? DIFGeneralConfigurationParameters.getInstance().getDefaultLanguage()
                                             : context.getLanguage());

                    throw new ParameterException(getDebugPrefix() + getMessages().getMessages(language).get("noUser"),
                            this);
                }

            // FIXME: This will fail since the value is not kept through all user instances
            return (T) context.getSession().getUser().getParameter(getId());
        }
        else
            return this.value;
    }

    /**
     * Gets value as document.
     *
     * @param context the context
     *
     * @return the value as document
     *
     * @exception ParameterException the parameter exception
     * @see pt.digitalis.dif.dem.objects.parameters.IParameter#getValueAsDocument(pt.digitalis.dif.controller.interfaces.IDIFContext)
     *         pt.digitalis.dif.dem.objects.parameters.IParameter#getValueAsDocument(pt.digitalis.dif.controller.interfaces.IDIFContext)
     */
    @Override
    public DocumentRepositoryEntry getValueAsDocument(IDIFContext context) throws ParameterException
    {
        // Normally a parameter does not implement this mathod...
        throw new ParameterException(getMessages().getMessages(context.getLanguage()).get("invalidUsage"), this);
    }

    /**
     * Gets value as string.
     *
     * @param context the context
     *
     * @return the value as string
     *
     * @exception ParameterException the parameter exception
     * @see pt.digitalis.dif.dem.objects.parameters.IParameter#getValueAsString(IDIFContext)
     *         pt.digitalis.dif.dem.objects.parameters.IParameter#getValueAsString(IDIFContext)
     */
    @Override
    public String getValueAsString(IDIFContext context) throws ParameterException
    {
        T currentValue = getValue(context);

        if (currentValue == null)
            return null;
        else
            return currentValue.toString();
    }

    /**
     * Has form been submited boolean.
     *
     * @param context the current stage context
     *
     * @return T if the linked form has been submitted
     */
    protected boolean hasFormBeenSubmited(IDIFContext context)
    {
        if (getFormLinked() == null)
            return false;
        else
        {

            Object submitStage = context.getRequest().getParameter(HTTPConstants.FORM_SUBMIT_STAGE);
            Object submitForm = context.getRequest().getParameter(HTTPConstants.FORM_SUBMIT_NAME);

            return (submitStage != null && submitForm != null && context.getStage().equals(submitStage) &&
                    getFormLinked().equals(submitForm.toString()));
        }
    }

    /**
     * Has value boolean.
     *
     * @return T when the parameter has a value set
     */
    protected boolean hasValue()
    {
        return this.value != null;
    }

    /**
     * Initialize.
     *
     * @param id                   the id
     * @param parentType           the parent type
     * @param parentID             the parent id
     * @param formConfigurableType the form configurable type
     * @param persistToRepository  the persist to repository
     * @param allowAnonymousAccess the allow anonymous access
     * @param parameterScope       the parameter scope
     * @param defaultValue         the default value
     * @param constraintDefinition the constraint definition
     * @param validators           the validators
     * @param rules                the rules
     *
     * @exception ParameterException     the parameter exception
     * @exception ConfigurationException the configuration exception
     * @see pt.digitalis.dif.dem.objects.parameters.IParameter#initialize(java.lang.String,
     *         pt.digitalis.dif.dem.Entity,
     *         java.lang.String, pt.digitalis.dif.dem.objects.parameters.FormConfigurableDefinition, boolean, boolean,
     *         pt.digitalis.dif.dem.objects.parameters.ParameterScope, java.lang.String, java.lang.String,
     *         java.util.Map,
     *         java.util.List) pt.digitalis.dif.dem.objects.parameters.IParameter#initialize(java.lang.String,
     *         pt.digitalis.dif.dem.Entity,
     *         java.lang.String, pt.digitalis.dif.dem.objects.parameters.FormConfigurableDefinition, boolean, boolean,
     *         pt.digitalis.dif.dem.objects.parameters.ParameterScope, java.lang.String, java.lang.String,
     *         java.util.Map,
     *         java.util.List)
     */
    @Override
    public void initialize(String id, Entity parentType, String parentID,
            FormConfigurableDefinition formConfigurableType, boolean persistToRepository, boolean allowAnonymousAccess,
            ParameterScope parameterScope, String defaultValue, String constraintDefinition,
            Map<String, IParameterValidator> validators, List<IParameterRule<T>> rules)
            throws ParameterException, ConfigurationException
    {

        this.id = id.toLowerCase();
        this.parentID = parentID.toLowerCase();
        this.parentType = parentType;
        this.allowAnonymous = allowAnonymousAccess;
        this.persistToRepository = persistToRepository;
        this.parameterScope = parameterScope;
        this.formConfigurableDef = formConfigurableType;

        if (validators == null)
            this.validators = new HashMap<String, IParameterValidator>();
        else
            this.validators = validators;

        if (rules == null)
            this.rules = new ArrayList<IParameterRule<T>>();
        else
            this.rules = rules;

        this.constraints = new HashMap<String, IParameterConstraint>();

        // Add automatic constraints
        if (!"".equals(automaticConstraints()))
        {
            if (constraintDefinition == null || "".equals(constraintDefinition))
                constraintDefinition = automaticConstraints();
            else
                constraintDefinition = automaticConstraints() + "," + constraintDefinition;
        }

        // If there are constraints defined...
        if (constraintDefinition != null && !"".equals(constraintDefinition))
        {

            String[] constraintStrings = constraintDefinition.split(",");

            // For each declared constraint
            for (String constraintString : constraintStrings)
            {
                String constraintID;
                int posEqual = constraintString.indexOf("=");

                constraintID = (posEqual == -1 ? constraintString : constraintString.substring(0, posEqual));
                constraintID = constraintID.trim().toLowerCase();

                if (constraintID.equals("required"))
                    // Required only for REQUEST parameters
                    this.required = true;
                else
                {
                    IParameterConstraint constraint =
                            DIFIoCRegistry.getRegistry().getImplementation(IParameterConstraint.class, constraintID);

                    if (constraint == null)
                    {
                        // Bad constraint name. Report error
                        throw new ParameterException("Inexistant constraint \"" + constraintID + "\"", this);
                    }
                    else
                    {
                        constraint.configureConstraint(constraintString);
                        constraint.setParameter(this);
                        constraints.put(constraintID, constraint);
                    }
                }
            }
        }

        if (defaultValue != null && !"".equals(defaultValue))
        {
            setValueFromString(defaultValue, null);
            this.defaultValue = value;
        }

        if (persistToRepository)
        {
            // TODO: Implement persistence. Should read the value if available from the repository.
        }
    }

    /**
     * Is allow anonymous boolean.
     *
     * @return the boolean
     *
     * @see pt.digitalis.dif.dem.objects.parameters.IParameter#isAllowAnonymous() pt.digitalis.dif.dem.objects.parameters.IParameter#isAllowAnonymous()
     */
    @Override
    public boolean isAllowAnonymous()
    {
        return allowAnonymous;
    }

    /**
     * Inspector for the 'firstInitialization' attribute.
     *
     * @return the firstInitialization value
     */
    public boolean isFirstInitialization()
    {
        return firstInitialization;
    }

    /**
     * Modifier for the 'firstInitialization' attribute.
     *
     * @param firstInitialization the new firstInitialization value to set
     */
    public void setFirstInitialization(boolean firstInitialization)
    {
        this.firstInitialization = firstInitialization;
    }

    /**
     * Is form configurable boolean.
     *
     * @return the boolean
     *
     * @see pt.digitalis.dif.dem.objects.parameters.IParameter#isFormConfigurable()
     *         pt.digitalis.dif.dem.objects.parameters.IParameter#isFormConfigurable()
     */
    @Override
    public boolean isFormConfigurable()
    {
        return formConfigurableDef != null && formConfigurableDef.isConfigurable() &&
               StringUtils.isNotBlank(this.getFormLinked());
    }

    /**
     * Sets form configurable.
     *
     * @param formConfigurableType the form configurable type
     *
     * @see pt.digitalis.dif.dem.objects.parameters.IParameter#setFormConfigurable(pt.digitalis.dif.dem.objects.parameters.FormConfigurableDefinition)
     *         pt.digitalis.dif.dem.objects.parameters.IParameter#setFormConfigurable(pt.digitalis.dif.dem.objects.parameters.FormConfigurableDefinition)
     */
    @Override
    public void setFormConfigurable(FormConfigurableDefinition formConfigurableType)
    {
        this.formConfigurableDef = formConfigurableType;
    }

    /**
     * Is persist to repository boolean.
     *
     * @return the persistToRepository
     */
    @Override
    public boolean isPersistToRepository()
    {
        return persistToRepository;
    }

    /**
     * Is referenced in a rule from another parameter boolean.
     *
     * @return the boolean
     *
     * @see pt.digitalis.dif.dem.objects.parameters.IParameter#isReferencedInARuleFromAnotherParameter()
     *         pt.digitalis.dif.dem.objects.parameters.IParameter#isReferencedInARuleFromAnotherParameter()
     */
    @Override
    public boolean isReferencedInARuleFromAnotherParameter()
    {
        return referencedInARuleFromAnotherParameter;
    }

    /**
     * Sets referenced in a rule from another parameter.
     *
     * @param referencedInARuleFromAnotherParameter the referenced in a rule from another parameter
     *
     * @see pt.digitalis.dif.dem.objects.parameters.IParameter#setReferencedInARuleFromAnotherParameter(boolean)
     *         pt.digitalis.dif.dem.objects.parameters.IParameter#setReferencedInARuleFromAnotherParameter(boolean)
     */
    @Override
    public void setReferencedInARuleFromAnotherParameter(boolean referencedInARuleFromAnotherParameter)
    {
        this.referencedInARuleFromAnotherParameter = referencedInARuleFromAnotherParameter;
    }

    /**
     * Is required boolean.
     *
     * @return the boolean
     *
     * @see pt.digitalis.dif.dem.objects.parameters.IParameter#isRequired() pt.digitalis.dif.dem.objects.parameters.IParameter#isRequired()
     */
    @Override
    public boolean isRequired()
    {
        return required;
    }

    /**
     * Sets required.
     *
     * @param required the required
     *
     * @see pt.digitalis.dif.dem.objects.parameters.IEditableParameter#setRequired(boolean)
     *         pt.digitalis.dif.dem.objects.parameters.IEditableParameter#setRequired(boolean)
     */
    @Override
    public void setRequired(boolean required)
    {
        this.required = required;
    }

    /**
     * Is string setter supported boolean.
     *
     * @return the boolean
     *
     * @see pt.digitalis.dif.dem.objects.parameters.IParameter#isStringSetterSupported()
     *         pt.digitalis.dif.dem.objects.parameters.IParameter#isStringSetterSupported()
     */
    @Override
    public boolean isStringSetterSupported()
    {
        return true;
    }

    /**
     * Logs a warning
     *
     * @param warning the warning message to log
     */
    protected void logWarning(String warning)
    {

        if (DIFLogger.getLogger().isDebugEnabled())
            DIFLogger.getLogger().warn(getDebugPrefix() + warning + "\n" + this.toString());
        else
            DIFLogger.getLogger().warn(getDebugPrefix() + warning);
    }

    /**
     * Refresh parameter value parameter error list.
     *
     * @param stageInstance the stage instance
     *
     * @return the parameter error list
     *
     * @exception ConfigurationException the configuration exception
     * @see pt.digitalis.dif.dem.objects.parameters.IParameter#refreshParameterValue(pt.digitalis.dif.dem.interfaces.IStageInstance)
     *         pt.digitalis.dif.dem.objects.parameters.IParameter#refreshParameterValue(pt.digitalis.dif.dem.interfaces.IStageInstance)
     */
    @Override
    public ParameterErrorList refreshParameterValue(IStageInstance stageInstance) throws ConfigurationException
    {
        IDIFContext context = (stageInstance == null ? null : stageInstance.getContext());

        if (stageInstance == null || context == null)
            return new ParameterErrorList(this, null);
        else
        {
            String language = context.getLanguage();
            ParameterErrorList list = new ParameterErrorList(this, null);

            // Read the value from the request
            Object requestValue = context.getRequest().getParameter(getId());

            if (requestValue != null &&
                DIFGeneralConfigurationParameters.getInstance().getTreatWhiteSpacesOnlyParametersAsNull())
            {
                // Test if the parameter content is white space only
                String requestValueAsString = requestValue.toString();

                if (requestValueAsString.trim().length() == 0)
                    requestValue = null;
            }

            boolean parameterExistsInRequest = context.getRequest().getParameters().containsKey(getId());

            // IMPLEMENTATION NOTE:
            //
            // The parameter will be updated based on the following conditions:
            // - Parameter Scope: REQUEST, STATIC, SESSION, USER
            // - Persist the value: Read it from a persistent repository on initialization
            // - Default present: If no previous value a default value may be used
            // - Context: The context may be a source for values for REQUEST, USER or SESSION scopes
            // - Form submission for parameters that are linkedToForm's

            if (requestValue == null && !parameterExistsInRequest)
            {
                if (getParameterScope().equals(ParameterScope.SESSION))
                {
                    String formName = this.getFormLinked();
                    boolean isLinkedToForm = StringUtils.isNotBlank(formName);
                    boolean wasFormSubmited = false;

                    if (isLinkedToForm)
                    {
                        String stageSubmited = StringUtils.toStringOrNull(
                                stageInstance.getContext().getRequest().getParameter(HTTPConstants.FORM_SUBMIT_STAGE));
                        String formSubmited = StringUtils.toStringOrNull(
                                stageInstance.getContext().getRequest().getParameter(HTTPConstants.FORM_SUBMIT_NAME));

                        wasFormSubmited = stageInstance.getID().equals(stageSubmited) && formName.equals(formSubmited);
                    }

                    // No value passed, read the value from the session
                    // Gets the session from the context
                    IDIFSession session = context.getSession();

                    if (session == null)
                    {
                        ParameterError error = new ParameterError(getMessages().getMessages(language).get("noSession"),
                                ParameterErrorType.NO_SESSION);
                        list.addError(error);
                    }
                    else if (session.containsAttribute(this.getParameterSessionId(parameterContext, stageInstance)))
                    {
                        // If the parameter form was submited and no value was received we want to override the
                        // parameter value in session
                        if (wasFormSubmited)
                        {
                            session.addAttribute(this.getParameterSessionId(parameterContext, stageInstance), null);
                        }
                        else
                        {
                            // If the parameter already exists in session
                            Object sessionValue =
                                    session.getAttribute(this.getParameterSessionId(parameterContext, stageInstance));

                            // Recover value in session
                            logger.debug(getDebugPrefix() + "Setting to \"" + sessionValue + "\" (read from Session)");

                            requestValue = sessionValue;
                        }
                    }
                }
                else if (getParameterScope().equals(ParameterScope.USER))
                {
                    // TODO: Check the problem of several copies of the same user so the parameter is not kept between
                    // requests

                    IDIFSession session = context.getSession();

                    if (session == null)
                    {
                        ParameterError error = new ParameterError(getMessages().getMessages(language).get("noSession"),
                                ParameterErrorType.NO_SESSION);
                        list.addError(error);
                    }
                    else
                    {
                        // Gets the user from the context
                        IDIFUser user = session.getUser();

                        if (user == null && !allowAnonymous)
                        {
                            ParameterError error = new ParameterError(getMessages().getMessages(language).get("noUser"),
                                    ParameterErrorType.NO_USER);
                            list.addError(error);
                        }
                        else if (user != null && user.containsParameter(getId()))
                        {
                            // If the parameter already exists in the user
                            requestValue = user.getParameter(getId());

                            logger.debug(getDebugPrefix() + "Setting to \"" + requestValue + "\" (read from User)");
                        }
                    }
                }
            }

            // IMPLEMENTATION NOTE:
            // If the Scope is Static there is nothing to do since the initializations have already taken care of it in
            // the CodeGen and this value will persist through all the JVM life BUT we must check the REQUEST for value
            // setting

            boolean callSetValue = false;
            // All validations and initializations have taken place. Set the value now if existent
            // If no value present in the request (omitted or passed as null)...
            if (requestValue == null)
            {

                if (isRequired())
                {
                    if (!isReferencedInARuleFromAnotherParameter())
                    {
                        // Request parameters with required must be set!
                        ParameterError error = new ParameterError(getMessages().getMessages(language).get("required"),
                                ParameterErrorType.MISSING);
                        list.addError(error);
                        list.addErrorList(setValue(null, stageInstance, true).getErrorList());
                    }
                }
                else
                {
                    if (!parameterExistsInRequest)
                    {
                        if (this.getParameterScope() != ParameterScope.STATIC || this.isFirstInitialization())
                        {
                            // Set if to the default value (or null if none is set)...
                            logger.debug(getDebugPrefix() + "No value passed - Setting to " +
                                         (defaultValue == null ? "null"
                                                               : "default \"" + defaultValue.toString() + "\""));
                            list.addErrorList(setDefaultValue(stageInstance, true).getErrorList());
                        }
                    }
                    else
                    {
                        callSetValue = true;
                    }

                    this.setFirstInitialization(false);
                }
            }
            else
            {
                callSetValue = true;
            }
            if (callSetValue)
            {
                logger.debug(getDebugPrefix() + "Setting to \"" + requestValue + "\" (read from request)");

                if (isStringSetterSupported())
                    list.addErrorList(setValueFromString(this.convertObjectToString(requestValue), stageInstance, true)
                            .getErrorList());
                else
                    list.addErrorList(setValue((T) requestValue, stageInstance, true).getErrorList());
            }

            return list;
        }
    }

    /**
     * Sets default value.
     *
     * @param stageInstance the stage instance
     *
     * @return the default value
     *
     * @exception ConfigurationException the configuration exception
     * @see pt.digitalis.dif.dem.objects.parameters.IParameter#setDefaultValue(pt.digitalis.dif.dem.interfaces.IStageInstance)
     *         pt.digitalis.dif.dem.objects.parameters.IParameter#setDefaultValue(pt.digitalis.dif.dem.interfaces.IStageInstance)
     */
    @Override
    public ParameterErrorList setDefaultValue(IStageInstance stageInstance) throws ConfigurationException
    {
        return setDefaultValue(stageInstance, false);
    }

    /**
     * Sets default value.
     *
     * @param stageInstance            the stage instance
     * @param initializationInProgress the initialization in progress
     *
     * @return the default value
     *
     * @exception ConfigurationException the configuration exception
     * @see pt.digitalis.dif.dem.objects.parameters.IParameter#setDefaultValue(pt.digitalis.dif.dem.interfaces.IStageInstance,
     *         boolean) pt.digitalis.dif.dem.objects.parameters.IParameter#setDefaultValue(pt.digitalis.dif.dem.interfaces.IStageInstance,
     *         boolean)
     */
    @Override
    public ParameterErrorList setDefaultValue(IStageInstance stageInstance, boolean initializationInProgress)
            throws ConfigurationException
    {
        return setValue(defaultValue, stageInstance, initializationInProgress);
    }

    /**
     * Sets value.
     *
     * @param value         the value
     * @param stageInstance the stage instance
     *
     * @return the value
     *
     * @exception ConfigurationException the configuration exception
     * @see pt.digitalis.dif.dem.objects.parameters.IParameter#setValue(java.lang.Object,
     *         pt.digitalis.dif.dem.interfaces.IStageInstance) pt.digitalis.dif.dem.objects.parameters.IParameter#setValue(java.lang.Object,
     *         pt.digitalis.dif.dem.interfaces.IStageInstance)
     */
    @Override
    public ParameterErrorList setValue(T value, IStageInstance stageInstance) throws ConfigurationException
    {
        return setValue(value, stageInstance, false);
    }

    /**
     * Sets value.
     *
     * @param value                    the value
     * @param stageInstance            the stage instance
     * @param initializationInProgress the initialization in progress
     *
     * @return the value
     *
     * @exception ConfigurationException the configuration exception
     * @see pt.digitalis.dif.dem.objects.parameters.IParameter#setValue(java.lang.Object,
     *         pt.digitalis.dif.dem.interfaces.IStageInstance, boolean) pt.digitalis.dif.dem.objects.parameters.IParameter#setValue(java.lang.Object,
     *         pt.digitalis.dif.dem.interfaces.IStageInstance, boolean)
     */
    @Override
    public ParameterErrorList setValue(T value, IStageInstance stageInstance, boolean initializationInProgress)
            throws ConfigurationException
    {

        ParameterErrorList list = new ParameterErrorList(this, value);
        IDIFContext context;

        if (stageInstance == null)
            context = null;
        else
            context = stageInstance.getContext();

        // Perform validation. Will create the error list if not valid
        list = validateParameterValue(value, stageInstance, initializationInProgress);

        if (context == null)
            // Null context is interpreted like an initialization so it will not try to update the repositories defined
            // by it's scope
            this.value = value;

        else
        {
            if (parameterScope == ParameterScope.SESSION)
            {
                if (context.getSession() == null)
                {
                    list.addError(new ParameterError(getMessages().getMessages(context.getLanguage()).get("noSession"),
                            ParameterErrorType.NO_SESSION));

                    this.value = null;
                }
                else
                {
                    context.getSession()
                            .addAttribute(this.getParameterSessionId(parameterContext, stageInstance), value);
                    this.value = value;
                }
            }
            else if (parameterScope == ParameterScope.USER)
            {
                if (context.getSession() == null || context.getSession().getUser() == null)
                {
                    if (!allowAnonymous)
                        list.addError(new ParameterError(getMessages().getMessages(context.getLanguage()).get("noUser"),
                                ParameterErrorType.NO_USER));

                    this.value = null;
                }
                else
                {
                    // FIXME: This will fail since the value is not kept through all user instances
                    context.getSession().getUser().setParameter(getId(), value);
                    this.value = value;
                }
            }
            else if (parameterScope == ParameterScope.REQUEST || parameterScope == ParameterScope.GLOBAL_REQUEST)
            {
                // The parameter will be thrown away after the request is processed
                this.value = value;
            }
            else if (parameterScope == ParameterScope.STATIC)
                // Static values are kept in the inner value attribute. The only ones that keep it here.
                this.value = value;

            if (isPersistToRepository())
            {
                // TODO: Persist to persistent repository
            }
        }

        return list;
    }

    /**
     * Sets value from string.
     *
     * @param value         the value
     * @param stageInstance the stage instance
     *
     * @return the value from string
     *
     * @exception ConfigurationException the configuration exception
     * @see pt.digitalis.dif.dem.objects.parameters.IParameter#setValueFromString(java.lang.String,
     *         pt.digitalis.dif.dem.interfaces.IStageInstance) pt.digitalis.dif.dem.objects.parameters.IParameter#setValueFromString(java.lang.String,
     *         pt.digitalis.dif.dem.interfaces.IStageInstance)
     */
    @Override
    public final ParameterErrorList setValueFromString(String value, IStageInstance stageInstance)
            throws ConfigurationException
    {

        return this.setValueFromString(value, stageInstance, false);
    }

    /**
     * To object formatter object formatter.
     *
     * @param format        the format
     * @param dumpedObjects the dumped objects
     *
     * @return the object formatter
     */
    @Override
    public ObjectFormatter toObjectFormatter(Format format, List<Object> dumpedObjects)
    {
        ObjectFormatter formatter = new ObjectFormatter(format, dumpedObjects);
        formatter.addItem("ID", this.id);
        formatter.addItem("Value", value);
        formatter.addItem("Value Type", this.value == null ? "n/a" : this.value.getClass().getCanonicalName());
        formatter.addItem("Parent", "(" + this.parentType + ") " + this.parentID);
        formatter.addItem("Scope", this.parameterScope);
        formatter.addItem("Parameter Context", this.parameterContext);
        formatter.addItem("Persist", this.persistToRepository);
        formatter.addItem("Required", this.required);
        formatter.addItem("Default Value", this.defaultValue);
        formatter.addItem("Link to Form", this.formLinked);
        formatter.addItem("Form Configurable", this.formConfigurableDef);
        formatter.addItem("Constraints", this.constraints);
        formatter.addItem("Validators", this.validators);
        formatter.addItem("Rules", this.rules);
        formatter.addItem("Referenced in a rule from another parameter", this.referencedInARuleFromAnotherParameter);

        return formatter;
    }

    /**
     * To string string.
     *
     * @return the string
     *
     * @see java.lang.Object#toString() java.lang.Object#toString()
     */
    @Override
    public String toString()
    {
        return toObjectFormatter(Format.TEXT, null).getFormatedObject();
    }

    /**
     * Validate a parameter value on type, constraints, validators and rules, if applied.
     *
     * @param value                    the value to validate
     * @param stageInstance            the stage where the parameter is being validated
     * @param initializationInProgress T if called within the dif parameter initialization
     *
     * @return the list of errors found upon validation
     *
     * @exception ConfigurationException the configuration exception
     */
    protected ParameterErrorList validateParameterValue(T value, IStageInstance stageInstance,
            boolean initializationInProgress) throws ConfigurationException
    {
        ParameterErrorList list = new ParameterErrorList(this, value);
        IDIFContext context;

        if (stageInstance == null)
            context = null;
        else
            context = stageInstance.getContext();

        // Validate all constraints
        for (Entry<String, IParameterConstraint> constraintEntry : getConstraints().entrySet())
        {

            ParameterConstraintResult constraintResult =
                    constraintEntry.getValue().getValidationResult(value, stageInstance);
            if (!constraintResult.isValid())
            {
                list.addError(new ParameterError(constraintEntry.getKey(), constraintResult));
            }
        }

        // Validate all validators
        for (Entry<String, IParameterValidator> validatorEntry : getValidators().entrySet())
        {
            if (!validatorEntry.getValue().validate(value, context))
            {
                ParameterError error =
                        new ParameterError(validatorEntry.getValue().validationErrorMessage(getLanguage(context)),
                                ParameterErrorType.VALIDATOR);
                error.setValidator(validatorEntry.getKey(), validatorEntry.getValue());

                list.addError(error);
            }
        }

        // Validate all rules
        for (IParameterRule<T> rule : getRules())
        {
            ParameterRuleResult result;
            try
            {
                result = rule.getValidationResult(stageInstance, value, initializationInProgress, this);

                if (!result.isValid())
                {
                    ParameterError error = new ParameterError(result.getErrorMessage(), ParameterErrorType.RULE);
                    error.setRule(rule);

                    list.addError(error);
                }
            }
            catch (ParameterException e)
            {
                ParameterError error = new ParameterError(e.getMessage(), ParameterErrorType.RULE);
                error.setRule(rule);

                list.addError(error);
            }
        }

        return list;
    }
}
