/**
 * 2007, Digitalis Informatica. All rights reserved. Distribuicao e Gestao de Informatica, Lda. Estrada de Paco de Arcos
 * num.9 - Piso -1 2780-666 Paco de Arcos Telefone: (351) 21 4408990 Fax: (351) 21 4408999 http://www.digitalis.pt
 */
package pt.digitalis.dif.dem.objects.parameters.types;

import pt.digitalis.dif.controller.interfaces.IDIFContext;
import pt.digitalis.dif.dem.interfaces.IStageInstance;
import pt.digitalis.dif.dem.objects.parameters.constraints.IParameterConstraint;
import pt.digitalis.dif.dem.objects.parameters.errors.ParameterError;
import pt.digitalis.dif.dem.objects.parameters.errors.ParameterErrorList;
import pt.digitalis.dif.dem.objects.parameters.errors.ParameterErrorType;
import pt.digitalis.dif.exception.objects.ParameterException;
import pt.digitalis.dif.startup.DIFGeneralConfigurationParameters;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;

/**
 * /** This class will define a numeric Parameter.
 * <p>
 * It will hold information relative to the Parameter value, ID key and validation constraints.
 *
 * @author Pedro Viegas <a href="mailto:pviegas@digitalis.pt">pviegas@digitalis.pt</a>
 * @author Rodrigo Gonalves <a href="mailto:rgoncalves@digitalis.pt">rgoncalves@digitalis.pt</a>
 * @created Nov 23, 2007
 */
public class LongParameter extends AbstractParameter<Number>
{

    /** The list of supported classes to define in the concrete implementations */
    @SuppressWarnings("serial")
    final static private List<String> supportedClasses = new ArrayList<String>()
    {

        {
            add(Long.class.getCanonicalName());
        }
    };

    /**
     * @see pt.digitalis.dif.dem.objects.parameters.types.AbstractParameter#automaticConstraints()
     */
    @Override
    protected String automaticConstraints()
    {
        return "numeric";
    }

    /**
     * @see pt.digitalis.dif.dem.objects.parameters.IParameter#getSupportedClasses()
     */
    public List<String> getSupportedClasses()
    {
        return supportedClasses;
    }

    /**
     * @see pt.digitalis.dif.dem.objects.parameters.IParameter#getValueAsBigDecimal(pt.digitalis.dif.controller.interfaces.IDIFContext)
     */
    public BigDecimal getValueAsBigDecimal(IDIFContext context) throws ParameterException
    {
        return BigDecimal.valueOf(getValueAsLong(context));
    }

    /**
     * @see pt.digitalis.dif.dem.objects.parameters.IParameter#getValueAsBoolean(IDIFContext)
     */
    public boolean getValueAsBoolean(IDIFContext context) throws ParameterException
    {
        return !getValueAsLong(context).equals(0L);
    }

    /**
     * @see pt.digitalis.dif.dem.objects.parameters.IParameter#getValueAsDate(IDIFContext)
     */
    public Date getValueAsDate(IDIFContext context)
    {
        return null;
    }

    /**
     * @see pt.digitalis.dif.dem.objects.parameters.IParameter#getValueAsDouble(IDIFContext)
     */
    public Double getValueAsDouble(IDIFContext context) throws ParameterException
    {
        return getValueAsLong(context).doubleValue();
    }

    /**
     * @see pt.digitalis.dif.dem.objects.parameters.IParameter#getValueAsLong(IDIFContext)
     */
    public Long getValueAsLong(IDIFContext context) throws ParameterException
    {
        return (Long) getValue(context);
    }

    /**
     * @see pt.digitalis.dif.dem.objects.parameters.IParameter#isNumeric()
     */
    public boolean isNumeric()
    {
        return true;
    }

    /**
     * @see pt.digitalis.dif.dem.objects.parameters.IParameter#setValueFromString(java.lang.String,
     *         pt.digitalis.dif.dem.interfaces.IStageInstance, boolean)
     */
    public ParameterErrorList setValueFromString(String value, IStageInstance stageInstance,
            boolean initializationInProgress)
    {

        ParameterErrorList list = new ParameterErrorList(this, value);
        ParameterError error;
        IParameterConstraint constraint = getConstraints().get("numeric");
        IDIFContext context;

        if (stageInstance == null)
            context = null;
        else
            context = stageInstance.getContext();

        // If the numeric constraint does not validate...
        if (!constraint.validateConstraint(value, stageInstance))
        {
            String language;

            if (context == null)
                language = DIFGeneralConfigurationParameters.getInstance().getDefaultLanguage();
            else
                language = context.getLanguage();
            error = new ParameterError(constraint.validationErrorMessage(language), ParameterErrorType.CONSTRAINT);
            error.setConstraint("numeric", constraint);

            list.addError(error);
        }
        else
        {
            try
            {
                if (value == null || "".equals(value) || "undefined".equals(value))
                    return setValue(null, stageInstance, initializationInProgress);
                else
                    return setValue(Long.parseLong(value.trim()), stageInstance, initializationInProgress);
            }
            catch (Exception e)
            {
                /*
                 * Unreachable exception: to get here Long#parseLong() would have to throw a NumberFormatException. This
                 * exception is raised if the string does not contain a parsable double. This would happen if a bad
                 * value is passed. This situation is avoided by the constraint validation mechanism. So if a bad value
                 * is passed the code never enters the 'else' branch and as such is never possible to get here.
                 */
                String language;

                if (context == null)
                    language = DIFGeneralConfigurationParameters.getInstance().getDefaultLanguage();
                else
                    language = context.getLanguage();

                error = new ParameterError(getMessages().getMessages(language).get("valueTooLong"),
                        ParameterErrorType.OTHER);
                error.setException(e);
                list.addError(error);
            }
        }

        return list;
    }
}
