/**
 * 2007, Digitalis Informatica. All rights reserved. Distribuicao e Gestao de Informatica, Lda. Estrada de Paco de Arcos
 * num.9 - Piso -1 2780-666 Paco de Arcos Telefone: (351) 21 4408990 Fax: (351) 21 4408999 http://www.digitalis.pt
 */
package pt.digitalis.dif.dem.objects.parameters.types;

import pt.digitalis.dif.controller.interfaces.IDIFContext;
import pt.digitalis.dif.dem.interfaces.IStageInstance;
import pt.digitalis.dif.dem.objects.parameters.errors.ParameterErrorList;
import pt.digitalis.dif.exception.objects.ParameterException;
import pt.digitalis.utils.common.CollectionUtils;
import pt.digitalis.utils.common.DateUtils;
import pt.digitalis.utils.config.ConfigurationException;

import java.math.BigDecimal;
import java.text.ParseException;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;

/**
 * /** This class will define a string array Parameter.
 * <p>
 * It will hold information relative to the Parameter value, ID key and validation constraints.
 *
 * @author Pedro Viegas <a href="mailto:pviegas@digitalis.pt">pviegas@digitalis.pt</a>
 * @author Rodrigo Gon�alves <a href="mailto:rgoncalves@digitalis.pt">rgoncalves@digitalis.pt</a>
 * @created Nov 23, 2007
 */
public class StringArrayParameter extends AbstractParameter<StringArray>
{

    /** The list of supported classes to define in the concrete implementations */
    @SuppressWarnings("serial")
    final static private List<String> supportedClasses = new ArrayList<String>()
    {

        {
            add(StringArray.class.getCanonicalName());
        }
    };

    /**
     * @see pt.digitalis.dif.dem.objects.parameters.types.AbstractParameter#convertObjectToString(java.lang.Object)
     */
    @Override
    protected String convertObjectToString(Object obj)
    {
        if (obj == null)
            return null;
        else if (obj instanceof StringArray)
        {
            StringArray obgAsStringArray = (StringArray) obj;

            return CollectionUtils.listToCommaSeparatedString(obgAsStringArray);
        }
        else
        {
            return obj.toString();
        }
    }

    /**
     * @see pt.digitalis.dif.dem.objects.parameters.IParameter#getSupportedClasses()
     */
    public List<String> getSupportedClasses()
    {
        return supportedClasses;
    }

    /**
     * @see pt.digitalis.dif.dem.objects.parameters.IParameter#getValueAsBigDecimal(pt.digitalis.dif.controller.interfaces.IDIFContext)
     */
    public BigDecimal getValueAsBigDecimal(IDIFContext context) throws ParameterException
    {
        try
        {
            return new BigDecimal(getValue(context).toString());
        }
        catch (Exception e)
        {
            throw new ParameterException("Error parsing value", e, this);
        }
    }

    /**
     * @see pt.digitalis.dif.dem.objects.parameters.IParameter#getValueAsBoolean(IDIFContext)
     */
    public boolean getValueAsBoolean(IDIFContext context) throws ParameterException
    {
        StringArray value = getValue(context);

        if (value.size() == 1)
            return Boolean.getBoolean(value.get(0));
        else
            return false;
    }

    /**
     * @see pt.digitalis.dif.dem.objects.parameters.IParameter#getValueAsDate(IDIFContext)
     */
    public Date getValueAsDate(IDIFContext context) throws ParameterException
    {
        StringArray value = getValue(context);

        if (value.size() == 1)
        {
            try
            {
                return DateUtils.stringToDate(value.get(0));
            }
            catch (ParseException e)
            {
                throw new ParameterException("Error parsing value", e, this);
            }
        }
        else
            return null;
    }

    /**
     * @see pt.digitalis.dif.dem.objects.parameters.IParameter#getValueAsDouble(IDIFContext)
     */
    public Double getValueAsDouble(IDIFContext context) throws ParameterException
    {
        StringArray value = getValue(context);

        if (value.size() == 1)
        {
            try
            {
                return Double.parseDouble(value.get(0));
            }
            catch (Exception e)
            {
                throw new ParameterException("Error parsing value", e, this);
            }
        }
        else
            return null;
    }

    /**
     * @see pt.digitalis.dif.dem.objects.parameters.IParameter#getValueAsLong(IDIFContext)
     */
    public Long getValueAsLong(IDIFContext context) throws ParameterException
    {
        StringArray value = getValue(context);

        if (value.size() == 1)
        {
            try
            {
                return Long.parseLong(value.get(0));
            }
            catch (Exception e)
            {
                throw new ParameterException("Error parsing value", e, this);
            }
        }
        else
            return null;
    }

    /**
     * @see pt.digitalis.dif.dem.objects.parameters.types.AbstractParameter#getValueAsString(pt.digitalis.dif.controller.interfaces.IDIFContext)
     */
    @Override
    public String getValueAsString(IDIFContext context) throws ParameterException
    {
        StringArray currentValue = getValue(context);

        if (currentValue == null)
            return null;
        else
            return CollectionUtils.listToCommaSeparatedString(currentValue);
    }

    /**
     * @see pt.digitalis.dif.dem.objects.parameters.IParameter#isNumeric()
     */
    public boolean isNumeric()
    {
        return false;
    }

    /**
     * @see pt.digitalis.dif.dem.objects.parameters.IParameter#setValueFromString(java.lang.String,
     *         pt.digitalis.dif.dem.interfaces.IStageInstance, boolean)
     */
    public ParameterErrorList setValueFromString(String value, IStageInstance stageInstance,
            boolean initializationInProgress) throws ConfigurationException
    {
        if (value == null)
            return setValue(null, stageInstance, initializationInProgress);
        else
        {
            StringArray valueAsStringArray = new StringArray();
            valueAsStringArray.setFromCommaSeparatedString(value);

            return setValue(valueAsStringArray, stageInstance, initializationInProgress);
        }
    }
}