/**
 * 2016, Digitalis Informatica. All rights reserved. Distribuicao e Gestao de Informatica, Lda. Estrada de Paco de Arcos
 * num.9 - Piso -1 2780-666 Paco de Arcos Telefone: (351) 21 4408990 Fax: (351) 21 4408999 http://www.digitalis.pt
 */

package pt.digitalis.dif.features;

import pt.digitalis.dif.ioc.DIFIoCRegistry;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * References all DIF feature that aren't provided by the dif-core.
 *
 * @author Luis Pinto <a href="mailto:lpinto@digitalis.pt">lpinto@digitalis.pt</a><br/>
 * @created Oct 6, 2016
 */
public class DIFFeatures
{

    /**
     * Gets the active {@link IDIFFeature} implementation.
     *
     * @param difFeatureInterface the {@link IDIFFeature}
     *
     * @return the o auth implementations
     */
    public static IDIFFeature getActiveImplementation(Class<? extends IDIFFeature> difFeatureInterface)
    {

        IDIFFeature featureImplementation = null;

        for (IDIFFeature featureImpl : getImplementations(difFeatureInterface))
        {
            if (featureImpl.isActive())
            {
                featureImplementation = featureImpl;
                break;
            }
        }

        return featureImplementation;
    }

    /**
     * Gets the active {@link IDIFFeature} implementation.
     *
     * @param feature the {@link Feature}
     *
     * @return the o auth implementations
     */
    public static IDIFFeature getActiveImplementation(Feature feature)
    {
        IDIFFeature featureImplementation = null;

        if (feature != null)
            featureImplementation = getActiveImplementation(getFeatures().get(feature.toString()));

        return featureImplementation;
    }

    /**
     * Get the map of features currently on DIF
     *
     * @return {@link Map} of {@link IDIFFeature}
     */
    public static Map<String, Class<? extends IDIFFeature>> getFeatures()
    {
        Map<String, Class<? extends IDIFFeature>> featuresMap = new HashMap<String, Class<? extends IDIFFeature>>();

        featuresMap.put(Feature.BASE_AUTHENTICATION.toString(), IDIFFeatureBaseAuthentication.class);
        featuresMap.put(Feature.ALTERNATIVE_AUTHENTICATION.toString(), IDIFFeatureAlternativeAuthentication.class);

        return featuresMap;
    }

    /**
     * Gets the {@link IDIFFeature} implementations.
     *
     * @param difFeatureInterface the {@link IDIFFeature}
     *
     * @return the o auth implementations
     */
    public static List<IDIFFeature> getImplementations(Class<? extends IDIFFeature> difFeatureInterface)
    {
        List<IDIFFeature> featureImplementations = new ArrayList<IDIFFeature>();
        if (difFeatureInterface != null)
        {
            for (IDIFFeature impl : DIFIoCRegistry.getRegistry().getImplementations(difFeatureInterface))
            {
                featureImplementations.add(impl);
            }
        }
        return featureImplementations;
    }

    /**
     * Gets the {@link IDIFFeature} implementations.
     *
     * @param feature the {@link Feature}
     *
     * @return the o auth implementations
     */
    public static List<IDIFFeature> getImplementations(Feature feature)
    {
        List<IDIFFeature> featureImplementations = new ArrayList<IDIFFeature>();

        if (feature != null)
            return getImplementations(getFeatures().get(feature.toString()));

        return featureImplementations;
    }

    /**
     * Gets the {@link IDIFFeature} implementations.
     *
     * @param difFeatureInterface the {@link IDIFFeature}
     *
     * @return the o auth implementations
     */
    public static Map<String, String> getImplementationsNames(Class<? extends IDIFFeature> difFeatureInterface)
    {
        Map<String, String> result = new HashMap<String, String>();

        if (difFeatureInterface != null)
            for (IDIFFeature impl : getImplementations(difFeatureInterface))
            {
                result.put(impl.getName(), impl.getName());
            }

        return result;
    }

    /**
     * Gets the {@link IDIFFeature} implementations names.
     *
     * @param feature the {@link Feature}
     *
     * @return the o auth implementations
     */
    public static Map<String, String> getImplementationsNames(Feature feature)
    {
        Map<String, String> result = new HashMap<String, String>();

        if (feature != null)
        {
            result = getImplementationsNames(getFeatures().get(feature.toString()));
        }

        return result;
    }
}
