/**
 * 2018, Digitalis Informatica. All rights reserved. Distribuicao e Gestao de Informatica, Lda. Estrada de Paco de Arcos
 * num.9 - Piso -1 2780-666 Paco de Arcos Telefone: (351) 21 4408990 Fax: (351) 21 4408999 http://www.digitalis.pt
 */

package pt.digitalis.dif.flightrecorder;

import org.apache.commons.collections4.map.HashedMap;
import pt.digitalis.dif.controller.interfaces.IDIFContext;
import pt.digitalis.dif.exception.DIFException;
import pt.digitalis.dif.startup.DIFInitializer;
import pt.digitalis.dif.utils.logging.DIFLogger;
import pt.digitalis.dif.utils.logging.DIFLoggerInterceptorImpl;
import pt.digitalis.log.LogLevel;
import pt.digitalis.utils.common.StringUtils;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Map;

/**
 * Will save full trace of all records according to the {@link FlightRecorderConfiguration} preferences.
 *
 * @author Pedro Viegas <a href="mailto:pviegas@digitalis.pt">pviegas@digitalis.pt</a><br/>
 * @created Jul 31, 2019
 */
// JSONFormatter: http://azimi.me/json-formatter-js/ ou melhor https://github.com/josdejong/jsoneditor
public class FlightRecorder
{

    /** list of classes to ignore when determining the class that issued the log action. */
    public static List<String> classesToIgnore = null;

    /** The index recording by ID. */
    static Map<Long, RecorderEntry> indexRecordingByID = new HashedMap<Long, RecorderEntry>();

    /** The index recording by ID. */
    static Map<String, Map<Long, RecorderEntry>> indexRecordingBySessionID =
            new HashedMap<String, Map<Long, RecorderEntry>>();

    /** The entries that are recorded by global parameters (on error, on slow request). */
    private static CircularFifoQueueForFlightRecordings globalEntries = new CircularFifoQueueForFlightRecordings();

    /** the app per thread repository instance. */
    private static ThreadLocal<RecorderEntry> recorderEntryPerThread = new ThreadLocal<RecorderEntry>();

    /** The entries that are recorded by user/session flagged to be saved. */
    private static CircularFifoQueueForFlightRecordings sessionEntries = new CircularFifoQueueForFlightRecordings();

    /** The sessions to record. */
    private static List<String> sessionsToRecord = new ArrayList<String>();

    /** The suspend data gathering - start suspended. */
    private static boolean suspendDataGathering = true;

    /** The users to record. */
    private static List<String> usersToRecord = new ArrayList<String>();

    static
    {
        classesToIgnore = new ArrayList<String>();
        classesToIgnore.addAll(DIFLoggerInterceptorImpl.classesToIgnore);
        classesToIgnore.add(FlightRecorder.class.getCanonicalName());
    }

    /**
     * Adds the session to record.
     *
     * @param sessionID the session ID
     */
    static public void addSessionToRecord(String sessionID)
    {
        sessionsToRecord.add(sessionID);
    }

    /**
     * Adds the user to record.
     *
     * @param userID the user ID
     */
    static public void addUserToRecord(String userID)
    {
        usersToRecord.add(userID);
    }

    /**
     * Fill recorder entry from context.
     *
     * @param entry
     * @param context the context
     */
    static private void fillRecorderEntryFromContext(RecorderEntry entry, IDIFContext context)
    {
        if (context != null)
        {
            entry.setSessionID(context.getSession().getSessionID());
            entry.setUser(context.getSession().getUser());
            entry.setClientDescriptor(context.getRequest().getClient());
        }
    }

    /**
     * Gets the existing recorder entry for current thread.
     *
     * @param type
     * @param context
     *
     * @return the existing recorder entry for current thread
     */
    static private RecorderEntry getExistingRecorderEntryForCurrentThread(ActivityType type, IDIFContext context)
    {
        RecorderEntry entry = recorderEntryPerThread.get();

        return entry;
    }

    /**
     * Create recorder entry for current thread.
     *
     * @param type    the type
     * @param context
     *
     * @return the or create recorder entry for current thread
     */
    static private RecorderEntry getOrCreateRecorderEntryForCurrentThread(ActivityType type, IDIFContext context)
    {
        RecorderEntry entry = getExistingRecorderEntryForCurrentThread(type, context);

        if (shouldCreateRecorderEntryForCurrentThread(type, context))
        {
            boolean diferentSessions = false;
            boolean diferentUsers = false;

            String currentSessionID = context != null ? context.getSession().getSessionID() : null;
            String existingEntrySessionID = entry != null ? entry.getSessionID() : null;

            String currentUserID =
                    context != null && context.getSession().isLogged() ? context.getSession().getUser().getID() : null;
            String existingEntryUserID = entry != null ? entry.getUserID() : null;

            if (context != null)
            {
                diferentSessions = (currentSessionID == null && existingEntrySessionID != null) ||
                                   (currentSessionID != null && existingEntrySessionID == null) ||
                                   (currentSessionID != null && !currentSessionID.equals(existingEntrySessionID));

                diferentUsers = (currentUserID == null && existingEntryUserID != null) ||
                                (currentUserID != null && existingEntryUserID == null) ||
                                (currentUserID != null && !currentUserID.equals(existingEntryUserID));
            }

            // When no entry exists or if the entry exists in the current thread but is associated to a different
            // session must create a new RecorderEntry
            if (entry == null || diferentSessions || diferentUsers)
            {
                StringBuffer buffer = new StringBuffer();
                buffer.append("New Flight Log entry ");
                if (diferentSessions)
                    buffer.append(" | new session");
                if (diferentUsers)
                    buffer.append(" | user change: " + StringUtils.nvl(existingEntryUserID, "none") + " -> " +
                                  StringUtils.nvl(currentUserID, "none"));

                DIFLogger.getLogger().debug(buffer.toString());

                entry = new RecorderEntry();
                fillRecorderEntryFromContext(entry, context);
                recorderEntryPerThread.set(entry);

                String sessionID = entry.getSessionID();
                String userID = entry.getAttributeAsString(RecorderEntry.Fields.USERID);

                boolean saveInSessionEntries = false;

                if (StringUtils.isNotBlank(sessionID) && getSessionsToRecord().contains(sessionID))
                    saveInSessionEntries = true;
                else if (StringUtils.isNotBlank(userID) && getUsersToRecord().contains(userID))
                    saveInSessionEntries = true;

                if (saveInSessionEntries)
                    sessionEntries.add(entry);
                else
                    globalEntries.add(entry);

                indexRecordingByID.put(entry.getId(), entry);

                if (StringUtils.isNotBlank(entry.getSessionID()))
                {
                    Map<Long, RecorderEntry> sessionRecordings = indexRecordingBySessionID.get(entry.getSessionID());

                    if (sessionRecordings == null)
                    {
                        sessionRecordings = new HashedMap<Long, RecorderEntry>();
                        indexRecordingBySessionID.put(entry.getSessionID(), sessionRecordings);
                    }

                    sessionRecordings.put(entry.getId(), entry);
                }
            }
        }

        return entry;
    }

    /**
     * Gets the recording.
     *
     * @param recordingID the recording ID
     *
     * @return the recording
     */
    public static RecorderEntry getRecording(Long recordingID)
    {
        return indexRecordingByID.get(recordingID);
    }

    /**
     * Gets the recordings.
     *
     * @return the recordings
     */
    public static List<RecorderEntry> getRecordings()
    {
        List<RecorderEntry> list = new ArrayList<RecorderEntry>();
        list.addAll(sessionEntries);
        list.addAll(globalEntries);

        return list;
    }

    /**
     * Gets the recordings for session ID.
     *
     * @param sessionID the session ID
     *
     * @return the recordings for session ID
     */
    public static Collection<RecorderEntry> getRecordingsForSessionID(String sessionID)
    {
        return indexRecordingBySessionID.get(sessionID).values();
    }

    /**
     * Inspector for the 'sessionsToRecord' attribute.
     *
     * @return the sessionsToRecord value
     */
    public static List<String> getSessionsToRecord()
    {
        return sessionsToRecord;
    }

    /**
     * Modifier for the 'sessionsToRecord' attribute.
     *
     * @param sessionsToRecord the new sessionsToRecord value to set
     */
    public static void setSessionsToRecord(List<String> sessionsToRecord)
    {
        FlightRecorder.sessionsToRecord = sessionsToRecord;
    }

    /**
     * Inspector for the 'usersToRecord' attribute.
     *
     * @return the usersToRecord value
     */
    public static List<String> getUsersToRecord()
    {
        return usersToRecord;
    }

    /**
     * Modifier for the 'usersToRecord' attribute.
     *
     * @param usersToRecord the new usersToRecord value to set
     */
    public static void setUsersToRecord(List<String> usersToRecord)
    {
        FlightRecorder.usersToRecord = usersToRecord;
    }

    /**
     * Checks if is active.
     *
     * @return true, if is active
     */
    public static boolean isActive()
    {
        if (suspendDataGathering || !DIFInitializer.isInitialized() ||
            !FlightRecorderConfiguration.getInstance().getActive())
        {
            return false;
        }
        else
        {
            FlightRecorderConfiguration fr = FlightRecorderConfiguration.getInstance();

            boolean isActiveForUserOrSession = !getSessionsToRecord().isEmpty() || !getUsersToRecord().isEmpty();
            boolean isActiveForGlobalTriggers =
                    fr.getRecordRequestLonguerThan() > 0 || fr.getRecordRequestsWithExceptions() ||
                    fr.getRecordRequestsWithSQLErrors();

            return isActiveForGlobalTriggers || isActiveForUserOrSession;
        }
    }

    /**
     * Checks if is suspended.
     *
     * @return true, if is suspended
     */
    public static boolean isSuspended()
    {
        return suspendDataGathering;
    }

    /**
     * Report exception.
     *
     * @param difException the dif exception
     */
    public static void reportException(DIFException difException)
    {
        RecorderEntry entry = getOrCreateRecorderEntryForCurrentThread(ActivityType.EXCEPTION, null);

        if (entry != null)
            entry.addException(difException);
    }

    /**
     * Adds the log entry.
     *
     * @param level   the level
     * @param message the message
     */
    public static void reportLog(LogLevel level, Object message)
    {
        RecorderEntry entry = getOrCreateRecorderEntryForCurrentThread(ActivityType.LOG, null);

        if (entry != null)
        {
            LogLevel desiredLogLevel = FlightRecorderConfiguration.getInstance().getLogLevelObjToKeep();

            // Report only desired LOGLevel or upper level logs (TRACE->DEBUG->WARN->INFO->ERROR->FATAL)
            if (desiredLogLevel.compareTo(level) < 1)
            {
                String className = null;
                StackTraceElement[] stackTrace = Thread.currentThread().getStackTrace();
                int traceItem = 0;

                do
                {
                    // determine className and if it is a class to ignore
                    className = stackTrace[traceItem++].getClassName();
                } while (classesToIgnore.contains(className) && traceItem < stackTrace.length);

                entry.addLog(className, level, message);
            }
        }
    }

    /**
     * Adds the request.
     *
     * @param context the context
     */
    public static void reportRequest(IDIFContext context)
    {
        RecorderEntry entry = getOrCreateRecorderEntryForCurrentThread(ActivityType.REQUEST, context);

        if (entry != null)
            entry.addRequest(context);
    }

    /**
     * Adds the SQL.
     *
     * @param log the log
     */
    public static void reportSQL(SQLExecutionLog log)
    {
        ActivityType type = log.isSuccess() ? ActivityType.SQL : ActivityType.SQL_ERROR;
        RecorderEntry entry = getOrCreateRecorderEntryForCurrentThread(type, null);

        if (entry != null)
            entry.addSQLLog(log);
    }

    /**
     * Report uncaught exception.
     *
     * @param thread    the thread
     * @param exception the exception
     */
    public static void reportUncaughtException(Thread thread, Throwable exception)
    {
        RecorderEntry entry = getOrCreateRecorderEntryForCurrentThread(ActivityType.EXCEPTION, null);

        if (entry != null)
            entry.addException(exception);
    }

    /**
     * Should create recorder entre for current thread.
     *
     * @param type    the type
     * @param context the context
     *
     * @return true, if successful
     */
    static private boolean shouldCreateRecorderEntryForCurrentThread(ActivityType type, IDIFContext context)
    {
        if (!isActive())
            return false;
        else
        {
            if (type == ActivityType.EXCEPTION &&
                FlightRecorderConfiguration.getInstance().getRecordRequestsWithExceptions())
                return true;

            if (type == ActivityType.SQL_ERROR &&
                FlightRecorderConfiguration.getInstance().getRecordRequestsWithSQLErrors())
                return true;

            if (context != null && getSessionsToRecord().contains(context.getRequest().getSession().getSessionID()))
                return true;

            if (context != null && context.getSession().getUser() != null &&
                getUsersToRecord().contains(context.getSession().getUser().getID()))
                return true;
        }

        return false;
    }

    /**
     * Startup.
     */
    public static void startup()
    {
        // Load the configuration to prevent cyclic method calls when it is initializing
        FlightRecorderConfiguration.getInstance();

        // Activate data gathering
        suspendDataGathering = false;
        DIFLogger.getLogger().info("Flight recorder service active.");
    }

    /**
     * Suspend.
     */
    public static void suspend()
    {
        suspendDataGathering = true;
        DIFLogger.getLogger().info("Flight recorder service suspended.");
    }
}
