/**
 * 2018, Digitalis Informatica. All rights reserved. Distribuicao e Gestao de Informatica, Lda. Estrada de Paco de Arcos
 * num.9 - Piso -1 2780-666 Paco de Arcos Telefone: (351) 21 4408990 Fax: (351) 21 4408999 http://www.digitalis.pt
 */

package pt.digitalis.dif.flightrecorder;

import pt.digitalis.dif.controller.interfaces.IDIFContext;
import pt.digitalis.dif.controller.objects.DIFRequest;
import pt.digitalis.dif.dem.interfaces.IStage;
import pt.digitalis.dif.dem.managers.IDEMManager;
import pt.digitalis.dif.exception.DIFException;
import pt.digitalis.dif.ioc.DIFIoCRegistry;
import pt.digitalis.dif.model.utils.AbstractBeanAttributes;
import pt.digitalis.dif.utils.ObjectFormatter;
import pt.digitalis.dif.utils.ObjectFormatter.Format;
import pt.digitalis.log.LogLevel;
import pt.digitalis.utils.common.CollectionUtils;
import pt.digitalis.utils.common.DateUtils;
import pt.digitalis.utils.common.StringUtils;

import java.text.ParseException;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.Map.Entry;
import java.util.TreeMap;

/**
 * The Class FlightRecorderLogEntry.
 *
 * @author Pedro Viegas <a href="mailto:pviegas@digitalis.pt">pviegas@digitalis.pt</a><br/>
 * @created Jul 31, 2019
 */
public class FlightRecorderLogEntry extends AbstractBeanAttributes
{

    /** The id counter. */
    private static long idCounter = 1;

    /** The activity type. */
    private ActivityType activityType = null;

    /** The class name. */
    private String className = null;

    /** The id. */
    private Long id = idCounter++;

    /** The log level. */
    private LogLevel logLevel = null;

    /** The entry object. */
    private ObjectFormatter object = null;

    /** The object preview. */
    private String objectPreview = null;

    /** The time stamp. */
    private Date timeStamp = null;

    /**
     * Instantiates a new flight recorder log entry.
     *
     * @param exception the exception
     */
    public FlightRecorderLogEntry(DIFException exception)
    {
        this.timeStamp = new Date();
        this.activityType = ActivityType.LOG;
        this.object = exception.toObjectFormatter(Format.JSON, null);

        List<String> list = new ArrayList<String>();
        list.add(exception.getClass().getCanonicalName());

        if (exception.getMessage() != null)
            list.add(exception.getMessage());

        if (exception.getCause() != null)
        {
            list.add("&nbsp;&nbsp;|&nbsp;Caused by: " + exception.getCause().getClass().getCanonicalName());

            if (exception.getCause().getMessage() != null)
                list.add("&nbsp;&nbsp;|&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;" + exception.getCause().getMessage());
        }

        this.objectPreview = CollectionUtils.listToSeparatedString(list, "<br/>");

        if (exception.getStackTrace() != null && exception.getStackTrace().length > 0)
            this.className =
                    exception.getStackTrace()[0].getClassName() + "[" + exception.getStackTrace()[0].getMethodName() +
                    ":" + exception.getStackTrace()[0].getLineNumber() + "]";
    }

    /**
     * Instantiates a new flight recorder log entry.
     *
     * @param context the context
     */
    public FlightRecorderLogEntry(IDIFContext context)
    {
        this.timeStamp = new Date();
        this.activityType = ActivityType.REQUEST;

        if (context != null && StringUtils.isNotBlank(context.getStage()))
        {
            IDEMManager demManager = DIFIoCRegistry.getRegistry().getImplementation(IDEMManager.class);
            IStage stage = demManager.getStage(context.getStage());

            if (stage != null)
                this.className = stage.getOriginalClassName();
        }

        this.object = context.toObjectFormatter(Format.JSON, null);

        List<String> list = new ArrayList<String>();
        list.add("StageID: " + context.getStage() + " (Action: " + context.getRequest().getRestAction().toString() +
                 (context.getRequest().isAjaxMode() ? " | AjaxCall" : "") +
                 (context.getRequest().isTemplateMode() ? " | TemplateMode" : "") +
                 (context.getRequest().isComponentMode() ? " | ComponentMode" : "") +
                 (context.getRequest().isPopupMode() ? " | PopupMode" : "") + ")");

        TreeMap<String, String> parameters = new TreeMap<String, String>();
        for (Entry<String, Object> entry : context.getRequest().getParameters().entrySet())
        {
            String value = StringUtils.toStringOrNull(entry.getValue());

            if (entry.getKey().contains(DIFRequest.PASSWORD_PARAMETER_ID) || entry.getKey().contains("password"))
                value = "*****";

            parameters.put(entry.getKey(), StringUtils.toStringOrNull(value));
        }

        list.add("Parameters: " + CollectionUtils.keyValueMapToString(parameters));

        this.objectPreview = CollectionUtils.listToSeparatedString(list, " | ");
    }

    /**
     * Instantiates a new flight recorder log entry.
     *
     * @param level     the level
     * @param message   the message
     * @param className the class name
     */
    public FlightRecorderLogEntry(LogLevel level, Object message, String className)
    {
        this.timeStamp = new Date();
        this.activityType = ActivityType.LOG;
        this.logLevel = level;
        this.object = new ObjectFormatter(Format.JSON, null).addItem("result", message);
        this.className = className;
        this.objectPreview = StringUtils.toStringOrNull(message);
    }

    /**
     * Instantiates a new flight recorder log entry.
     *
     * @param sqlExecutionLog the sql execution log
     */
    public FlightRecorderLogEntry(SQLExecutionLog sqlExecutionLog)
    {
        this.activityType = sqlExecutionLog.isSuccess() ? ActivityType.SQL : ActivityType.SQL_ERROR;
        this.timeStamp = new Date();
        this.object = new ObjectFormatter(Format.JSON, null).addItem("sqlExecution", sqlExecutionLog);

        if (sqlExecutionLog.isSuccess())
            this.objectPreview = sqlExecutionLog.getSql();
        else
        {
            List<String> list = new ArrayList<String>();

            if (sqlExecutionLog.getException() != null)
            {
                Exception exception = sqlExecutionLog.getException();

                list.add(exception.getClass().getCanonicalName());

                if (exception.getMessage() != null)
                    list.add(exception.getMessage());

                if (exception.getCause() != null)
                {
                    list.add("&nbsp;&nbsp;|&nbsp;Caused by: " + exception.getCause().getClass().getCanonicalName());

                    if (exception.getCause().getMessage() != null)
                        list.add("&nbsp;&nbsp;|&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;" + exception.getCause().getMessage());
                }
            }

            list.add(sqlExecutionLog.getSql());

            this.objectPreview = CollectionUtils.listToSeparatedString(list, "<br/>");
        }
    }

    /**
     * Instantiates a new flight recorder log entry.
     *
     * @param exception the exception
     */
    public FlightRecorderLogEntry(Throwable exception)
    {
        this.timeStamp = new Date();
        this.activityType = ActivityType.LOG;
        this.object = new ObjectFormatter(Format.JSON, null);

        this.object.addItemIfNotNull("Cause", exception.getCause());
        this.object.addItemIfNotNull("Message", exception.getMessage());
        this.object.addItemIfNotNull("StackTrace", exception.getStackTrace());

        List<String> list = new ArrayList<String>();
        list.add(exception.getClass().getCanonicalName());

        if (exception.getMessage() != null)
            list.add(exception.getMessage());

        if (exception.getCause() != null)
        {
            list.add("&nbsp;&nbsp;|&nbsp;Caused by: " + exception.getCause().getClass().getCanonicalName());

            if (exception.getCause().getMessage() != null)
                list.add("&nbsp;&nbsp;|&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;" + exception.getCause().getMessage());
        }

        this.objectPreview = CollectionUtils.listToSeparatedString(list, "<br/>");

        if (exception.getStackTrace() != null && exception.getStackTrace().length > 0)
            this.className =
                    exception.getStackTrace()[0].getClassName() + "[" + exception.getStackTrace()[0].getMethodName() +
                    ":" + exception.getStackTrace()[0].getLineNumber() + "]";
    }

    /**
     * Inspector for the 'activityType' attribute.
     *
     * @return the activityType value
     */
    public ActivityType getActivityType()
    {
        return activityType;
    }

    /**
     * Modifier for the 'activityType' attribute.
     *
     * @param activityType the new activityType value to set
     */
    public void setActivityType(ActivityType activityType)
    {
        this.activityType = activityType;
    }

    /**
     * Gets the attribute no graph navigation.
     *
     * @param attributeName the attribute name
     *
     * @return the attribute no graph navigation
     *
     * @see pt.digitalis.dif.model.utils.AbstractBeanAttributes#getAttributeNoGraphNavigation(java.lang.String)
     */
    @Override
    protected Object getAttributeNoGraphNavigation(String attributeName)
    {
        if ("id".equals(attributeName))
            return id;
        if ("activityType".equals(attributeName))
            return activityType;
        else if ("object".equals(attributeName))
            return object;
        else if ("content".equals(attributeName))
            return object.getJsonObject().toString();
        else if ("className".equals(attributeName))
            return className;
        else if ("logLevel".equals(attributeName))
            return logLevel;
        else if ("timeStamp".equals(attributeName))
            return timeStamp;
        else if ("objectPreview".equals(attributeName))
            return objectPreview;
        else
            return null;
    }

    /**
     * Inspector for the 'className' attribute.
     *
     * @return the className value
     */
    public String getClassName()
    {
        return className;
    }

    /**
     * Modifier for the 'className' attribute.
     *
     * @param className the new className value to set
     */
    public void setClassName(String className)
    {
        this.className = className;
    }

    /**
     * Inspector for the 'id' attribute.
     *
     * @return the id value
     */
    public Long getId()
    {
        return id;
    }

    /**
     * Modifier for the 'id' attribute.
     *
     * @param id the new id value to set
     */
    public void setId(Long id)
    {
        this.id = id;
    }

    /**
     * Inspector for the 'logLevel' attribute.
     *
     * @return the logLevel value
     */
    public LogLevel getLogLevel()
    {
        return logLevel;
    }

    /**
     * Modifier for the 'logLevel' attribute.
     *
     * @param logLevel the new logLevel value to set
     */
    public void setLogLevel(LogLevel logLevel)
    {
        this.logLevel = logLevel;
    }

    /**
     * Inspector for the 'object' attribute.
     *
     * @return the object value
     */
    public ObjectFormatter getObject()
    {
        return object;
    }

    /**
     * Modifier for the 'object' attribute.
     *
     * @param object the new object value to set
     */
    public void setObject(ObjectFormatter object)
    {
        this.object = object;
    }

    /**
     * @return the objectPreview
     */
    public String getObjectPreview()
    {
        return objectPreview;
    }

    /**
     * @param objectPreview the objectPreview to set
     */
    public void setObjectPreview(String objectPreview)
    {
        this.objectPreview = objectPreview;
    }

    /**
     * Inspector for the 'timeStamp' attribute.
     *
     * @return the timeStamp value
     */
    public Date getTimeStamp()
    {
        return timeStamp;
    }

    /**
     * Modifier for the 'timeStamp' attribute.
     *
     * @param timeStamp the new timeStamp value to set
     */
    public void setTimeStamp(Date timeStamp)
    {
        this.timeStamp = timeStamp;
    }

    /**
     * Sets the attribute.
     *
     * @param attributeName  the attribute name
     * @param attributeValue the attribute value
     *
     * @see pt.digitalis.utils.common.IBeanAttributes#setAttribute(java.lang.String, java.lang.Object)
     */
    @Override
    public void setAttribute(String attributeName, Object attributeValue)
    {
        if ("activityType".equals(attributeName))
            activityType = (ActivityType) attributeValue;
        else if ("object".equals(attributeName))
            object = (ObjectFormatter) attributeValue;
        else if ("className".equals(attributeName))
            className = (String) attributeValue;
        else if ("logLevel".equals(attributeName))
            logLevel = (LogLevel) attributeValue;
        else if ("timeStamp".equals(attributeName))
            timeStamp = (Date) attributeValue;
        else if ("id".equals(attributeName))
            id = (Long) attributeValue;
        else if ("objectPreview".equals(attributeName))
            objectPreview = (String) attributeValue;
    }

    /**
     * Sets the attribute from string.
     *
     * @param attributeName  the attribute name
     * @param attributeValue the attribute value
     *
     * @see pt.digitalis.utils.common.IBeanAttributes#setAttributeFromString(java.lang.String, java.lang.String)
     */
    @Override
    public void setAttributeFromString(String attributeName, String attributeValue)
    {
        if ("activityType".equals(attributeName))
            activityType = ActivityType.valueOf(attributeValue);
        else if ("className".equals(attributeName))
            className = attributeValue;
        else if ("logLevel".equals(attributeName))
            logLevel = LogLevel.valueOf(attributeValue);
        else if ("logLevel".equals(attributeName))
            id = Long.parseLong(attributeValue);
        else if ("objectPreview".equals(attributeName))
            objectPreview = attributeValue;
        else if ("timeStamp".equals(attributeName))
        {
            try
            {
                timeStamp = DateUtils.stringToDate(attributeValue);
            }
            catch (ParseException e)
            {
                e.printStackTrace();
            }
        }
    }
}
